/*
 *	fhist - file history and comparison tools
 *	Copyright (C) 2000, 2001 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions for reading input
 */

#include <ac/string.h>

#include <fcheck.h>
#include <input/private.h>
#include <mem.h>


#ifdef input_name
#undef input_name
#endif

const char *
input_name(fp)
	input_ty	*fp;
{
	return fp->vptr->name(fp);
}


#ifdef input_length
#undef input_length
#endif

long
input_length(fp)
	input_ty	*fp;
{
	return fp->vptr->length(fp);
}


#ifdef input_ftell
#undef input_ftell
#endif

long
input_ftell(fp)
	input_ty	*fp;
{
	return fp->vptr->ftell(fp) - fp->pushback_len;
}


#ifdef input_read
#undef input_read
#endif

long
input_read(fp, data, len)
	input_ty	*fp;
	void		*data;
	long		len;
{
	if (len <= 0)
		return 0;
	if (fp->pushback_len > 0)
	{
		fp->pushback_len--;
		*(char *)data = fp->pushback_buf[fp->pushback_len];
		return 1;
	}
	return fp->vptr->read(fp, data, len);
}


#ifdef input_getc
#undef input_getc
#endif

int
input_getc(fp)
	input_ty	*fp;
{
	if (fp->pushback_len > 0)
	{
		fp->pushback_len--;
		return fp->pushback_buf[fp->pushback_len];
	}
	return fp->vptr->get(fp);
}


#ifdef input_ungetc
#undef input_ungetc
#endif

void
input_ungetc(fp, c)
	input_ty	*fp;
	int		c;
{
	if (c < 0)
		return;
	if (fp->pushback_len >= fp->pushback_max)
	{
		fp->pushback_max = 16 + 2 * fp->pushback_max;
		fp->pushback_buf =
			mem_change_size(fp->pushback_buf, fp->pushback_max);
	}
	fp->pushback_buf[fp->pushback_len++] = c;
}


void
input_delete(fp)
	input_ty	*fp;
{
	if (fp->vptr->destruct)
		fp->vptr->destruct(fp);
	if (fp->pushback_buf)
		mem_free(fp->pushback_buf);
	fp->pushback_buf = 0;
	fp->pushback_len = 0;
	fp->pushback_max = 0;
	fp->vptr = 0; /* paranoia */
	mem_free(fp);
}


#include <cmalloc.h>

/*
 * Routine to read in the next line from a file, no matter how long it is.
 * This returns a pointer to the string, and also indirectly its length.
 * The string is normally returned from a static string, which is reused
 * for each call.  But if keep is non-zero, then the returned string
 * belongs to the caller and must later be freed.  The returned line
 * is ended with a newline and null character, but the returned length
 * does not count the null character.  Returns 0 on an error, with the
 * error stored in the fp structure.
 */

char *
input_readline(fp, retlen, keep, is_bin_file)
	input_ty	*fp;		/* file to read line from */
	long		*retlen;	/* ret len of line if non-null */
	int		keep;
	int		*is_bin_file;
{
	char		*dp;		/* destination pointer */
	size_t		totallen;	/* total length of data */
	static char	*linebuffer;	/* common line buffer */
	static size_t	linelength;	/* total length of line */

	if (linelength == 0)
	{
		/* allocate line buffer */
		linelength = 16;
		linebuffer = r_alloc_and_check(linelength + 1);
	}
	totallen = 0;
	for (;;)
	{
		int c = input_getc(fp);
		if (c < 0)
		{
			if (totallen == 0)
				return 0;
			warning_last_line_unterminated(input_name(fp));
			c = '\n';
		}
		if (c == 0)
		{
			*is_bin_file = 1;
			c = 0x80;
		}
		if (totallen >= linelength)
		{
			/*
			 * doubling gives O(1) behaviour
			 * over the life of the prog
			 */
			linelength = linelength * 2 + 16;
			linebuffer = r_realloc_and_check(linebuffer, linelength + 1);
		}
		linebuffer[totallen++] = c;
		if (c == '\n')
			break;
	}
	linebuffer[totallen] = 0;

	if (retlen)
		*retlen = totallen;
	if (!keep)
		dp = linebuffer;
	else
	{
		dp = cm_alloc_and_check(totallen + 1);
		memcpy(dp, linebuffer, (size_t)totallen);
	}
	return dp;
}


void
input_skip_lines(fp, numlines)
	input_ty	*fp;		/* file to read line from */
	int		numlines;
{
	while (numlines > 0)
	{
		int c = input_getc(fp);
		if (c < 0)
			break;
		if (c == '\n')
			--numlines;
	}
}
