/* object.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* functions for dealing with Objects  */  
/*
 * $Log: object.c,v $
 * Revision 1.4  2000/12/17 00:57:42  moz
 * examples, filled open splines, highlight_objects
 *
 * Revision 1.3  2000/12/06 20:56:03  moz
 * GPL stuff.
 *
 * Revision 1.2  2000/08/21 01:30:31  moz
 * Re-calculate cache on spline enlarge to avoid low-res artefacts.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:30  moz
 * CVS Import
 *
 * Revision 1.43  2000/03/13 01:50:12  moz
 * Fixed add_object0() bug.
 *
 * Revision 1.42  2000/03/12 23:33:49  moz
 * Fixed stupid typo reversing text lines !
 *
 * Revision 1.41  2000/03/12 04:13:40  moz
 * Derry to node ellipses correctly.
 *
 * Revision 1.40  2000/03/11 19:10:15  moz
 * Fix calc to recalc_text_box().
 *
 * Revision 1.39  2000/03/11 18:19:50  moz
 * Adjust text box position to stay static on scale.
 *
 * Revision 1.38  2000/03/06 02:28:42  moz
 * Compile fixes.
 *
 * Revision 1.37  2000/02/23 02:20:41  moz
 * Was forgetting to NULL polyline.pic !.
 *
 * Revision 1.36  2000/02/21 03:53:13  moz
 * Backed out debugging message.
 *
 * Revision 1.35  2000/02/18 21:14:15  moz
 * polyline.pic fix.
 * Some compile fixes.
 *
 * Revision 1.34  2000/02/18 04:06:27  moz
 * Was accessing freed mem in remove_derry() ...
 *
 * Revision 1.33  2000/02/18 03:04:57  moz
 * kill_object() should free derries.
 *
 * Revision 1.32  2000/02/17 22:25:56  moz
 * Backed out debugging change.
 *
 * Revision 1.31  2000/02/04 02:17:38  moz
 * kill_object() should free arcellipse arrows.
 *
 * Revision 1.30  2000/02/04 01:54:59  moz
 * switch_icons() needs view
 *
 * Revision 1.29  2000/01/29 23:51:37  moz
 * Small fix.
 *
 * Revision 1.28  2000/01/29 04:13:16  moz
 * Removed more wrong v_error().
 *
 * Revision 1.27  2000/01/29 03:58:37  moz
 * Default for get_nearest_point isn't an error.
 *
 * Revision 1.26  2000/01/26 18:50:23  moz
 * kill_object should free arrows.
 *
 * Revision 1.25  2000/01/26 18:19:53  moz
 * Changes to allow FIG_DEBUG_LIST_ALLOC.
 *
 * Revision 1.24  1999/11/29 20:34:40  moz
 * BBSIZE() renamed to avoid IRIX param.h namespace clash.
 *
 * Revision 1.23  1999/11/15 02:06:34  moz
 * Use rounded trig.
 * Pick smallest object under cursor.
 *
 * Revision 1.22  1999/08/17 00:35:24  moz
 * Fix malloc_node.
 *
 * Revision 1.21  1999/08/08 20:55:05  moz
 * From clean up of structs.
 *
 * Revision 1.20  1999/07/29 20:44:35  moz
 * Make sure rottext resize rubber box is redrawn by apply_scale...().
 *
 * Revision 1.19  1999/07/04 15:19:49  moz
 * Only draw scaled compounds when finished.
 *
 * Revision 1.18  1999/07/04 03:13:59  moz
 * Changes to derrying to an ellipse.
 *
 * Revision 1.17  1999/05/31 23:05:18  moz
 * Spline curves contribute interpolated/angular points to attachment.
 *
 * Revision 1.16  1999/05/22 02:52:56  moz
 * Correctly recalculate a scaled compound's bounding box.
 *
 * Revision 1.15  1999/05/19 17:08:37  moz
 * 1.0 Checkin.
 *
 * Revision 1.14  1999/05/05 00:38:05  moz
 * Derry to arcellipses.
 *
 * Revision 1.13  1999/05/04 20:09:57  moz
 * Remove derries on rotate and scale.
 *
 * Revision 1.12  1999/05/04 20:01:45  moz
 * Move targetee.
 *
 * Revision 1.11  1999/05/04 15:50:17  moz
 * Merged in duplicate_object() and kill_object().
 *
 * Revision 1.10  1999/05/03 05:54:49  moz
 * Scale text in compound.
 *
 * Revision 1.9  1999/04/27 16:58:26  moz
 * Set text angle to zero if close (3 degrees).
 *
 * Revision 1.8  1999/04/27 16:55:59  moz
 * Flip object.
 *
 * Revision 1.7  1999/04/27 04:11:33  moz
 * Use calc_text_outer_box.
 * -Wall appeasement.
 *
 * Revision 1.6  1999/04/26 19:59:55  moz
 * move_object function added.
 *
 * Revision 1.5  1999/04/25 19:39:37  moz
 * Rotate text's bounding box.
 *
 * Revision 1.4  1999/04/22 22:16:46  moz
 * Don't register undos for scaling subobjects.
 *
 * Revision 1.3  1999/04/22 20:55:18  moz
 * Rescale compounds within compounds correctly.
 *
 * Revision 1.2  1999/04/22 16:16:44  moz
 * Imperfect fix for scaling arc circles.
 *
 * Revision 1.1  1999/03/30 00:05:37  moz
 * Initial revision
 *
 */    

#include <math.h> 
#include <string.h> 
#include "include/figurine.h"
#include "include/extern.h"

Tree malloc_node(void);

/* this calculates the correct position for a scaled object with a tied corner  */ 
/* given a topleft corner, returns a new one  */  
void 
corner_magic(Object *ob, long *x, long *y, double rx, double ry)
{
	switch (state.tied_corner) 
		{
		case TOPLEFT:
			break;

		case TOPRIGHT:
				*x = *x + (ob->bbox.x2-ob->bbox.x1) - R(ob->bbox.x2-ob->bbox.x1,abs(rx)); 
			break;
		
		case BOTTOMLEFT:
				*y = *y + (ob->bbox.y2-ob->bbox.y1) - R(ob->bbox.y2-ob->bbox.y1,abs(ry)); 
			break;

		case BOTTOMRIGHT:
				*x = *x + (ob->bbox.x2-ob->bbox.x1) - R(ob->bbox.x2-ob->bbox.x1,abs(rx)); 
				*y = *y + (ob->bbox.y2-ob->bbox.y1) - R(ob->bbox.y2-ob->bbox.y1,abs(ry)); 
			break; 
		
		default:
			v_error(state.tied_corner);
			break;
		};

}
 
Tree 
malloc_node(void)
{
	return (Tree)malloc(sizeof(Node));
}

/* add an object to object tree and list  */  
Tree 
add_object(Tree t, List *l, Object *ob)
{
	if ((t=add_object0(t,ob))!=NULL)
		*l = add_to_list(*l,(ulong)ob,ob->type,(void *)ob);
	return t;
}

/* insert an object in object tree  */  
Tree 
add_object0(Tree t, Object *ob)
{
	Tree tp,tp2; 

	if (t==NULL) /* empty tree  */ 
		{
		tp = malloc_node();
		if (tp==NULL)
			return NULL;

		tp->left = NULL;
		tp->right = NULL;
		tp->ob = ob;
		tp->bbox = ob->bbox;
		tp->up = NULL; 

		return tp;
		}
	else if (t->ob!=NULL) /* node */ 
		{
		tp = malloc_node();
		if (tp==NULL)
			return NULL;
		tp->left = NULL;
		tp->right = NULL;
		tp->ob = ob;
		tp->bbox = ob->bbox;
		 
		tp2 = malloc_node();
		if (tp2==NULL)
			return NULL;
		tp2->left = t; /* this is used in algorithm ! */ 
		tp2->right = tp;
		tp2->up = NULL; 
		tp->up = tp2;
		t->up = tp2;
		tp2->ob = NULL;
		tp2->bbox = merge_boxes(tp2->left->bbox, tp2->right->bbox);
		
		return tp2;
		}
	else
		{
		/* not at a node yet  */  
		/* if no recursing */  
		if (!is_contained_v(t->left->bbox,ob->bbox) && !is_contained_v(t->right->bbox,ob->bbox))
			{ 
			tp = malloc_node();
			if (tp==NULL)
				return NULL;
			tp->left = NULL;
			tp->right = NULL;
			tp->ob = ob;
			tp->bbox = ob->bbox;

			tp2 = malloc_node();
			if (tp2==NULL)
				return NULL;
				 
			if ( (FIG_BBSIZE(merge_boxes(t->left->bbox,ob->bbox))-FIG_BBSIZE(t->left->bbox)) <
			   (FIG_BBSIZE(merge_boxes(t->right->bbox,ob->bbox))-FIG_BBSIZE(t->right->bbox)) )
				{
				tp2->left = t->left;
				tp2->right = tp;
				tp->up = tp2;
				t->left->up = tp2;
				tp2->ob = NULL;
				tp2->bbox = merge_boxes(tp2->left->bbox,tp2->right->bbox);
				t->left = tp2; 
				tp2->up = t; 
				t->bbox = merge_boxes(t->left->bbox,t->right->bbox); 
				} 
			else 
				{
				tp2->left = t->right;
				tp2->right = tp;
				tp->up = tp2;
				t->right->up = tp2;
				tp2->ob = NULL;
				tp2->bbox = merge_boxes(tp2->left->bbox,tp2->right->bbox);
				t->right = tp2; 
				tp2->up = t; 
				t->bbox = merge_boxes(t->left->bbox,t->right->bbox); 
				}; 
			 
			 	return t;
			}
		else if (!is_contained_v(t->left->bbox,t->right->bbox) &&
					!is_contained_v(t->right->bbox,t->left->bbox))
			{
			/* now ob is contained with in left or right, and they are disjointish, so recurse down */  
			if (is_contained_v(t->left->bbox,ob->bbox))
				{
				t->left = add_object0(t->left,ob);
				t->bbox = merge_boxes(t->left->bbox,t->right->bbox);
				t->left->up = t; 
				return t;
				}
			else
				{
				t->right = add_object0(t->right,ob);
				t->bbox = merge_boxes(t->left->bbox,t->right->bbox);
				t->right->up = t; 
				return t;
				};
			}
		else if (is_contained_v(t->left->bbox,t->right->bbox))
			{
			/* now right is inside left, so join right with ob */ 
			if (is_contained_v(t->right->bbox,ob->bbox))
				{
				t->right = add_object0(t->right,ob);
				t->bbox = merge_boxes(t->left->bbox, t->right->bbox);
				t->right->up = t; 
				return t;
				}
			else
				{ 
				tp = malloc_node();
				if (tp==NULL)
					return NULL;
				tp->left = NULL;
				tp->right = NULL;
				tp->ob = ob;
				tp->bbox = ob->bbox;
				tp2 = malloc_node();
				if (tp2==NULL)
					return NULL;
				tp2->left = tp;
				tp2->right = t->right;
				tp->up = tp2;
				t->right->up = tp2;
				tp2->ob = NULL;
				tp2->bbox = merge_boxes(tp2->left->bbox,tp2->right->bbox);
				t->right = tp2;
				tp2->up = t; 
				t->bbox = merge_boxes(t->left->bbox,t->right->bbox); 
				return t; 
				};
			}
		else
			{
			/* now left is inside right, so join left with ob */ 
			if (is_contained_v(t->left->bbox,ob->bbox))
				{
				t->left = add_object0(t->left,ob);
				t->bbox = merge_boxes(t->left->bbox, t->right->bbox);
				t->left->up = t; 
				return t;
				}
			else
				{ 
				tp = malloc_node();
				if (tp==NULL)
					return NULL;
				tp->left = NULL;
				tp->right = NULL;
				tp->ob = ob;
				tp->bbox = ob->bbox;
				tp2 = malloc_node();
				if (tp2==NULL)
					return NULL;
				tp2->left = tp;
				tp2->right = t->left;
				tp->up = tp2; 
				t->left->up = tp2;
				tp2->ob = NULL;
				tp2->bbox = merge_boxes(tp2->left->bbox,tp2->right->bbox);
				t->left = tp2;
				tp2->up = t; 
				t->bbox = merge_boxes(t->left->bbox,t->right->bbox); 
				return t; 
				};

			};
		};
		
} 

/* return leaf of tree t containing ob  */  
Node *
get_object_node(Tree t, Object *ob)
{
	Node *t2;

	if (t==NULL)
		return NULL;
	
	if (t->left==NULL && t->right==NULL)
		{
		if (ob==t->ob)
			return t;
		else
			return NULL;
		};

	if ((t2=get_object_node(t->left,ob))!=NULL)
		return t2; 
		 
	if ((t2=get_object_node(t->right,ob))!=NULL)
		return t2; 
		 
	return NULL;
}

/* find an object by its ticket value  */  
/* this is used for derrying support  */  
/* node ellipses must be special-cased */ 
Object *
get_object_by_ticket(Tree t, ulong ticket)
{
	Object *ob; 

	if (t==NULL)
		return NULL;
	
	if (t->left==NULL && t->right==NULL)
		{
		if (ticket==t->ob->ticket)
			return t->ob;
		else if (t->ob->type==TEXT && t->ob->ob.text.node && ticket==t->ob->ob.text.ellipse->ticket)
			return t->ob->ob.text.ellipse;
		else
			return NULL;
		};

	if ((ob=get_object_by_ticket(t->left,ticket))!=NULL)
		return ob; 
		 
	if ((ob=get_object_by_ticket(t->right,ticket))!=NULL)
		return ob; 
		 
	return NULL;
}

/* take object off tree and list  */  
Tree 
trash_object(Tree t, List *lo, Object *ob)
{
	t = delete_object(t,ob);
	*lo = delete_from_list(*lo, (ulong)ob);
	return t; 
}

/* take object off tree  */  
Tree 
delete_object(Tree t, Object *ob)
{
	Node *a;
	 
	if (t==NULL)
		return NULL;
	
	if (t->left==NULL && t->right==NULL)
		{
		if (t->ob==ob)
			{
			free(t);
			return NULL;
			}
		else
			return t;
		};
	
	if (t->left->ob!=NULL)
		{
		/* left of tree  */
		if (t->left->ob==ob)
			{
			a = t->right;
			t->right->up = NULL; 
			free(t->left); 
			free(t);
			return a;
			};
		};
	
	if (t->right->ob!=NULL)
		{
		/* right of tree  */ 
		if (t->right->ob==ob)
			{
			a = t->left;
			t->left->up = NULL; 
			free(t->right); 
			free(t);
			return a;
			};
		};
	
	if (t->left->ob==NULL)
		{ 
		t->left = delete_object(t->left,ob);
		t->left->up = t;
		};
	
	if (t->right->ob==NULL)
		{ 
		t->right = delete_object(t->right,ob);
		t->right->up = t;
		};

	t->bbox = merge_boxes(t->left->bbox,t->right->bbox); 
	return t;
}

/* for small bbox changes, just propogate changes upwards  */  
void 
resize_bbox(Tree t, Object *ob)
{
	if (t==NULL)
		return;
	
	if (t->left==NULL && t->right==NULL)
		{
		if (t->ob->ticket==ob->ticket)
			t->bbox = ob->bbox;
			 
		return;
		}
	else
		{
		resize_bbox(t->left,ob);
		
		resize_bbox(t->right,ob);

		t->bbox = merge_boxes(t->left->bbox,t->right->bbox);
		return; 
		};
}

/* for bbox changes, we need to reinsert to maintain a good tree  */  
Tree 
change_bbox(Tree t, Object *ob)
{
	t = delete_object(t,ob);
	t = add_object0(t,ob);
	return t; 
}

Node *
object_at_point(View *v, Tree t, long x, long y)
{
	List match=NULL;
	List l; 
	Node *best=NULL;
	ulong bsize=ULONG_MAX; 

	match = object_at_point0(match, v, t, x, y);
	l = match; 
	
	while (match!=NULL)
		{
		if (FIG_BBSIZE(((Node *)match->data)->ob->bbox) < bsize)
			{
			bsize = FIG_BBSIZE(((Node *)match->data)->ob->bbox);
			best = (Node *)match->data;
			};
		match=match->next;
		};

	delete_list(l); 
	return best;


}

/* return object at document coords x,y  */  
List
object_at_point0(List match, View *v, Tree t, long x, long y)
{
	if (t==NULL)
		return match;
	
	if (t->left==NULL && t->right==NULL)
		{ 
		if (is_in_bbox(x,y,t->bbox.x1-P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->bbox.y1-P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->bbox.x2+P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->bbox.y2+P2D(SELECT_PIXEL_LOOSENESS,v)))
			{
			match = add_to_list(match,0,0,(void *)t);
			return match;
			} 
		else 
			return match;
		}
	else
		{
		if (is_in_bbox(x,y,t->left->bbox.x1-P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->left->bbox.y1-P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->left->bbox.x2+P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->left->bbox.y2+P2D(SELECT_PIXEL_LOOSENESS,v)))
			match = object_at_point0(match, v, t->left,x,y);

		if (is_in_bbox(x,y,t->right->bbox.x1-P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->right->bbox.y1-P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->right->bbox.x2+P2D(SELECT_PIXEL_LOOSENESS,v),
								 t->right->bbox.y2+P2D(SELECT_PIXEL_LOOSENESS,v)))
			match = object_at_point0(match, v, t->right,x,y);

			return match;

		};
}

/* return object at pixel coords x,y  */  
Node *
object_at_point_p(View *v, Tree t, long x, long y)
{
	return object_at_point(v, t, XP2D(x,v), YP2D(y,v));
}

/* area coherence test: should we change object selection ?  */  
Boolean 
still_same_object(Node *n, long x, long y)
{
	if (n==NULL)
		return FALSE;
	
	if (!is_in_bbox(x,y,n->bbox.x1,n->bbox.y1,n->bbox.x2,n->bbox.y2))
		return FALSE;
	
	if (n->up==NULL) /* only one object in tree !  */ 
		return TRUE; 

	if (n->up->right!=n && n->up->right!=NULL && is_in_bbox(x,y,n->up->right->bbox.x1,n->up->right->bbox.y1,n->up->right->bbox.x2,n->up->right->bbox.y2))
		return FALSE;

	if (n->up->left!=n && n->up->left!=NULL && is_in_bbox(x,y,n->up->left->bbox.x1,n->up->left->bbox.y1,n->up->left->bbox.x2,n->up->left->bbox.y2))
		return FALSE;

	return TRUE; 
}

Boolean 
still_same_object_p(View *v, Node *n, long x, long y)
{
	return still_same_object(n,XP2D(x,v),YP2D(y,v));
}

/* return depth-sorted list of objects that intersect bb  */  
List 
intersecting_objects(Tree t, VRegion bb, List l)
{
	List l2;

	if (t==NULL)
		return l;
	else if (t->left==NULL && t->right==NULL) /* leaf node */ 
		{
 		if (intersects_v(bb,t->bbox))  
			{ 
			/* why like this ? for FIG_DEBUG_LIST_ALLOC */  
			l2 = add_to_list(l,ULONG_MAX-t->ob->depth,0,(void *)t->ob); 
			return l2; 
			} 
 		else 
 			return l; 
		}
	else
		{
		if (t->left!=NULL && intersects_v(bb,t->left->bbox))
			l = intersecting_objects(t->left,bb,l);
			
		if (t->right!=NULL && intersects_v(bb,t->right->bbox))
			return intersecting_objects(t->right,bb,l);

		return l;
		};

}
 
/* return depth-sorted list of objects that intersect cx,cy  */  
List 
intersecting_objects_point(Tree t, long cx, long cy, List l)
{
	List l2;

	if (t==NULL)
		return l;
	else if (t->left==NULL && t->right==NULL) /* leaf node */ 
		{
 		if (is_in_object(cx,cy,t->ob))
			{ 
			/* why like this ? for FIG_DEBUG_LIST_ALLOC */  
			l2 = add_to_list(l,0,0,(void *)t->ob); 
			return l2; 
			} 
 		else 
 			return l; 
		}
	else
		{
		if (t->left!=NULL && is_in_bbox(cx,cy,t->left->bbox.x1,t->left->bbox.y1,t->left->bbox.x2,t->left->bbox.y2))
			l = intersecting_objects_point(t->left,cx,cy,l);
			
		if (t->right!=NULL && is_in_bbox(cx,cy,t->right->bbox.x1,t->right->bbox.y1,t->right->bbox.x2,t->right->bbox.y2))
			return intersecting_objects_point(t->right,cx,cy,l);

		return l;
		};
}
 
/* free all mem held by ob */ 
void 
kill_object(Object *ob)
{
	List l,l2;

	switch (ob->type)
		{
		case TEXT:
			if (ob->ob.text.node && ob->ob.text.ellipse!=NULL)
				free(ob->ob.text.ellipse);

			l = ob->ob.text.lines;

			while (l!=NULL)
				{
				l2 = TEXTLINE(l)->sections;
				while (l2!=NULL)
					{
					free(TEXTSEC(l2)); 
					l2 = l2->next;
					};
				delete_list(TEXTLINE(l)->sections);
				free(TEXTLINE(l)); 
				l = l->next; 
				};
			delete_list(ob->ob.text.lines);
			l=ob->derries;
			while (l!=NULL)
				{
				free(DERRY(l));
				l=l->next;
				};
			delete_list(ob->derries);
			free(ob);
			break; 
		
		case POLYLINE:
		case POLYGON:
			l = ob->ob.polyline.points;
			 
			while (l!=NULL)
				{
				free (POINT(l)); 
				l = l->next;
				};
				 
			if (ob->ob.polyline.pic) 
				free (ob->ob.polyline.pic);

                        if (ob->farrow)
				free(ob->farrow);
                        if (ob->barrow)
				free(ob->barrow);

			delete_list(ob->ob.polyline.points);
			l=ob->derries;
			while (l!=NULL)
				{
				free(DERRY(l));
				l=l->next;
				};
			delete_list(ob->derries);
			free(ob);
			break;

		case SPLINE:
		case ARC: 
		 	l = ob->ob.spline.points;
			while (l!=NULL)
				{
				free(SPOINT(l)); 
				l = l->next;
				};
				 
			delete_list(ob->ob.spline.points);
	
		 	l = ob->ob.spline.cache;
			while (l!=NULL)
				{
				free(POINT(l)); 
				l = l->next;
				};
				 
                        if (ob->farrow)
				free(ob->farrow);
                        if (ob->barrow)
				free(ob->barrow);

			l=ob->derries;
			while (l!=NULL)
				{
				free(DERRY(l));
				l=l->next;
				};
			delete_list(ob->derries);
			delete_list(ob->ob.spline.cache);
			free(ob); 
			break;

		case ARCELLIPSE:
                        if (ob->farrow)
				free(ob->farrow);
                        if (ob->barrow)
				free(ob->barrow);
			l=ob->derries;
			while (l!=NULL)
				{
				free(DERRY(l));
				l=l->next;
				};
			delete_list(ob->derries);
			free(ob); 
			break;

		case ELLIPSE:
		case ROUNDBOX:
			l=ob->derries;
			while (l!=NULL)
				{
				free(DERRY(l));
				l=l->next;
				};
			delete_list(ob->derries);
			free(ob); 
			break;
			
		case COMPOUND:
			 
			l = ob->ob.compound.obs;
			while (l!=NULL)
				{
				kill_object(OB(l));
				l = l->next;
				};
			delete_list(ob->ob.compound.obs);
			l=ob->derries;
			while (l!=NULL)
				{
				free(DERRY(l));
				l=l->next;
				};
			delete_list(ob->derries);
			free(ob);
			break;
		
		default:
			v_error(ob->type); 
			break;
		};
}

/* duplicate an object. the ticket value is altered  */  
Object *
duplicate_object(Object *ob)
{
	List l,l2;
	Object *nob;
	TextLine *tl; 
	TextSection *ts; 
	VPoint *p;
	SPoint *sp;
	uint c=0; 
	uint c2=0; 

	nob = (Object *)malloc(sizeof(Object));

	nob->type = ob->type;
	nob->ticket = ob_ticket++;
	nob->derries = NULL; 
	nob->depth = ob->depth;
	nob->colour = ob->colour;
	nob->fillcolour = ob->fillcolour;
	nob->ls = ob->ls;
	nob->lw = ob->lw;
	nob->es = ob->es;
	nob->js = ob->js;
	nob->fs = ob->fs;
	nob->dash = ob->dash;
	nob->gap = ob->gap;
	nob->bbox = ob->bbox;
	if (ob->farrow!=NULL)
		nob->farrow = make_arrow(*ob->farrow);
	else 
		nob->farrow = NULL;
	if (ob->barrow!=NULL)
		nob->barrow = make_arrow(*ob->barrow);
	else 
		nob->barrow = NULL;

	switch (ob->type)
		{
		case TEXT:
			if (ob->ob.text.node && ob->ob.text.ellipse!=NULL)
				nob->ob.text.ellipse = (void *)duplicate_object(ob->ob.text.ellipse);

			nob->ob.text.node = ob->ob.text.node; 
			nob->ob.text.bbox = ob->ob.text.bbox;
			nob->ob.text.angle = ob->ob.text.angle;
			nob->ob.text.lines = NULL; 
			l = ob->ob.text.lines;

			while (l!=NULL)
				{
				tl = (TextLine *)malloc(sizeof(TextLine));
				tl->y = TEXTLINE(l)->y;
				tl->h = TEXTLINE(l)->h;
				tl->just = TEXTLINE(l)->just; 
				tl->sections = NULL; 
				l2 = TEXTLINE(l)->sections;
				while (l2!=NULL)
					{
					ts = (TextSection *)malloc(sizeof(TextSection));

					ts->num = TEXTSEC(l2)->num;
					ts->size = TEXTSEC(l2)->size;
					strcpy(ts->text,TEXTSEC(l2)->text);
					ts->x = TEXTSEC(l2)->x;
					ts->y = TEXTSEC(l2)->y;
					ts->w = TEXTSEC(l2)->w;
					ts->h = TEXTSEC(l2)->h;
					tl->sections = add_to_list(tl->sections, c,0,(void *)ts); 
					c++; 

					l2 = l2->next;
					};
				nob->ob.text.lines = add_to_list(nob->ob.text.lines,c2,0, (void *)tl);
				c2++; 
				l = l->next; 
				};
			break; 
		
		case POLYLINE:
		case POLYGON:
			l = ob->ob.polyline.points;
			 
			if (ob->ob.polyline.pic)
				{
				nob->ob.polyline.pic = malloc(strlen(ob->ob.polyline.pic)+1);
				strcpy(nob->ob.polyline.pic, ob->ob.polyline.pic);
				}
			else
				nob->ob.polyline.pic = NULL;

			nob->ob.polyline.points = NULL;
			
			while (l!=NULL)
				{
				p = (VPoint *)malloc(sizeof(VPoint));

				p->x = POINT(l)->x;
				p->y = POINT(l)->y;
				p->derried = FALSE; 
				nob->ob.polyline.points = add_to_list(nob->ob.polyline.points,c,0,(void *)p);
				c++; 
				
				l = l->next;
				};
			break;

		case ROUNDBOX:
			nob->ob.roundbox.radius = ob->ob.roundbox.radius;
			break;

		case SPLINE:
		case ARC: 
			nob->ob.spline.closed = ob->ob.spline.closed;
			nob->ob.spline.cache = NULL;
			nob->ob.spline.points = NULL; 
			 
		 	l = ob->ob.spline.points;
			while (l!=NULL)
				{
				sp = (SPoint *)malloc(sizeof(SPoint));

				sp->x = SPOINT(l)->x;
				sp->y = SPOINT(l)->y;
				sp->derried = FALSE; 
				sp->s = SPOINT(l)->s;
				nob->ob.spline.points = add_to_list(nob->ob.spline.points,c,0,(void *)sp); 
				c++; 
				l = l->next;
				};
				break;

		case ELLIPSE:
			nob->ob.ellipse.centre.x = ob->ob.ellipse.centre.x;
			nob->ob.ellipse.centre.y = ob->ob.ellipse.centre.y;
			nob->ob.ellipse.xradius = ob->ob.ellipse.xradius;
			nob->ob.ellipse.yradius = ob->ob.ellipse.yradius;
			break; 
			
		case ARCELLIPSE:
			nob->ob.arcellipse.centre.x = ob->ob.arcellipse.centre.x;
			nob->ob.arcellipse.centre.y = ob->ob.arcellipse.centre.y;
			nob->ob.arcellipse.xradius = ob->ob.arcellipse.xradius;
			nob->ob.arcellipse.yradius = ob->ob.arcellipse.yradius;
			nob->ob.arcellipse.start = ob->ob.arcellipse.start;
			nob->ob.arcellipse.end = ob->ob.arcellipse.end;
			nob->ob.arcellipse.open = ob->ob.arcellipse.open;
			break; 
			
		case COMPOUND:
			nob->ob.compound.obs = NULL;
			 
			l = ob->ob.compound.obs;
			while (l!=NULL)
				{
				Object *tob; 
				tob = duplicate_object(OB(l)); 
				nob->ob.compound.obs = add_to_list(nob->ob.compound.obs,ULONG_MAX-OB(l)->depth,0,(void *)tob);
				l = l->next;
				};
			break;
			 
		default:
			v_error(ob->type);
			break;
		};
		 
	return nob; 
}
  
/* move an object by document offset dx,dy  */  
void
move_object(View *view, Object *ob, long dx, long dy)
{
	List l;
 
	if (view->gridon)
		{
		dx = snap(view->grid_x,ob->bbox.x1+dx) - ob->bbox.x1;
		dy = snap(view->grid_y,ob->bbox.y1+dy) - ob->bbox.y1;
		};

	ob->bbox.x1 += dx; 
	ob->bbox.y1 += dy; 
	ob->bbox.x2 += dx; 
	ob->bbox.y2 += dy; 
	
	if (ob->type==COMPOUND)
		compound_move(view,ob,ob->bbox.x1-dx,ob->bbox.y1-dy,ob->bbox.x1,ob->bbox.y1);
	else if (ob->type==TEXT)
		{
		ob->ob.text.bbox.x1 += dx; 
		ob->ob.text.bbox.y1 += dy; 
		ob->ob.text.bbox.x2 += dx; 
		ob->ob.text.bbox.y2 += dy; 
		};
	
	if (ob->type==POLYLINE || ob->type==POLYGON)
		{
		l=ob->ob.polyline.points;
		while (l!=NULL)
			{
			if (POINT(l)->derried)
				{
				/* derries to somewhere, so should stay where it is  */  
				POINT(l)->x -= dx; 
				POINT(l)->y -= dy; 
				recalc_polyline_bbox(ob, FALSE);
				view->doc->o = change_bbox(view->doc->o, ob);
				};
			l=l->next; 
			};
		}
	else if (ob->type==SPLINE || ob->type==ARC)
		{
		l=ob->ob.spline.points;
		while (l!=NULL)
			{
			if (SPOINT(l)->derried)
				{
				/* derries to somewhere, so should stay where it is  */  
				SPOINT(l)->x -= dx; 
				SPOINT(l)->y -= dy; 
				delete_cache_list(ob->ob.spline.cache);
				ob->ob.spline.cache=NULL;
				ob->ob.spline.cache = compute_spline_segment(ob->ob.spline.points,ob->ob.spline.closed);
				recalc_polyline_bbox(ob, TRUE);
				view->doc->o = change_bbox(view->doc->o, ob);
				};
			l=l->next; 
			};
		};

	view->doc->o = change_bbox(view->doc->o, ob); 
	view->selected_object = get_object_node(view->doc->o,ob);
	view->highlighted_object = view->selected_object;
	switch_icons(view); 
	 
	/* if derries on object, they need to be moved  */   
	if (ob->type==TEXT && ob->ob.text.node && ob->ob.text.ellipse->derries!=NULL)
		derry_move(view,ob->ob.text.ellipse,ob->ob.text.ellipse->bbox.x1-dx,ob->ob.text.ellipse->bbox.y1-dy);
	else if (ob->derries!=NULL)
		derry_move(view,ob,ob->bbox.x1-dx,ob->bbox.y1-dy);
}

/* flip object ob horizontally  */  
void
flip_object_x(View *view, Object *ob)
{
	flip_object_x0(view,ob,TRUE);
}

void
flip_object_x0(View *view, Object *ob,Boolean needs_undo)
{ 
	List l;
	long dx; 
	long tx; 
	double t; 

	if (needs_undo) 
		{ 
		register_undo(UNDO_OB_PROP, ob, view->doc);  
		store_redraw_object(ob); 
		}; 

	dx = ob->bbox.x2-ob->bbox.x1;

	switch(ob->type)
		{
		case POLYLINE:
		case POLYGON:
			l = ob->ob.polyline.points;
			while (l!=NULL)
				{
				POINT(l)->x = dx - POINT(l)->x; 
				l = l->next;
				};
			break;

		case SPLINE:
		case ARC:
			l = ob->ob.spline.points;
			while (l!=NULL)
				{
				SPOINT(l)->x = dx - SPOINT(l)->x; 
				l = l->next;
				};
			l = ob->ob.spline.cache;
			while (l!=NULL)
				{
				POINT(l)->x = dx - POINT(l)->x; 
				l = l->next;
				};
			break;

		case ARCELLIPSE:
			if (ob->ob.arcellipse.start>ob->ob.arcellipse.end)
				{
				t = ob->ob.arcellipse.start;
				ob->ob.arcellipse.start = ob->ob.arcellipse.end;
				ob->ob.arcellipse.end = t; 
				};
			ob->ob.arcellipse.start = ((180*64.0)/PI) * flip_angle_x((ob->ob.arcellipse.start*PI)/(64.0*180)); 
			ob->ob.arcellipse.end = ((180*64.0)/PI) * flip_angle_x((ob->ob.arcellipse.end*PI)/(64.0*180)); 
			t = ob->ob.arcellipse.start;
			ob->ob.arcellipse.start = ob->ob.arcellipse.end;
			ob->ob.arcellipse.end = t; 
			ob->ob.arcellipse.end += 360*64; 
			break; 
			 
		case COMPOUND:
			l = ob->ob.compound.obs;
			while (l!=NULL)
				{
				OB(l)->bbox.x1 -= ob->bbox.x1;
				tx = dx - OB(l)->bbox.x1;
				OB(l)->bbox.x2 -= ob->bbox.x1;
				OB(l)->bbox.x1 = dx - OB(l)->bbox.x2;
				OB(l)->bbox.x2 = tx;
				OB(l)->bbox.x1 += ob->bbox.x1; 
				OB(l)->bbox.x2 += ob->bbox.x1; 
				flip_object_x0(view,OB(l),FALSE); 
				l = l->next; 
				};
			break;
		
		default:
			/* not handled */ 
			break;
		};

	if (needs_undo) 
		send_stored_redraw_object(view, ob);  
}

/* flip object ob vertically  */  
void
flip_object_y(View *view, Object *ob)
{
	flip_object_y0(view,ob,TRUE);
}

void
flip_object_y0(View *view, Object *ob,Boolean needs_undo)
{ 
	List l;
	long dy; 
	long ty; 
	double t; 

	if (needs_undo) 
		{ 
		register_undo(UNDO_OB_PROP, ob, view->doc);  
		store_redraw_object(ob); 
		}; 

	dy = ob->bbox.y2-ob->bbox.y1;

	switch(ob->type)
		{
		case POLYLINE:
		case POLYGON:
			l = ob->ob.polyline.points;
			while (l!=NULL)
				{
				POINT(l)->y = dy - POINT(l)->y; 
				l = l->next;
				};
			break;

		case SPLINE:
		case ARC:
			l = ob->ob.spline.points;
			while (l!=NULL)
				{
				SPOINT(l)->y = dy - SPOINT(l)->y; 
				l = l->next;
				};
			l = ob->ob.spline.cache;
			while (l!=NULL)
				{
				POINT(l)->y = dy - POINT(l)->y; 
				l = l->next;
				};
			break;

		case ARCELLIPSE:
			ob->ob.arcellipse.start = ((180*64.0)/PI) * flip_angle_y((ob->ob.arcellipse.start*PI)/(64.0*180)); 
			ob->ob.arcellipse.end = ((180*64.0)/PI) * flip_angle_y((ob->ob.arcellipse.end*PI)/(64.0*180)); 
			t = ob->ob.arcellipse.start;
			ob->ob.arcellipse.start = ob->ob.arcellipse.end;
			ob->ob.arcellipse.end = t; 
			ob->ob.arcellipse.end += 360*64; 
			break; 
			 
		case COMPOUND:
			l = ob->ob.compound.obs;
			while (l!=NULL)
				{
				OB(l)->bbox.y1 -= ob->bbox.y1;
				ty = dy - OB(l)->bbox.y1;
				OB(l)->bbox.y2 -= ob->bbox.y1;
				OB(l)->bbox.y1 = dy - OB(l)->bbox.y2;
				OB(l)->bbox.y2 = ty;
				OB(l)->bbox.y1 += ob->bbox.y1; 
				OB(l)->bbox.y2 += ob->bbox.y1; 
				flip_object_y0(view,OB(l),FALSE); 
				l = l->next; 
				};
			break;
		
		default:
			/* not handled */ 
			break;	
		};

	if (needs_undo) 
		send_stored_redraw_object(view, ob);  
}

/* rotate an object by angle  */  
void 
apply_rotate_to_object(View *view, Object *ob, double angle)
{
	double cosa, sina;
	long cx,cy; 
	List l; 
	long x,y; 
	Object *s; 

	 
	register_undo(UNDO_OB_PROP, ob, view->doc);  
	 
	cosa = cosround(angle);
	sina = sinround(angle);
	cx = (ob->bbox.x2-ob->bbox.x1)/2;
	cy = (ob->bbox.y2-ob->bbox.y1)/2;

	SET_CLIP_WINDOW(handlegc, view->draw_window); 
	 
 	store_redraw_object(ob); 
	 
	switch(ob->type)
		{
		case TEXT:
			ob->ob.text.angle = ob->ob.text.angle+angle;
			/* if it's a small rotation, default to 0.0  */  
			if (abs(ob->ob.text.angle)<((3*PI)/180)) 
				ob->ob.text.angle=0.0;

			calc_text_outer_box_adjust(ob,0,0); 
			 
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);
			view->selected_object = get_object_node(view->doc->o, s);
			view->highlighted_object = view->selected_object;
			break;

		case POLYLINE:
		case POLYGON: 
			l = ob->ob.polyline.points;

			while (l!=NULL)
				{
				x = cx+(cosa*(POINT(l)->x-cx))-(sina*(POINT(l)->y-cy)); 
				y = cy+(sina*(POINT(l)->x-cx))+(cosa*(POINT(l)->y-cy)); 
				POINT(l)->x = x;
				POINT(l)->y = y;
				/* remove if derry  */  
				if (POINT(l)->derried)
					remove_derry(view,ob,POINT(l)); 
				l=l->next;
				};
				 
  			recalc_polyline_bbox(ob, FALSE);
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break;
		
		case SPLINE:
		case ARC: 
			l = ob->ob.spline.points;

			while (l!=NULL)
				{
				x = cx+(cosa*(SPOINT(l)->x-cx))-(sina*(SPOINT(l)->y-cy)); 
				y = cy+(sina*(SPOINT(l)->x-cx))+(cosa*(SPOINT(l)->y-cy)); 
				SPOINT(l)->x = x;
				SPOINT(l)->y = y;
				/* remove if derry  */  
				if (POINT(l)->derried)
					remove_derry(view,ob,POINT(l)); 
				l=l->next;
				};
				 
			if (!ob->ob.spline.cache)
				ob->ob.spline.cache = compute_spline_segment(ob->ob.spline.points, ob->ob.spline.closed);

			l = ob->ob.spline.cache;

			while (l!=NULL)
				{
				x = cx+(cosa*(POINT(l)->x-cx))-(sina*(POINT(l)->y-cy)); 
				y = cy+(sina*(POINT(l)->x-cx))+(cosa*(POINT(l)->y-cy)); 
				POINT(l)->x = x;
				POINT(l)->y = y;
				l=l->next;
				};
				 
  			recalc_polyline_bbox(ob, TRUE);
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break;
		
		default:
			/* object type not handled */ 
			break; 
		};	

  	send_stored_redraw_object(view, ob);  
}

/* scale a compound object  */  
/* must be treated differently to pass a centre of rotation  */  
void
apply_scale_to_compound(View *view, Object *ob, double x, double y, long x1, long y1)
{
	List l; 
	Object *s; 

	l = ob->ob.compound.obs;
			
	ob->bbox.x2 = ob->bbox.x1+1;
	ob->bbox.y2 = ob->bbox.y1+1;
	 
	while (l!=NULL)
		{
		if (OB(l)->type==TEXT)
			{
			OB(l)->ob.text.bbox.x1 = x1 + R(OB(l)->ob.text.bbox.x1-x1,x);
			OB(l)->ob.text.bbox.y1 = y1 + R(OB(l)->ob.text.bbox.y1-y1,y);
			OB(l)->ob.text.bbox.x2 = x1 + R(OB(l)->ob.text.bbox.x2-x1,x); 
			OB(l)->ob.text.bbox.y2 = y1 + R(OB(l)->ob.text.bbox.y2-y1,y); 
			}
		else
			{
			OB(l)->bbox.x1 = x1 + R(OB(l)->bbox.x1-x1,x);
			OB(l)->bbox.y1 = y1 + R(OB(l)->bbox.y1-y1,y);
			OB(l)->bbox.x2 = x1 + R(OB(l)->bbox.x2-x1,x); 
			OB(l)->bbox.y2 = y1 + R(OB(l)->bbox.y2-y1,y); 
			}; 
		if (OB(l)->type==COMPOUND)
			apply_scale_to_compound(view,OB(l),x,y,x1,y1); 
		else 
			apply_scale_to_object(view,OB(l),x,y,FALSE); 
		ob->bbox = merge_boxes(ob->bbox,OB(l)->bbox); 
		view->doc->o = trash_object(view->doc->o,&view->doc->lo,OB(l)); 
		l = l->next;
		}; 
		 
	s = view->selected_object->ob; 
  	view->doc->o = change_bbox(view->doc->o, ob);  
	view->selected_object = get_object_node(view->doc->o, s); 
	view->highlighted_object = view->selected_object; 
}

/* scale object ob by ratios x and y  */  
void 
apply_scale_to_object(View *view, Object *ob, double x, double y, Boolean part_of_compound)
{
	List l,l2; 
	Object *s; 
	long ox=0,oy=0;

	x = abs(x);
	y = abs(y);
	 
	if (part_of_compound)  
		{ 
		register_undo(UNDO_OB_PROP, ob, view->doc);
		/* ugh !  */  
		if (ob->type==TEXT)
			{
			VRegion bb;
			bb.x1 = ob->bbox.x1; bb.y1 = ob->bbox.y1;
			bb.x2 = ob->bbox.x1 + R(ob->bbox.x2-ob->bbox.x1,x); 
			bb.y2 = ob->bbox.y1 + R(ob->bbox.y2-ob->bbox.y1,y); 
			ob->bbox = merge_boxes(ob->bbox,bb); 
			}; 
		store_redraw_object(ob); 
		}; 
	 
	corner_magic(ob, &ob->bbox.x1, &ob->bbox.y1, x, y); 
	switch(ob->type)
		{
		case TEXT:
			if (ob->ob.text.angle!=0.0)
				{
				double sina,cosa;
				long cx,cy; 
				sina = sinround(ob->ob.text.angle);
				cosa = cosround(ob->ob.text.angle);
				cx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2 + ob->ob.text.bbox.x1;
				cy = (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2 + ob->ob.text.bbox.y1;
				ox = cx+(cosa*(ob->ob.text.bbox.x1-cx))-(sina*(ob->ob.text.bbox.y1-cy)); 
				oy = cy+(sina*(ob->ob.text.bbox.x1-cx))+(cosa*(ob->ob.text.bbox.y1-cy)); 
				}; 

			corner_magic(ob, &ob->ob.text.bbox.x1, &ob->ob.text.bbox.y1, x, y); 
			(x<y) ? (y=x) : (x=y);
			if (part_of_compound)
				{
				ob->ob.text.bbox.x2 = R(ob->ob.text.bbox.x2-ob->ob.text.bbox.x1,x);
				ob->ob.text.bbox.x2 += ob->ob.text.bbox.x1; 
				ob->ob.text.bbox.y2 = R(ob->ob.text.bbox.y2-ob->bbox.y1,y);
				ob->ob.text.bbox.y2 += ob->ob.text.bbox.y1;
				}; 
			l = ob->ob.text.lines;
			while (l!=NULL)
				{
				l2 = TEXTLINE(l)->sections;
				while (l2!=NULL)
					{
					TEXTSEC(l2)->size=y*TEXTSEC(l2)->size;
					l2=l2->next;
					};
				l=l->next; 
				};
			recalc_text_box(view,ob,FALSE,FALSE); 
			if (ob->ob.text.angle!=0.0)
				{
				double sina,cosa;
				long cx,cy; 
				long nx,ny; 
				sina = sinround(ob->ob.text.angle);
				cosa = cosround(ob->ob.text.angle);
				cx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2 + ob->ob.text.bbox.x1;
				cy = (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2 + ob->ob.text.bbox.y1;
				nx = cx+(cosa*(ob->ob.text.bbox.x1-cx))-(sina*(ob->ob.text.bbox.y1-cy)); 
				ny = cy+(sina*(ob->ob.text.bbox.x1-cx))+(cosa*(ob->ob.text.bbox.y1-cy)); 
				ob->bbox.x1 -= nx - ox; 
				ob->bbox.y1 -= ny - oy; 
				ob->bbox.x2 -= nx - ox; 
				ob->bbox.y2 -= ny - oy; 
				ob->ob.text.bbox.x1 -= nx - ox; 
				ob->ob.text.bbox.y1 -= ny - oy; 
				ob->ob.text.bbox.x2 -= nx - ox; 
				ob->ob.text.bbox.y2 -= ny - oy; 
				}; 
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break; 
			 
		case ROUNDBOX:
			if (part_of_compound)
				{
				ob->bbox.x2 = R(ob->bbox.x2-ob->bbox.x1,x);
				ob->bbox.x2 += ob->bbox.x1;
				ob->bbox.y2 = R(ob->bbox.y2-ob->bbox.y1,y);
				ob->bbox.y2 += ob->bbox.y1;
				}; 
			s = view->selected_object->ob;
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break; 

		case ELLIPSE:
			ob->ob.ellipse.centre.x = R(ob->ob.ellipse.centre.x,x);
			ob->ob.ellipse.centre.y = R(ob->ob.ellipse.centre.y,y);
			ob->ob.ellipse.xradius = R(ob->ob.ellipse.xradius,x);
			ob->ob.ellipse.yradius = R(ob->ob.ellipse.yradius,y);
			ob->bbox.x2 = ob->bbox.x1 + ob->ob.ellipse.centre.x + ob->ob.ellipse.xradius; 
			ob->bbox.y2 = ob->bbox.y1 + ob->ob.ellipse.centre.y + ob->ob.ellipse.yradius; 
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break;
	
		case ARCELLIPSE:
			/* must constrain to circular shape */  
			/* this works very badly in compounds, but there doesn't
				seem to be a better way to ensure arc is contained */  
			(x<y) ? (y=x) : (x=y);
			ob->ob.arcellipse.centre.x = R(ob->ob.arcellipse.centre.x,x);
			ob->ob.arcellipse.centre.y = R(ob->ob.arcellipse.centre.y,y);
			ob->ob.arcellipse.xradius = R(ob->ob.arcellipse.xradius,x);
			ob->ob.arcellipse.yradius = R(ob->ob.arcellipse.yradius,y);
			ob->bbox.x2 = ob->bbox.x1 + ob->ob.arcellipse.centre.x + ob->ob.arcellipse.xradius; 
			ob->bbox.y2 = ob->bbox.y1 + ob->ob.arcellipse.centre.y + ob->ob.arcellipse.yradius; 
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break;
	
		case POLYGON: 
		case POLYLINE:
			l = ob->ob.polyline.points;

			while (l!=NULL)
				{
				POINT(l)->x = R(POINT(l)->x,x);
				POINT(l)->y = R(POINT(l)->y,y);
				if (POINT(l)->derried)
					remove_derry(view,ob,POINT(l)); 
				l = l->next;
				};

			recalc_polyline_bbox(ob, FALSE);
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break;
		
		case SPLINE:
		case ARC: 
			l = ob->ob.spline.points; 

			while (l!=NULL)
				{
				SPOINT(l)->x = R(SPOINT(l)->x,x);
				SPOINT(l)->y = R(SPOINT(l)->y,y);
				if (POINT(l)->derried)
					remove_derry(view,ob,POINT(l)); 
				l = l->next;
				};
			
			delete_cache_list(ob->ob.spline.cache);
			ob->ob.spline.cache = compute_spline_segment(ob->ob.spline.points,ob->ob.spline.closed);
			recalc_polyline_bbox(ob, TRUE);
			s = view->selected_object->ob; 
  			view->doc->o = change_bbox(view->doc->o, ob);  
			view->selected_object = get_object_node(view->doc->o, s); 
			view->highlighted_object = view->selected_object; 
			break;
			 
		case COMPOUND:
			apply_scale_to_compound(view,ob,x,y,ob->bbox.x1,ob->bbox.y1); 
			break; 
		
		default:
			v_error(ob->type);
			break; 
		};

	if (part_of_compound) 
		send_stored_redraw_object(view, ob); 
}

/* request for a derry. Search through objects intersecting this point
	and pick out the contributed point that is closest. The caller then
	sets up the connection with the returned value *rob */
Boolean 
get_nearest_point(View *view, Tree t, long cx, long cy, long *x,long *y,Object **rob) 
{
	List l=NULL; 
	List l2; 
	List points; 
	VRegion bb; 
	double rcx, rcy; 
	double tx,ty; 
	double ax,ay; 
	double bx,by; 
	long ix, iy; 
	Object *ob; 
	double theta; 
	double theta2; 
	Boolean found=FALSE; 
	double nearest=600000000.0; /* far away  */ 
	 
	if (t==NULL)
		return FALSE;

	/* look thirty pixels in every direction  */  
	bb.x1 = cx - P2D(30,view);
	bb.y1 = cy - P2D(30,view);
	bb.x2 = cx + P2D(30,view);
	bb.y2 = cy + P2D(30,view);
	l = intersecting_objects(t,bb,l);
	l2=l;
	while (l!=NULL)
		{
		ob = OB(l);

		switch (ob->type)
			{
			case POLYLINE:
			case POLYGON:
				rcx = cx - ob->bbox.x1; 
				rcy = cy - ob->bbox.y1;

				points = ob->ob.polyline.points;

				while (points!=NULL)
					{
					if (sqrt((double)((POINT(points)->x-rcx)*(POINT(points)->x-rcx))+((POINT(points)->y-rcy)*(POINT(points)->y-rcy)))
						 < nearest)
						{
						nearest = sqrt((double)((POINT(points)->x-rcx)*(POINT(points)->x-rcx))+((POINT(points)->y-rcy)*(POINT(points)->y-rcy)));
						*x = ob->bbox.x1 + POINT(points)->x;
						*y = ob->bbox.y1 + POINT(points)->y;
						found = TRUE;
						*rob = ob; 
						};
					points = points->next;
					}; 
				break;

			case SPLINE:
			case ARC: 
				if (ob->ob.spline.closed)
					break;

				rcx = cx - ob->bbox.x1;
				rcy = cy - ob->bbox.y1;

				points = ob->ob.spline.points;

				while (points!=NULL)
					{
					/* only jump if curve is near point */  
					/* ideally, this should take account of when curve passes through,
						but has an approximated shape factor (three-point straight line) */  
					if (SPOINT(points)->s<SHAPE_ATTACH_THRESHOLD && 
					sqrt((double)((SPOINT(points)->x-rcx)*(SPOINT(points)->x-rcx))+((SPOINT(points)->y-rcy)*(SPOINT(points)->y-rcy)))
						 < nearest)
						{
						nearest = sqrt((double)((SPOINT(points)->x-rcx)*(SPOINT(points)->x-rcx))+((SPOINT(points)->y-rcy)*(SPOINT(points)->y-rcy)));
						*x = ob->bbox.x1 + SPOINT(points)->x;
						*y = ob->bbox.y1 + SPOINT(points)->y;
						found = TRUE;
						*rob = ob; 
						};
					points = points->next;
					}; 
				break;

			case TEXT:
				if (!ob->ob.text.node)
					break; 
				else
					{
					long xr,yr;
					Object *eob=ob->ob.text.ellipse;

					/* only attempt to derry to node's ellipse  */  
					xr = eob->ob.ellipse.xradius;
					yr = eob->ob.ellipse.yradius;
					
					theta = atan2((((double)yr)/xr)*(cy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0)),
								  (((double)yr)/xr)*(((cx-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0)))));
				

					tx = xr * cosround(theta);
					ty = yr * sinround(theta);
					ix = tx + ob->bbox.x1 + (ob->bbox.x2-ob->bbox.x1)/2;
					iy = ty + ob->bbox.y1 + (ob->bbox.y2-ob->bbox.y1)/2;
					if (sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy))) < nearest) 
						{ 
						/* kludge factor  */  
						/* this is to attempt to determine whether we are on the ellipse line or not already  */  
						/* it's far from perfect */  
						theta2 = atan2((((double)yr)/xr)*(iy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0)),
										  (((double)yr)/xr)*(((ix-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0)))));
						ax = (double)cx-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0);
						ay = (double)cy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0);
						bx = (double)xr;
						by = (double)yr;
						if (sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy)))*(50*abs(1-((ax*ax)/(bx*bx)+(ay*ay)/(by*by)))) > 500.0)
							{
							*x = ix;
							*y = iy;
							}
						else
							{
							*x = cx;
							*y = cy;
							}; 
						found = TRUE;
						*rob = eob; 
						nearest = sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy))); 
						};	 
					};
				break;

			case ARCELLIPSE:
				theta = atan2((((double)ob->ob.arcellipse.yradius)/ob->ob.ellipse.xradius)*(cy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0)),
								  (((double)ob->ob.arcellipse.yradius)/ob->ob.ellipse.xradius)*(((cx-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0)))));
				
				tx = ob->ob.arcellipse.xradius * cosround(theta);
				ty = ob->ob.arcellipse.yradius * sinround(theta);
				ix = tx + ob->bbox.x1 + (ob->bbox.x2-ob->bbox.x1)/2;
				iy = ty + ob->bbox.y1 + (ob->bbox.y2-ob->bbox.y1)/2;
				if (sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy))) < nearest) 
					{ 
					/* kludge factor  */  
					/* this is to attempt to determine whether we are on the ellipse line or not already  */  
					/* it's far from perfect */  
					theta2 = atan2((((double)ob->ob.arcellipse.yradius)/ob->ob.arcellipse.xradius)*(iy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0)),
									  (((double)ob->ob.arcellipse.yradius)/ob->ob.arcellipse.xradius)*(((ix-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0)))));
					ax = (double)cx-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0);
					ay = (double)cy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0);
					bx = (double)ob->ob.arcellipse.xradius;
					by = (double)ob->ob.arcellipse.yradius;
					if (sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy)))*(50*abs(1-((ax*ax)/(bx*bx)+(ay*ay)/(by*by)))) > 500.0)
						{
						*x = ix;
						*y = iy;
						}
					else
						{
						*x = cx;
						*y = cy;
						}; 
					*rob = ob; 
					found = TRUE;
					nearest = sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy))); 
					};
				break;
				 
			case ELLIPSE:
				theta = atan2((((double)ob->ob.ellipse.yradius)/ob->ob.ellipse.xradius)*(cy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0)),
								  (((double)ob->ob.ellipse.yradius)/ob->ob.ellipse.xradius)*(((cx-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0)))));
				
				tx = ob->ob.ellipse.xradius * cosround(theta);
				ty = ob->ob.ellipse.yradius * sinround(theta);
				ix = tx + ob->bbox.x1 + (ob->bbox.x2-ob->bbox.x1)/2;
				iy = ty + ob->bbox.y1 + (ob->bbox.y2-ob->bbox.y1)/2;
				if (sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy))) < nearest) 
					{ 
					/* kludge factor  */  
					/* this is to attempt to determine whether we are on the ellipse line or not already  */  
					/* it's far from perfect */  
					theta2 = atan2((((double)ob->ob.ellipse.yradius)/ob->ob.ellipse.xradius)*(iy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0)),
									  (((double)ob->ob.ellipse.yradius)/ob->ob.ellipse.xradius)*(((ix-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0)))));
					ax = (double)cx-(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2.0);
					ay = (double)cy-(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2.0);
					bx = (double)ob->ob.ellipse.xradius;
					by = (double)ob->ob.ellipse.yradius;
					if (sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy)))*(50*abs(1-((ax*ax)/(bx*bx)+(ay*ay)/(by*by)))) > 500.0)
						{
						*x = ix;
						*y = iy;
						}
					else
						{
						*x = cx;
						*y = cy;
						}; 
					*rob = ob; 
					found = TRUE;
					nearest = sqrt((double)((cx-ix)*(cx-ix))+((cy-iy)*(cy-iy))); 
					};
				break;
			
			default:
				/* not an error, just an uninteresting object */ 
				break;
			}; 
		 
		l = l->next;
		};
	delete_list(l2);
	return found;
}

/* Adds a derry to the host object */  
/* the point is flagged as derried in the source object  */  
void 
add_derry(Object *ob, Object *src, VPoint *point)
{
	Derry *d;

	d = (Derry *)malloc(sizeof(Derry));

	d->ticket = src->ticket;
	d->point = point;
	
	ob->derries = add_to_list(ob->derries,src->ticket,0,(void *)d);
}

/* to move all derryees  */  
void 
derry_move(View *v,Object *ob,long oldx,long oldy)
{
	long x,y; 
	long dx,dy;
	Object *nob=NULL; 
	List l; 

	dx = ob->bbox.x1-oldx;
	dy = ob->bbox.y1-oldy;
	 
	l=ob->derries;

	while (l!=NULL)
		{
		nob = get_object_by_ticket(v->doc->o,DERRY(l)->ticket); 
		if (nob!=NULL)
			{
			store_redraw_object(nob);
			x = nob->bbox.x1 + DERRY(l)->point->x;
			y = nob->bbox.y1 + DERRY(l)->point->y;
			
			x += dx;
			y += dy;
			DERRY(l)->point->x = x - nob->bbox.x1; 
			DERRY(l)->point->y = y - nob->bbox.y1; 
			if (nob->type==SPLINE || nob->type==ARC)
				{
				if (nob->type==ARC)
					{
					SPoint *p1,*p2,*p3;
					short a,b; 
					p1 = SPOINT(nob->ob.spline.points);
					p2 = SPOINT(nob->ob.spline.points->next);
					p3 = SPOINT(nob->ob.spline.points->next->next);
					bent_midpoint(p1->x,	p1->y,p3->x,p3->y, 0.261799388, &a, &b);
					p2->x = a;
					p2->y = b;
					}; 
				delete_cache_list(nob->ob.spline.cache);
				nob->ob.spline.cache = compute_spline_segment(nob->ob.spline.points,nob->ob.spline.closed);
				}; 
			recalc_polyline_bbox(nob,(nob->type==SPLINE || nob->type==ARC)); 
			v->doc->o = change_bbox(v->doc->o, nob);
			send_stored_redraw_object(v,nob);
			};
		 
		l = l->next;
		};
}

/* remove a no-longer-valid attachment  */  
void 
remove_derry(View *v,Object *ob, VPoint *p)
{
	List l,l2;
	Derry *d; 
	Object *nob=NULL; 

	l = v->doc->lo;

	while (l!=NULL)
		{
		if (OB(l)->derries!=NULL)
			{
			l2 = OB(l)->derries;
			while (l2!=NULL)
				{
				nob = get_object_by_ticket(v->doc->o,DERRY(l2)->ticket); 
				if (nob==ob && DERRY(l2)->point==p)
					{
					d = DERRY(l2);
					OB(l)->derries = delete_from_list_v(OB(l)->derries,l2->data);
					free(d);
					l2 = OB(l)->derries; 
					}
				else
					l2 = l2->next;
				};
			}; 
		l = l->next;
		};
	p->derried=FALSE; 
}
