/* save.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* Saving of fig file  */  
/*
 * $Log: save.c,v $
 * Revision 1.6  2001/01/31 15:38:06  movement
 * pre11 - fix netscape, add metapost support, credit foggy
 *
 * Revision 1.6  2001/01/29 20:57:28  moz
 * Make netscape work again, metapost support.
 *
 * Revision 1.5  2000/12/17 00:57:42  moz
 * examples, filled open splines, highlight_objects
 *
 * Revision 1.4  2000/12/08 22:33:34  moz
 * Clean up possibly unsafe tmp file handling.
 *
 * Revision 1.3  2000/12/06 20:56:04  moz
 * GPL stuff.
 *
 * Revision 1.2  2000/09/05 00:54:51  moz
 * Make system() calls secure.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:30  moz
 * CVS Import
 *
 * Revision 1.26  2000/03/12 04:14:20  moz
 * Save out node ellipse derries.
 *
 * Revision 1.25  2000/03/07 21:46:48  moz
 * Compile fixes.
 *
 * Revision 1.24  2000/03/04 17:16:01  moz
 * Fix for pre7 unfilled breakage.
 *
 * Revision 1.23  2000/02/27 15:20:21  moz
 * shades.fig now works as xfig.
 *
 * Revision 1.22  2000/02/27 14:34:29  moz
 * Fixed most fillcolour problems with tints and shades.
 *
 * Revision 1.21  2000/02/22 22:55:37  moz
 * Merged export.c
 * Fixed node text positioning bug.
 *
 * Revision 1.20  2000/02/18 21:19:05  moz
 * polyline.pic usage changed.
 *
 * Revision 1.19  2000/01/29 21:01:38  moz
 * Initialise strings.
 *
 * Revision 1.18  1999/11/15 02:04:54  moz
 * Use rounded trig.
 * Position multi-line rotated text correctly.
 * Don't create empty text objects.
 *
 * Revision 1.17  1999/08/08 20:55:10  moz
 * From clean up of structs.
 *
 * Revision 1.16  1999/06/17 19:09:37  moz
 * Save out attachment information.
 *
 * Revision 1.15  1999/05/24 17:29:24  moz
 * If an object is in a compound, use compound's depth.
 *
 * Revision 1.14  1999/05/22 23:39:27  moz
 *  Pedantic ANSI.
 *
 * Revision 1.13  1999/05/22 02:52:16  moz
 * Escape on next text sections as well.
 *
 * Revision 1.12  1999/05/22 02:49:34  moz
 * Escape out backslashes and the like.
 *
 * Revision 1.11  1999/05/19 17:08:16  moz
 * 1.0 Checkin.
 *
 * Revision 1.10  1999/05/04 14:40:27  moz
 * Switch arc arrows correctly.
 *
 * Revision 1.9  1999/04/29 01:34:55  moz
 * Save the correct x,y value for the text.
 *
 * Revision 1.8  1999/04/27 16:56:04  moz
 * Flip text angle before saving.
 *
 * Revision 1.7  1999/04/27 04:11:57  moz
 * -Wall appeasement
 *
 * Revision 1.6  1999/04/24 22:35:09  moz
 * Save out user-defined colours.
 *
 * Revision 1.5  1999/04/23 20:40:27  moz
 * Append similar text to output.
 *
 * Revision 1.4  1999/04/23 20:21:21  moz
 * 2 is the value for special LaTeX text.
 *
 * Revision 1.3  1999/04/21 18:03:51  moz
 * Fixes for fitting midpoint of arc.
 *
 * Revision 1.2  1999/04/12 19:05:47  moz
 * Various fixes for fig2dev.
 *
 * Revision 1.1  1999/03/30 00:05:58  moz
 * Initial revision
 *
 */    

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h> 
 
#include "include/figurine.h"
#include "include/extern.h"

/* see the file Doc/FORMAT3.2 for a description of the file format  */  
 
static FILE *fl=NULL; 
static Document *doc; 
static Object *incompoundob=NULL; 
/* these are used for aligning pic with top right on export */  
static long minx;
static long miny;

void 
linestyle_out(int ls)
{
	fprintf(fl,"%d ",ls); 
}

void 
colour_out(int col)
{
	fprintf(fl,"%d ",col-STARTOFCOLOURS);
}
 
void 
fillcolour_out(Object *ob)
{
 	if (ob->fs>0 && ob->fs<20 && (ob->fillcolour==FILLCWHITE || ob->fillcolour==FILLCBLACK))  
		fprintf(fl,"%d ",ob->colour-STARTOFCOLOURS); 
	else 
		fprintf(fl,"%d ",ob->fillcolour-STARTOFFILLCOLOURS);
}
 
void 
depth_out(ulong depth)
{
	/* must convert into FIG3.2 0-999 depth and account for compounds */  
	if (incompoundob==NULL)
		fprintf(fl,"%d ",(int)(max(ULONG_MAX-999,depth)-(ULONG_MAX-999)));
	else
		fprintf(fl,"%d ",(int)(max(ULONG_MAX-999,incompoundob->depth)-(ULONG_MAX-999)));
}
 
void 
fillstyle_out(Object *ob)
{
	if (ob->fs>0 && ob->fs<20)
		{
		if (ob->fillcolour==FILLCWHITE)
			{
			if (ob->colour==CBLACK)
				fprintf(fl,"%d ",ob->fs);
			else
				fprintf(fl,"%d ",(20-ob->fs)+20);
			} 
		else	
			fprintf(fl,"%d ",20-ob->fs); 
		} 
	else 
		{ 
		if (ob->fs==SOLID)
			fprintf(fl,"20 ");
		else if (ob->fs==20)
			fprintf(fl,"0 ");
		else if (ob->fs==-1)
			fprintf(fl,"-1 "); 
		else 
			fprintf(fl,"%d ",ob->fs+20);
		}; 
}

void 
joinstyle_out(int js)
{
	switch (js)
		{
		case JoinRound:
			fprintf(fl,"2 ");
			break;

		case JoinBevel:
			fprintf(fl,"1 ");
			break;

		case JoinMiter:
			fprintf(fl,"0 ");
			break;
		}; 
}

void 
capstyle_out(int es)
{
	switch (es)
		{
		case CapButt:
			fprintf(fl,"0 "); 
			break;

		case CapRound:
			fprintf(fl,"1 "); 
			break;

		case CapProjecting:
			fprintf(fl,"2 "); 
			break;
		};
}
 
void 
arrow_out(Arrow *arrow)
{
	if (arrow==NULL)
		return;
	
	fprintf(fl, "%d %d %f %f %f\n",arrow->type,arrow->filled,(double)arrow->lw,(arrow->w/80.0)*doc->ppi,(arrow->h/80.0)*doc->ppi);
}
 
void
derry_out(Object *ob)
{
	List l;
	Object *dob; 

	l = ob->derries;
	while (l!=NULL)
		{
		dob = get_object_by_ticket(doc->o,DERRY(l)->ticket);
		if (dob!=NULL)
			fprintf(fl,"###FIGURINE DERRY %p %ld %ld\n",dob,DERRY(l)->point->x+dob->bbox.x1,DERRY(l)->point->y+dob->bbox.y1); 
			 
		l = l->next; 
		};
}
  
void
derry_check(Object *ob)
{
	List l;

	if (ob->type==SPLINE || ob->type==ARC)
		l = ob->ob.spline.points;
	else
		l = ob->ob.polyline.points;

	while (l!=NULL)
		{
		if (POINT(l)->derried)
			fprintf(fl,"###FIGURINE DERRY_SOURCE %p %ld %ld\n",ob,POINT(l)->x+ob->bbox.x1,POINT(l)->y+ob->bbox.y1);

		l=l->next;
		};
	
}

void 
polyline_out(Object *ob)
{
	List l; 
	int c=0; 
	
	fprintf(fl,"2 1 ");
	linestyle_out((int)ob->ls); 
	fprintf(fl,"%d ",ob->lw);
	colour_out((int)ob->colour);
	fillcolour_out(ob);
	depth_out(ob->depth); 
	fprintf(fl,"-1 -1 "); 
	fprintf(fl,"8.000 ");
	joinstyle_out(ob->js);
	capstyle_out(ob->es);
	fprintf(fl,"0 ");
	fprintf(fl,"%d ",ob->farrow!=NULL);
	fprintf(fl,"%d ",ob->barrow!=NULL);
	 
	l = ob->ob.polyline.points;

	while (l!=NULL)
		{
		c++;
		l = l->next;
		};

	fprintf(fl,"%d\n",c); 
	arrow_out(ob->farrow);
	arrow_out(ob->barrow);
	
	l = ob->ob.polyline.points;

	fprintf(fl,"\t"); 
	while (l!=NULL)
		{
		fprintf(fl," %ld %ld", (POINT(l)->x+ob->bbox.x1)-minx, (POINT(l)->y+ob->bbox.y1)-miny);
		l = l->next;
		};
	
	fprintf(fl,"\n");
	 
	if (ob->derries!=NULL)
		derry_out(ob); 
		 
	derry_check(ob);

}

void 
spline_out(Object *ob)
{
	List l; 
	int c=0; 
	
	fprintf(fl,"3 ");
	if (ob->ob.spline.closed)
		fprintf(fl,"5 ");
	else
		fprintf(fl,"4 ");
	linestyle_out((int)ob->ls); 
	fprintf(fl,"%d ",ob->lw);
	colour_out((int)ob->colour);
	fillcolour_out(ob);
	depth_out(ob->depth); 
	fprintf(fl,"-1 "); 
	fillstyle_out(ob);
	fprintf(fl,"8.000 ");
	capstyle_out(ob->es);
	fprintf(fl,"%d ",ob->farrow!=NULL);
	fprintf(fl,"%d ",ob->barrow!=NULL);
	 
	l = ob->ob.polyline.points;

	while (l!=NULL)
		{
		c++;
		l = l->next;
		};

	fprintf(fl,"%d\n",c); 
	arrow_out(ob->farrow);
	arrow_out(ob->barrow);
	
	l = ob->ob.spline.points;

	fprintf(fl,"\t"); 
	while (l!=NULL)
		{
		fprintf(fl," %ld %ld", (SPOINT(l)->x+ob->bbox.x1)-minx, (SPOINT(l)->y+ob->bbox.y1)-miny);
		l = l->next;
		};
	
	fprintf(fl,"\n");
	
	l = ob->ob.spline.points;
	while (l!=NULL)
		{
		fprintf(fl," %f", SPOINT(l)->s);
		l = l->next;
		};
	
	fprintf(fl,"\n");
	 
	if (ob->derries!=NULL)
		derry_out(ob); 
		 
	derry_check(ob);

}

void 
polygon_out(Object *ob)
{
	List l; 
	int c=0; 
	int subtype; 
	
	fprintf(fl,"2 ");
	if (ob->ob.polyline.pic) 
		subtype = 5;
	else
		subtype = 3;
	fprintf(fl,"%d ",subtype);
	linestyle_out((int)ob->ls); 
	fprintf(fl,"%d ",ob->lw);
	colour_out((int)ob->colour);
	fillcolour_out(ob);
	depth_out(ob->depth); 
	fprintf(fl,"-1 "); 
	fillstyle_out(ob); 
	fprintf(fl,"8.000 ");
	joinstyle_out(ob->js);
	capstyle_out(ob->es);
	fprintf(fl,"0 0 0 ");
	 
	l = ob->ob.polyline.points;

	while (l!=NULL)
		{
		c++;
		l = l->next;
		};

	if (subtype==5) /* picture  */  
		{ 
		fprintf(fl,"%d\n",c);
		fprintf(fl,"0 %s\n",ob->ob.polyline.pic);
		}
	else
		fprintf(fl,"%d\n",c+1); 
	
	l = ob->ob.polyline.points;

	fprintf(fl,"\t"); 
	while (l!=NULL)
		{
		fprintf(fl," %ld %ld", (POINT(l)->x+ob->bbox.x1)-minx, (POINT(l)->y+ob->bbox.y1)-miny);
		l = l->next;
		};
	
	if (subtype==3)
		fprintf(fl," %ld %ld", (POINT(ob->ob.polyline.points)->x+ob->bbox.x1)-minx, 
				(POINT(ob->ob.polyline.points)->y+ob->bbox.y1)-miny);
		 
	fprintf(fl,"\n");
	 
	if (ob->derries!=NULL)
		derry_out(ob); 
		 
	derry_check(ob);

}

void 
roundbox_out(Object *ob)
{
	fprintf(fl,"2 4 ");
	linestyle_out((int)ob->ls); 
	fprintf(fl,"%d ",ob->lw);
	colour_out((int)ob->colour);
	fillcolour_out(ob);
	depth_out(ob->depth); 
	fprintf(fl,"-1 "); 
	fillstyle_out(ob); 
	fprintf(fl,"8.000 ");
	joinstyle_out(ob->js);
	capstyle_out(ob->es);
	fprintf(fl,"%d ",ob->ob.roundbox.radius); 
	fprintf(fl,"0 0 ");
	 
	fprintf(fl,"5\n"); 
	fprintf(fl,"%ld %ld %ld %ld %ld %ld %ld %ld %ld %ld\n",
			  ob->bbox.x1-minx, ob->bbox.y1-miny,
			  ob->bbox.x2-minx, ob->bbox.y1-miny,
			  ob->bbox.x2-minx, ob->bbox.y2-miny,
			  ob->bbox.x1-minx, ob->bbox.y2-miny,
			  ob->bbox.x1-minx, ob->bbox.y1-miny);
}

void 
ellipse_out(Object *ob)
{
	fprintf(fl,"1 1 ");
	linestyle_out((int)ob->ls); 
	fprintf(fl,"%d ",ob->lw);
	colour_out((int)ob->colour);
	fillcolour_out(ob);
	depth_out(ob->depth); 
	fprintf(fl,"-1 "); 
	fillstyle_out(ob); 
	fprintf(fl,"8.000 1 0.000 ");
	fprintf(fl,"%ld %ld %ld %ld ", (ob->ob.ellipse.centre.x+ob->bbox.x1)-minx, (ob->ob.ellipse.centre.y+ob->bbox.y1)-miny, 
			  ob->ob.ellipse.xradius, ob->ob.ellipse.yradius);
	fprintf(fl,"%ld %ld %ld %ld\n", 
			  (ob->ob.ellipse.centre.x+ob->bbox.x1+ob->ob.ellipse.xradius)-minx, 
			  (ob->ob.ellipse.centre.y+ob->bbox.y1+ob->ob.ellipse.yradius)-miny, 
			  (ob->ob.ellipse.centre.x+ob->bbox.x1+ob->ob.ellipse.xradius)-minx, 
			  (ob->ob.ellipse.centre.y+ob->bbox.y1+ob->ob.ellipse.yradius)-miny); 
			   
	if (ob->derries!=NULL)
		derry_out(ob); 
		 
}

void 
arcellipse_out(Object *ob)
{
	double start,mid,end; 
	long mx,my; 
 
	fprintf(fl,"5 ");
	if (ob->ob.arcellipse.open)
		fprintf(fl, "1 ");
	else
		fprintf(fl, "2 ");
	linestyle_out((int)ob->ls); 
	fprintf(fl,"%d ",ob->lw);
	colour_out((int)ob->colour);
	fillcolour_out(ob);
	depth_out(ob->depth); 
	fprintf(fl,"-1 "); 
	if (ob->ob.arcellipse.open)
		fprintf(fl,"-1 ");
	else
		fillstyle_out(ob); 
	fprintf(fl,"8.000 ");
	capstyle_out(ob->es);
	fprintf(fl,"0 ");
	fprintf(fl,"%d ",ob->farrow!=NULL);
	fprintf(fl,"%d ",ob->barrow!=NULL);
	fprintf(fl,"%f %f ",    (double)(ob->ob.arcellipse.centre.x+ob->bbox.x1)-minx, 
				(double)(ob->ob.arcellipse.centre.x+ob->bbox.y1)-miny);
	start = (ob->ob.arcellipse.start/64.0)*(PI/180.0); 
	end = (ob->ob.arcellipse.end/64.0)*(PI/180.0); 
	mid = ((end-start)/2.0)+start; 
	mx = (ob->bbox.x1 + (ob->bbox.x2-ob->bbox.x1)/2)-minx;
	my = (ob->bbox.y1 + (ob->bbox.y2-ob->bbox.y1)/2)-miny;
	/* we do min & max on start/end to ensure the correct direction of sweep in xfig  */  
	fprintf(fl,"%ld ", (long)(mx+ob->ob.arcellipse.xradius*cosround(max(start,end))));
	fprintf(fl,"%ld ", (long)(my-ob->ob.arcellipse.yradius*sinround(max(start,end))));
  	fprintf(fl,"%ld ", (long)(mx+ob->ob.arcellipse.xradius*cosround(mid))); 
  	fprintf(fl,"%ld ", (long)(my-ob->ob.arcellipse.yradius*sinround(mid)));  
	fprintf(fl,"%ld ", (long)(mx+ob->ob.arcellipse.xradius*cosround(min(start,end))));
	fprintf(fl,"%ld\n",(long)(my-ob->ob.arcellipse.yradius*sinround(min(start,end))));
	/* these are also switched  */  
	arrow_out(ob->barrow);
	arrow_out(ob->farrow);
	 
	if (ob->derries!=NULL)
		derry_out(ob); 
		 
}

void 
font_out(int num)
{
	if (num<=34)
		fprintf(fl,"%d ",num);
	else
		fprintf(fl,"%d ",num-34);
}

void 
text_out(Object *ob)
{
	List l,l2,l3; 
	VFont *vf; 
	unsigned char *c; 
	long x,y,h;	


	if (ob->ob.text.node)
		{
		Object *eob=ob->ob.text.ellipse;
		fprintf(fl,"1 1 ");
		linestyle_out((int)eob->ls); 
		fprintf(fl,"%d ",eob->lw);
		colour_out((int)eob->colour);
		fillcolour_out(eob);
		depth_out(eob->depth); 
		fprintf(fl,"-1 "); 
		fillstyle_out(eob); 
		fprintf(fl,"8.000 1 0.000 ");
		fprintf(fl,"%ld %ld %ld %ld ",  (ob->bbox.x1-minx)+(ob->bbox.x2-ob->bbox.x1)/2, 
						(ob->bbox.y1-miny)+(ob->bbox.y2-ob->bbox.y1)/2,
						(ob->bbox.x2-ob->bbox.x1)/2,(ob->bbox.y2-ob->bbox.y1)/2);
		fprintf(fl,"%ld %ld %ld %ld\n", 
				  ((ob->bbox.x1-minx)+(ob->bbox.x2-ob->bbox.x1)/2)+(ob->bbox.x2-ob->bbox.x1)/2, 
				  ((ob->bbox.y1-miny)+(ob->bbox.y2-ob->bbox.y1)/2)+(ob->bbox.y2-ob->bbox.y1)/2, 
				  ((ob->bbox.x1-minx)+(ob->bbox.x2-ob->bbox.x1)/2)+(ob->bbox.x2-ob->bbox.x1)/2, 
				  ((ob->bbox.y1-miny)+(ob->bbox.y2-ob->bbox.y1)/2)+(ob->bbox.y2-ob->bbox.y1)/2);
		if (eob->derries)
			derry_out(eob); 
		};
			   
	l = ob->ob.text.lines;

	while (l!=NULL)
		{
		l2 = TEXTLINE(l)->sections;

		while (l2!=NULL)
			{
			 
			/* check whether the text is empty (can happen still, but shouldn't) */   
			if (streq(TEXTSEC(l2)->text,""))
				{ 
				l2=l2->next; 
				continue; 
				}; 

			fprintf(fl,"4 0 ");
			colour_out((int)ob->colour);
			depth_out(ob->depth);
			fprintf(fl,"-1 ");
			font_out(TEXTSEC(l2)->num);
			fprintf(fl,"%lu ", (ulong)TEXTSEC(l2)->size);
			fprintf(fl,"%f ", flip_angle_y(ob->ob.text.angle));
			if (TEXTSEC(l2)->num>34)
				fprintf(fl,"2 ");
			else
				fprintf(fl,"4 ");
			fprintf(fl,"%lu ",(ulong)TEXTLINE(l)->h);
			fprintf(fl,"%lu ",(ulong)TEXTSEC(l2)->w);
			l3 = doc->views;
			while ((vf = get_font(TEXTSEC(l2)->num,ZPO(TEXTSEC(l2)->size,VIEW(l3))))==NULL && l!=NULL)
				l3 = l3->next;
				 
			if (vf==NULL)
				h = TEXTLINE(l)->y+TEXTLINE(l)->h;
			else
				h = TEXTLINE(l)->y+P2D(vf->x->max_bounds.ascent, VIEW(l3));
					 
			if (ob->ob.text.angle!=0.0)
				{
				double sina, cosa;
				long cx,cy; 
				 
				sina = sinround(ob->ob.text.angle);
				cosa = cosround(ob->ob.text.angle);

				cx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2; 
				cy = (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2; 
				 
				x = ob->bbox.x1 + (ob->ob.text.bbox.x1-ob->bbox.x1) + cx 
			                   	+ (cosa*(TEXTSEC(l2)->x-cx)) - (sina*(h-cy)); 

				y = ob->bbox.y1 + (ob->ob.text.bbox.y1-ob->bbox.y1) + cy 
				                + (sina*(TEXTSEC(l2)->x-cx)) + (cosa*(h-cy));

				}
			else
				{
				x = (ob->bbox.x1)+TEXTSEC(l2)->x;
				y = (ob->bbox.y1) + h;
				};
			
			if (ob->ob.text.node) 
				fprintf(fl,"%ld %ld ",  x+((ob->bbox.x2-ob->bbox.x1)/2)-(ob->ob.text.bbox.x1-ob->bbox.x1)-minx, 
							y+((ob->bbox.y2-ob->bbox.y1)/2)-(ob->ob.text.bbox.y1-ob->bbox.y1)-miny);
			else 
				fprintf(fl,"%ld %ld ", x-minx, y-miny);
			
			c = (unsigned char *)TEXTSEC(l2)->text;
			while (*c!='\0')
				{
				if (*c=='\\') /* escape a backslash  */ 
					fprintf(fl,"%s", "\\\\"); 
				else if (*c>127) /* 8-bit ASCII */  
					fprintf(fl,"\\%o", *c); 
				else
					fprintf(fl,"%c", *c);
				c++;
				};
 
			/* if next section is same font, append to this line  */
			while (l2->next!=NULL && TEXTSEC(l2->next)->num==TEXTSEC(l2)->num && TEXTSEC(l2->next)->size==TEXTSEC(l2)->size)
				{
				c = (unsigned char *)TEXTSEC(l2->next)->text;
				while (*c!='\0')
					{
					if (*c=='\\') /* escape a backslash  */ 
						fprintf(fl,"%s", "\\\\"); 
					else if (*c>127) /* 8-bit ASCII */  
						fprintf(fl,"\\%o", *c); 
					else
						fprintf(fl,"%c", *c);
					c++;
					};
				l2 = l2->next;
				};

			/* delimiter  */  
			fprintf(fl,"\\001\n");

			l2 = l2->next;
			};
		l = l->next;
		};
}

void 
compound_out(Object *ob)
{
	List l;

	fprintf(fl,"6 %ld %ld %ld %ld\n",ob->bbox.x1-minx,ob->bbox.y1-miny,ob->bbox.x2-minx,ob->bbox.y2-miny);
	 
	 
	l = ob->ob.compound.obs;
	incompoundob = ob; 
	 
	while (l!=NULL)
		{
		switch (OB(l)->type)
			{
			case POLYLINE:
				polyline_out(OB(l));
				break;

			case SPLINE:
			case ARC: 
				spline_out(OB(l));
				break;

			case POLYGON:
				polygon_out(OB(l));
				break;

			case ROUNDBOX:
				roundbox_out(OB(l));
				break; 
		
			case ELLIPSE:
				ellipse_out(OB(l));
				break; 
			
			case ARCELLIPSE:
				arcellipse_out(OB(l));
				break;
		
			case TEXT:
				text_out(OB(l));
				break;
			
			case COMPOUND:
				compound_out(OB(l));
				break;
			
			default:
				v_error(OB(l)->type);
				break;
			};
		l = l->next;
		}; 
	
	incompoundob=NULL; 
	 
	/* delimiter  */  
	fprintf(fl,"-6\n"); 

}

void 
save(Document *d)
{
 	char file[FIGURINE_PATH_MAX]="";  
	strncpy(file,d->pathname,FIGURINE_PATH_MAX/2);
	strncat(file,d->filename,FIGURINE_PATH_MAX/2);

	save_name(d,file); 
}

void 
save_name(Document *d,char *file)
{
	List l; 
	char s[FIGURINE_PATH_MAX]=""; 


	fl = fopen(file,"w");

	if (fl==NULL)
		{
	   strcpy(s,"File ");
	   strncat(s,file,FIGURINE_PATH_MAX-40);
	   strcat(s," could not be opened for writing.");
	   stk_open_info(s);
		return;
		};
		 
	doc = d; 
	 
	/* don't shift on save */  
	minx=0;
	miny=0;

	fprintf(fl,"#FIG 3.2\n");
	if (doc->landscape)
		fprintf(fl,"Landscape\n");
	else
		fprintf(fl,"Portrait\n");
	if (doc->centred)
		fprintf(fl,"Center\n");
	else
		fprintf(fl,"Flush Left\n");
	fprintf(fl,"Inches\n");
	/* *sigh* two spaces to handle fig2dev 3.2.3 bug */  
	fprintf(fl,"%s  \n",doc->canvas_name);
	fprintf(fl,"100.00\n");
	if (doc->mpage)
		fprintf(fl,"Multiple\n");
	else
		fprintf(fl,"Single\n");
	fprintf(fl,"%d\n",doc->transcolour);
	fprintf(fl,"%ld 2\n",(long)doc->ppi);

	/* output user-defined colours  */  
	l = doc->cols;
	while (l!=NULL)
		{
		fprintf(fl,"0 %ld %s\n",UCOLOUR(l)->tag,UCOLOUR(l)->name);
		l=l->next;
		};
		
	l = doc->lo;
	 
	while (l!=NULL)
		{
		switch (l->type)
			{
			case POLYLINE:
				polyline_out(OB(l));
				break;

			case SPLINE:
			case ARC: 
				spline_out(OB(l));
				break;

			case POLYGON:
				polygon_out(OB(l));
				break;

			case ROUNDBOX:
				roundbox_out(OB(l));
				break; 
		
			case ELLIPSE:
				ellipse_out(OB(l));
				break; 
			
			case ARCELLIPSE:
				arcellipse_out(OB(l));
				break;
		
			case TEXT:
				text_out(OB(l));
				break;
			
			case COMPOUND:
				compound_out(OB(l));
				break;
				
			default:
				v_error((int)l->type);
				break;
			};
		l = l->next;
		};
	 
	  
	fclose(fl);  
}

static FILE *get_tmpfile(char **name)
{
	int fd;

	do {
		*name = tempnam(NULL, "foo");
		fd = open(*name, O_CREAT | O_EXCL | O_TRUNC | O_RDWR, 0600);
	} while (fd == -1);
	
	return fdopen(fd, "w");
}

int fig2dev_version(void);

/* we need to do this to work round command format breakage in 3.2.3 */  
int fig2dev_version()
{
	char *tfile;
	char *tfile2;
	char s[FIGURINE_PATH_MAX]=""; 
	char t[FIGURINE_PATH_MAX]=""; 
	FILE *fl; 
	FILE *fl2; 
	int ret; 

	fl = get_tmpfile(&tfile);

	if (fl==NULL)
		{
		strcpy(s,"Temporary file ");
		strncat(s,tfile,FIGURINE_PATH_MAX-40);
		strcat(s," could not be opened for writing.");
		stk_open_info(s);
		free(tfile);
		return 322;
		};

	fl2 = get_tmpfile(&tfile2);
	 
	if (fl2==NULL)
		{
		strcpy(s,"Temporary file ");
		strncat(s,tfile2,FIGURINE_PATH_MAX-40);
		strcat(s," could not be opened for writing.");
		stk_open_info(s);
		fclose(fl);
		free(tfile);
		free(tfile2);
		return 322;
		};

	fclose(fl2);

	/* simple test file */
	fprintf(fl,"#FIG 3.2\nLandscape\nFlush Left\nInches\nLetter  \n100.00\nSingle\n-2\n1200 2\n");
	fprintf(fl,"4 0 -1 0 0 0 12 0.0000 4 180 945 8100 5775 Living Room\001\n");
	fclose(fl);

	strcpy(t,"sh -c \"fig2dev -Leps ");
	strcat(t,tfile);
	strcat(t," ");
	strcat(t,tfile2);
	strcat(t," >/dev/null 2>/dev/null\" "); 
	 
	/* this should fail on < 3.2.3 */ 
	ret = system(t);
	remove(tfile);
	remove(tfile2);

	free(tfile);
	free(tfile2);

	if (ret)
		return 322;
	else
		return 323;
}
 
char *
export_name(Document *d,char *filename, ExportType format)
{
	List l; 
	FILE *fl2;
	int fig2dev=322; 
	char *file; 
	char *tfile;
	char s[FIGURINE_PATH_MAX]=""; 
	char t[FIGURINE_PATH_MAX]=""; 
	char cmd[FIGURINE_PATH_MAX]=""; 

	if (!filename)
		{
		fl2 = get_tmpfile(&file);
		if (fl2==NULL)
			{
			strcpy(s,"Temporary file ");
			strncat(s,tfile,FIGURINE_PATH_MAX-40);
			strcat(s," could not be opened for writing.");
			stk_open_info(s);
			free(file); 
			return NULL;
			};
		fclose(fl2);
		}
	else
		file = filename;

	/* temp fig files  */  
	fl = get_tmpfile(&tfile); 

	if (fl==NULL)
		{
		strcpy(s,"Temporary file ");
		strncat(s,tfile,FIGURINE_PATH_MAX-40);
		strcat(s," could not be opened for writing.");
		stk_open_info(s);
		free(tfile); 
		return file;
		};
		 
	if (system("sh -c \"fig2dev -V\" >/dev/null 2>&1 ")!=0)
		{
		strcpy(s,"fig2dev is not available on your path.");
		strcat(s," Cannot export file.");
		stk_open_info(s);
		free(tfile); 
		return file;
		};
		
	/* determine fig2dev version */  
	fig2dev = fig2dev_version(); 
	 
	doc = d; 
	 
	fprintf(fl,"#FIG 3.2\n");
	if (doc->landscape)
		fprintf(fl,"Landscape\n");
	else
		fprintf(fl,"Portrait\n");
	fprintf(fl,"Flush Left\n");
	fprintf(fl,"Inches\n");
	/* two spaces for fig2dev 3.2.3 bug */  
	fprintf(fl,"%s  \n",doc->canvas_name);
	fprintf(fl,"100.00\n");
	fprintf(fl,"Single\n");
	fprintf(fl,"%d\n",doc->transcolour);
	fprintf(fl,"%ld 2\n",(long)doc->ppi);

	/* calculate amount to subtract to move whole pic to topleft  */  
	if (doc->o!=NULL)
		{
 		minx = doc->o->bbox.x1; 
 		miny = doc->o->bbox.y1; 
		/* if PS, we seem to need to move 2 inches to right  */  
		if (format==EXPORT_PS)
			minx-=doc->ppi*2;
		};
	
	/* export user-defined colours  */  
	l = doc->cols;
	while (l!=NULL)
	   {
	   fprintf(fl,"0 %ld %s\n",UCOLOUR(l)->tag,UCOLOUR(l)->name);
	   l=l->next;
	   };

	l = doc->lo;
	
	while (l!=NULL)
		{
		switch (l->type)
			{
			case POLYLINE:
				polyline_out(OB(l));
				break;

			case SPLINE:
			case ARC: 
				spline_out(OB(l));
				break;

			case POLYGON:
				polygon_out(OB(l));
				break;

			case ROUNDBOX:
				roundbox_out(OB(l));
				break; 
		
			case ELLIPSE:
				ellipse_out(OB(l));
				break; 
			
			case ARCELLIPSE:
				arcellipse_out(OB(l));
				break;
		
			case TEXT:
				text_out(OB(l));
				break;
			
			case COMPOUND:
				compound_out(OB(l));
				break;
			
			default:
				v_error((int)l->type);
				break;
			};
		l = l->next;
		};
	 
	fclose(fl);
	
	/* run fig2dev on tfile  */  
	
	strcpy(cmd,"fig2dev ");

	switch (format)
		{
		case EXPORT_LATEX_BOX:
			strcat(cmd,"-Lbox "); 
			break;
			
		case EXPORT_LATEX_PICTURE:
			strcat(cmd,"-Llatex "); 
			break;

		case EXPORT_EPIC:
			strcat(cmd,"-Lepic "); 
			break;

		case EXPORT_EEPIC:
			strcat(cmd,"-Leepic "); 
			break;

		case EXPORT_EEPICEMU:
			strcat(cmd,"-Leepicemu "); 
			break;

		case EXPORT_PICTEX:
			strcat(cmd,"-Lpictex "); 
			break;
			
		case EXPORT_IBMGL:
			strcat(cmd,"-Libmgl "); 
			if (!doc->landscape)
				strcat(cmd,"-P ");
			break;

		case EXPORT_EPS:
			if (fig2dev>322) 
				strcat(cmd,"-Leps -e ");
			else
				strcat(cmd,"-Lps -e ");
			sprintf(s,"-z %s ",doc->canvas_name); 
			strcat(cmd,s); 
			if (doc->landscape)
				strcat(cmd,"-l xxx ");
			else
				strcat(cmd,"-p xxx ");
			sprintf(s,"-n '%s' ", strip_dirname(file));
			strcat(cmd,s); 
			break;

		case EXPORT_PS:
			strcat(cmd,"-Lps "); 
			if (doc->centred)
				strcat(cmd,"-c ");
			if (doc->landscape)
				strcat(cmd,"-l xxx ");
			else
				strcat(cmd,"-p xxx ");
			/*  in 3.2.2 and below this is how we show ps instead of eps */  
			if (fig2dev <= 322)
				strcat(cmd,"-P "); 
			sprintf(s,"-z %s ",doc->canvas_name); 
			strcat(cmd,s); 
			sprintf(s,"-n '%s' ", strip_dirname(file));
			strcat(cmd,s); 
			break;
			 
		case EXPORT_PSLATEX:
			strcat(cmd,"-Lpstex "); 
			if (doc->centred)
				strcat(cmd,"-c ");
			if (doc->landscape)
				strcat(cmd,"-l xxx ");
			else
				strcat(cmd,"-p xxx ");
			sprintf(s,"-z %s ",doc->canvas_name); 
			strcat(cmd,s); 
			sprintf(s,"-n '%s' ", strip_dirname(file));
			strcat(cmd,s); 
			 
			/* latex file now */  
			strcpy(s,"fig2dev -Lpstex_t -p '");
			strcat(s,strip_dirname(file));
			strcat(s,"' '");
			strcat(s,tfile);
			strcat(s,"' '");
			strcat(s,file);
			strcat(s,"_t'");
			if (system(s)!=0)
				{
				strcpy(t,"fig2dev command ");
				strcat(t, s);
				strcat(t," failed. Cannot export file.");
				stk_open_info(t);
				remove(tfile);
				return file;
				};
			break;
			 
		case EXPORT_TEXTYL:
			strcat(cmd,"-Ltextyl "); 
			break;
			 
		case EXPORT_TPIC:
			strcat(cmd,"-Ltpic "); 
			break;
			 
		case EXPORT_PIC:
			strcat(cmd,"-Lpic "); 
			break;
			 
		case EXPORT_METAFONT:
			strcat(cmd,"-Lmf "); 
			break;
			 
		case EXPORT_METAPOST:
			strcat(cmd,"-Lmp ");
			break;

		case EXPORT_MULTIMETAPOST:
			strcat(cmd,"-Lmmp ");
			break;

		case EXPORT_AUTOCAD:
			strcat(cmd,"-Lacad "); 
			break;
			 
		case EXPORT_PCX:
			strcat(cmd,"-Lpcx "); 
			break;
			 
		case EXPORT_PNG:
			strcat(cmd,"-Lpng "); 
			break;
			 
		case EXPORT_GIF:
			strcat(cmd,"-Lgif "); 
			break;
			 
		case EXPORT_JPEG:
			strcat(cmd,"-Ljpeg "); 
			break;
			 
		case EXPORT_TIFF:
			strcat(cmd,"-Ltiff "); 
			break;
			 
		case EXPORT_PPM:
			strcat(cmd,"-Lppm "); 
			break;
			 
		case EXPORT_XBM:
			strcat(cmd,"-Lxbm "); 
			break;
			 
		case EXPORT_XPM:
			strcat(cmd,"-Lxpm "); 
			break;
		};

	sprintf(s,"-m %f ",doc->magnify/100.0); 
	strcat(cmd,s);
	strcat(cmd,"'"); 
	strcat(cmd,tfile);
	strcat(cmd,"' '");
	strcat(cmd,file);
	strcat(cmd,"'"); 
	 
	/* cmdline has been constructed, so run it */   
	if (system(cmd)!=0)
		{
		strcpy(s,"fig2dev command ");
		strcat(s, cmd);
		strcat(s," failed. Cannot export file.");
		stk_open_info(s);
		};

	/* get rid of tfile  */  
	remove(tfile);
	free(tfile);
	
	return(file);
}
