/* spline.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* drawing of spline */  
/*
 * $Log: spline.c,v $
 * Revision 1.3  2000/12/17 00:57:42  moz
 * examples, filled open splines, highlight_objects
 *
 * Revision 1.2  2000/12/06 20:56:04  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:30  moz
 * CVS Import
 *
 * Revision 1.15  2000/03/15 00:16:15  moz
 * Deal with disable_motion - zoom out point edit not fixed.
 *
 * Revision 1.14  2000/03/09 23:29:05  moz
 * Joinstyle should always be JoinRound.
 *
 * Revision 1.13  2000/03/07 20:45:36  moz
 * Compile fixes.
 *
 * Revision 1.12  2000/02/18 04:07:16  moz
 * Free temporary cache contents.
 *
 * Revision 1.11  2000/01/29 21:01:56  moz
 * Use get_nearest_point correctly.
 *
 * Revision 1.10  2000/01/28 16:44:22  moz
 * Use edit_view to stop droppings on other views.
 *
 * Revision 1.9  2000/01/26 18:21:01  moz
 * Can't use ++ in add_to_list() now.
 *
 * Revision 1.8  1999/11/15 02:04:37  moz
 * Use rounded trig.
 *
 * Revision 1.7  1999/05/19 17:08:13  moz
 * 1.0 Checkin.
 *
 * Revision 1.6  1999/04/27 19:02:19  moz
 * Fixed coredump on ob==NULL
 *
 * Revision 1.5  1999/04/27 04:12:04  moz
 * -Wall appeasement.
 *
 * Revision 1.4  1999/04/23 00:40:06  moz
 * redraw_view_window change.
 *
 * Revision 1.3  1999/04/15 22:30:34  moz
 * Can edit shape factor of start and finish when closed spline now.
 *
 * Revision 1.2  1999/04/15 22:19:15  moz
 * Fixed final bbox calculation for interpolated closed splines.
 *
 * Revision 1.1  1999/03/30 00:06:02  moz
 * Initial revision
 *
 */    

#include <math.h> 
#include "include/figurine.h"
#include "include/extern.h" 

/* the spline model used is X-Spline, Blanc and Schlick  */  
 
XPoint xp[MAXPOINTS]; 
XPoint hp[MAXPOINTS]; 
static Boolean held = FALSE;
static ListEl *edited_point=NULL; 
static View *edit_view=NULL; 
static Object *cpoly=NULL; 
static uint pcount=0; 
static long orig_x=0;
static long orig_y=0;
static int ticket=0; 

extern int disable_motion;

void delete_cache_list(List points)
{
	List l = points;

	while (l!=NULL)
		{
		if (l->data!=NULL) 
			free(l->data);
		l = l->next;
		};
	
	delete_list(points);
}

/* computes the curve given the control points  */  
List 
compute_spline_segment(List points, Boolean closed)
{
	List l=NULL;
	double step;
	int k = 0; 
	ListEl *p0,*p1,*p2,*p3; 

	ticket=0;
	 
	if (points==NULL)
		return NULL;
	
	if (points->next==NULL)
		return NULL;
	
	if (points->next->next==NULL)
		closed=FALSE; /* don't attempt to close a line  */ 
		 
	/* so at least three control points */  
	
	/* this is a good compromise between looks and speed  */  
	step=0.08;

	if (!closed)
		{
		/* first control point used twice */ 

		p0 = points;
		p1 = points;
		p2 = points->next;
		if (p2->next==NULL)
			p3 = p2;
		else
			p3 = p2->next;

		for (k=0; ; k++)
			{
				 
			l = spline_compute(l, step, k, SPOINT(p0), SPOINT(p1), SPOINT(p2), SPOINT(p3), SPOINT(p1)->s, SPOINT(p2)->s); 
			if (p3->next==NULL)
				break;
			p0 = p1;
			p1 = p2;
			p2 = p3;
			p3 = p3->next;
			};
		
		p0 = p1;
		p1 = p2;
		p2 = p3;
		l = spline_compute(l, step, k, SPOINT(p0), SPOINT(p1), SPOINT(p2), SPOINT(p3), SPOINT(p1)->s, SPOINT(p2)->s); 
		}
	else
		{
		/* closed spline, loop the curve round  */  
		p0 = points;
		p1 = p0->next;
		p2 = p1->next;
		p3 = p2->next;

	   for (k=0 ; p3!=NULL ; k++) 
			{
			l = spline_compute(l, step, k, SPOINT(p0), SPOINT(p1), SPOINT(p2), SPOINT(p3), SPOINT(p1)->s, SPOINT(p2)->s); 
			p0 = p1;
			p1 = p2;
			p2 = p3;
			p3 = p3->next;
			};

		p3 = points;

		l = spline_compute(l, step, k, SPOINT(p0), SPOINT(p1), SPOINT(p2), SPOINT(p3), SPOINT(p1)->s, SPOINT(p2)->s); 
		 
		k++;
		p0 = p1;
		p1 = p2;
		p2 = p3;
		p3 = p3->next;
		l = spline_compute(l, step, k, SPOINT(p0), SPOINT(p1), SPOINT(p2), SPOINT(p3), SPOINT(p1)->s, SPOINT(p2)->s); 
		k++;
		p0 = p1;
		p1 = p2;
		p2 = p3;
		p3 = p3->next;
		l = spline_compute(l, step, k, SPOINT(p0), SPOINT(p1), SPOINT(p2), SPOINT(p3), SPOINT(p1)->s, SPOINT(p2)->s); 
		};	
		
	return l;
}

/* just to make below a bit clear  */  
/* simple mean average of blending functions  */  
#define SPLINE_CALC(d) (blending[0]*p0->d + \
			blending[1]*p1->d +                \
		   blending[2]*p2->d +                \
		   blending[3]*p3->d) /               \
		   (blending[0] + blending[1] + blending[2] + blending[3])

/* three different blenders  */  
double 
f_blend(double n, double d)
{
	double p = 2*d*d;
	double u = n/d;
	double u2 = u*u;

	return (u*u2*(10-p + (2*p-15)*u + (6-p)*u2)); 
}

double 
g_blend(double u, double q)
{
	return (u*(q + u*(2*q + u*(8-12*q + u*(14*q-11 + u*(4-5*q))))));
}

double 
h_blend(double u, double q)
{
	double u2 = u*u;
	return (u*(q + u*(2*q + u2*(-2*q - u*q))));
}

/* the next four functions determine the effect of the shape factor on the curve  */  
void 
compute_negative_s1(double t, double s, double *b1, double *b2)
{
	*b1 = h_blend(-t, -s);
	*b2 = g_blend(t, -s);
}

void 
compute_negative_s2(double t, double s, double *b1, double *b2)
{
	*b1 = g_blend(1-t, -s);
	*b2 = h_blend(t-1, -s);
}

void 
compute_positive_s1(int k, double t, double s, double *b1, double *b2)
{
	double Tk;

	Tk = k + 1 + s;
	*b1 = (t+k+1<Tk) ? f_blend(t+k+1-Tk, k-Tk) : 0.0;
	Tk = k + 1 - s; 
	*b2 = f_blend(t+k+1-Tk, k+2-Tk); 
}
 
void 
compute_positive_s2(int k, double t, double s, double *b1, double *b2)
{
	double Tk;

	Tk = k + 2 + s;
	*b1 = f_blend(t+k+1-Tk, k+1-Tk); 
	Tk = k + 2 - s; 
	*b2 = (t+k+1>Tk) ? f_blend(t+k+1-Tk, k+3-Tk) : 0.0;
}
 
/* compute one segment of the spline  */  
List 
spline_compute(List l, double step, int k, SPoint *p0, SPoint *p1, SPoint *p2, SPoint *p3, double s1, double s2)
{
	VPoint *vp;
	double blending[4];
	double t;

	for (t=0.0; t<1; t+=step)
		{
		if (s1 < 0 && s2 < 0)
			{
			compute_negative_s1(t, s1, &blending[0], &blending[2]);
			compute_negative_s2(t, s2, &blending[1], &blending[3]);
			}
		else if (s1 < 0)
			{
			compute_negative_s1(t, s1, &blending[0], &blending[2]);
			compute_positive_s2(k, t, s2, &blending[1], &blending[3]);
			}
		else if (s2 < 0) 
			{ 
			compute_positive_s1(k, t, s1, &blending[0], &blending[2]);
			compute_negative_s2(t, s2, &blending[1], &blending[3]);
			}
		else /* both positive or =0 */ 
			{ 
			compute_positive_s1(k, t, s1, &blending[0], &blending[2]);
			compute_positive_s2(k, t, s2, &blending[1], &blending[3]);
			}

		vp = (VPoint *)malloc(sizeof(VPoint));
		vp->x = SPLINE_CALC(x);
		vp->y = SPLINE_CALC(y);
		vp->derried = FALSE; 
		l = add_to_list(l, (ulong)ticket, 0, (void *)vp);
		ticket++; 
		}
	
	return l;
}
 
void 
spline_button(BEvent *bev, View *view)
{
	Object *ob; 
	List l; 
	SPoint *p1; 

	if (bev->button==Button2) 
		{
		return; 
		}; 

	if (bev->button==Button3 && state.busy_drawing)
		{
		/* cancel the current line */ 
		if (!held) 
			XUngrabPointer(display,CurrentTime);
		
		toggle_spline_line(view, bev->x, bev->y);
			 
		if (cpoly->ob.spline.points->next==NULL)
		   {
		   view->doc->o = trash_object(view->doc->o, &view->doc->lo,  cpoly);
		   kill_object(cpoly);
		   cpoly = NULL;
			}; 
		
		if (cpoly!=NULL)
			{
			/* must set final shape factor to 0.0 and redraw  */  
			l = cpoly->ob.spline.points;
			while (l->next!=NULL)
				l = l->next;

			if (!cpoly->ob.spline.closed) 
				SPOINT(l)->s = 0.0;
			else
				SPOINT(l)->s = view->current_sfactor;

			delete_cache_list(cpoly->ob.spline.cache);
			cpoly->ob.spline.cache=NULL;
			 
			/* we need to recalc and redisplay in some situations  */  
			store_redraw_object(cpoly);  
			cpoly->ob.spline.cache = compute_spline_segment(cpoly->ob.spline.points,cpoly->ob.spline.closed);
			recalc_polyline_bbox(cpoly, TRUE);
			view->doc->o = change_bbox(view->doc->o, cpoly); 
			send_stored_redraw_object(view,cpoly); 
			}; 
		 
		state.busy_drawing = FALSE;
		edit_view=NULL; 
		held = FALSE;
		cpoly = NULL;
		pcount = 0;
		 
		}
	else if (bev->button==Button1)
		{
		long x,y;

		if (!P_IN_DOC(bev->x, bev->y, view) && state.busy_drawing)
			{
			/* ok, cancel the particular line we're drawing  */
			state.busy_drawing = FALSE;
			pcount = 0;
			cpoly = NULL;
			 
			if (!held)
				XUngrabPointer(display,CurrentTime);
			else 
				held = FALSE;
			
			toggle_spline_line(view, bev->x, bev->y);
			edit_view=NULL; 
			}
		else
			{
			switch (bev->type)
				{
				case BUTTON_HELD:
					
					if (state.busy_drawing)
						{

						}
					else
						{
						Object *rob=NULL; 
						/* start drawing */  
						state.busy_drawing = TRUE;
						held = TRUE;
						edit_view=view; 
						if (state.shift_down &&
							get_nearest_point(view,view->doc->o, XP2D(bev->x,view),YP2D(bev->y,view),&x,&y,&rob))
							{
							if (view->guide_lines && view->guide_lines_displayed)
								{
								toggle_guidelines(view);
								view->guide_lines_displayed = FALSE;
								};
							orig_x = x;
							orig_y = y;
							bev->x = XD2P(x,view); 
							bev->y = YD2P(y,view); 
							mouse_x = bev->x;
							mouse_y = bev->y;
							XWarpPointer(display, None, view->draw_window->win, 0, 0, 1, 1, 
											 bev->x,bev->y);
							}   
						else 
							{
							if (view->gridon)
								{
								orig_x = GXP2D(bev->x,view);
								orig_y = GYP2D(bev->y,view);
								}
							else
								{
								orig_x = XP2D(bev->x,view);
								orig_y = YP2D(bev->y,view);
								};	
							}; 
							 
 						toggle_spline_line(view, bev->x, bev->y);  
						/* add start of spline control points !  */ 
						ob = (Object *)malloc(sizeof(Object));
						p1 = (SPoint *)malloc(sizeof(SPoint));
						p1->x = 0;
						p1->y = 0;
						p1->derried = rob!=NULL; 
						if (!view->closed) 
							p1->s = 0.0; /* angular to starting point if open */ 
						else
							p1->s = view->current_sfactor;
						ob->bbox.x1 = orig_x;
						ob->bbox.y1 = orig_y;
						ob->bbox.x2 = orig_x;
						ob->bbox.y2 = orig_y;
						ob->ls = view->linestyle; 
						ob->lw = view->linewidth; 
						ob->es = view->endstyle;
						ob->js = JoinRound;
						if (view->barrow_on && !view->closed)
							ob->barrow = make_arrow(view->barrow);
						else
							ob->barrow = NULL;
						if (view->farrow_on && !view->closed)
							ob->farrow = make_arrow(view->farrow);
						else
							ob->farrow = NULL; 
						ob->colour = view->colour; 
						if (view->fillon)
							ob->fs = view->fillstyle;
						else
							ob->fs = NONE;
						ob->fillcolour = view->fillcolour; 
						ob->type = SPLINE;
						ob->ticket = ob_ticket++; 
						ob->derries = NULL; 
						ob->depth = view->doc->ob_depth--; 
						ob->ob.spline.closed = view->closed; 
						 
						ob->ob.spline.points = create_list(0,0,(void *)p1);
						ob->ob.spline.cache = NULL; 
						cpoly = ob; 
						pcount = 1; 

						register_undo(UNDO_PASTE,ob,view->doc); 
						if (rob!=NULL)
							add_derry(rob,ob,(VPoint *)p1);
						view->doc->o = add_object(view->doc->o, &view->doc->lo, ob);
						
						}; 

					break; /* BUTTON_HELD  */ 

				case BUTTON_CLICKED:
				case BUTTON_RELEASED: 

					if (held)
						{
						XGrabPointer(display, view->draw_window->win, False, ButtonPressMask | ButtonReleaseMask | PointerMotionMask | ButtonMotionMask 
										 | Button1MotionMask, GrabModeAsync, GrabModeAsync, None, None, CurrentTime); 
						}
					else
						{
						VRegion bb; 
						Object *rob=NULL; 

						toggle_spline_line(view, bev->x, bev->y);
						 
						/* add line to existing object */ 
						bb.x1 = orig_x;
						bb.y1 = orig_y;
						if (state.shift_down && get_nearest_point(view,view->doc->o, XP2D(bev->x,view),YP2D(bev->y,view),&x,&y,&rob))
							{
							if (view->guide_lines && view->guide_lines_displayed)
								{
								toggle_guidelines(view);
								view->guide_lines_displayed = FALSE;
								};
							bb.x2 = x;
							bb.y2 = y;
							bev->x = XD2P(x,view); 
							bev->y = YD2P(y,view); 
							mouse_x = bev->x;
							mouse_y = bev->y;
							XWarpPointer(display, None, view->draw_window->win, 0, 0, 1, 1, 
											 bev->x,bev->y);
							}
						else
							{
							if (view->gridon) 
								{ 
								bb.x2 = GXP2D(bev->x,view);
								bb.y2 = GYP2D(bev->y,view);
								} 
							else
								{
								bb.x2 = XP2D(bev->x,view);
								bb.y2 = YP2D(bev->y,view);
								}; 
							};
						
						p1 = (SPoint *)malloc(sizeof(SPoint));
						
						p1->x = bb.x2;
						p1->y = bb.y2;
						p1->derried = rob!=NULL; 
						p1->s = view->current_sfactor; 
						orig_x = p1->x;
						orig_y = p1->y;
						normalise_rectangle(&bb.x1,&bb.y1,&bb.x2, &bb.y2);
						l = cpoly->ob.spline.points;

						/* unrelavitise coordinates before bb merge  */  
						while (l != NULL)
							{
							((SPoint *)l->data)->x = ((SPoint *)l->data)->x+cpoly->bbox.x1;
							((SPoint *)l->data)->y = ((SPoint *)l->data)->y+cpoly->bbox.y1;
							l = l->next; 
							}; 

						cpoly->bbox = merge_boxes(cpoly->bbox,bb); 
						cpoly->ob.spline.points = add_to_list(cpoly->ob.spline.points, pcount, 0, (void *)p1); 
						pcount++; 
					
						l = cpoly->ob.spline.points;
						while (l != NULL)
							{
							((SPoint *)l->data)->x = ((SPoint *)l->data)->x-cpoly->bbox.x1;
							((SPoint *)l->data)->y = ((SPoint *)l->data)->y-cpoly->bbox.y1;
							l = l->next; 
							}; 

						delete_cache_list(cpoly->ob.spline.cache); 
						cpoly->ob.spline.cache = compute_spline_segment(cpoly->ob.spline.points,FALSE);
						recalc_polyline_bbox(cpoly, TRUE);
						view->doc->o = change_bbox(view->doc->o, cpoly); 
						bb.x2++; bb.y2++; 
						 
						if (rob!=NULL)
							add_derry(rob,cpoly,(VPoint *)p1);
 						send_redraw_object(view,cpoly); 
						/* carry on drawing  */  
						toggle_spline_line(view,bev->x, bev->y);
					
						}; 
					 
					held = FALSE; 
					break;
				};
			}; 
		}; 
}

void 
toggle_spline_line(View *view, int x, int y)
{
	VLine *pl;
	VLine line;
	Boolean clipped; 

	if (edit_view!=view)
		return;

	line.x1 = XD2P(orig_x,view);
	line.y1 = YD2P(orig_y,view);
	 
	if (view->gridon)
		{
		line.x2 = XD2P(GXP2D(x,view),view);
		line.y2 = YD2P(GYP2D(y,view),view);
		}
	else
		{
		line.x2 = x;
		line.y2 = y;
		};

	pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 0); 
		 
  	if (pl!=NULL)    
		XDrawLine(display,view->draw_window->win, blackxorgc, pl->x1, pl->y1, pl->x2, pl->y2);
}

void 
spline_point_button(BEvent *bev, View *v)
{
	if (bev->type==BUTTON_HELD)
		{
		if (!state.editing_point && !is_in_bbox(XP2D(bev->x,v),YP2D(bev->y,v),
													 v->edited_object->bbox.x1-P2D(HANDLE_PIXEL_SIZE,v),
													 v->edited_object->bbox.y1-P2D(HANDLE_PIXEL_SIZE,v), 
													 v->edited_object->bbox.x2+P2D(HANDLE_PIXEL_SIZE,v),
													 v->edited_object->bbox.y2+P2D(HANDLE_PIXEL_SIZE,v)))
			{
			if (state.editing_point) 
				{ 
				toggle_point_spline(v,bev->x,bev->y); 
				state.editing_point = FALSE; 
				};
			send_redraw_object(v,v->edited_object->ob);
			v->edited_object = NULL;
			} 
		else if (!state.editing_point)
			{
			double near; 
			ListEl *n;

			n = polyline_nearest_point(v->edited_object->ob->ob.spline.points, 
												XP2D(bev->x,v)-v->edited_object->ob->bbox.x1, 
												YP2D(bev->y,v)-v->edited_object->ob->bbox.y1, &near);

			if (D2P(near,v)<POINTER_JUMP_SIZE)
				{ 
				XWarpPointer(display, None, v->draw_window->win, 0, 0, 1, 1, 
								 XD2P(SPOINT(n)->x+v->edited_object->ob->bbox.x1,v), 
								 YD2P(SPOINT(n)->y+v->edited_object->ob->bbox.y1,v));
				state.editing_point = TRUE;
				edited_point = n; 
				toggle_point_spline(v,bev->x, bev->y);
				orig_x = bev->x;
				orig_y = bev->y;
				};
			}
		else if (bev->button == Button3)
			{
			toggle_point_spline(v,bev->x, bev->y);
			state.editing_point = FALSE;
			}; 
		}
	else
		{ /* CLICKED or RELEASED  */ 
		Object *rob=NULL; 
		Object *s=NULL; 
		long x,y;

		/* if released, place it and redraw object  */  
		if (bev->type == BUTTON_RELEASED && state.editing_point && P_IN_DOC(bev->x,bev->y,v))
			{
			register_undo(UNDO_OB_PROP,v->edited_object->ob,v->doc); 
			toggle_point_spline(v,bev->x, bev->y);
			if (SPOINT(edited_point)->derried)
				remove_derry(v,v->edited_object->ob,(VPoint *)SPOINT(edited_point));

			if (!state.control_down) 
				{ 
				if (state.shift_down &&
					get_nearest_point(v,v->doc->o, XP2D(bev->x,v),YP2D(bev->y,v),&x,&y,&rob))
					{
					if (v->guide_lines && v->guide_lines_displayed)
						{
						toggle_guidelines(v);
						v->guide_lines_displayed = FALSE;
						};
					SPOINT(edited_point)->x = x-v->edited_object->ob->bbox.x1;
					SPOINT(edited_point)->y = y-v->edited_object->ob->bbox.y1;
					bev->x = XD2P(x,v); 
					bev->y = YD2P(y,v); 
					mouse_x = bev->x;
					mouse_y = bev->y;
					XWarpPointer(display, None, v->draw_window->win, 0, 0, 1, 1, 
									 bev->x,bev->y);
					add_derry(rob,v->edited_object->ob,(VPoint *)SPOINT(edited_point));	
					SPOINT(edited_point)->derried = TRUE;
					}    
				else 
					{ 
					if (v->gridon)
						{
						SPOINT(edited_point)->x = GXP2D(bev->x, v)-v->edited_object->ob->bbox.x1;
						SPOINT(edited_point)->y = GYP2D(bev->y, v)-v->edited_object->ob->bbox.y1;
						}
					else
						{
						SPOINT(edited_point)->x = XP2D(bev->x, v)-v->edited_object->ob->bbox.x1;
						SPOINT(edited_point)->y = YP2D(bev->y, v)-v->edited_object->ob->bbox.y1;
						}; 
					};
				}
			else if (!(edited_point->prev==NULL || edited_point->next==NULL)
						|| v->edited_object->ob->ob.spline.closed)
				{
				double f;
				long sx, sy; 
				 
				sx =  min(P2D(50,v),max(P2D(-50,v),XP2D(bev->x, v) - ((SPOINT(edited_point)->x)+v->edited_object->ob->bbox.x1)));
				sy =  min(P2D(50,v),max(P2D(-50,v),YP2D(bev->y, v) - ((SPOINT(edited_point)->y)+v->edited_object->ob->bbox.y1)));
				f = (double)max(abs(sx),abs(sy));
				/* minus shape factor if "behind" point */  
				if (sx<0 || sy<0)
					f = -f;
				f = f/P2D(50,v);
				if (abs(f)<0.2)
					f = 0.0; /* angular  */ 
				SPOINT(edited_point)->s = f;
				};
				
 
			store_redraw_object(v->edited_object->ob); 
			delete_cache_list(v->edited_object->ob->ob.spline.cache);
			v->edited_object->ob->ob.spline.cache = compute_spline_segment(v->edited_object->ob->ob.spline.points,v->edited_object->ob->ob.spline.closed);
 			recalc_polyline_bbox(v->edited_object->ob,TRUE);	  
			s = v->edited_object->ob; 
			v->doc->o = change_bbox(v->doc->o, v->edited_object->ob);	
			v->edited_object = get_object_node(v->doc->o,s);
			send_stored_redraw_object(v,v->edited_object->ob); 
			state.editing_point = FALSE; 
			}
		else if (state.editing_point)
			{
			toggle_point_spline(v,bev->x,bev->y);
			state.editing_point = FALSE;
			};
		}; 
}

void 
spline_point_motion(View *v, long x, long y)
{
	long tx, ty;
	 
	/* if drawing, resize as appropriate */
	if (state.editing_point)
		{
		Object *rob=NULL; 
		int test=-1,test2=-1; /* func returns -1 if no change */
				 
		/* complex logic for after a zoom */
		if (!disable_motion)
			toggle_point_spline(v,orig_x,orig_y);
		else if (disable_motion==1)
			disable_motion--;
		else
			{
			disable_motion--;
			return;
			};
			 
		if (state.shift_down && get_nearest_point(v,v->doc->o, XP2D(x,v),YP2D(y,v),&tx,&ty,&rob))
			{
			if (v->guide_lines && v->guide_lines_displayed)
				{
				toggle_guidelines(v);
				v->guide_lines_displayed = FALSE;
				};
			mouse_x = XD2P(tx,v);
			mouse_y = YD2P(ty,v);
			x = mouse_x;
			y = mouse_y;
			XWarpPointer(display, None, v->draw_window->win, 0, 0, 1, 1, 
				 mouse_x,mouse_y);
			};  
			 
		orig_x = x;
		orig_y = y;
		 
		if ( x < MOVE_BOUNDARY) /* move left */  
			test = nudge_ruler_x(v->ruler_x_window, v, LEFT);
		else if ( x >= ((int)v->draw_window->w)-MOVE_BOUNDARY) /* move right */ 
			test = nudge_ruler_x(v->ruler_x_window, v, RIGHT);
				
		if ( y < MOVE_BOUNDARY) /* move up */  
			test2 = nudge_ruler_y(v->ruler_y_window, v, UP); 
		else if ( y >= ((int)v->draw_window->h)-MOVE_BOUNDARY) /* move down */ 
			test2 = nudge_ruler_y(v->ruler_y_window, v, DOWN); 
				
		if (test==-1 && test2==-1) /* i.e. not scrolled  */ 
			{
			if (!disable_motion)
				toggle_point_spline(v, x, y);
			else
				disable_motion--;
			}    
		else
			{ 
			redraw_view_window(v); 
			state.rubberon=FALSE;
			toggle_point_spline(v,mouse_x,mouse_y); 
			};
		}; 
}

void 
toggle_point_spline(View *v, long x, long y)
{
	List l=NULL,l2,l3;
	SPoint p,pprev,pnext,pprevprev,pnextnext; 
	Boolean clipped = FALSE; 
	int xpp; 
	VLine line; VLine *pl; 
	
	/* we only need to recompute the relevant section of the spline 2 control points either way */  
	if (edited_point->prev!=NULL)
		{
		if (edited_point->prev->prev!=NULL)
			{
			pprevprev = *SPOINT(edited_point->prev->prev);
			pprevprev.x += v->edited_object->ob->bbox.x1;
			pprevprev.y += v->edited_object->ob->bbox.y1;
			l = add_to_list(l,0,0,(void *)&pprevprev);
			};
	
		pprev = *SPOINT(edited_point->prev);
		pprev.x += v->edited_object->ob->bbox.x1;
		pprev.y += v->edited_object->ob->bbox.y1;
		l = add_to_list(l,1,0,(void *)&pprev);
		};
	
	if (!state.control_down) 
		{ 
		if (v->gridon)
			{
			p.x = GXP2D(x,v);
			p.y = GYP2D(y,v);
			}
		else
			{
			p.x = XP2D(x,v);
			p.y = YP2D(y,v);
			};
	
		p.s = SPOINT(edited_point)->s;
		}
	else if (!(edited_point->prev==NULL || edited_point->next==NULL)
				|| v->edited_object->ob->ob.spline.closed)
		{
		double f;
		p.x = SPOINT(edited_point)->x+v->edited_object->ob->bbox.x1;
 		p.y = SPOINT(edited_point)->y+v->edited_object->ob->bbox.y1; 
		f =  min(P2D(50,v),max(P2D(-50,v),XP2D(x, v) - p.x));
		f = f/P2D(50,v);
		if (abs(f)<0.2)
			f = 0.0; /* angular  */ 
 		p.s = f; 
		} 
	else
		{
		p.x = SPOINT(edited_point)->x+v->edited_object->ob->bbox.x1;
 		p.y = SPOINT(edited_point)->y+v->edited_object->ob->bbox.y1; 
		p.s = SPOINT(edited_point)->s; 
		}; 


	l = add_to_list(l,2,0,(void *)&p);
	
	if (edited_point->next!=NULL)
		{
		pnext = *SPOINT(edited_point->next);
		pnext.x += v->edited_object->ob->bbox.x1;
		pnext.y += v->edited_object->ob->bbox.y1;
		l = add_to_list(l,3,0,(void *)&pnext);
		if (edited_point->next->next!=NULL)
			{
			pnextnext = *SPOINT(edited_point->next->next);
			pnextnext.x += v->edited_object->ob->bbox.x1;
			pnextnext.y += v->edited_object->ob->bbox.y1;
			l = add_to_list(l,4,0,(void *)&pnextnext);
			};
		};
	
	l2 = compute_spline_segment(l,FALSE);

	xpp=0; 
	l3 = l2;
	line.x1 = XD2P(POINT(l3)->x,v);
	line.y1 = YD2P(POINT(l3)->y,v);
	xp[xpp].x = line.x1;
	xp[xpp].y = line.y1;
	xpp++;

	while (l3->next!=NULL)
		{
		line.x1 = XD2P(POINT(l3)->x,v);
		line.y1 = YD2P(POINT(l3)->y,v);
		line.x2 = XD2P(POINT(l3->next)->x,v);
		line.y2 = YD2P(POINT(l3->next)->y,v);
 		pl = clip_line(0,0,(long)v->draw_window->w,(long)v->draw_window->h, &line, &clipped, 1); 
 		if (pl!=NULL)   
			{
			xp[xpp].x = pl->x2;
			xp[xpp].y = pl->y2;
			xpp++;
			};
		
		if (clipped || xpp==MAXPOINTS)
			{
			XDrawLines(display,v->draw_window->win, blackxorgc, xp, xpp, CoordModeOrigin);
			/* continue from last line  */  
			xp[0].x = XD2P(POINT(l3->next)->x,v); 
			xp[0].y =  YD2P(POINT(l3->next)->y,v); 
			xpp=1;
			};
		l3 = l3->next;
		};
		 
	if (xpp!=0)
		XDrawLines(display,v->draw_window->win, blackxorgc, xp, xpp, CoordModeOrigin);
	
	/* don't need to delete contents, auto vars in scope */ 
	delete_list(l);
	 
	delete_cache_list(l2);
}

void 
draw_spline(Object *ob, View *view, GC gc, long x, long y, double rx, double ry, double angle)
{
	int xpp=0,hpp;
	List l;
	VLine line; 
	VLine *pl; 
	Boolean clipped=FALSE; 
	
	if (ob->ob.spline.cache==NULL) 
		{ 
		if (cpoly!=ob) /* ie. not drawing this  */  
			ob->ob.spline.cache = compute_spline_segment(ob->ob.spline.points,ob->ob.spline.closed);
		else
			ob->ob.spline.cache = compute_spline_segment(ob->ob.spline.points,FALSE);
		};
	if (ob->ob.spline.cache==NULL)
		return;
		 
	if (angle!=0.0)
		{
		double sina, cosa;
		long cx,cy;

		cosa = cosround(angle);
		sina = sinround(angle);
		cx = (ob->bbox.x2-ob->bbox.x1)/2;
		cy = (ob->bbox.y2-ob->bbox.y1)/2;

		xpp=0; 
		l = ob->ob.spline.cache;
		line.x1 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l)->x-cx)),view) - D2P((sina*(POINT(l)->y-cy)),view));
		line.y1 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l)->x-cx)),view) + D2P((cosa*(POINT(l)->y-cy)),view));
		xp[xpp].x = line.x1;
		xp[xpp].y = line.y1;
		xpp++;

		while (l->next!=NULL)
			{
         line.x1 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l)->x-cx)),view) - D2P((sina*(POINT(l)->y-cy)),view));
         line.y1 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l)->x-cx)),view) + D2P((cosa*(POINT(l)->y-cy)),view));
         line.x2 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(l->next)->x-cx)),view) - D2P((sina*(POINT(l->next)->y-cy)),view));
         line.y2 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l->next)->x-cx)),view) + D2P((cosa*(POINT(l->next)->y-cy)),view));
			pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2*ob->lw); 
			if (pl!=NULL)   
				{
				xp[xpp].x = pl->x2;
				xp[xpp].y = pl->y2;
				xpp++;
				};
			
			if (clipped || xpp==MAXPOINTS)
				{
				XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
				/* continue from last line  */  
            xp[0].x = XD2P(x,view) + D2P(cx,view) +  (D2P((cosa*(POINT(l->next)->x-cx)),view) - D2P((sina*(POINT(l->next)->y-cy)),view));
            xp[0].y = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(l->next)->x-cx)),view) + D2P((cosa*(POINT(l->next)->y-cy)),view));
												 
				xpp=1;
				};
			l = l->next;
			};
			 
		if (xpp!=0)
			XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
		
		if (ob!=cpoly && ob->fs!=NONE && gc!=blackxorgc)
			XFillPolygon(display,view->draw_window->win, fillgc, xp, xpp, Complex, CoordModeOrigin);    

		if (ob->ob.spline.closed)
			{
			/* last little line  */ 
		 	line.x1 = xp[xpp-1].x;
			line.y1 = xp[xpp-1].y;
			line.x2 = XD2P(x,view) + D2P(cx,view) + (D2P((cosa*(POINT(ob->ob.spline.cache)->x-cx)),view) - D2P((sina*(POINT(ob->ob.spline.cache)->y-cy)),view));
         line.y2 = YD2P(y,view) + D2P(cy,view) + (D2P((sina*(POINT(ob->ob.spline.cache)->x-cx)),view) + D2P((cosa*(POINT(ob->ob.spline.cache)->y-cy)),view));
			pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2*ob->lw); 
			if (pl!=NULL)   
				XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
			}; 
		}
	else
		{
		if (state.tied_corner!=NOTSCALING)
			corner_magic(ob, &x, &y, rx, ry);

		rx = abs(rx);
		ry = abs(ry);
		
		l = ob->ob.spline.cache;
		hpp=0;
		while (l!=NULL && hpp<MAXPOINTS)
			{
			hp[hpp].x = XD2P(R(POINT(l)->x,rx)+x,view);
			hp[hpp].y = YD2P(R(POINT(l)->y,ry)+y,view);
			hpp++;
			l = l->next;
			};
			
		if (ob!=cpoly && ob->fs!=NONE && gc!=blackxorgc)
			XFillPolygon(display,view->draw_window->win,fillgc, hp, hpp, Complex, CoordModeOrigin);    

		if (gc==blackxorgc || ob->lw!=0)
			{
			xpp=0; 
			l = ob->ob.spline.cache;
			line.x1 = XD2P(R(POINT(l)->x,rx)+x,view);
			line.y1 = YD2P(R(POINT(l)->y,ry)+y,view);
			xp[xpp].x = line.x1;
			xp[xpp].y = line.y1;
			xpp++;

			while (l->next!=NULL)
				{
				line.x1 = XD2P(R(POINT(l)->x,rx)+x,view);
				line.y1 = YD2P(R(POINT(l)->y,ry)+y,view);
				line.x2 = XD2P(R(POINT(l->next)->x,rx)+x,view);
				line.y2 = YD2P(R(POINT(l->next)->y,ry)+y,view);
				pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2*ob->lw); 
				if (pl!=NULL)   
					{
					xp[xpp].x = pl->x2;
					xp[xpp].y = pl->y2;
					xpp++;
					};
				
				if (clipped || xpp==MAXPOINTS)
					{
					XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
					/* continue from last line  */  
					xp[0].x =  XD2P(R(POINT(l->next)->x,rx)+x,view);
					xp[0].y =  YD2P(R(POINT(l->next)->y,ry)+y,view);
					xpp=1;
					};
				l = l->next;
				};
				 
			if (xpp!=0)
				XDrawLines(display,view->draw_window->win, gc, xp, xpp, CoordModeOrigin);
				
			if (ob->ob.spline.closed)
				{
				/* last little line  */ 
				line.x1 = xp[xpp-1].x;
				line.y1 = xp[xpp-1].y;
				line.x2 = XD2P(R(POINT(ob->ob.spline.cache)->x,rx)+x,view);      
				line.y2 = YD2P(R(POINT(ob->ob.spline.cache)->y,ry)+y,view); 
				pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2*ob->lw); 
				if (pl!=NULL)   
					XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
				} 
			else
				{
				if (ob->barrow!=NULL && rx==1.0 && ry==1.0 && cpoly!=ob)
					{
					long x1,y1,x2,y2;

					x2 = XD2P(POINT(ob->ob.spline.points)->x+x,view); 
					y2 = YD2P(POINT(ob->ob.spline.points)->y+y,view); 
					if (ob->ob.spline.points->next!=NULL)
						{
						x1 = XD2P(POINT(ob->ob.spline.points->next)->x+x,view); 
						y1 = YD2P(POINT(ob->ob.spline.points->next)->y+y,view); 
						}
					else
						{
						x1 = XD2P(POINT(ob->ob.spline.points)->x+x,view); 
						y1 = YD2P(POINT(ob->ob.spline.points)->y+y,view); 
						};
					draw_arrow(view,gc,ob->barrow,x1,y1,x2,y2);
					};

				if (ob->farrow!=NULL && rx==1.0 && ry==1.0 && cpoly!=ob)
					{
					/* init to zero to keep -Wall happy  */  
					long x1=0,y1=0,x2,y2;

					l = ob->ob.spline.points;

					while (l->next!=NULL)
						{
						if (l->next->next==NULL)
							{
							x1 = XD2P(POINT(l)->x+x,view);
							y1 = YD2P(POINT(l)->y+y,view); 
							}; 
						l = l->next;
						};

					x2 = XD2P(POINT(l)->x+x,view);
					y2 = YD2P(POINT(l)->y+y,view); 
		
					draw_arrow(view,gc,ob->farrow,x1,y1,x2,y2);
					};
				}; 
			};

				 
		/* handles after everything else  */  
		if ((state.current_icon==SPLINEICON && state.busy_drawing && ob==cpoly) || 
			 (view->edited_object!=NULL && view->edited_object->ob == ob))
			{
			l = ob->ob.spline.points;
			while (l!=NULL)
				{
				if (P_IN_DOC(XD2P(R(SPOINT(l)->x,rx)+x,view),YD2P(R(SPOINT(l)->y,ry)+y,view),view))
					draw_handle(view, XD2P(R(SPOINT(l)->x,rx)+x,view),YD2P(R(SPOINT(l)->y,ry)+y,view));
				l=l->next; 
				};
			};
		};
}
