/* text.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* contains functions assoc. with text editing and drawing */
/*
 * $Log: text.c,v $
 * Revision 1.2  2000/12/06 20:56:06  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:31  moz
 * CVS Import
 *
 * Revision 1.7  2000/05/03 22:42:26  moz
 * Cope better with font being NULL in new_text().
 *
 * Revision 1.6  2000/03/13 02:47:07  moz
 * Fill in useless Object parameters.
 *
 * Revision 1.5  2000/03/11 19:11:11  moz
 * Fixed stupid typos in text_object_at_pixel_point().
 * Don't adjust text position on load of rotated text.
 *
 * Revision 1.4  2000/03/11 18:20:05  moz
 * Compile fix for previous revision.
 *
 * Revision 1.3  2000/03/11 17:59:26  moz
 * Place properly when changing rotated text size.
 *
 * Revision 1.2  2000/03/10 18:01:20  moz
 * Fix for compile fix.
 *
 * Revision 1.1  2000/03/10 17:48:17  moz
 * Initial revision
 *
 * Revision 1.24  2000/03/07 21:05:26  moz
 * Compile fixes.
 *
 * Revision 1.23  2000/03/05 21:18:25  moz
 * Removed old clipping stuff and tidied up a bit.
 *
 * Revision 1.22  2000/03/05 20:42:24  moz
 * text_object_at_pixel_point() selects only text objects.
 *
 * Revision 1.21  2000/03/05 00:07:47  moz
 * Fix coredump when placing cursor in empty object.
 *
 * Revision 1.20  2000/02/18 03:04:32  moz
 * Free empty text line.
 *
 * Revision 1.19  2000/01/28 15:59:45  moz
 * Hopefully checking isprint() will not allow control chars.
 *
 * Revision 1.18  1999/11/15 02:02:28  moz
 * Use rounded trig.
 * Don't dump core on cs==NULL.
 *
 * Revision 1.17  1999/08/08 20:55:56  moz
 * From clean up of structs.
 *
 * Revision 1.16  1999/07/29 20:44:27  moz
 * Draw rubber boxes for resizing text.
 *
 * Revision 1.15  1999/05/24 01:01:41  moz
 * Don't create nil-dimension pixmaps.
 *
 * Revision 1.14  1999/05/19 17:06:53  moz
 * 1.0 Checkin.
 *
 * Revision 1.13  1999/05/04 15:50:38  moz
 * Moved list functions to list.c
 *
 * Revision 1.12  1999/04/28 16:35:54  moz
 * fixed bug with calc_text_outer_box.
 *
 * Revision 1.11  1999/04/28 16:31:39  moz
 * Various text bug fixes.
 *
 * Revision 1.10  1999/04/28 15:07:07  moz
 * Reqrite of recalc_text_box.
 * Fixed two views editing_text problem.
 *
 * Revision 1.9  1999/04/27 20:22:30  moz
 * place_cursor now works for rotated text.
 *
 * Revision 1.8  1999/04/27 04:12:12  moz
 * Rotated cursor added.
 *
 * Revision 1.7  1999/04/26 19:22:17  moz
 * Fixed up drawing of rotated text correctly.
 *
 * Revision 1.6  1999/04/25 21:25:26  moz
 * First pass at rotating text.
 *
 * Revision 1.5  1999/04/25 00:21:39  moz
 * Draw outline when deciding rotation.
 *
 * Revision 1.4  1999/04/24 22:23:06  moz
 * gc_colour needs view parameter.
 *
 * Revision 1.3  1999/04/22 22:17:15  moz
 * Ignore control characters on non-Linux OSs.
 *
 * Revision 1.2  1999/04/22 21:26:46  moz
 * Fix centre and right-justified text entry.
 *
 * Revision 1.1  1999/03/30 00:06:58  moz
 * Initial revision
 *
 */    

/* see mouse_button.c for general behaviour */  
 
#include <math.h> 
#include <X11/keysym.h> 
#include <string.h> 
#include <ctype.h> 
#include "include/figurine.h"  
#include "include/extern.h"

static ListEl *cl=NULL; /* current text line  */ 
static ListEl *cs=NULL; /* current section on line */  
static Object *mob=NULL;
static int cx; /* current position of text cursor  */ 
static int cy;
static int charpos; /* in each 19-char section */ 
View *editview=NULL; /* which view are we editing in */  

/* note: do NOT use with nodes */  
/* with text, ob->bbox describes the furthest
	reaches of the text. when rotated, ob->ob.text.bbox
	is kept, and ob->bbox is calculated by rotating the
	points. The final position is adjusted by dx,dy */  
void
calc_text_outer_box_adjust(Object *ob, long dx, long dy)
{
	if (ob->ob.text.angle==0.0)
		{
		ob->bbox.x1 = ob->ob.text.bbox.x1;
		ob->bbox.y1 = ob->ob.text.bbox.y1;
		ob->bbox.x2 = ob->ob.text.bbox.x2;
		ob->bbox.y2 = ob->ob.text.bbox.y2;
		}
	else
		{
		VRegion a1,a2,a3,a4;
		double sina,cosa;
		long ccx,ccy;
		sina = sinround(ob->ob.text.angle);
		cosa = cosround(ob->ob.text.angle);
		ccx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2 + ob->ob.text.bbox.x1;
		ccy = (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2 + ob->ob.text.bbox.y1;
		a1.x1 = ccx + (cosa*(ob->ob.text.bbox.x1-ccx))-(sina*(ob->ob.text.bbox.y1-ccy));
		a1.y1 = ccy + (sina*(ob->ob.text.bbox.x1-ccx))+(cosa*(ob->ob.text.bbox.y1-ccy));
		a1.x2 = ccx + (cosa*(ob->ob.text.bbox.x2-ccx))-(sina*(ob->ob.text.bbox.y1-ccy));
		a1.y2 = ccy + (sina*(ob->ob.text.bbox.x2-ccx))+(cosa*(ob->ob.text.bbox.y1-ccy));
		a2.x1 = ccx + (cosa*(ob->ob.text.bbox.x2-ccx))-(sina*(ob->ob.text.bbox.y1-ccy));
		a2.y1 = ccy + (sina*(ob->ob.text.bbox.x2-ccx))+(cosa*(ob->ob.text.bbox.y1-ccy));
		a2.x2 = ccx + (cosa*(ob->ob.text.bbox.x2-ccx))-(sina*(ob->ob.text.bbox.y2-ccy));
		a2.y2 = ccy + (sina*(ob->ob.text.bbox.x2-ccx))+(cosa*(ob->ob.text.bbox.y2-ccy));
		a3.x1 = ccx + (cosa*(ob->ob.text.bbox.x2-ccx))-(sina*(ob->ob.text.bbox.y2-ccy));
		a3.y1 = ccy + (sina*(ob->ob.text.bbox.x2-ccx))+(cosa*(ob->ob.text.bbox.y2-ccy));
		a3.x2 = ccx + (cosa*(ob->ob.text.bbox.x1-ccx))-(sina*(ob->ob.text.bbox.y2-ccy));
		a3.y2 = ccy + (sina*(ob->ob.text.bbox.x1-ccx))+(cosa*(ob->ob.text.bbox.y2-ccy));
		a4.x1 = ccx + (cosa*(ob->ob.text.bbox.x1-ccx))-(sina*(ob->ob.text.bbox.y2-ccy));
		a4.y1 = ccy + (sina*(ob->ob.text.bbox.x1-ccx))+(cosa*(ob->ob.text.bbox.y2-ccy));
		a4.x2 = ccx + (cosa*(ob->ob.text.bbox.x1-ccx))-(sina*(ob->ob.text.bbox.y1-ccy));
		a4.y2 = ccy + (sina*(ob->ob.text.bbox.x1-ccx))+(cosa*(ob->ob.text.bbox.y1-ccy));
		normalise_rectangle(&a1.x1,&a1.y1,&a1.x2,&a1.y2);
		normalise_rectangle(&a2.x1,&a2.y1,&a2.x2,&a2.y2);
		normalise_rectangle(&a1.x1,&a1.y1,&a1.x2,&a1.y2);
		normalise_rectangle(&a2.x1,&a2.y1,&a2.x2,&a2.y2);
		normalise_rectangle(&a3.x1,&a3.y1,&a3.x2,&a3.y2);
		normalise_rectangle(&a4.x1,&a4.y1,&a4.x2,&a4.y2);
		ob->bbox = merge_boxes(a1,merge_boxes(a2,merge_boxes(a3,a4)));
		ob->bbox.x1 -= dx; 
		ob->bbox.y1 -= dy;
		ob->bbox.x2 -= dx; 
		ob->bbox.y2 -= dy;
		ob->ob.text.bbox.x1 -= dx; 
		ob->ob.text.bbox.y1 -= dy;
		ob->ob.text.bbox.x2 -= dx; 
		ob->ob.text.bbox.y2 -= dy;
		};
}

void 
set_text_just(View *view)
{
	if (mob!=NULL && cl!=NULL)
		{
		TEXTLINE(cl)->just = view->justification;
		recalc_text_box(view,mob,FALSE,FALSE); 
		};
}

void 
unselect_text(View *view)
{
	state.editing_text = FALSE;
	view->text_object = NULL; 
	cl = NULL; cs = NULL;

	if (mob!=NULL && TEXTLINE(mob->ob.text.lines)->sections==NULL)
		{
		view->doc->o = trash_object(view->doc->o, &view->doc->lo, mob);
		send_redraw_object(view,mob); 
		if (mob->ob.text.ellipse!=NULL)
			free(mob->ob.text.ellipse);
		free(TEXTLINE(mob->ob.text.lines));
		free(mob->ob.text.lines); 
		free(mob);
		} 
	else if (mob!=NULL)
		send_redraw_object(view,mob); 
	mob = NULL;
	charpos = 0; 
	cx = 0;
	cy = 0;
}

/* place the text cursor at document position  */  
/* x,y must be relative to ob.text.bbox, in DOCUMENT CO-ORDS  */ 
void 
place_cursor(View *view, int x, int y)
{
	List l,l2,ol=NULL; 
	uint i=0;
	VFont *vf; 

	/* rotate place point back if necessary  */  
	if (mob->ob.text.angle!=0.0)
		{
		double sina,cosa;
		long ccx,ccy;
		long tx; 

		sina = sinround(-mob->ob.text.angle); 
		cosa = cosround(-mob->ob.text.angle); 
		ccx = (mob->ob.text.bbox.x2-mob->ob.text.bbox.x1)/2;
		ccy = (mob->ob.text.bbox.y2-mob->ob.text.bbox.y1)/2;
		
		tx = ccx + (cosa*(x-ccx))-(sina*(y-ccy)); 
		y = ccy + (sina*(x-ccx))+(cosa*(y-ccy)); 
		x = tx; 
		};
		 
	l = mob->ob.text.lines;

	while (l!=NULL)
		{
		if (y>=TEXTLINE(l)->y && y <=(TEXTLINE(l)->y+TEXTLINE(l)->h))
			{
			l2 = TEXTLINE(l)->sections;
			while (l2!=NULL)
				{
				if (x>=TEXTSEC(l2)->x && x <=(TEXTSEC(l2)->x+TEXTSEC(l2)->w))
					{
					vf = get_font(TEXTSEC(l2)->num,ZPO(TEXTSEC(l2)->size,view)); 
					if (vf==NULL)
						return; 
					while (x-TEXTSEC(l2)->x > P2D(XTextWidth(vf->x,TEXTSEC(l2)->text,(int)i),view) && i<=strlen(TEXTSEC(l2)->text))
						i++;
					charpos = i;
					cs = l2;
					cl = l;
					cx = TEXTSEC(l2)->x + P2D(XTextWidth(vf->x,TEXTSEC(l2)->text,(int)i),view);
					cy = TEXTLINE(l)->y; 
					break;
					};
				ol = l2; 
				l2 = l2->next; 
				};
				 
			/* new empty string */
			if (!ol)
				return;

			if (l2==NULL && x <= mob->ob.text.bbox.x2 - mob->ob.text.bbox.x1)
				{
				vf = get_font(TEXTSEC(ol)->num,ZPO(TEXTSEC(ol)->size,view)); 
				if (vf==NULL)
					return; 
				charpos = strlen(TEXTSEC(ol)->text);
				cs = ol;
				cl = l;
				cx = TEXTSEC(ol)->x + P2D(XTextWidth(vf->x,TEXTSEC(ol)->text,(int)strlen(TEXTSEC(ol)->text)),view);
				cy = TEXTLINE(l)->y; 
				}; 
			break;
			};
		l = l->next;
		};
}

/* create a new text object  */  
void 
new_text(BEvent *bev, View *view)
{
	VFont *vf;
	TextLine *tl; 
	VRegion bb; 
	 
	/* start a new object */  
	state.editing_text=TRUE; 
	mob = (Object *)malloc(sizeof(Object));

	mob->type = TEXT;
	mob->ticket = ob_ticket++;
	mob->derries = NULL; 
	mob->depth = view->doc->ob_depth--;
	mob->colour = view->colour;
	mob->fillcolour = view->fillcolour;
	mob->ls = view->linestyle; 
	mob->lw = view->linewidth;
	mob->fs = view->fillstyle;
	mob->es = view->endstyle;
	mob->js = view->endstyle;
	mob->farrow = NULL;
	mob->barrow = NULL;
	mob->ob.text.node = (state.current_icon==NODEICON);
	mob->ob.text.ellipse = NULL;
	mob->ob.text.angle = 0.0; 
	
	bb = mob->bbox; 
	 
	cx = cy = 0; 
	tl = (TextLine *)malloc(sizeof(TextLine)); 
	tl->just = view->justification; 
	mob->ob.text.lines = insert_before(NULL,NULL, (ulong)cy, 0, (void *)tl, &cl);
	TEXTLINE(cl)->y = 0;
	TEXTLINE(cl)->w = 0; 
	TEXTLINE(cl)->sections = NULL; 

	vf = get_font(view->fontnum, ZPO(view->fontsize,view));

	/* give up */ 
	if (vf==NULL)
		{ 
		state.editing_text = FALSE;
		view->text_object = NULL;
		cl = NULL; cs = NULL;      
		charpos=cx=cy=0;
		free(tl);
		free(mob->ob.text.lines);
	       	free(mob);
		return;
		}; 
		 
	if (view->gridon)
		{
		mob->ob.text.bbox.x1 = GXP2D(bev->x,view) - P2D(2,view);
		mob->ob.text.bbox.y1 = GYP2D(bev->y - vf->x->max_bounds.ascent,view);
		}
	else
		{
		mob->ob.text.bbox.x1 = XP2D(bev->x,view) - P2D(2,view);
		mob->ob.text.bbox.y1 = YP2D(bev->y - vf->x->max_bounds.ascent,view);
		};

	recalc_text_box(view,mob,FALSE,FALSE); /* also makes ellipse */ 
	view->doc->o = add_object(view->doc->o, &view->doc->lo, mob);
	register_undo(UNDO_PASTE,mob,view->doc); 
	view->text_object = mob;
	send_redraw_object(view,mob);
}

Object *text_object_at_pixel_point(View *view, int x, int y);

Object *
text_object_at_pixel_point(View *view, int x, int y)
{
	List l,l2;
	ulong depth=ULONG_MAX; 
	Object *foundob=NULL; 

	l = intersecting_objects_point(view->doc->o, XP2D(x,view), YP2D(y,view), NULL);
	l2 = l; 
	while (l2!=NULL)
		{
		if (OB(l2)->type==TEXT && OB(l2)->depth<=depth)
			{
			depth = OB(l2)->depth;
			foundob = OB(l2);
			};
		l2=l2->next;
		};

	delete_list(l); 
	
	return foundob;
}

void 
text_button(BEvent *bev, View *view)
{
	 
	if (bev->button==Button1 && (bev->type==BUTTON_RELEASED || bev->type==BUTTON_CLICKED) && 
	   (!state.editing_text || editview!=view))
		{
		Object *fob;

		fob = text_object_at_pixel_point(view,bev->x,bev->y);

		if (view!=editview)
			{ 
			if (editview!=NULL) 
				unselect_text(editview);
			editview = view;
			};

		if (fob)
			{
			mob = fob;
			state.editing_text = TRUE; 
			editview = view; 
			view->text_object = mob; 
			place_cursor(view,XP2D(bev->x,view)-mob->ob.text.bbox.x1,YP2D(bev->y,view)-mob->ob.text.bbox.y1);
			/* ensure the cursor's there  */  
			if (cl==NULL)
				{
				cl = mob->ob.text.lines;
				cs = TEXTLINE(cl)->sections;
				charpos = 0;
				cx = cy = 0; 
				}; 
			send_redraw_object(view,mob); 
			} 
		else
			{ 
			new_text(bev,view); 
			editview=view;
			};
		} 
	else if (state.editing_text && (bev->type==BUTTON_RELEASED || bev->type==BUTTON_CLICKED)
				&& bev->button!=Button2) 
		{
		if (!is_in_bbox(bev->x, bev->y, XD2P(mob->ob.text.bbox.x1,view),
			 YD2P(mob->ob.text.bbox.y1,view), XD2P(mob->ob.text.bbox.x2,view), YD2P(mob->ob.text.bbox.y2,view)))
			{
			Object *fob;
			 
			unselect_text(view); 
			if (state.current_icon==ARCICON)
				unselect_arc(view);

			fob = text_object_at_pixel_point(view,bev->x,bev->y);
			
			if (fob)
				{
				mob = fob;
				state.editing_text = TRUE; 
				view->text_object = mob; 
				place_cursor(view,XP2D(bev->x,view)-mob->ob.text.bbox.x1,YP2D(bev->y,view)-mob->ob.text.bbox.y1);
				/* ensure the cursor's there  */  
				if (cl==NULL)
					{
					cl = mob->ob.text.lines;
					cs = TEXTLINE(cl)->sections;
					charpos = 0;
					cx = cy = 0; 
					}; 
				send_redraw_object(view,mob); 
				} 
			else if (state.current_icon==TEXTICON && bev->button!=Button3)
				new_text(bev,view);
			}
		else
			{ 
			place_cursor(view,XP2D(bev->x,view)-mob->ob.text.bbox.x1,YP2D(bev->y,view)-mob->ob.text.bbox.y1);
			/* ensure the cursor's there  */  
			if (cl==NULL)
				{
				cl = mob->ob.text.lines;
				cs = TEXTLINE(cl)->sections;
				charpos = 0;
				cx = cy = 0; 
				}; 
			send_redraw_object(view,mob); 
			}; 
		};
}

/* called by zoom.c - we need accurate metrics on font
	at current zoom. This doesn't affect the stored text.bbox size
	(zoom==FALSE) */  
void 
recalc_text_boxes(View *view)
{
	List l;

	l = view->doc->lo;
	
	while (l!=NULL)
		{
		if (OB(l)->type==TEXT)
			recalc_text_box(view,OB(l),TRUE,FALSE);
		l = l->next;
		};
}
 
/* recalculate the size of the ob->ob.text.bbox  */  
void
recalc_text_box(View *view, Object *ob, Boolean zoom, Boolean load)
{
	List l;
	List l2;
	long x=0,y=0,dx=0,ox=0,oy=0;
	long w=0;
	long maxw=0,maxh=0;
	VFont *vf; 

	if (ob->ob.text.angle!=0.0)
		{
		double sina,cosa;
		long ccx,ccy;
		sina = sinround(ob->ob.text.angle);
		cosa = cosround(ob->ob.text.angle);
		ccx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2 + ob->ob.text.bbox.x1;
		ccy = (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2 + ob->ob.text.bbox.y1;
		ox = ccx + (cosa*(ob->ob.text.bbox.x1-ccx))-(sina*(ob->ob.text.bbox.y1-ccy));
		oy = ccy + (sina*(ob->ob.text.bbox.x1-ccx))+(cosa*(ob->ob.text.bbox.y1-ccy));
		};

	l = ob->ob.text.lines;

	while (l!=NULL)
		{
		x=0;w=0;maxh=0; 

		l2 = TEXTLINE(l)->sections;
		TEXTLINE(l)->y = y; 

		while (l2!=NULL)
			{
			vf = get_font(TEXTSEC(l2)->num, ZPO(TEXTSEC(l2)->size,view));
			if (vf!=NULL)
				{
				TEXTSEC(l2)->x = w;
				TEXTSEC(l2)->w = P2D(XTextWidth(vf->x, TEXTSEC(l2)->text, (int)strlen(TEXTSEC(l2)->text)),view);
				w += TEXTSEC(l2)->w;
				if (strlen(TEXTSEC(l2)->text)) 
					maxh=max(maxh,P2D(vf->x->max_bounds.ascent+vf->x->max_bounds.descent,view)); 
				}
			else
				{
				TEXTSEC(l2)->x = w;
				w += TEXTSEC(l2)->w;
				};
			l2 = l2->next; 
			};
			 
		TEXTLINE(l)->w = w;
		TEXTLINE(l)->h = maxh;
		y += TEXTLINE(l)->h;
		maxw = max(maxw,w);
		 
		l = l->next; 
		};
		
	if (!zoom)
		{
		/* calculate lowerright corner  */  
		vf = get_font(view->fontnum, ZPO(view->fontsize,view));
		ob->ob.text.bbox.x2 = ob->ob.text.bbox.x1 + max(P2D(3,view),maxw)+5;
		if (vf!=NULL)
			{ 
			if (y==0)
				{ 
				if (TEXTLINE(ob->ob.text.lines)->sections!=NULL &&
				    !strlen(TEXTSEC(TEXTLINE(ob->ob.text.lines)->sections)->text)) 
					{ 
					TextSection *tts=TEXTSEC(TEXTLINE(ob->ob.text.lines)->sections);
					vf = get_font(tts->num, ZPO(tts->size,view));
					}; 
				ob->ob.text.bbox.y2 = ob->ob.text.bbox.y1 + 
				                      P2D(vf->x->max_bounds.ascent+vf->x->max_bounds.descent,view);
				} 
			else 
				ob->ob.text.bbox.y2 = ob->ob.text.bbox.y1 + y;
			}
		else
			ob->ob.text.bbox.y2 = ob->ob.text.bbox.y1 + y;
		};
	
	cx = 0; 
	if (cs!=NULL)
		{
		vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
		if (vf!=NULL) 
			cx = TEXTSEC(cs)->x + P2D(XTextWidth(vf->x,TEXTSEC(cs)->text,charpos),view);    
		};

	/* handle justification  */  
	l = ob->ob.text.lines;
	while (l!=NULL)
		{
		if (state.editing_text && mob==ob && cl==l)
			cy = TEXTLINE(l)->y;
		
		switch (TEXTLINE(l)->just)
			{
			case LEFT:
				dx = 0; 
				break;
			case CENTRE:
				dx = ((ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)-TEXTLINE(l)->w)/2;
				break;
			case RIGHT:
				dx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)-TEXTLINE(l)->w;
				break;
			};

		l2 = TEXTLINE(l)->sections;
		while (l2!=NULL)
			{
			TEXTSEC(l2)->x += dx; 
			if (l2==cs)
				cx += dx;
			l2 = l2->next;
			};

		l = l->next;
		};
		 
	if (!zoom)
		{
		if (ob->ob.text.node)
			{
			Object *e=ob->ob.text.ellipse;
			/* create/update ellipse  */ 
			if (ob->ob.text.ellipse==NULL)
				{
				ob->ob.text.ellipse = (Object *)malloc(sizeof(Object));
				e=ob->ob.text.ellipse;
				e->type = ELLIPSE;
				e->ticket = ob_ticket++;
				e->derries = NULL;
				e->depth = ob->depth+1;
				e->ls = view->linestyle;
				e->es = view->endstyle; 
				e->js = JoinRound; 
				e->lw = view->linewidth;
				e->colour = view->colour;
				e->fillcolour = view->fillcolour;
				e->farrow = NULL;
				e->barrow = NULL;
				if (view->fillon)
					e->fs = view->fillstyle;
				else
					e->fs = NONE;

				e->dash = 5;
				e->gap = 4;
				};

			ob->bbox.x1 = ob->ob.text.bbox.x1 - (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2;
			ob->bbox.y1 = ob->ob.text.bbox.y1 - (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2;
			ob->bbox.x2 = ob->ob.text.bbox.x1 + 3*(ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2;
			ob->bbox.y2 = ob->ob.text.bbox.y1 + 3*(ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2;
			e->ob.ellipse.centre.x = (ob->bbox.x2-ob->bbox.x1)/2;
			e->ob.ellipse.centre.y = (ob->bbox.y2-ob->bbox.y1)/2;
			e->ob.ellipse.xradius = (ob->bbox.x2-ob->bbox.x1)/2;
			e->ob.ellipse.yradius = (ob->bbox.y2-ob->bbox.y1)/2;
			}
		else
			{
			if (load || ob->ob.text.angle==0.0)
				calc_text_outer_box_adjust(ob,0,0); 
			else
				{
				double sina,cosa;
				long ccx,ccy;
				long ddx,ddy; 
				sina = sinround(ob->ob.text.angle);
				cosa = cosround(ob->ob.text.angle);
				ccx = (ob->ob.text.bbox.x2-ob->ob.text.bbox.x1)/2 + ob->ob.text.bbox.x1;
				ccy = (ob->ob.text.bbox.y2-ob->ob.text.bbox.y1)/2 + ob->ob.text.bbox.y1;
				ddx = ccx + ((long)((cosa*(ob->ob.text.bbox.x1-ccx))-(sina*(ob->ob.text.bbox.y1-ccy)))) - ox;
				ddy = ccy + ((long)((sina*(ob->ob.text.bbox.x1-ccx))+(cosa*(ob->ob.text.bbox.y1-ccy)))) - oy; 
				calc_text_outer_box_adjust(ob, ddx, ddy);
				};
			};

		resize_bbox(view->doc->o,ob);
		};
}

/* insert a char into a string str  */  
void 
insert_char(char *str, char c, int pos)
{
	char a[20]="";

	strncpy(a,str,(uint)pos);
	a[strlen(a)+1]='\0';
	a[strlen(a)]=c;
	strcat(a,&str[pos]);
	strcpy(str,a);
}

/* create a new text section  */ 
/* text is split into TextLines; each
	Line has one or more TextSections with
	different fonts and text */ 
TextSection *
new_section(View *view)
{
	TextSection *ts;

	ts = (TextSection *)malloc(sizeof(TextSection));

	ts->isReturn = FALSE;
	ts->num = view->fontnum;
	ts->size = view->fontsize;
	ts->x = cx;
	ts->w = 0;
	strcpy(ts->text,"");
	return ts;
}

/* add a char at the current position  */  
/* must take care in end-of-line situations
	and not to overflow one textsection string buffer */  
void 
add_char(View *view, char *c)
{
	TextSection *ts; 

	if (cs==NULL)
		{ 
		/* only when no sections on line */  
		ts = new_section(view); 
		TEXTLINE(cl)->sections = add_to_list(TEXTLINE(cl)->sections,0,0,(void *)ts); 
		cs = TEXTLINE(cl)->sections; 
		}; 
	
	if ((view->fontnum!=TEXTSEC(cs)->num || view->fontsize!=TEXTSEC(cs)->size)
		 && charpos==(int)strlen(TEXTSEC(cs)->text) && (cs->next==NULL || streq(TEXTSEC(cs->next)->text,"")))
		{
		if (charpos==0)
			{ 
			send_redraw_object(view,mob); 
			TEXTSEC(cs)->size = view->fontsize;
			TEXTSEC(cs)->num = view->fontnum;
			}
		else
			{
			ts = new_section(view); 
			if (cs->next!=NULL)
				{
				ts->size = TEXTSEC(cs)->size;
				ts->num = TEXTSEC(cs)->num;
				};
			TEXTLINE(cl)->sections = insert_after(TEXTLINE(cl)->sections,cs,0,0,(void *)ts, &cs);
			}; 
		insert_char(TEXTSEC(cs)->text,*c,0); 
		charpos = 1;
		}
	else if (strlen(TEXTSEC(cs)->text)<19)
		{ 
		insert_char(TEXTSEC(cs)->text,*c,charpos);
		charpos++;
		}
	else
		{
		/* doesn't fit in current textsection, create new */ 
		if (charpos == 19)
			{
			/* at end */ 
			ts = new_section(view);
			if (cs->next!=NULL)
				{
				ts->size = TEXTSEC(cs)->size;
				ts->num = TEXTSEC(cs)->num;
				};
			TEXTLINE(cl)->sections = insert_after(TEXTLINE(cl)->sections,cs,0,0,(void *)ts, &cs);
			charpos = 0;
			insert_char(TEXTSEC(cs)->text,*c,charpos++);
			}
		else if (charpos == 0)
			{
			/* before cs */ 
			ts = new_section(view);
			if (cs->next!=NULL)
				{
				ts->size = TEXTSEC(cs)->size;
				ts->num = TEXTSEC(cs)->num;
				};
			TEXTLINE(cl)->sections = insert_before(TEXTLINE(cl)->sections,cs,0,0,(void *)ts, &cs);
			charpos = 0;
			insert_char(TEXTSEC(cs)->text,*c,charpos++);
			}
		else
			{
			/* inbetween must split */ 
			ts = new_section(view);
			if (cs->next!=NULL)
				{
				ts->size = TEXTSEC(cs)->size;
				ts->num = TEXTSEC(cs)->num;
				};
			strcpy(ts->text,&TEXTSEC(cs)->text[charpos]);
			TEXTSEC(cs)->text[charpos] = '\0';
			TEXTLINE(cl)->sections = insert_after(TEXTLINE(cl)->sections,cs,0,0,(void *)ts, &cs);
			insert_char(TEXTSEC(cs)->text,*c,0); 
			charpos = 1;
			};
		};
}

/* handle a keypress  */  
void 
text_key(View *view, XKeyPressedEvent *report)
{
	KeySym keysym;
	char str[50]="";
	int num=0; 
	VFont *vf; 
	 
	num = XLookupString(report,str,sizeof(str), &keysym, NULL);
	str[num]='\0'; 

	if (mob==NULL)
		return;

	if (cs!=NULL)
		vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
	else
		vf = get_font(view->fontnum, ZPO(view->fontsize,view));
	 
	if (vf==NULL)
		return;

	switch (keysym)
		{
		/* handle creation of a new line  */  
		case XK_Return:
		case XK_KP_Enter: 
			if (cs==NULL || charpos==(int)strlen(TEXTSEC(cs)->text))
				{
				TextLine *tl; 
				TextSection *ts;

				tl = (TextLine *)malloc(sizeof(TextLine)); 
				tl->just = view->justification; 
				mob->ob.text.lines = insert_after(mob->ob.text.lines,cl,0,0,(void *)tl, &cl);
				ts = new_section(view); 
				if (cs==NULL)
					{ 
					TEXTLINE(cl)->sections = add_to_list(NULL, 0, 0, (void *)ts); 
					cs=TEXTLINE(cl)->sections;
					} 
				else	 
					{ 
					ts->size = TEXTSEC(cs)->size;
					ts->num = TEXTSEC(cs)->num;
					TEXTLINE(cl)->sections = insert_after(NULL,NULL,0,0,(void *)ts, &cs); 
					}; 

				charpos = 0; 
				store_redraw_object(mob); 
				recalc_text_box(view,mob,FALSE,FALSE); 
				view->doc->o = change_bbox(view->doc->o, mob);
				send_stored_redraw_object(view,mob); 
				}
			else
				{
				/* must split */  
				TextLine *tl; 
				TextSection *ts;
				ListEl *le; 
				VRegion bb; 
				 
				store_redraw_object(mob); 
				tl = (TextLine *)malloc(sizeof(TextLine)); 
				tl->just = view->justification; 
				mob->ob.text.lines = insert_after(mob->ob.text.lines,cl,0,0,(void *)tl, &cl);
				ts = new_section(view);
				ts->size = TEXTSEC(cs)->size;
				ts->num = TEXTSEC(cs)->num;
				strcpy(ts->text,&TEXTSEC(cs)->text[charpos]);
				TEXTSEC(cs)->text[charpos] = '\0';
				le = cs->next; 
				cs->next = NULL;
				TEXTLINE(cl)->sections = insert_after(NULL,NULL,0,0,(void *)ts, &cs);
				cs->prev = NULL;
				cs->next = le; 
				if (le!=NULL)
					le->prev = cs;
				charpos = 0;
				bb = mob->bbox;
					recalc_text_box(view,mob,FALSE,FALSE); 
				view->doc->o = change_bbox(view->doc->o ,mob);
				send_stored_redraw_object(view,mob); 
				};
			break;

		case XK_Left:
			if (cs!=NULL)
				{
				if (charpos>0) 
					{
					charpos--;
					vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
					cx = TEXTSEC(cs)->x + P2D(XTextWidth(vf->x, TEXTSEC(cs)->text, charpos),view);
					send_redraw_object(view,mob); 
					}
				else if (cs->prev!=NULL)
					{
					cs = cs->prev; 
					charpos = max(0,((int)strlen(TEXTSEC(cs)->text))-1);
					vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
					cx = TEXTSEC(cs)->x + P2D(XTextWidth(vf->x, TEXTSEC(cs)->text, charpos),view);
					send_redraw_object(view,mob); 
					}
				else if (cl->prev!=NULL)
					{
					List l,pl=NULL;

					cl = cl->prev;
					l = TEXTLINE(cl)->sections;

					while (l!=NULL)
						{
						pl = l;
						l = l->next;
						}; 
					
					cs = pl;
					charpos = strlen(TEXTSEC(cs)->text); 
					vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
					cx = TEXTSEC(cs)->x + P2D(XTextWidth(vf->x, TEXTSEC(cs)->text, charpos),view);
					cy = TEXTLINE(cl)->y;
					send_redraw_object(view,mob); 
					}; 
				}; 
			break;

		case XK_Right:
			if (cs!=NULL)
				{
				if (charpos<19 && (int)strlen(TEXTSEC(cs)->text)>charpos)
					{
					charpos++;
					vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
					cx = TEXTSEC(cs)->x + P2D(XTextWidth(vf->x, TEXTSEC(cs)->text, charpos),view);
					send_redraw_object(view,mob); 
					}
				else if (cs->next!=NULL)
					{
					cs = cs->next;
					charpos = min(strlen(TEXTSEC(cs)->text),1);
					vf = get_font(TEXTSEC(cs)->num, ZPO(TEXTSEC(cs)->size,view));
					cx = TEXTSEC(cs)->x + P2D(XTextWidth(vf->x,TEXTSEC(cs)->text,charpos),view);
					send_redraw_object(view,mob); 
					}
				else if (cl->next!=NULL)
					{
					cl = cl->next;
					cs = TEXTLINE(cl)->sections;

					charpos = 0;
					cx = TEXTSEC(cs)->x;
					cy = TEXTLINE(cl)->y;
					send_redraw_object(view,mob); 
					}; 
				}; 
			break;

		case XK_Up:
			if (cl->prev!=NULL)
				{
				if (mob->ob.text.angle!=0.0)
					{
					double sina,cosa;
					long crx,cry;
					long tx,ty; 
					sina = sinround(mob->ob.text.angle);
					cosa = cosround(mob->ob.text.angle);
					crx = (mob->ob.text.bbox.x2-mob->ob.text.bbox.x1)/2; 
					cry = (mob->ob.text.bbox.y2-mob->ob.text.bbox.y1)/2; 
					tx = crx + (cosa*(cx-crx))-(sina*(TEXTLINE(cl->prev)->y+3-cry)); 
					ty = cry + (sina*(cx-crx))+(cosa*(TEXTLINE(cl->prev)->y+3-cry)); 
					place_cursor(view,tx,ty);
					}
				else
					place_cursor(view,cx,TEXTLINE(cl->prev)->y+3);
				send_redraw_object(view,mob); 
				}; 
			break;

		case XK_Down:
			if (cl->next!=NULL)
				{
				if (mob->ob.text.angle!=0.0)
					{
					double sina,cosa;
					long crx,cry;
					long tx,ty; 
					sina = sinround(mob->ob.text.angle);
					cosa = cosround(mob->ob.text.angle);
					crx = (mob->ob.text.bbox.x2-mob->ob.text.bbox.x1)/2; 
					cry = (mob->ob.text.bbox.y2-mob->ob.text.bbox.y1)/2; 
					tx = crx + (cosa*(cx-crx))-(sina*(TEXTLINE(cl->next)->y+3-cry)); 
					ty = cry + (sina*(cx-crx))+(cosa*(TEXTLINE(cl->next)->y+3-cry)); 
					place_cursor(view,tx,ty);
					}
				else
					place_cursor(view,cx,TEXTLINE(cl->next)->y+3);
				send_redraw_object(view,mob); 
				}; 
			break;
		
		case XK_BackSpace:
			if (cs!=NULL)
				{
				if (charpos>0) 
					{
					VRegion bb;

					store_redraw_object(mob); 
					strcpy(&TEXTSEC(cs)->text[charpos-1],&TEXTSEC(cs)->text[charpos]);
					charpos--;
					bb = mob->bbox;
					recalc_text_box(view,mob,FALSE,FALSE);
					view->doc->o = change_bbox(view->doc->o ,mob);
					send_stored_redraw_object(view,mob);
					} 
				else if (cs->prev!=NULL)
					{
					VRegion bb;
					int i; 

					store_redraw_object(mob); 
					cs = cs->prev;
					bb = mob->bbox;
					i = strlen(TEXTSEC(cs)->text)-1;
					charpos = max(i,0);
					TEXTSEC(cs)->text[charpos]='\0';
					recalc_text_box(view,mob,FALSE,FALSE); 
					view->doc->o = change_bbox(view->doc->o ,mob);
					send_stored_redraw_object(view,mob); 
					}
				else if (cl->prev!=NULL)
					{
					VRegion bb;
					List l;
					List l2; 
					
					store_redraw_object(mob); 
					bb = mob->bbox;
					l = TEXTLINE(cl->prev)->sections;
					while (l->next!=NULL)
						l = l->next;

					if (l==TEXTLINE(cl->prev)->sections && strlen(TEXTSEC(l)->text)==0)
						{
						/* nothing on previous line */  
						free(TEXTSEC(l));
						free(l);
						TEXTLINE(cl->prev)->sections = cs;
						cs->prev = NULL;
						if (cl->next!=NULL)
							cl->next->prev = cl->prev;
						cl->prev->next = cl->next;
						l2 = cl; 
						cl = cl->prev; 
						free(TEXTLINE(l2)); 
						free(l2); 
						}
					else
						{
						/* something there */ 
						l->next = cs;
						cs->prev = l;
						if (cl->next!=NULL)
							cl->next->prev = cl->prev;
						cl->prev->next = cl->next;
						l2 = cl;
						cl = cl->prev;
						free(TEXTLINE(l2)); 
						free(l2);
						}; 
						
					charpos = 0;
					recalc_text_box(view,mob,FALSE,FALSE); 
					cx = TEXTSEC(cs)->x;
					cy = TEXTLINE(cl)->y;
					view->doc->o = change_bbox(view->doc->o ,mob);
					send_stored_redraw_object(view,mob); 
					};
				}; 
			break;

		case XK_Delete:
			if (cs!=NULL)
				{
				if (charpos<19 && (int)strlen(TEXTSEC(cs)->text)>charpos)
					{
					VRegion bb;

					store_redraw_object(mob); 
					strcpy(&TEXTSEC(cs)->text[charpos],&TEXTSEC(cs)->text[charpos+1]);
					bb = mob->bbox;
					recalc_text_box(view,mob,FALSE,FALSE); 
					view->doc->o = change_bbox(view->doc->o ,mob);
					send_stored_redraw_object(view,mob); 
					} 
				else if (cs->next!=NULL)
					{
					VRegion bb;

					store_redraw_object(mob); 
					cs = cs->next;
					charpos = 0;
					strcpy(TEXTSEC(cs)->text,TEXTSEC(cs)->text+1);
					bb = mob->bbox;
					recalc_text_box(view,mob,FALSE,FALSE); 
					view->doc->o = change_bbox(view->doc->o ,mob);
					send_stored_redraw_object(view,mob); 
					}
				else if (cl->next!=NULL)
					{
					VRegion bb;
					List l;
					List l2; 
					
					bb = mob->bbox;
					store_redraw_object(mob); 
					l = TEXTLINE(cl->next)->sections;

					if (strlen(TEXTSEC(l)->text)==0 && l->next==NULL)
						{
						/* nothing on next line */  
						free(TEXTSEC(l));
						free(l);
						l2 = cl->next;
						if (cl->next->next!=NULL)
							cl->next->next->prev = cl;
						cl->next = cl->next->next;
						free (TEXTLINE(l2)); 
						free (l2);
						}
					else
						{
						/* something there */ 
						if (strlen(TEXTSEC(cs)->text)!=0)
							{
							cs->next = l; 
							l->prev = cs;
							l2 = cl->next;
							if (cl->next->next!=NULL)
								cl->next->next->prev = cl;
							cl->next = cl->next->next;
							free (TEXTLINE(l2));
							free (l2);
							} 
						else
							{
							TEXTLINE(cl)->sections = l;
							free (TEXTSEC(cs));
							free (cs);
							cs = l;
							l2 = cl->next;
							if (cl->next->next!=NULL)
								cl->next->next->prev = cl;
							cl->next = cl->next->next;
							free (TEXTLINE(l2));
							free (l2);
							};
						};
						
					recalc_text_box(view,mob,FALSE,FALSE); 
					view->doc->o = change_bbox(view->doc->o ,mob);
					send_stored_redraw_object(view,mob); 
					}; 
				}
			break;

		default:
			if (!streq(str,"") && isprint(*str))
				{
				add_char(view, str);
				recalc_text_box(view,mob,FALSE,FALSE); 
				view->doc->o = change_bbox(view->doc->o ,mob);
				send_redraw_object(view,mob); 
				};
			break;
		};

}

/* draw text  */  
void 
draw_text(Object *ob, View *view, GC gc, long x, long y, double rx, double ry, double angle)
{
	Boolean clipped;
	List l,l2; 
	VLine line; VLine *pl;
	VFont *vf; 
	Drawable drawable; 
	/* init to zero to appease -Wall  */  
	GC tempgc=0; 
	uint pixw=0,pixh=0; 
	XImage *xi; 
	long dx,dy; 
	double sina,cosa;
	long x1,y1,x2,y2; 
	Boolean rotcursor=FALSE; 

	if (ob->ob.text.node)
		{
		/* draw the ellipse  */  
		gc_lw(view,ugc, ob->ob.text.ellipse); 
		gc_line(ugc, ob->ob.text.ellipse);
		gc_fill(fillgc, ob->ob.text.ellipse);
		gc_colour(view,ugc, ob->ob.text.ellipse);
		gc_fillcolour(view,fillgc, ob->ob.text.ellipse);           
 		draw_object(ob->ob.text.ellipse,view,gc,x,y,rx,ry,0.0,0,0,0,0);
		/* set our text colour back */ 
		gc_colour(view,ugc, ob);
		};
		
	/* calc pixel co-ords of text.bbox  */ 
	x1 = XD2P(x,view) + D2P(ob->ob.text.bbox.x1-ob->bbox.x1,view);
	y1 = YD2P(y,view) + D2P(ob->ob.text.bbox.y1-ob->bbox.y1,view);
	x2 = XD2P(x,view) + R(D2P((ob->ob.text.bbox.x2 - ob->ob.text.bbox.x1) + 
				ob->ob.text.bbox.x1-ob->bbox.x1,view),rx); 
	y2 = YD2P(y,view) + R(D2P((ob->ob.text.bbox.y2 - ob->ob.text.bbox.y1) + 
				ob->ob.text.bbox.y1-ob->bbox.y1,view),ry); 
	 
	if ((ob->ob.text.angle+angle)!=0.0 && (rx!=1.0 || ry!=1.0))
		{
		/* this is just lazy, need to fix the positioning properly before drawing
			an accurate box around rottext */  
		
		x1 = XD2P(x,view);
		y1 = YD2P(y,view);
		x2 = XD2P(x,view) + R(D2P(ob->bbox.x2 - ob->bbox.x1,view),rx); 
		y2 = YD2P(y,view) + R(D2P(ob->bbox.y2 - ob->bbox.y1,view),ry); 
		line.x1 = x1; line.y1 = y1; line.x2 = x2; line.y2 = y1; 
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = x2; line.y1 = y1; line.x2 = x2; line.y2 = y2; 
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = x2; line.y1 = y2; line.x2 = x1; line.y2 = y2; 
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = x1; line.y1 = y2; line.x2 = x1; line.y2 = y1; 
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, gc, pl->x1, pl->y1, pl->x2, pl->y2);
		}
	else if ((ob->ob.text.angle+angle)!=0.0 && (gc==blackxorgc || (state.editing_text && view->text_object==ob)))
		{
		long ccx,ccy; 
		GC bgc; 

		/* draw rotated box of text  */  
		 
		if (gc!=blackxorgc)
			bgc = dashgc; 
		else
			bgc = blackxorgc;
			 
		sina = sinround(ob->ob.text.angle+angle);
		cosa = cosround(ob->ob.text.angle+angle);
		 
		ccx = XD2P(x+(ob->bbox.x2-ob->bbox.x1)/2,view); 
		ccy = YD2P(y+(ob->bbox.y2-ob->bbox.y1)/2,view); 
		line.x1 = ccx + (cosa*(x1-ccx))-(sina*(y1-ccy));
		line.y1 = ccy + (sina*(x1-ccx))+(cosa*(y1-ccy));
		line.x2 = ccx + (cosa*(x2-ccx))-(sina*(y1-ccy));
		line.y2 = ccy + (sina*(x2-ccx))+(cosa*(y1-ccy));
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = ccx + (cosa*(x2-ccx))-(sina*(y1-ccy));
		line.y1 = ccy + (sina*(x2-ccx))+(cosa*(y1-ccy));
		line.x2 = ccx + (cosa*(x2-ccx))-(sina*(y2-ccy));
		line.y2 = ccy + (sina*(x2-ccx))+(cosa*(y2-ccy));
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = ccx + (cosa*(x2-ccx))-(sina*(y2-ccy));
		line.y1 = ccy + (sina*(x2-ccx))+(cosa*(y2-ccy));
		line.x2 = ccx + (cosa*(x1-ccx))-(sina*(y2-ccy));
		line.y2 = ccy + (sina*(x1-ccx))+(cosa*(y2-ccy));
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = ccx + (cosa*(x1-ccx))-(sina*(y2-ccy));
		line.y1 = ccy + (sina*(x1-ccx))+(cosa*(y2-ccy));
		line.x2 = ccx + (cosa*(x1-ccx))-(sina*(y1-ccy));
		line.y2 = ccy + (sina*(x1-ccx))+(cosa*(y1-ccy));
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		}
	else if ((ob->ob.text.angle+angle)==0.0 && (gc==blackxorgc || (state.editing_text && view->text_object==ob)))
		{
		GC bgc; 

		/* draw straight bbox of text  */ 
		 
		if (gc!=blackxorgc)
			bgc = dashgc; 
		else
			bgc = blackxorgc;
			 
		line.x1 = x1;
		line.y1 = y1;
		line.x2 = x2;
		line.y2 = y1;
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = x2;
		line.y1 = y1;
		line.x2 = x2;
		line.y2 = y2;
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = x2;
		line.y1 = y2;
		line.x2 = x1;
		line.y2 = y2;
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		line.x1 = x1;
		line.y1 = y2;
		line.x2 = x1;
		line.y2 = y1;
		pl = clip_line(0,0,(long)view->draw_window->w,(long)view->draw_window->h, &line, &clipped, 2);
		if (pl!=NULL)
			XDrawLine(display,view->draw_window->win, bgc, pl->x1, pl->y1, pl->x2, pl->y2);
		};

	if (gc==blackxorgc)
		return;

	l = ob->ob.text.lines;

	if (ob->ob.text.angle!=0.0)
		{
		XGCValues gcv; 
		 
		/* rotated. create a pixmap of text.bbox size ready for drawing onto */
			 
		pixw = D2P(ob->ob.text.bbox.x2-ob->ob.text.bbox.x1,view); 
		pixh = D2P(ob->ob.text.bbox.y2-ob->ob.text.bbox.y1,view);
			 
		if (pixw<1 || pixh <1)
			return; 
				 
		drawable = XCreatePixmap(display,view->draw_window->win, pixw, pixh, 1);
		gcv.foreground = WhitePixel(display,screen);
		tempgc = XCreateGC(display,drawable,GCForeground,&gcv); 
 		XFillRectangle(display,drawable,tempgc,0,0,pixw,pixh);
		gc = tempgc;
		gcv.foreground = BlackPixel(display,screen);
 		XChangeGC(display,tempgc,GCForeground,&gcv);
		dx = 0;
		dy = 0;
		} 
	else
		{ 
		/* unrotated, draw text straight to view window  */  
	
		drawable = view->draw_window->win;
		gc = ugc; 
		dx = ob->ob.text.bbox.x1; 
		dy = ob->ob.text.bbox.y1; 
		}; 

	while (l!=NULL)
		{
		l2 = TEXTLINE(l)->sections;

		while (l2!=NULL)
			{
			/* get the font, set it, and draw the text  */  
			vf = get_font(TEXTSEC(l2)->num, ZPO(TEXTSEC(l2)->size,view));
			if (vf!=NULL)
				{
				XSetFont(display, gc, vf->x->fid); 
				if (ob->ob.text.angle!=0.0)
					XDrawString(display,drawable, gc, D2P(TEXTSEC(l2)->x,view),
						D2P(TEXTLINE(l)->y,view)+vf->x->max_bounds.ascent, 
						TEXTSEC(l2)->text, (int)strlen(TEXTSEC(l2)->text));
				else
					XDrawString(display,drawable, gc, XD2P(dx+TEXTSEC(l2)->x,view),
						YD2P(dy+TEXTLINE(l)->y,view)+vf->x->max_bounds.ascent, 
						TEXTSEC(l2)->text, (int)strlen(TEXTSEC(l2)->text));
					
				if (l==cl && l2==cs)
					{
					/* cursor is here, set the values and draw if not rotated  */  
					line.x1 = XD2P(cx+ob->ob.text.bbox.x1,view);
					line.y1 = YD2P(TEXTLINE(l)->y+ob->ob.text.bbox.y1,view); 
					line.x2 = XD2P(cx+ob->ob.text.bbox.x1,view);
					line.y2 = YD2P(TEXTLINE(l)->y+TEXTLINE(cl)->h+ob->ob.text.bbox.y1,view);
					if (ob->ob.text.angle==0.0)
						{
						pl = clip_line(0,0, (long)view->draw_window->w,(long)view->draw_window->h, 
							&line, &clipped, 2);
							 
						if (pl!=NULL)
							XDrawLine(display,view->draw_window->win, 
								handlegc, pl->x1, pl->y1, pl->x2, pl->y2);
						}
					else
						rotcursor = TRUE; 
					}; 
				}	
			else
				{
				uint w;
				uint h; 

				/* we can't get font from server, do very poor 
					estimate and greek in solid colour  */  
				if (l2->next==NULL)
					w = D2P(ob->bbox.x2-(ob->bbox.x1+TEXTSEC(l2)->x),view); 
				else
					w = D2P(TEXTSEC(l2->next)->x - TEXTSEC(l2)->x,view);
					
				if (l->next==NULL)
					h = D2P(ob->bbox.y2-(ob->bbox.y1+TEXTLINE(l)->y),view);
				else
					h = D2P(TEXTLINE(l->next)->y - TEXTLINE(l)->y,view);

				XFillRectangle(display,drawable, gc, 
					XD2P(ob->ob.text.bbox.x1+TEXTSEC(l2)->x,view),
					YD2P(ob->ob.text.bbox.y1+TEXTLINE(l)->y,view), w, h);
				};
										
			l2=l2->next;
			}; 
		l = l->next;
		};
		
	if (ob->ob.text.angle!=0.0)
		{
		long px,py;
		long i,j; 
		long ccx,ccy; 
		/* now turn pixmap into ximage, and rotate point by point  */  
		/* slow but simple */
		sina = sinround(ob->ob.text.angle); 
		cosa = cosround(ob->ob.text.angle); 
		 
		xi = XGetImage(display,drawable,0,0,pixw,pixh,1,XYPixmap);
		ccx = pixw/2;
		ccy = pixh/2;
		for (i=0;i<(long)pixw;i++)
			{
			for (j=0;j<(long)pixh;j++)
				{ 
				if (!XGetPixel(xi,(int)i,(int)j))
					{ 
					px = ccx+(cosa*(i-ccx))-(sina*(j-ccy));
					py = ccy+(sina*(i-ccx))+(cosa*(j-ccy)); 
					XDrawPoint(display,view->draw_window->win,ugc,
						XD2P(ob->bbox.x1+(ob->ob.text.bbox.x1-ob->bbox.x1),view)+px,
						YD2P(ob->bbox.y1+(ob->ob.text.bbox.y1-ob->bbox.y1),view)+py);
					}; 
				};
			};
		XDestroyImage(xi); 
		XFreePixmap(display,drawable); 
		XFreeGC(display,tempgc);
			
		if (rotcursor)
			{
			long cenx,ceny,cx1,cy1,cx2,cy2;

			/* draw the rotated cursor  */  
			cenx = XD2P(ob->bbox.x1+(ob->bbox.x2-ob->bbox.x1)/2,view); 
			ceny = YD2P(ob->bbox.y1+(ob->bbox.y2-ob->bbox.y1)/2,view); 
			cx1 = cenx+(cosa*(line.x1-cenx))-(sina*(line.y1-ceny));
			cy1 = ceny+(sina*(line.x1-cenx))+(cosa*(line.y1-ceny)); 
			cx2 = cenx+(cosa*(line.x2-cenx))-(sina*(line.y2-ceny));
			cy2 = ceny+(sina*(line.x2-cenx))+(cosa*(line.y2-ceny)); 
			XDrawLine(display,view->draw_window->win,handlegc,cx1,cy1,cx2,cy2);
			};
		};
}
