/* undo.c  */ 
/* COPYRIGHT (C) 2000 THE VICTORIA UNIVERSITY OF MANCHESTER and John Levon
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details. 
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
/* handle undo feature  */  
/*
 * $Log: undo.c,v $
 * Revision 1.2  2000/12/06 20:56:06  moz
 * GPL stuff.
 *
 * Revision 1.1.1.1  2000/08/21 01:05:31  moz
 *
 *
 * Revision 1.1.1.1  2000/07/19 22:45:31  moz
 * CVS Import
 *
 * Revision 1.17  2000/02/18 01:18:55  moz
 * Free compound undos.
 * Free undos from dead documents.
 *
 * Revision 1.16  2000/02/17 22:15:27  moz
 * Clear up of code, stage 1.
 * Compound/uncompound still not done.
 *
 * Revision 1.15  2000/02/15 01:47:08  moz
 * Fix for some undo problems.
 *
 * Revision 1.14  2000/02/09 02:47:27  moz
 * Added debugging info.
 * kill the right object when falling off end of redo stack.
 * reset state when nulifying_sel_object.
 *
 * Revision 1.13  2000/01/25 23:47:41  moz
 * Fixes for free_list().
 * Allow undolevels==0.
 *
 * Revision 1.12  1999/11/15 02:01:28  moz
 * Name change.
 * Pick up default:s.
 *
 * Revision 1.11  1999/05/22 23:39:40  moz
 *  Pedantic ANSI.
 *
 * Revision 1.10  1999/05/19 17:06:48  moz
 * 1.0 Checkin.
 *
 * Revision 1.9  1999/05/02 22:53:17  moz
 * Commented out debugging messages.
 *
 * Revision 1.8  1999/05/02 22:49:14  moz
 * Handle UNDO_UNCOMPOUND
 *
 * Revision 1.7  1999/04/29 22:49:31  moz
 * Redo added.
 *
 * Revision 1.6  1999/04/29 22:00:21  moz
 * Rewritten to work with compound objects on list.
 *
 * Revision 1.5  1999/04/29 17:18:41  moz
 * Undo functions rewrite in preparation for redo.
 *
 * Revision 1.4  1999/04/27 07:39:33  moz
 * Various fixes to handle UNDO_COMPOUND
 *
 * Revision 1.3  1999/04/27 04:12:30  moz
 * Fixed bug in push_undo.
 *
 * Revision 1.2  1999/04/22 22:17:35  moz
 * Add facility to undo compound makes.
 *
 * Revision 1.1  1999/03/30 00:07:00  moz
 * Initial revision
 *
 */    

#include "include/figurine.h"
#include "include/extern.h"

/* #define UNDO_DEBUG */


/* 0 == undostack, 1 == redostack */  
void kill_undo(Undo *u, int redo)
{
	if (!redo)
		{
		switch (u->type)
			{
			case UNDO_OB_PROP:
#ifdef UNDO_DEBUG
				fprintf(stderr,"kill undo OB_PROP ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				kill_object(u->ob); 
				break;
			
			case UNDO_CUT:
#ifdef UNDO_DEBUG
				fprintf(stderr,"kill undo CUT ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				kill_object(u->ob); 
				break;
		
			case UNDO_PASTE:
#ifdef UNDO_DEBUG
				fprintf(stderr,"kill undo PASTE ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				/* realob was in picture so has been killed already */  
				break;
		
			case UNDO_COMPOUND:
#ifdef UNDO_DEBUG
				fprintf(stderr,"kill undo COMPOUND ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				break;
			
			case UNDO_UNCOMPOUND:
#ifdef UNDO_DEBUG
				fprintf(stderr,"kill undo UNCOMPOUND ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				/* members may still exist, should only free this object */  
				delete_list(u->real_ob->ob.compound.obs); 
				free(u->real_ob); 
				break;
			
			default:
				v_error(u->type);
				break;
			};
		}
	else
		{
		switch (u->type)
			{
			case UNDO_OB_PROP:
#ifdef UNDO_DEBUG
				fprintf(stderr,"redo OB_PROP ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				kill_object(u->real_ob); 
				break;
			
			case UNDO_CUT:
#ifdef UNDO_DEBUG
				fprintf(stderr,"redo CUT ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				break;
			
			case UNDO_PASTE:
#ifdef UNDO_DEBUG
				fprintf(stderr,"redo PASTE ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				kill_object(u->real_ob); 
				break;
			
			case UNDO_COMPOUND:
#ifdef UNDO_DEBUG
				fprintf(stderr,"redo COMPOUND ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				/* members may still exist, should only free this object */  
				delete_list(u->real_ob->ob.compound.obs); 
				free(u->real_ob); 
				break;
			
			case UNDO_UNCOMPOUND:
#ifdef UNDO_DEBUG
				fprintf(stderr,"redo UNCOMPOUND ob %p rob %p\n",u->ob,u->real_ob); 
#endif
				break;
			
			default:
				v_error(u->type);
				break;
			};
		};
} 

/* pop top undo off  */  
Undo *
pop_undo(List *stack)
{
	Undo *u;
	List l;

	if (*stack==NULL)
 		return NULL;
	
	u = UNDO(*stack);
	l = *stack; 
	*stack = (*stack)->next;
	free(l);
	if (*stack!=NULL) 
		(*stack)->prev = NULL;
	return u; 
}

void clear_redo_stack()
{
	List l,l2;

	l = redostack;

	while (l!=NULL)
		{
		l2=l->next;
		kill_undo(UNDO(l),TRUE); 
		free(UNDO(l));
		free(l);
		l = l2;
		};
		 
	redostack = NULL; 
}

/* push undo onto stack  */  
/* if stack full, drop oldest undo  */  
List
push_undo(List stack, Undo *u)
{
	ListEl *le;
	List l; 
	List l2; 
	int c=0; 

	le = (ListEl *)malloc(sizeof(ListEl));
	if (le==NULL)
		return stack;

	le->data = (void *)u;
	le->prev = NULL;
	le->next = stack;
	if (stack!=NULL)
		stack->prev = le;
	
	l = stack;

	l2 = l;

	while (l!=NULL)
		{
		c++; 
		l2 = l; 
		l = l->next; 
		};
	
	if (c>undolevels-1)
		{
#ifdef UNDO_DEBUG
		fprintf(stderr,"Removing %d, ob %p, rob %p\n",UNDO(l2)->type,UNDO(l2)->ob,UNDO(l2)->real_ob); 
		if (stack==redostack)
			fprintf(stderr,"Trying to free redo !\n");
#endif
		if (l2->prev!=NULL)
			l2->prev->next = NULL;

		kill_undo(UNDO(l2),FALSE); 
		free(UNDO(l2));
		free(l2);
		};

	return le;
}

/* when we perform an undo, objects change  */  
/* this ensures all object refs are the correct ones  */  
void
replace_undo_refs(Object *old,Object *new)
{
	List l=undostack,l2;

	while (l!=NULL)
		{
		if (UNDO(l)->real_ob==old)
			UNDO(l)->real_ob=new;
			 
		if (UNDO(l)->type==UNDO_UNCOMPOUND)
			{
			l2 = UNDO(l)->real_ob->ob.compound.obs;
			while (l2!=NULL)
				{
				if (OB(l2)==old)
					l2->data=(void *)new;
				l2=l2->next;
				};
			};
		l=l->next;
		};
}

/* deselect stuff  */  
void
nullify_sel_object(Document *doc)
{
	View *v;
	List l=doc->views;

#ifdef UNDO_DEBUG
	fprintf(stderr,"nullify_sel_object called\n");
#endif
	while (l!=NULL)
		{
		v = VIEW(l); 
		if (v->highlighted_object!=NULL)
			{ 
			send_redraw_object(VIEW(doc->views),v->highlighted_object->ob); 
			v->highlighted_object = NULL;
			}; 
		if (v->selected_object!=NULL)
			{ 
			send_redraw_object(VIEW(doc->views),v->selected_object->ob); 
			v->selected_object = NULL;
			}; 
		if (v->edited_object!=NULL)
			{ 
			send_redraw_object(VIEW(doc->views),v->edited_object->ob); 
			v->edited_object = NULL;
			}; 
		if (v->text_object!=NULL)
			unselect_text(v);
		l = l->next; 
		};
		
	state.editing_object=state.editing_point=state.editing_text=FALSE;             
}
 
/* push an undo from the program on the stack and   */   
/* store the required objects  */  
void 
register_undo(int type, Object *ob, Document *doc)
{
	Undo *u; 

	if (!undolevels)
		return;

	if ((u = (Undo *)malloc(sizeof(Undo)))==NULL)
		return;
	
	switch (type)
		{
		case UNDO_COMPOUND:
			u->real_ob = ob;
			u->ob = NULL;
			break;

		case UNDO_UNCOMPOUND:
			u->real_ob = ob;
			u->ob = NULL;
			break;

		case UNDO_OB_PROP:
			u->real_ob = ob;
			u->ob = undo_duplicate_object(ob);
			replace_undo_refs(ob,u->ob); 
			break;

		case UNDO_CUT:
			u->real_ob = NULL;
			u->ob = undo_duplicate_object(ob);
			replace_undo_refs(ob,u->ob); 
			break;

		case UNDO_PASTE:
			u->real_ob = ob;
			u->ob = NULL;
			break;
		};

	u->type = type;
	u->doc = doc;

#ifdef UNDO_DEBUG
		fprintf(stderr,"Registering %d, ob %p, rob %p\n",u->type,u->ob,u->real_ob); 
#endif
	undostack = push_undo(undostack,u);
	/* it makes no sense to redo now */  
	clear_redo_stack(); 
}

/* perform the top undo  */  
void
do_undo()
{
	Undo *u;
	View *v=NULL;
	List l; 

	/* no undo whilst busy  */  
	if (state.busy_drawing || state.editing_text || state.editing_object)
		return;
		
	if (!undolevels)
		return;

	if ((u = pop_undo(&undostack))==NULL)
		return;
		 
	if (!is_in_list_v(docs,(void *)u->doc)) 
		{
		kill_undo(u,FALSE); 
		free(u);
		return;
		}; 

	v = VIEW(u->doc->views);
	
#ifdef UNDO_DEBUG
		fprintf(stderr,"Undoing %d, ob %p, rob %p\n",u->type,u->ob,u->real_ob); 
#endif
	/* do the undo  */ 
	switch (u->type)
		{
		case UNDO_CUT:
			nullify_sel_object(u->doc); 
			u->doc->o = add_object(u->doc->o, &u->doc->lo, u->ob);
			send_redraw_object(v,u->ob);
			break;

		case UNDO_PASTE:
			nullify_sel_object(u->doc);
			send_redraw_object(v,u->real_ob);
			u->doc->o = trash_object(u->doc->o,&u->doc->lo,u->real_ob);
			break;

		case UNDO_OB_PROP:
			nullify_sel_object(u->doc); 
			u->doc->o = trash_object(u->doc->o,&u->doc->lo,u->real_ob);
			send_redraw_object(v,u->real_ob);
			u->doc->o = add_object(u->doc->o,&u->doc->lo,u->ob);
			send_redraw_object(v,u->ob);
			replace_undo_refs(u->real_ob,u->ob); 
			break;

		case UNDO_COMPOUND:
			nullify_sel_object(u->doc); 
			send_redraw_object(v,u->real_ob);
			l = u->real_ob->ob.compound.obs;
			while (l!=NULL)
				{
				u->doc->o = add_object(u->doc->o,&u->doc->lo,OB(l));
				l=l->next;
				};
			u->doc->o = trash_object(u->doc->o,&u->doc->lo,u->real_ob); 
			break; 
			
		case UNDO_UNCOMPOUND:
			nullify_sel_object(u->doc); 
			send_redraw_object(v,u->real_ob);
			/* remember real object is on canvas */  
			l = u->real_ob->ob.compound.obs;
			while (l!=NULL)
				{
				u->doc->o = trash_object(u->doc->o,&u->doc->lo,OB(l)); 
				l=l->next; 
				};
			u->doc->o = add_object(u->doc->o,&u->doc->lo,u->real_ob); 
			break; 
		default:
			v_error(u->type);
			break;
		};
	
	/* add it to redo stack  */  
	redostack = push_undo(redostack,u);
}

/* ensures references within a compound remain correct  */  
Object *undo_duplicate_object(Object *ob)
{
	Object *nob;
	Object *cnob; 
	List l; 

	nob = duplicate_object(ob);

	if (nob->type==COMPOUND)
		{
		l=nob->ob.compound.obs;
		while (l!=NULL)
			{
			kill_object(OB(l)); 
			l=l->next;
			};
		delete_list(nob->ob.compound.obs);
		nob->ob.compound.obs=NULL;
		l=ob->ob.compound.obs;
		while (l!=NULL)
			{
			cnob = duplicate_object(OB(l));
			nob->ob.compound.obs = add_to_list(nob->ob.compound.obs,ULONG_MAX-OB(l)->depth,0,(void *)cnob);
			replace_undo_refs(OB(l), cnob);
			l=l->next;
			};
		}; 
	 
	return nob;
}

/* pop off redo stack and do in reverse  */  
void
do_redo()
{
	Undo *u;
	View *v=NULL;
	List l; 

	if (state.editing_text || state.editing_object)
		return;

	if (!undolevels)
		return;
		 
	if ((u = pop_undo(&redostack))==NULL)
		return;
		 
	if (!is_in_list_v(docs,(void *)u->doc))
		{
		kill_undo(u,TRUE);
		free(u);
		return; 
		};
	
	v = VIEW(u->doc->views);

#ifdef UNDO_DEBUG
		fprintf(stderr,"Redoing %d, ob %p, rob %p\n",u->type,u->ob,u->real_ob); 
#endif
	switch(u->type)
		{
		case UNDO_OB_PROP:
			/* replace new object with old object */  
			nullify_sel_object(u->doc); 
			send_redraw_object(v,u->ob);
			u->doc->o = trash_object(u->doc->o,&u->doc->lo,u->ob);
			u->doc->o = add_object(u->doc->o,&u->doc->lo,u->real_ob);
			send_redraw_object(v,u->real_ob);
			replace_undo_refs(u->ob,u->real_ob);
			break;

		case UNDO_COMPOUND:
			nullify_sel_object(u->doc); 
			l = u->real_ob->ob.compound.obs;
			while (l!=NULL)
				{
				u->doc->o = trash_object(u->doc->o,&u->doc->lo,OB(l)); 
				l=l->next;
				};
			u->doc->o = add_object(u->doc->o,&u->doc->lo,u->real_ob);
			send_redraw_object(v,u->real_ob);
			break;

		case UNDO_UNCOMPOUND:
			nullify_sel_object(u->doc);
			send_redraw_object(v,u->real_ob);
			l = u->real_ob->ob.compound.obs;
			while (l!=NULL)
				{
				u->doc->o = add_object(u->doc->o,&u->doc->lo,OB(l));
				l=l->next;
				};
			u->doc->o = trash_object(u->doc->o,&u->doc->lo,u->real_ob); 
			break; 
			
		case UNDO_CUT:
			nullify_sel_object(u->doc); 
			send_redraw_object(v,u->ob);
			u->doc->o = trash_object(u->doc->o, &u->doc->lo, u->ob);
			break;

		case UNDO_PASTE:
			nullify_sel_object(u->doc); 
			u->doc->o = add_object(u->doc->o,&u->doc->lo,u->real_ob);
			send_redraw_object(v,u->real_ob);
			break;
			 
		default:
			v_error(u->type);
			break; 
		};
	/* place back onto undo stack  */  
	undostack=push_undo(undostack,u); 
}
