/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomevfs/gnome-vfs-init.h>
#include <glade/glade.h>
#include "file-utils.h"
#include "fr-process.h"
#include "window.h"
#include "typedefs.h"
#include "preferences.h"


static void prepare_app     (poptContext pctx);
static void initialize_data ();
static void release_data    ();

GList *window_list = NULL;
GList *viewer_list = NULL;
Preferences preferences;

static gchar *add_to = NULL;
static gint   add;
static gchar *extract_to = NULL;
static gint   extract;
static gchar *default_url = NULL;

struct poptOption options[] = {
	{ "add-to", 'a', POPT_ARG_STRING, &add_to, 0,
	  N_("Add files to the specified archive and quit the program"),
	  N_("ARCHIVE") },

	{ "add", 'd', POPT_ARG_NONE, &add, 0,
	  N_("Add files asking the name of the archive and quit the program"),
	  0 },

	{ "extract-to", 'e', POPT_ARG_STRING, &extract_to, 0,
          N_("Extract archives to the specified directory and quit the program"),
          N_("DIRECTORY") },

	{ "extract", 'f', POPT_ARG_NONE, &extract, 0,
          N_("Extract archives asking the destination directory and quit the program"),
          0 },

	{ "default-dir", 0, POPT_ARG_STRING, &default_url, 0,
	  " ",
	  N_("DIRECTORY") },

	{ NULL, '\0', 0, NULL, 0 }
};


/* -- Main -- */

int main (int argc, char **argv)
{
	poptContext pctx;

	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	g_thread_init (NULL);
	if (gnome_init_with_popt_table ("File Roller", VERSION, argc, argv, options, 0, &pctx) != 0) {
		g_warning ("Cannot initialize Gnome.");
		return 1;
	}

	if (! gnome_vfs_init ()) 
                g_error ("Cannot initialize the Virtual File System.");

	glade_gnome_init ();

	initialize_data ();
	prepare_app (pctx);
	poptFreeContext (pctx);

	gtk_main ();

	release_data ();

	return 0;
}


/* Initialize application data. */

static void 
initialize_data ()
{
	gchar *pixmap_file;

	pixmap_file = gnome_pixmap_file ("file-roller.png");
        gnome_window_icon_set_default_from_file (pixmap_file);
        g_free (pixmap_file);

	preferences_load ();
}


/* Free application data. */

static void 
release_data ()
{
	preferences_save ();

	while (viewer_list != NULL) {
		FRProcess *proc;
		proc = viewer_list->data;
		fr_process_stop (proc);
	}
}


/* Create the windows. */


gchar *
get_path_from_url (gchar *url)
{
	gchar *path;
	GnomeVFSURI *uri;

	if (url == NULL)
		return NULL;

	uri = gnome_vfs_uri_new (url);
	if (! gnome_vfs_uri_is_local (uri))
		return NULL;

	path = gnome_vfs_uri_to_string (uri, 
					GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
	gnome_vfs_uri_unref (uri);

	return path;
}


static void 
prepare_app (poptContext pctx)
{
	gchar *path;
	gchar *default_dir;

	/* create the config dir if necessary. */

	path = g_strconcat (g_get_home_dir (),
			    "/.file-roller",
			    NULL);
	ensure_dir_exists (path);
	g_free (path);

	/**/

	default_dir = get_path_from_url (default_url);

	if (poptPeekArg (pctx) == NULL) { 
		/* no archive specified. */
		FRWindow *window;
		window = window_new ();
		gtk_widget_show (window->app);
	} else {
		gchar *archive;

		if ((add_to != NULL) || (add == 1)) {
			FRWindow *window;
			GList *file_list = NULL;
			gchar *filename;

			window = window_new ();
			if (default_dir != NULL)
				window_set_default_dir (window, default_dir);
			gtk_widget_show (window->app);

			while ((filename = poptGetArg (pctx)) != NULL) {
				gchar *path;

				if (! g_path_is_absolute (filename)) {
					char *current_dir;
					current_dir = g_get_current_dir ();
					path = g_strconcat (current_dir, 
							    "/", 
							    filename, 
							    NULL);
					g_free (current_dir);
				} else
					path = g_strdup (filename);
				file_list = g_list_prepend (file_list, path);
			}
			file_list = g_list_reverse (file_list);

			if (add_to != NULL)
				window_archive__open_add_close (window, 
								add_to,
								file_list);
			else 
				window_archive__open_add_close (window, 
								NULL,
								file_list);

			if (default_dir != NULL)
				g_free (default_dir);

			return;
		}

		while ((archive = poptGetArg (pctx)) != NULL) {
			FRWindow *window;

			window = window_new ();
			if (default_dir != NULL)
				window_set_default_dir (window, default_dir);
			gtk_widget_show (window->app);

			if (extract_to != NULL)
				window_archive__open_extract_close (window, archive, extract_to);
			else if (extract == 1) 
				window_archive__open_extract_close (window, archive, NULL);
			else
				window_archive_open (window, archive);
		}
	}

	if (default_dir != NULL)
		g_free (default_dir);
}
