/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */

/* 
 *** file menu applet
 *** Copyright (C) 2001 Benjamin Kahn <xkahn@ximian.com>
 *** 
 *** This program is free software; you can redistribute it and/or
 *** modify it under the terms of the GNU General Public License
 *** as published by the Free Software Foundation; either version 2
 *** of the License, or (at your option) any later version.
 *** 
 *** This program is distributed in the hope that it will be useful,
 *** but WITHOUT ANY WARRANTY; without even the implied warranty of
 *** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *** GNU General Public License for more details.
 *** 
 *** You should have received a copy of the GNU General Public License
 *** along with this program; if not, write to the Free Software
 *** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 *** 02111-1307, USA.
 ***/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk/gdkx.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <applet-widget.h>
#include <gnome.h>
#include <config.h>

#include <glade/glade.h>
#include <gconf/gconf-client.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include "main.h"
#include "scroll-menu.h"
#include "gicon.h"

GtkWidget *applet;
GtkWidget *menu_bar;
GtkWidget *documents_item;
GtkWidget *documents_menu;
GtkWidget *action_menu = NULL;
GtkTooltips *menu_tips;

GladeXML *preferences;

GHashTable *directories;

void create_new_file_menuitem (GnomeVFSFileInfo *item, 
			       directory_menu *dm, 
			       char *full_path, 
			       char *mime_type);
void create_new_directory_submenu (GnomeVFSFileInfo *item, 
				   directory_menu *dm, 
				   char *full_path);
static gboolean load_directory_contents (const gchar *rel_path, 
					 GnomeVFSFileInfo *file_info,
					 gboolean recursing_will_loop,
					 gpointer data, 
					 gboolean *recurse);

void doabout(AppletWidget * widget, gpointer data)
{
	
	gchar *authors[] = {
	        "Benjamin Kahn <xkahn@ximian.com>",
		NULL
	};
	
	gtk_widget_show(gnome_about_new(PACKAGE, VERSION,
					"Copyright 2001",
					(const gchar **) authors,
					"File Manager Menu Applet", NULL));

}

GtkWidget *
pixmap_menu_item_new (const char *text, const GdkPixbuf *icon, int truncate)
{
	GtkWidget *item;
	GtkWidget *label;
	char *tip = NULL;

	item = gtk_pixmap_menu_item_new ();

	if (icon && gnome_preferences_get_menus_have_icons ()) {
		GtkWidget *pixmap;
		GdkPixmap *pixmap_return;
		GdkBitmap *mask_return;
		
		gdk_pixbuf_render_pixmap_and_mask (icon, &pixmap_return, &mask_return, 10);
		pixmap = gtk_pixmap_new (pixmap_return, mask_return);

		if(pixmap) {
			gtk_widget_show (pixmap);
			gtk_pixmap_menu_item_set_pixmap
				(GTK_PIXMAP_MENU_ITEM (item), pixmap);
		}
	}

	if (text) {
		if ((truncate != 0) && (strlen (text) > truncate)) {
			tip = strdup (text);
			gtk_tooltips_set_tip(GTK_TOOLTIPS (menu_tips), item,
					     tip, tip);
			text[truncate-3] = text[truncate-2] = text[truncate-1] = '.';
			text[truncate] = '\0';
		}
		label = gtk_label_new (text);
		gtk_object_set_data (GTK_OBJECT(item), "label", strdup (text));
		gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
		gtk_container_add (GTK_CONTAINER (item), label);
		gtk_widget_show (label);
		
	}

	return item;
}

void
remove_old_menus (GtkWidget *item, gpointer menu)
{
	if (GTK_IS_MENU (GTK_MENU_ITEM (item)->submenu)) {
		gtk_container_foreach (GTK_CONTAINER (GTK_MENU_ITEM (item)->submenu), 
				       remove_old_menus, 
				       GTK_MENU_ITEM (item)->submenu);
	}

	if (GTK_IS_MENU_ITEM (item)) {
		gtk_widget_hide (item);
		gtk_container_remove (GTK_CONTAINER (menu), item);
		/* FIXME: We need to remove the label, and the pixmap. */
	}
}



void exec_document_wrap (GtkWidget *widget, void *application_name)
{
	gchar *file_path;
	GnomeVFSMimeApplication *app_name = application_name;
	GdkEventButton *event = (GdkEventButton*)gtk_get_current_event();

	file_path = (gchar *) gtk_object_get_data (GTK_OBJECT (widget), "full_path");	

	if (event->button == 1) {
		if (!app_name) {
			exec_document (file_path);
		} else {
			gtk_signal_emit_by_name (GTK_MENU_BAR (documents_item->parent), "deactivate", NULL);
			exec_application (app_name->command, file_path, app_name->name, app_name->requires_terminal);
		}
	}
}

void
show_directory_menu (GtkWidget *menu, void *path)
{
	DIR *directory;
	struct dirent *item;
	directory_menu *dm;
	GtkWidget *tearoff;
	GnomeVFSResult result;

	GnomeVFSFileInfo *dir_info;

	dir_info = gnome_vfs_file_info_new ();
	result = gnome_vfs_get_file_info (path, dir_info, (GNOME_VFS_FILE_INFO_GET_MIME_TYPE
						   | GNOME_VFS_FILE_INFO_FOLLOW_LINKS));
	if (result != GNOME_VFS_OK) {
		fprintf (stderr, "file_menu_applet: %s: %s (%d)\n",
			 (char *)path, gnome_vfs_result_to_string (result), result);
		return;
	}
	
	dm = (directory_menu *) g_hash_table_lookup (directories, menu);

	if (!dm) {
		dm = g_new (directory_menu, 1);
		dm->mtime = -1;
		dm->name = strdup (path);
	}

	if ((int) dir_info->mtime == (int) dm->mtime) {
		/*		printf ("Don't need to regenerate.\n");*/
		return;
	} else if (dm->mtime != 0) {
		/* We need to deallocate the old menu before we can regenerate it. */
		gtk_container_foreach (GTK_CONTAINER (dm->dmenu), remove_old_menus, dm->dmenu);
		tearoff = gtk_tearoff_menu_item_new ();
		gtk_widget_show (tearoff);
		gtk_menu_append (GTK_MENU (dm->dmenu), tearoff);
	}

	dm->mtime = dir_info->mtime;

	gnome_vfs_file_info_unref (dir_info);

	//	directory = opendir (path);
	/*	result = gnome_vfs_directory_visit 
		(path, (GNOME_VFS_FILE_INFO_FOLLOW_LINKS
		 | GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE),
		 NULL, GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK,
		 (GnomeVFSDirectoryVisitFunc) load_directory_contents,
		 dm);*/

	result = gnome_vfs_directory_visit 
		(path, (GNOME_VFS_FILE_INFO_FOLLOW_LINKS
		 | GNOME_VFS_FILE_INFO_GET_MIME_TYPE),
		 NULL, GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK,
		 (GnomeVFSDirectoryVisitFunc) load_directory_contents,
		 dm);

	/*gtk_widget_show (dm->tearoff);*/
	
	return;
}

static gboolean
load_directory_contents (const gchar *rel_path, 
			 GnomeVFSFileInfo *file_info,
			 gboolean recursing_will_loop,
			 gpointer data, 
			 gboolean *recurse) {
	directory_menu *dm = (directory_menu *) data;
	gchar *full_path;

	*recurse = FALSE;

	if (file_info->name[0] == '.') {
		return TRUE;        /* We ignore dot-files for now. FIXME: use nautilus prefs? */
	}
	
	/* Maybe there are events waiting.  This will make it seem to multitask.*/
	while (gtk_events_pending ()) gtk_main_iteration ();  /* This should no longer be needed. */
	
	/* Assemble the full path name. */
	full_path = g_strdup_printf ("%s/%s", (char *) dm->name, (char *) file_info->name);
	
	switch (file_info->type) {
	case GNOME_VFS_FILE_TYPE_REGULAR:
		
		/* FILE */
		create_new_file_menuitem (file_info, dm, full_path, (char *)file_info->mime_type);
		gtk_menu_reposition (GTK_MENU (dm->dmenu));    /* The new item could have overextended the menu. */
		break;
		
	case GNOME_VFS_FILE_TYPE_DIRECTORY:
		
		/* DIRECTORY */
		create_new_directory_submenu (file_info, dm, full_path);
		gtk_menu_reposition (GTK_MENU (dm->dmenu));    /* The new item could have overextended the menu. */
		break;
	default:
		break;
	}
	
	/* 	gnome_vfs_file_info_unref (file_info); */
	
	return TRUE;
}

void 
create_new_directory_submenu (GnomeVFSFileInfo *item, directory_menu *dm, char *full_path)
{
	GtkWidget *tearoff;
	GtkWidget *menuitem;
	GtkWidget *mmenu;

	directory_menu *dm_new;

	/*	menuitem = pixmap_menu_item_new (item->name, gicon_get_directory_icon_scaled (15, 15)); */
	menuitem = pixmap_menu_item_new (item->name, flist_mime_icon_load (full_path, item->type, NULL), 40); 
	
	mmenu = scroll_menu_new ();
	tearoff = gtk_tearoff_menu_item_new ();
	
	gtk_widget_show (menuitem);
	gtk_widget_show (tearoff);
	
	dm_new = g_new (directory_menu, 1);
	dm_new->name = g_strdup (full_path);
	dm_new->mtime = -1;
	dm_new->dmenu = mmenu;
	dm_new->tearoff = tearoff;
	
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), mmenu);
	gtk_menu_append (GTK_MENU (mmenu), tearoff);
	/* gtk_menu_insert (GTK_MENU (dm->dmenu), menuitem, 1);*/
	scroll_menu_insert_sorted (GTK_MENU (dm->dmenu), menuitem);
	
	g_hash_table_insert (directories, menuitem, dm_new);
	
	gtk_signal_connect (GTK_OBJECT (menuitem), "button_press_event", 
			    GTK_SIGNAL_FUNC (maybe_popup_menu), (gpointer) dm->dmenu);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate", 
			    GTK_SIGNAL_FUNC (show_directory_menu), (gpointer) full_path); 
	
	gtk_object_set_data (GTK_OBJECT (menuitem), "full_path", (gpointer) full_path);

	set_drag_stuff_on_directory_menu (menuitem, (gchar *) full_path);
	
}

void 
create_new_file_menuitem (GnomeVFSFileInfo *item, directory_menu *dm, char *full_path, char *mime_type)
{
	GtkWidget *menuitem;

	/*	menuitem = pixmap_menu_item_new (item->name, gicon_get_icon_for_file_scaled 
		(full_path, 15, 15, TRUE));*/
	menuitem = pixmap_menu_item_new (item->name, flist_mime_icon_load 
					 (full_path, item->type, mime_type), 40);
	gtk_object_set_data (GTK_OBJECT (menuitem), "full_path", (gpointer) full_path);
	
	gtk_signal_connect (GTK_OBJECT (menuitem), "button_press_event", 
			    GTK_SIGNAL_FUNC (maybe_popup_menu), (gpointer) dm->dmenu);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
			    GTK_SIGNAL_FUNC (exec_document_wrap), NULL);
	
	gtk_widget_show (menuitem);
	/* gtk_menu_append (GTK_MENU (dm->dmenu), menuitem);*/
	scroll_menu_insert_sorted (GTK_MENU (dm->dmenu), menuitem);

	set_drag_stuff_on_file_menu (menuitem, (gchar *) full_path);
	
}

static ORBit_MessageValidationResult
accept_all_cookies (CORBA_unsigned_long request_id,
		    CORBA_Principal *principal,
		    CORBA_char *operation)
{
	/* allow ALL cookies */
	return ORBIT_MESSAGE_ALLOW_ALL;
}

gint main(gint argc, gchar * argv[])
{
	GtkWidget *tearoff;
	directory_menu *dm;
	gchar *sd;
	gchar *menu_label;
	gchar *dir_watch;
	gboolean cache_menus;
	guint n1, n2, n3;
	guint handlerid;

	GError* error = NULL;
	GConfClient* client = NULL;
	GtkWidget* config;
	
	/* init all the services we are using. */
	glade_gnome_init();
	applet_widget_init(PACKAGE, VERSION, argc, argv, NULL, 0, NULL);
	directories = g_hash_table_new (NULL, NULL);

	if (!gnome_vfs_init ()) {
		fprintf (stderr, "file_menu_applet: Cannot initialize the GNOME Virtual File System.\n");
		return 1;
	}
	
	if (!gconf_init(argc, argv, &error)) {
		g_assert(error != NULL);
		g_warning("GConf init failed:\n  %s", error->message);
		g_error_free(error);
		error = NULL;
		return 1;
	}
	
	/* Shut ORBit up. */
	ORBit_set_request_validation_handler (accept_all_cookies); 

	/* Get Preferences and Defaults */
	client = gconf_client_get_default();
	gconf_client_add_dir(client, GCONF_KEY, GCONF_CLIENT_PRELOAD_NONE, NULL);
	
	menu_label = gconf_client_get_string (client, GCONF_KEY "/menu_title", NULL);
	dir_watch = gconf_client_get_string (client, GCONF_KEY "/directory_to_watch", NULL);
	cache_menus = gconf_client_get_bool (client, GCONF_KEY "/cache_menus", NULL);
	
	if (!menu_label)
		menu_label = strdup ("Documents");
	if (!dir_watch)
		dir_watch = strdup (getenv ("HOME"));

	/* Create */
	applet = applet_widget_new(PACKAGE);
	menu_bar = gtk_menu_bar_new ();
	documents_item = gtk_menu_item_new_with_label (menu_label);
	documents_menu = scroll_menu_new ();
	tearoff = gtk_tearoff_menu_item_new();
	menu_tips = gtk_tooltips_new();
	dm = g_new (directory_menu, 1);
	sd = dir_watch;

	gtk_object_set_data(GTK_OBJECT(applet), "client", client);
	
	/* Configure */
	gtk_menu_bar_set_shadow_type (GTK_MENU_BAR (menu_bar), GTK_SHADOW_NONE);
	

	/* Preferences redux */
	n1 = gconf_client_notify_add(client, GCONF_KEY "/menu_title", 
				     set_menu_title, documents_item, NULL, NULL);
	n2 = gconf_client_notify_add(client, GCONF_KEY "/directory_to_watch", 
				     change_watched_directory, NULL, NULL, NULL);
	n3 = gconf_client_notify_add(client, GCONF_KEY "/cache_menus", 
				     change_cache_setting, NULL, NULL, NULL);

	/* Pack */
	applet_widget_add(APPLET_WIDGET(applet), menu_bar);
	gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), documents_item);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (documents_item), documents_menu);
	gtk_menu_append (GTK_MENU (documents_menu), tearoff);
	dm->name = sd;
	dm->mtime = -1;
	dm->dmenu = documents_menu;
	dm->tearoff = tearoff;
	
	/* Display */
	gtk_widget_show (menu_bar);
	gtk_widget_show (documents_item);
	gtk_widget_show (applet);
	gtk_tooltips_enable (menu_tips);
	g_hash_table_insert (directories, documents_item, dm);

	/* Callbacks */

	set_drag_stuff_on_menu_bar (documents_item);

        applet_widget_register_stock_callback((AppletWidget *) applet,
                                              "Properties",
                                              GNOME_STOCK_MENU_PROP,
                                              "Properties", show_preferences_menu,
                                              (gpointer) applet);
	applet_widget_register_stock_callback((AppletWidget *) applet,
					      "About...",
					      GNOME_STOCK_MENU_ABOUT,
					      "About...", doabout, NULL);
	handlerid = gtk_signal_connect (GTK_OBJECT (documents_item), 
					"activate", 
					GTK_SIGNAL_FUNC (show_directory_menu), 
					(gpointer) sd);
	
	/* We need to remember the handler so we can remove it again. */
	gtk_object_set_data (GTK_OBJECT (documents_item), "handler", (gpointer) handlerid);
	gtk_object_set_data (GTK_OBJECT (documents_item), "dm", (gpointer) dm);

	/* cleanup and exit */
	gtk_main();

	/* So we need to detach cleanly from gconf.  Ugh. */
	gconf_client_notify_remove (client, n1);
	gconf_client_notify_remove (client, n2);
	gtk_object_unref(GTK_OBJECT(client));

	return 0;
}

