/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.1
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.1 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.1 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cstdlib>
#include <iostream>
#include <cmath>
#include "vectorfield.h"

using namespace std;

VectorField::VectorField(int x, int y)
{
	__xRes = x;
	__yRes = y;

	__data = NULL;
	
	int n = 0;
	char* f = fillField(n);
	while(f)
	{
		__field.push_back(f);
		n++;
		f = fillField(n);
	}

	if(__field.empty())
	{
		cerr << "ERROR: unable to calculate any vectorfields. out of memory?" << endl;
		exit(EXIT_FAILURE);
	}

	__data = __field.at(0);
	cout << "* successfully calculated " << __field.size() << " different vectorfields" << endl;
}

VectorField::~VectorField()
{
	for(unsigned int i = 0; i < __field.size(); i--)
	{
		free(__field[i]);
	}
	__field.clear();
}

char* VectorField::get()
{
	return (char*)__data;
}

void VectorField::change()
{
	int n = rand() % __field.size();
	while(__data == __field[n]) n = rand() % __field.size();
	__data = __field[n];
}

int VectorField::width()
{
	return __xRes;
}

int VectorField::height()
{
	return __yRes;
}

void VectorField::randomize(char* vec, int n)
{
	return;
	/* experimental */
	if(n % 2 == 0) n++;
	*vec += (rand() % n) - n / 2;
	*(vec + 1) += (rand() % n) - n / 2;
}

char* VectorField::fillField(int n)
{
	char* v;
	double d;
	char* f = (char*)malloc(__xRes * __yRes * 2);
	if(!f) return NULL;
	switch(n)
	{
		case 0:
			// linear vectors showing away from a horizontal mirror axis
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*v = 0;
					*(v + 1) = (y < __yRes / 2) ? (__yRes / 100) : (-__yRes / 100);
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 1:
			// linear vectors showing away from a vertical mirror axis
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*v = (x < __xRes / 2) ? (__xRes / 100) : (-__xRes / 100);
					*(v + 1) = 0;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 2:
			// radial vectors showing away from the center
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*v = -(x - __xRes / 2) / 25;
					*(v + 1) = -(y - __yRes / 2) / 25;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 3:
			// tangential vectors (right)
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*(v + 1) = -(x - __xRes / 2) / 25;
					*v = (y - __yRes / 2) / 25;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 4:
			// tangential vectors (left)
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*(v + 1) = (x - __xRes / 2) / 25;
					*v = -(y - __yRes / 2) / 25;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 5:
			// tangential-radial vectors (left)
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*(v + 1) = (x - __xRes / 2) / 25 - (y - __yRes / 2) / 25;;
					*v = -(y - __yRes / 2) / 25 - (x - __xRes / 2) / 25;;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 6:
			// tangential-radial vectors (right)
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*(v + 1) = -(x - __xRes / 2) / 25 - (y - __yRes / 2) / 25;;
					*v = (y - __yRes / 2) / 25 - (x - __xRes / 2) / 25;;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 7:
			// radial sinewave vectors
			d = 25 / (double)__xRes;
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					double r = sqrt(pow((double)(x - __xRes / 2), 2) + pow((double)(y - __yRes / 2), 2)) * d;
					*v = (char)(-(x - __xRes / 2) / 25 - (double)((y - __yRes / 2) / 50) * sin(r));
					*(v + 1) = (char)(-(y - __yRes / 2) / 25 + (double)((x - __xRes / 2) / 50) * sin(r));
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 8:
			// hyperbolic vectors
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*v = (y - __yRes / 2) / 25;
					*(v + 1) = (x - __xRes / 2) / 25;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 9:
			// hyperbolic vectors
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*v = -(y - __yRes / 2) / 25;
					*(v + 1) = -(x - __xRes / 2) / 25;
					randomize(v, 2);
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		case 10:
			// purely random
			for(int x = 0; x < __xRes; x++)
			{
				for(int y = 0; y < __yRes; y++)
				{
					v = f + 2 * x + y * 2 * __xRes;
					*v = (rand() % __xRes - __xRes / 2) / 50;
					*(v + 1) = (rand() % __yRes - __yRes / 2) / 50;
					validateX(x, v);
					validateY(y, v + 1);
				}
			}
			return f;
		default:
			// index too high. return nothing.
			free(f);
			return NULL;
	}
}
