/*
 *      Copyright (C) 1997-2000 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *     This program is a small utility to manipulate certain fields of
 *     the volume table of a floppy tape cartridge.  To be used with
 *     the QIC-40/80/3010/3020 floppy-tape driver "ftape" for Linux.
 */

char src[] = "$RCSfile: vtblc.c,v $";
char rev[] = "$Revision: 1.16 $";
char dat[] = "$Date: 2001/08/11 00:16:33 $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/ioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <locale.h>
#include <libintl.h>
#define _(String) gettext (String)

#include <linux/ftape.h>
#include <linux/ftape-vendors.h>
#include <linux/zftape.h>
#include <linux/ftape-header-segment.h>

#include "ftvt.h"
#include "vtblc-version.h"

extern void *xmalloc(size_t size);
extern void *xrealloc(void *old, size_t size);

/* The name this program was run with.  */
char *program_name;

static char *tape_dev = FTAPE_DEF_TAPE;

static u_int8_t hseg[FT_SEGMENT_SIZE];
static int fmt_code;
static ftvt volumes[FTVT_MAX_VOLUMES];
static u_int8_t vtbl_buffer[FT_SEGMENT_SIZE];
static int num_volumes;
static int tagged;
static int trunc;

static const char *short_options = "f:Vdvqh?p::a:t::m:#:b";

static const struct option long_options[] =
{
	{"file",               1, 0, 'f'}, /* full path to device */
	{"version",            0, 0, 'V'}, /* version information */
	{"debug",              0, 0, 'd'}, /* switch on debugging, unused */
	{"verbose",            0, 0, 'v'}, /* verbosity level, unused */ 
	{"quiet",              0, 0, 'q'}, /* be quiet */
	{"usage",              0, 0, 'h'},
	{"help",               0, 0, 'h'},
	{"print",              2, 0, 'p'}, /* pretty print the vtbl */
	{"append",             1, 0, 'a'}, /* append an entry at the end */
	{"truncate",           2, 0, 't'}, /* truncate to given size */
	{"modify",             1, 0, 'm'}, /* modify given vtbl */
	{"vtbl-entry",         1, 0, '#'}, /* vtbl to print or modify */ 
	{"bsm",                0, 0, 'b'}, /* print the bsm */
	{0, 0, 0, 0}
};

static const char *vtbl_str_tags[] =
{
	"label",
	"date",
	"start",
	"end",
	"tagged",
	NULL
};

static enum {
	vtbl_none   = -1,
	vtbl_label  = 0,
	vtbl_date   = 1,
	vtbl_start  = 2,
	vtbl_end    = 3,
	vtbl_tagged = 4
} vtbl_tag = vtbl_none;

static enum {
	none     = 0,
	append   = 1,
	modify   = 2,
} op_mode = none;

static int vtbl_no      = -1;
static char *label      = NULL;
static char *datestr    = NULL;
static int setdate      = 0;
static int start_seg    = -1;
static int end_seg      = -1;
static int vtbl_print   = 0;
static int vtbl_maxsize = -1;
static int verbose      = 1;
static int debug        = 0;
static int taggedinput  = 0;
static int bsm          = 0;

static void usage(FILE *file, int exit_val, char *message, ...);

int main(int argc, char *argv[])
{
	char *subopts, *value;
	int c;
	int tapefd;

	program_name = argv[0];

	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);

	while (1) {
		int option_index = 0;
		
		c = getopt_long(argc, argv, short_options, long_options,&option_index);

		if (c == -1) {
            break;
		}

		switch(c) {
		case 0:
			break;
		case '#':
			vtbl_no = strtol(optarg, NULL, 0);
			break;
		case 'a':
		case 'm':
			subopts = optarg;
			while (subopts && *subopts != '\0') {
				vtbl_tag = getsubopt(&subopts, vtbl_str_tags, &value);
				if (value == NULL &&
					vtbl_tag != vtbl_date && vtbl_tag != vtbl_tagged &&
					vtbl_tag != -1) {
					usage(stderr, 1, _("The option \"%s\" needs a value!\n"),
						  vtbl_str_tags[vtbl_tag]);
				}
				switch(vtbl_tag) {
				case vtbl_label:
					label = value;
					break;
				case vtbl_date:
					datestr = value;
					setdate = 1;
					break;
				case vtbl_start:
					start_seg = strtol(value, NULL, 0);
					break;
				case vtbl_end:
					end_seg = strtol(value, NULL, 0);
					break;
				case vtbl_tagged:
					taggedinput = 1;
					break;
				default:
					usage(stderr, 1, _("Unknown option: %s\n"), value);
					break;
				}
			}
			if (c == 'a') {
				op_mode = append;
			} else {
				op_mode = modify;
			}
			break;
		case 'p':
			vtbl_print = 1;
			if (optarg && !strcmp(optarg, "tagged")) {
				tagged = 1;
			} else {
				tagged = 0;
			}
			break;
		case 't':
			if (optarg) {
				vtbl_maxsize = strtol(optarg, NULL, 0);
			}
			trunc = 1;
			break;
		case 'f':
			tape_dev = optarg;
			break;
		case 'b':
			bsm = 1;
			break;
		case 'V':
			printf(version_string);
			exit(0);
			break;
		case 'd':
			debug ++;
			break;
		case 'v':
			verbose ++;
			break;
		case 'q':
			verbose = 0;
			break;
		case 'h':
		case '?':
			usage(stdout, 0, NULL);
			break;
		default:
			fprintf (stderr, _("?? getopt returned character code 0%o ??\n"), c);
			exit(1);
			break;
		}
	}
	if (op_mode == none && !trunc) {
		vtbl_print ++;
	}	
	ftvt_set_ctrl((tagged ? FTVT_TAGGED : 0) | (verbose ? FTVT_VERBOSE : 0));
	if ((tapefd = ftvt_open(tape_dev,
							(op_mode == none && !trunc)
							? O_RDONLY : O_RDWR)) == -1) {
		exit(1);
	}
	if ((fmt_code = ftvt_read_header_segment(tapefd, hseg)) == -1) {
		(void)ftvt_close(tapefd);
		exit(1);
	}
	if (bsm) {
		ftvt_init_bsm(hseg, fmt_code, 0);
		ftvt_print_bad_sector_map(verbose);
		(void)ftvt_close(tapefd);
		exit(0);
	}
	if ((num_volumes = ftvt_read_vtbl(tapefd, hseg, fmt_code, volumes,
									  vtbl_buffer)) == -1) {
		(void)ftvt_close(tapefd);
		exit(1);
	}
	if (taggedinput) { /* implies modify or append */
		if ((num_volumes = ftvt_parse_tagged(volumes, fmt_code)) == -1) {
			(void)ftvt_close(tapefd);
			exit(1);
		}
#if 0
		print_vtbl(volumes, num_volumes);
		(void)tape_close(tapefd);
		exit(0);
#endif
	} else if (op_mode == modify || op_mode == append) {
		if (op_mode == append) {
			if (ftvt_add_volume(volumes, num_volumes) == -1) {
				(void)ftvt_close(tapefd);
				exit(1);
			}
			if (end_seg == -1 || start_seg == -1) {
				fprintf(stderr, _("Missing start- or ending segment\n"));
			}
			op_mode = modify;
			vtbl_no = -1;
		}
		if (setdate && ftvt_set_date(volumes, num_volumes, datestr, vtbl_no)) {
			(void)ftvt_close(tapefd);
			exit(1);
		}
		if (label && ftvt_set_label(volumes, num_volumes, label, vtbl_no)) {
			(void)ftvt_close(tapefd);
			exit(1);
		}
		if (start_seg != -1 && end_seg != -1 &&
			ftvt_set_bounds(volumes, num_volumes, start_seg, end_seg,
							vtbl_no)) {
			(void)ftvt_close(tapefd);
			exit(1);
		}
		if (ftvt_set_id(volumes, num_volumes, "VTBL", vtbl_no)) {
			(void)ftvt_close(tapefd);
			exit(1);
		}
	}
	if (trunc) {
		if (vtbl_maxsize == -1) {
			vtbl_maxsize = num_volumes - 1;
		}
		if (vtbl_maxsize < 0 || vtbl_maxsize > num_volumes) {
			fprintf(stderr, _("Volume number too big or negative: %d\n"), 
					vtbl_no);
			(void)ftvt_close(tapefd);
			exit(1);
		}
		num_volumes = vtbl_maxsize;
	}
	if (vtbl_print) {
		ftvt_print(volumes, num_volumes);
	}
	if (ftvt_write(tapefd, volumes, vtbl_buffer, num_volumes, trunc)) {
		(void)ftvt_close(tapefd);
		exit(1);
	}
	if (ftvt_close(tapefd)) {
		exit(1);
	} else {
		exit(0);
	}
	exit(0);
}

static void usage(FILE *file, int exit_val, char *message, ...)
{
	va_list ap;
	if (message) {
		va_start(ap, message);
		vfprintf(file, message, ap);
		va_end(ap);
	}
	fprintf(file, version_string);
	fprintf(file, "\n");
	fprintf(file,
_("Usage: %s [OPTIONS]\n"
"Manipulate the volume table of a floppy tape cartridge, for use with\n"
"%s, %s\n"
"\n"
"Mandatory or optional arguments to long options are mandatory or optional\n"
"for short options too. Unique abbreviations for long options are "
"accepted.\n"
"\n"
"  -f, --file=FILE       Tape device to use. Default is "
"\"%s\".\n"
"  -h, --help            Print this help.\n"
"  -?                    Same as \'-h\'.\n"
"      --usage           Same as \'-h\'.\n"
"  -V, --version         Print version information.\n"
"  -d, --debug           Unused yet.\n"
"  -v, --verbose         Be verbose (default).\n"
"  -q, --quiet           Don't be verbose.\n"
"  -b, --bsm             Print the bad sector map and quit.\n"
"  -#, --vtbl-entry=NR   Specify the volume number for \"--print\" and\n"
"                        \"--modify\".\n"
"  -p, --print[=tagged]  Print the volume table entry given by\n"
"                        \"--vtbl-entry\" or the entire table if\n"
"                        \"--vtbl-entry\" has been omitted.\n"
"                        If no other action is specified then \"--print\"\n"
"                        is the default.\n"
"                        If the argument \"tagged\" is supplied, print\n"
"                        the volume table in a format that may in turn\n"
"                        serve as input for \"--modify=tagged\" (see below).\n"
"  -t, --truncate[=SIZE] Truncate the volume table to SIZE entries by\n"
"                        filling the remainder with zero bytes. If SIZE\n"
"                        is omitted then the last entry is deleted.\n"
"  -a, --append          Append an entry to the end of the volume table.\n"
"  -m, --modify          Modify the volume table entry given by\n"
"                        \"--vtbl-entry\". If \"--vtbl-entry\" has been\n"
"                        omitted then the last entry is modified.\n"
"\n"
"\"--append\" and \"--modify\" understand the following sub-options:\n"
"\n"
"      label=LABEL       The description for this volume table entry.\n"
"                        LABEL will be truncated to 44 bytes.\n"
"      date[=DATE]       The time stamp for this entry. DATE is parsed\n"
"                        by strptime(3) using the \"%%T %%D\" format (same\n"
"                        as \"%%H:%%M:%%S %%m/%%d/%%y\").\n"
"                        If DATE is omitted the current local time is used\n"
"                        instead.\n"
"      start=SEG_ID      The starting segment for this volume table entry.\n"
"      end=SEG_ID        The final segment for this volume table entry.\n"
"      tagged            Read the complete volume table entry from stdin\n"
"                        in a tagged format. The input stream consists of\n"
"                        pairs of keywords and values, one pair per line.\n"
"                        The keywords are separated from the values by\n"
"                        space characters (SPACE or TAB).\n"
"      `vtblc` understands the following keywords and values:\n"
"\n"
"      VTBL START\n"
"      VTBL END          Marks the start of vtbl data. Maybe useful when\n"
"                        sending volume table entries via email :-).\n"
"                        However, the entire volume table data must be\n"
"                        surrounded by a \"VTBL START\" - \"VTBL END\" pair.\n"
"      ENTRY NUM         Starts the description for the volume NUM.\n"
"      ENTRY END         Here END is not a placeholder, but means the\n"
"                        word \"END\". Ends the description for the volume\n"
"                        table entry previously started by \"ENTRY NUM\".\n"
"                        Everything between \"ENTRY NUM\" and \"ENTRY END\"\n"
"                        modifies the volume number NUM. This provides means\n"
"                        to modify the entire volume table in a single run.\n"
"\n"
"      SIGNATURE \"SIG\"   Valid signature string. Must be one out of\n"
"                       \"VTBL\", \"XTBL\", \"EXVT\" and \"UTID\".\n"
"                        If this tag is used it MUST be the first entry\n"
"                        after the \"ENTRY\" tag. If the \"SIGNATURE\" tag\n"
"                        is missing, \"VTBL\" is assumed.\n"
"\n"
"The following entry is for signatures other than \"VTBL\":\n"
"\n"
"      ENTRY_DATA \"HEX\"  Data following the signature \"HEX\" is a string\n"
"                        of hexadecimal byte values, e.g. \"0x01 0x4f ...\",\n"
"                        of length at most 124. If less bytes are presented\n"
"                        the entry will be padded with zeroes.\n"
"\n"
"Following entries are for for signature \"VTBL\":\n"
"\n"
"      START STARTSEG    First segment of this volume.\n"
"      END ENDSEG        Last segment of this volume.\n"
"      NUM_SEGMENTS NUM  Number of segments occupied (optional).\n"
"                        If omitted, will be set to (ENDSEG-STARTSEG+1)\n"
"      DESCRIPTION \"DESC\" Description for this volume table. Will be\n"
"                        truncated to 44 characters.\n"
"      DATE \"DATESTR\"   Date for the volume. If \"DATESTR\" is ommitted,\n"
"                        then the current local time is used.\n"
"      FLAG_VENDOR_SPECIFIC VAL\n"
"      FLAG_MULTI_CARTRIDGE VAL\n"
"      FLAG_NOT_VERIFIED VAL\n"
"      FLAG_REDIRECTION_INHIBIT VAL\n"
"      FLAG_SEGMENT_SPANNING VAL\n"
"      FLAG_DIRECTORY_LAST VAL\n"
"      FLAG_RESERVED_6 VAL\n"
"      FLAG_RESERVED_7 VAL\n"
"      MULTI_CARTRIDGE_COUNT VAL\n"
"\n"
"The following tags are understood only if FLAG_VENDOR_SPECIFIC is 0\n"
"or if FLAG_VENDOR_SPECIFIC is 1 and QIC_MAJOR is 113:\n"
"\n"
"      QIC_MAJOR VAL     Major number of QIC spec cartridge conforms to,\n"
"                        only for cartridges with more than 65536 segments.\n"
"                        VAL should be 113 for all cases.\n"
"      QIC_MINOR VAL     Minor revision number of QIC spec. Only for\n"
"                        cartridges with more thatn 65536 segments.\n"
"                        At the time of this writing, QIC-113, Rev. G, is\n"
"                        is the latest one, so VAL should be 7.\n"
"      VENDOR_EXTENSION \"HEX\" Vendor extension data. \"HEX\" is a string\n"
"                        of hexadecimal byte values, i.e. \"0x01 0x4f ...\"\n"
"                        The maximal lenght of this string is 14 bytes for\n"
"                        cartridges with more thatn 2^16 = 65536 segments\n"
"                        and 26 bytes for smaller cartridges\n"
"      PASSWORD \"HEX\"  Password. \"HEX\" as above.\n"
"      DIRECTORY_SIZE VAL\n"
"      DATA_SIZE VAL64   64 bit value (a decimal count)\n"
"      OS_VERSION HEX    Here \"HEX\" is a hexadecimal 2 bytes value.\n"
"      SOURCE_DRIVE DRVSTR Source drive. \"DRVSTR\" is a 16 byte string.\n"
"      DEVICE HEXBYTE    A single byte in hexadecimal notation.\n"
"      RESERVED_1 HEXBYTE\n"
"      COMPRESSION_FLAGS HEXBYTE\n"
"      FORMAT HEYBYTE\n"
"      RESERVED_2 HEXBYTE\n"
"      RESERVED_3 HEXBYTE\n"
"\n"
"If FLAG_VENDOR_SPECIFIC is 1, then the following tag is allowed:\n"
"\n"
"      VENDOR_DATA \"HEX\"  A string of hexadecimal values of lenght at most\n"
"                        72 to fill the remainder of the volume table entry.\n"
"                        Smaller strings will be padded with zero bytes.\n"
"\n"
"Strings with spaces have to be surrounded by double quotes \"...\"\n"
"\"VAL\" is some decimal number.\n"
"\n"),
			program_name, FTAPE_VERSION, ZFTAPE_VERSION, FTAPE_DEF_TAPE);
	exit(exit_val);
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
