#!/usr/bin/pl -q -t main -f
% -*-Prolog-*-

/*

                          Firewall Builder

                 Copyright (C) 2001 Vadim Kurland, Vadim Zaliva

  Author:  Vadim Zaliva <lord@crocodile.org>

  $Id: fwcompiler.pl,v 1.17 2001/10/22 06:45:45 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  To get a copy of the GNU General Puplic License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

% For interactive start use: 
%     pl -g "load_files('fwcompiler.pl'), consult('t.pl')"

% Data

negative_action(deny  , accept ).
negative_action(reject, accept ).
negative_action(accept, deny   ).

action_name(deny  , 'DENY'  ).
action_name(accept, 'ACCEPT').
action_name(reject, 'REJECT').

% Rules

% ruleHostNegation(SRC, SRC_NEG, ACTION, NSRC, NACTION).

ruleHostNegation(SRC, false, ACTION, SRC, ACTION).
ruleHostNegation(SRC, true , ACTION, SRC, NACTION) :- negative_action(NACTION, ACTION).
ruleHostNegation(_  , true , ACTION, ANYHOSTID, ACTION) :- anynetwork(ANYHOSTID, _).

% ruleSrvNegation(SRV, SRV_NEG, ACTION, NSRV, NACTION).

ruleSrvNegation(SRC, false, ACTION, SRC, ACTION).
ruleSrvNegation(SRC, true , ACTION, SRC, NACTION) :- negative_action(NACTION, ACTION).
ruleSrvNegation(_  , true , ACTION, ANYSRVID, ACTION) :- anyipservice(ANYSRVID, _).

objects_list_member(Object, Lst)  :- member(Member, Lst), objects(Member, Object).
service_list_member(Service, Lst) :- member(Member, Lst), services(Member, Service).

objects(Id, Id) :- ip_object(Id, _, _).
objects(GroupId, MemberId) :- objectgroup(GroupId, _, GroupMember), 
        objects(GroupMember, MemberId).

services(Id, Id) :- some_service(Id, _).
services(GroupId, MemberId) :- servicegroup(GroupId, _, GroupMember), 
        services(GroupMember, MemberId).


% This predicate represents policy rule
% where all groups are open, all negations are processed
% and source, service and destination lists are opened.
normalizedPolicyRule(ID, FWID, NUM, NSRC, NDST, NSRV, FACTION, LOG) :- 
        policyRule(ID, FWID, NUM, SRCLIST, SRC_NEG, DSTLIST, DST_NEG, SRVLIST, SRV_NEG, ACTION, LOG),
        objects_list_member(SRC, SRCLIST), 
        ruleHostNegation(SRC, SRC_NEG, ACTION, NSRC, SACTION),
        objects_list_member(DST, DSTLIST), 
        ruleHostNegation(DST, DST_NEG, SACTION, NDST, DACTION),
        service_list_member(SRV, SRVLIST),
        ruleSrvNegation(SRV, SRV_NEG, DACTION, NSRV, FACTION).



% atomicPolicyRule(ID, FWID, NUM, IFACE, DIR, LIST, SRC, DST, SRV, ACTION, LOG, OPTIONS)
atomicPolicyRule(ID, FWID, NUM, SRC, DST, SRV, ACTION, LOG) :- normalizedPolicyRule(ID, FWID, NUM, SRC, DST, SRV, ACTION, LOG).

atomicPolicyRule(ID, FWID, NUM, SRC, FWID, SRV, ACTION, LOG) :- 
        firewalloption('firewall_is_part_of_any_and_networks' , FWID, true),
        normalizedPolicyRule(ID, FWID, NUM, SRC, DST, SRV, ACTION, LOG), 
        firewall(FWID, _, _), 
        anynetwork(DST, _).

atomicPolicyRule(ID, FWID, NUM, FWID, DST, SRV, ACTION, LOG) :- 
        firewalloption('firewall_is_part_of_any_and_networks' , FWID, true),
        normalizedPolicyRule(ID, FWID, NUM, SRC, DST, SRV, ACTION, LOG), 
        firewall(FWID, _, _), 
        anynetwork(SRC, _).

% InterfacePolicyRule:  id, iface_id, num, src, src_neg, dst, dst_neg, srv, srv_neg, action, log
% PolicyRule:           id, fw_id, num, src, src_neg, dst, dst_neg, srv, srv_neg, action, log

ip_object(Id, Name, IP) :- host(Id, Name,IP).
ip_object(Id, Name, IP) :- firewall(Id, Name,IP).
ip_object(Id, Name, IP) :- network(Id, Name, IP, _).
ip_object(Id, Name, [0,0,0,0]) :- anynetwork(Id, Name).

some_service(Id, Name) :- icmpservice(Id, Name, _, _).
some_service(Id, Name) :- ipservice(Id, Name, _, _, _, _, _, _, _).
some_service(Id, Name) :- tcpservice(Id, Name, _, _, _, _, _, _, _, _).
some_service(Id, Name) :- udpservice(Id, Name, _, _, _, _).
some_service(Id, Name) :- customservice(Id, Name, _, _).
some_service(Id, Name) :- anyipservice(Id, Name).

display_ip(OStream, [A0|[A1|[A2|[A3|_]]]]) :- 
        print(OStream,A0), display(OStream,'.'),
        print(OStream,A1), display(OStream,'.'),
        print(OStream,A2), display(OStream,'.'),
        print(OStream,A3).

print_rules(OStream, _, []) :- nl(OStream).

print_rules(OStream, Firewall, [[NUM, _, _, SRC, DST, SRV, ACTION, _]|OtherRules]) :-
        ip_object(SRC,_,SrcHostIp), 
        ip_object(DST,_,DstHostIp), 
        action_name(ACTION, ActionName), 
        display(OStream, '#'), display(OStream, NUM), display(OStream, ': '),
        display_ip(OStream, SrcHostIp), display(OStream, ' '),
        display_ip(OStream, DstHostIp),  display(OStream, ' '),
        some_service(SRV, SrvName),
        display(OStream, SrvName),  display(OStream, ' '),
        display(OStream, ActionName),
        nl(OStream),
        print_rules(OStream, Firewall, OtherRules).

% Checks if there are host and firewall with same IP
validate_duplicate_host_fw_IP :-
        firewall(_, FNAME , IP), host(_ , HNAME, IP), 
        display('Warning: host "'), display(HNAME), 
        display('" and firewall "'), display(FNAME),
        display('" are using same IP: '), 
        current_output(ErrStream), display_ip(ErrStream,IP), nl, fail.
validate_duplicate_host_fw_IP.

duplicate_hosts(ID0, ID1) :-
        host(ID0, HNAME , IP), host(ID1, HNAME, IP),
        compare(<, ID0, ID1).

% Checks if there are 2 hosts with same IP
validate_duplicate_host_host_IP :-
        host(ID0, H0NAME , IP), host(ID1 , H1NAME, IP),
        compare(<, ID0, ID1),
        not(duplicate_hosts(ID0, ID1)),
        display('Warning: host "'), display(H0NAME), 
        display('" and host "'), display(H1NAME),
        display('" are using same IP: '), 
        current_output(ErrStream), display_ip(ErrStream,IP), nl, fail.
validate_duplicate_host_host_IP.

% Checks if there are 2 hosts with same IP and name 
validate_duplicate_hosts :-
        host(ID0, HNAME , IP), host(ID1, HNAME, IP),
        duplicate_hosts(ID0, ID1),
        display('Warning: 2 hosts entries with name "'), display(HNAME), 
        display('" and IP '), 
        current_output(ErrStream), display_ip(ErrStream,IP), 
        display(' but 2 different IDs: "'), 
        display(ID0), display('" and "'), display(ID1), display('"'),
        nl, fail.
validate_duplicate_hosts.

% Check data for consitency and issue warnings if any
% problems are found.
validate_data :-
        validate_duplicate_hosts,
        validate_duplicate_host_host_IP,
        validate_duplicate_host_fw_IP.

main :- 
        display('# FWBuilder Prlog Compiler'), nl,
        current_prolog_flag(argv, Arguments),
        append(_SytemArgs, [--|[A0|[A1|[A2]]]], Arguments), !,
        display('# Loading Data file: '), print(A0), nl,
        consult(A0),
        validate_data,
        open(A1, write, OStream),
        display('# Generating file: '), print(A1), nl,
        setof([NUM, ID, FWID, SRC, DST, SRV, ACTION, LOG], atomicPolicyRule(ID, FWID, NUM, SRC, DST, SRV, ACTION, LOG), AllRules),
        print_rules(OStream, A2, AllRules),
        close(OStream),
        display('# Done.'), nl,
        halt.

main :- display('Usage: ./fwcompiler.pl <srcfile> <dstfile> <fwid>'), nl, halt(1).


