/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ExecBgr-2.cc,v 1.5 2001/09/22 01:39:29 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <signal.h>


#include "ExecBgr.hh"


using namespace libfwbuilder;


/************************************************************************
 *
 *    popen taken directly from Richard Stevens  Advanced Pogramming
 *    In the Unix Environment
 *
 *    Standard popen is inconvenient because there is no way to find out
 *    pid of the child process
 */

/* ptr to array allocated at run-time */
static int	maxfd;	/* from our open_max(), {Prog openmax} */
static pid_t   *childpid = NULL;


FILE* ExecBgr::popen(const char *cmd, 
		     const char* argv[], 
		     const char *type)  throw(FWException)
{
    int		i, pfd[2];
    pid_t	pid;
    FILE	*fp;

    /* only allow "r" or "w" */
    if ((type[0] != 'r' && type[0] != 'w') || type[1] != 0) {
	errno = EINVAL;		/* required by POSIX.2 */
	throw FWException(string("popen error: Invalid type '")+type[0]+"'");
	return(NULL);
    }

    if (childpid == NULL) {		/* first time through */
				/* allocate zeroed out array for child pids */
	maxfd = open_max();
	if ( (childpid = (pid_t*)calloc(maxfd, sizeof(pid_t))) == NULL) {
	    throw FWException("Memory allocation failure");
	    return(NULL);
	}
    }

    if (pipe(pfd) < 0) {
	throw FWException(string("pipe failed: ")+strerror(errno));
	return(NULL);	/* errno set by pipe() */
    }
    
    if ( (pid = fork()) < 0) {
	throw FWException(string("Could not fork: ")+strerror(errno));
	return(NULL);	/* errno set by fork() */
    }
    else if (pid == 0) {	/* child */
	if (*type == 'r') {
	    close(pfd[0]);
	    if (pfd[1] != STDOUT_FILENO) {
		dup2(pfd[1], STDOUT_FILENO);
		close(pfd[1]);
	    }
	} else {
	    close(pfd[1]);
	    if (pfd[0] != STDIN_FILENO) {
		dup2(pfd[0], STDIN_FILENO);
		close(pfd[0]);
	    }
	}
	/* close all descriptors in childpid[] */
	for (i = 0; i < maxfd; i++)
	    if (childpid[i] > 0)	close(i);

//	cerr << "Running '" << cmd << "' with arguments: " << endl;
//	for (i=0; argv[i]!=NULL; i++)
//	    cerr << i << ": '" << argv[i] << "'" << endl;

	dup2(STDOUT_FILENO, STDERR_FILENO);

	execvp(cmd, (char* const*)argv);

// if we got here there was an error

	char errmsg[2048];

	strcpy(errmsg,"Exec error (");
	strcat(errmsg,cmd);
	strcat(errmsg,") ");
	strcat(errmsg,strerror(errno));
	strcat(errmsg,"\n");

	write(STDOUT_FILENO, errmsg, strlen(errmsg) );

	_exit(1);
    }
    
    pgm_pid=pid;     

    /* parent */
    if (*type == 'r') {
	close(pfd[1]);
	if ( (fp = fdopen(pfd[0], type)) == NULL)      return(NULL);
    } else {
	close(pfd[0]);
	if ( (fp = fdopen(pfd[1], type)) == NULL)      return(NULL);
    }
    childpid[fileno(fp)] = pid;	/* remember child pid for this fd */
    return(fp);
}

int  ExecBgr::pclose(FILE *fp)
{
  int		fd, stat;
  pid_t	pid;

  if (childpid == NULL)
    return(-1);		/* popen() has never been called */

  fd = fileno(fp);
  if ( (pid = childpid[fd]) == 0)
    return(-1);		/* fp wasn't opened by popen() */

  childpid[fd] = 0;
  if (fclose(fp) == EOF)
    return(-1);

  while (waitpid(pid, &stat, 0) < 0)
    if (errno != EINTR)
      return(-1);	/* error other than EINTR from waitpid() */

  return(stat);	/* return child's termination status */
}

#ifdef	OPEN_MAX
static int	openmax = OPEN_MAX;
#else
static int	openmax = 0;
#endif

#define	OPEN_MAX_GUESS	256		/* if OPEN_MAX is indeterminate */
/* we're not guaranteed this is adequate */

int ExecBgr::open_max(void)
{
  if (openmax == 0) {		/* first time through */
    errno = 0;
    if ( (openmax = sysconf(_SC_OPEN_MAX)) < 0) {
      if (errno == 0)
	openmax = OPEN_MAX_GUESS;	/* it's indeterminate */
      else
	cerr << "sysconf error for _SC_OPEN_MAX";
    }
  }
  return(openmax);
}

/*******************************************************************/

