/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: FWObjectDatabaseGUI.cc,v 1.13 2001/12/24 06:02:36 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <iostream.h>

#include "config.h"

#include "fwbuilder/FWObject.hh"
#include "FWObjectDatabaseGUI.hh"

#include "fwbuilder/FWOptions.hh"

#include "fwbuilder/Host.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/Interval.hh"
#include "fwbuilder/IPService.hh"
#include "fwbuilder/ICMPService.hh"
#include "fwbuilder/TCPService.hh"
#include "fwbuilder/UDPService.hh"
#include "fwbuilder/CustomService.hh"

#include "fwbuilder/Rule.hh"
#include "fwbuilder/RuleElement.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/InterfacePolicy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/ObjectGroup.hh"
#include "fwbuilder/ServiceGroup.hh"
#include "fwbuilder/IntervalGroup.hh"

#include "FileSel.hh"
#include "MessageDialog.hh"
#include "main_window.hh"
#include "Resources.hh"

using namespace libfwbuilder;

/****************************************************************/

Group*             FWObjectDatabaseGUI::ScratchPad = NULL;

const string FWObjectDatabaseGUI::DB_TEMPLATE_FILE_NAME = "objects_init.xml" ;

FWObjectDatabaseGUI::FWObjectDatabaseGUI() : FWObjectDatabase()
{
    ScratchPad        = new Group();
    ScratchPad->setName("ScratchPad");
}

FWObjectDatabaseGUI::FWObjectDatabaseGUI(const string &filename) throw(FWException)
:FWObjectDatabase()
{
    ScratchPad        = new Group();
    ScratchPad->setName("ScratchPad");
    load(filename);
}

FWObjectDatabaseGUI::~FWObjectDatabaseGUI()
{
}

bool FWObjectDatabaseGUI::saveIfModified() throw(FWException)
{
    MessageDialog::DlgReturnValue action;
    
    action=MessageDialog::Question(
        "Some objects have been modified but not saved.
Do you want to save changes now ?");
    
    switch (action) 
    {
    case MessageDialog::OK_YES: save(); break; 
    case MessageDialog::NO:             break;         
    case MessageDialog::CANCEL:         return(false);
    }
    return(true);
}

void FWObjectDatabaseGUI::load(const string &f) throw(FWException)
{
    if(isDirty(true)) 
    {   // checking all the object recursively
	if (!saveIfModified()) 
            return;  // "Cancel"
    }
    
    if(f=="") 
    {
	FileSel *fs=new FileSel("Select file",
				Preferences::global_prefs->getWdir(),
				"*.xml");
	string df=fs->run();
	delete fs;
	if (df=="") return;
	data_file = df;
    } else
	data_file = f;
    
    if(data_file=="") 
        return;
    
    MessageBoxUpgradePredicate upgrade_predicate;
    FWObjectDatabase::load(data_file, &upgrade_predicate);

    ScratchPad->clearChildren();
    copyToScratchPad();
    
    if (main_w)
	main_w->showStatusbarMsg(string("Loaded data: ")+data_file);
}

void FWObjectDatabaseGUI::save() throw(FWException)
{
    if (data_file=="") 
    {
	saveAs();
	return;
    }

    saveFile(data_file);
}

void FWObjectDatabaseGUI::saveAs() throw(FWException)
{
    string df;

    FileSel *fs=new FileSel("Select file",
			    Preferences::global_prefs->getWdir(),
			    "*.xml");
    df=fs->run();
    delete fs;
    if (df=="") return;
    data_file=df;

    saveFile(data_file);
}

void FWObjectDatabaseGUI::saveFile(const string &filename) throw(FWException)
{
    FWObjectDatabase::saveFile(filename);
    if(main_w)
	main_w->showStatusbarMsg(string("Saved data as: ")+filename);
}


/*
 *  This method finds a scratch pad copy of object "obj"
 */
FWObject* FWObjectDatabaseGUI::getScratchCopy(FWObject *obj)
{
    if (obj==FWObjectDatabaseGUI::db) return FWObjectDatabaseGUI::ScratchPad;

    FWObject *p=FWObjectDatabaseGUI::ScratchPad->getById( obj->getId(), true );
    return p;
}

void FWObjectDatabaseGUI::copyToScratchPad()
{
    FWObject *p=ScratchPad;    *p=*this;
}

FWObject* FWObjectDatabaseGUI::create(const string& type_name)
{
    FWObject   *nobj=FWObjectDatabase::create(type_name);
    if(nobj==NULL)
        cerr << "unknown node: " << type_name << endl;
    
    return(nobj);
}


bool FWObjectDatabaseGUI::validateObjectForPositionInTree(FWObject* target,
							  FWObject* obj)
{
    if ( ! target->validateChild(obj) ) return false;

/* few  more strict checks for system groups */
    string where_id=target->getId();

    if (where_id==FWObjectDatabase::db->std.FirewallsId &&
	obj->getTypeName()!=Firewall::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.HostsId &&
	obj->getTypeName()!=Host::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.NetworksId &&
	obj->getTypeName()!=Network::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.IPServicesId &&
	obj->getTypeName()!=IPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.ICMPServicesId &&
	obj->getTypeName()!=ICMPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.TCPServicesId &&
	obj->getTypeName()!=TCPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.UDPServicesId &&
	obj->getTypeName()!=UDPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabase::db->std.CustomServicesId &&
	obj->getTypeName()!=CustomService::TYPENAME) return false;

    if (where_id==FWObjectDatabase::db->std.TimeId &&
	obj->getTypeName()!=Interval::TYPENAME) return false;

    return true;
}


FWObject*  FWObjectDatabaseGUI::newObject(const string& where_id , FWObject *o)
{
    if (o->getName()=="") {  // no name, use type description string instead
	o->setName( Resources::global_res->getObjResourceStr(o,"description") );
    }

    FWObject *s=db->create( o->getTypeName() );

    db->addAt(where_id,o);
    *s = *o;
    ScratchPad->addAt(where_id,s);

    return o;
}



FWObject*  FWObjectDatabaseGUI::newFirewall()
{
    FWObject *fw=db->create("Firewall");
    fw->add( new FirewallOptions() );
    fw->add( new Policy() );
    fw->add( new NAT() );
    (Firewall::cast(fw))->setDefaults();
    return newObject( db->std.FirewallsId , fw );
}

FWObject*  FWObjectDatabaseGUI::newHost()
{
    FWObject *h=db->create("Host");
//    FWObject *i=new Interface();
//    i->setName("unknown");
//    h->add( i );
    h->add( new HostOptions() );
    return newObject( db->std.HostsId , h);
}

FWObject*  FWObjectDatabaseGUI::newNetwork()
{
    FWObject *o=db->create("Network");
    return newObject( db->std.NetworksId,o );
}

FWObject*  FWObjectDatabaseGUI::newObjectGroup()
{
    FWObject *o=db->create("ObjectGroup");
    return newObject( db->std.ObjectGroupsId,o);
}

FWObject*  FWObjectDatabaseGUI::newIPService()
{
    FWObject *o=db->create("IPService");
    return newObject( db->std.IPServicesId, o);
}

FWObject*  FWObjectDatabaseGUI::newICMPService()
{
    FWObject *o=db->create("ICMPService");
    return newObject( db->std.ICMPServicesId,o);
}

FWObject*  FWObjectDatabaseGUI::newTCPService()
{
    FWObject *o=db->create("TCPService");
    return newObject( db->std.TCPServicesId,o);
}

FWObject*  FWObjectDatabaseGUI::newUDPService()
{
    FWObject *o=db->create("UDPService");
    return newObject( db->std.UDPServicesId,o);
}

FWObject*  FWObjectDatabaseGUI::newCustomService()
{
    FWObject *o=db->create("CustomService");
    return newObject( db->std.CustomServicesId,o);
}

FWObject*  FWObjectDatabaseGUI::newServiceGroup()
{
    FWObject *o=db->create("ServiceGroup");
    return newObject( db->std.ServiceGroupsId,o);
}

FWObject*  FWObjectDatabaseGUI::newInterval()
{
    FWObject *o=db->create("Interval");
    return newObject( db->std.TimeId,o);
}


