/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyListElement.cc,v 1.49 2001/12/27 06:48:47 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include <iostream.h>

#include "config.h"

#include "glademm_support.hh"
#include "helpers.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "BuiltinDialog.hh"
#include "gen_popup_menu.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Firewall.hh"
#include "Resources.hh"
#include "Preferences.hh"
#include "CommentDialog.hh"
#include "RuleOptionsDialog.hh"

#include "FWObjectDatabaseGUI.hh"

#include "fwbuilder/FWObjectReference.hh"
#include "fwbuilder/FWServiceReference.hh"
#include "fwbuilder/FWIntervalReference.hh"

#include "PolicyListElement.hh"
#include "PolicyListItem.hh"
#include "PolicyList.hh"

#include "fwbuilder/FWOptions.hh"

using namespace libfwbuilder;

static string obj_n; // to store name of the object to be opened in editor


/*******************************************************************/


PolicyListElement::PolicyListElement(gint r,gint c) : Gtk::Frame()
{
    row=r;
    col=c;
    width=height=0;

    set_flags( GTK_CAN_FOCUS );

    set_name("PolicyListElement");

    vbox=manage(new Gtk::VBox());
    add(*vbox);
    vbox->show();

    drag_dest_set ( GTK_DEST_DEFAULT_ALL,
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
    

    rel=NULL;
}

PolicyListElement::PolicyListElement(gint r,gint c, RuleElement *re) : Gtk::Frame()
{
    PolicyListItem    *itm;
    FWObject          *o, *oref;

    rel=re;

    row=r;
    col=c;
    width=height=0;

    set_flags( GTK_CAN_FOCUS );

    set_name("PolicyListElement");

    vbox=manage(new Gtk::VBox());
    add(*vbox);
    vbox->show();

    drag_dest_set ( GTK_DEST_DEFAULT_ALL,
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
    
    change_any_to_orig=false;
    if (RuleElementTSrc::isA(re) ||
	RuleElementTDst::isA(re) ||
	RuleElementTSrv::isA(re) )  change_any_to_orig=true;
    

    vector<FWObject*>::iterator m;
    for (m = rel->begin(); m!= rel->end(); ++m) {
	o=(*m);
	oref=NULL;
	if (FWObjectReference::isA(o) || FWServiceReference::isA(o) ||
	    FWIntervalReference::isA(o) )
	    oref=(FWReference::cast(o))->getPointer();

	if (oref!=NULL) {
	    itm= manage(new PolicyListObjectItem(oref, 
						 rel->getNeg(), 
						 change_any_to_orig) );
	    vbox->pack_start( *itm , true , true , 0 );
	    itm->show();
	}

    }

    selected_item=get_first_child();
    show_all();
}


Gtk::Widget* PolicyListElement::get_first_child()
{
    Gtk::Box_Helpers::Child    *ch;
    ch=*(vbox->children().begin());
    if (ch!=NULL) return ch->get_widget();
    else          return NULL;
}

Gtk::Widget* PolicyListElement::get_last_child()
{
    Gtk::Box_Helpers::Child    *ch;
    ch=*(vbox->children().rbegin());
    if (ch!=NULL) return ch->get_widget();
    else          return NULL;
}

void PolicyListElement::set_data_changed_flag(bool f)
{
    PolicyList *pl=(PolicyList*)get_user_data();
    pl->set_data_changed_flag( f );
}

/*
 *   We keep track of our physical size
 */
void PolicyListElement::size_allocate_impl(GtkAllocation* all)
{
    x=all->x;
    y=all->y;
    width=all->width;
    height=all->height;

    Gtk::Frame::size_allocate_impl(all);
}



void PolicyListElement::remove_item_from_policy(PolicyListItem* pi)
{
    assert (rel!=NULL);

    if (pi==NULL) return;
    if (rel->isAny()) return;

    FWObject *obj = pi->getObject();
    assert(obj!=NULL);

    vbox->remove(*pi);   // do not need to delete because it is managed
    
    rel->removeRef(obj);  // adds "any" if obj was the last one

    if (rel->isAny()) {

	obj=obj->getRoot()->getById( rel->getAnyElementId() , true );

	PolicyListItem* itm= 
	    manage(new PolicyListObjectItem( obj, 
					     rel->getNeg(), 
					     change_any_to_orig) );
	vbox->pack_start( *itm , true , true , 0 );
	itm->show();

	select();

	if (rel->getNeg())  toggleNeg();
    }	

    selected_item=get_first_child();

    set_data_changed_flag(true);
}

bool PolicyListElement::add_item_to_policy(FWObject *obj)
{
    if (rel==NULL) return(false);
    if (! rel->validateChild(obj) ) return(false);

    if (rel->getAnyElementId()==obj->getId()) return(false);

/*
 * we do not want duplicates
 * calling getById with third parameter to follow references
 */
    if (rel->getById(obj->getId(),true,true)!=NULL) return(false);
    
    if (rel->isAny()) {
	// element contains "Any". Remove corresponding PolicyListItem
	// from PolicyListElement before adding new item 
	//
        // there should be only one child widget, too
	//
	// this operation invalidates  selected_item
/*
	Gtk::Box_Helpers::BoxList::iterator bl_i;
	Gtk::Box_Helpers::Child    *ch;
	Gtk::Widget                *cc;
	bl_i=vbox->children().begin();
	if ( (ch=*bl_i)!=0 ) {
	    cc=ch->get_widget();
	    vbox->remove( *cc );
	}
*/
	Gtk::Widget *w=get_first_child();
	if (w) vbox->remove(*w);

	selected_item=NULL;

//	rel->clearChildren();
    }


    rel->addRef(obj);

    deselect();

    PolicyListItem* itm= 
	manage(new PolicyListObjectItem( obj, 
					 rel->getNeg(),change_any_to_orig) );
    vbox->pack_start( *itm , true , true , 0 );
    itm->show();
	
    selected_item=get_last_child();
	
    select();
	
    set_data_changed_flag(true);

    return(true);
}


void PolicyListElement::toggleNeg()
{
    rel->toggleNeg();

    Gtk::Box_Helpers::BoxList::iterator bl_i;
    for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); bl_i++) {
	((PolicyListItem*)(*bl_i)->get_widget())->setNeg( rel->getNeg() );
    }

    set_data_changed_flag(true);
}

void PolicyListElement::activate()
{
    set_state(GTK_STATE_PRELIGHT);
    Gtk::Box_Helpers::BoxList::iterator bl_i;
    for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); bl_i++) {
       ((PolicyListItem*)(*bl_i)->get_widget())->set_state(GTK_STATE_PRELIGHT);
    }
}

void PolicyListElement::deactivate()
{
    set_state(GTK_STATE_NORMAL);
    Gtk::Box_Helpers::BoxList::iterator bl_i;
    for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); bl_i++) {
	((PolicyListItem*)(*bl_i)->get_widget()) ->set_state(GTK_STATE_NORMAL);
    }
}


void PolicyListElement::select(bool select_row)
{
    if (select_row) {
	PolicyList* pp=(PolicyList*)get_parent();
    
	pp->deselect_current();
	pp->activate_row(row);
	pp->set_current_selected(this);
    }

    set_state(GTK_STATE_PRELIGHT);
    request_focus();
}

void PolicyListElement::set_current_selected(Gtk::Widget *itm)
{
    selected_item=itm;
}

void PolicyListElement::select_first_child()
{
    selected_item=get_first_child();
    if (selected_item!=NULL)
	((PolicyListItem*)selected_item)->set_state(GTK_STATE_SELECTED);
}

void PolicyListElement::select_last_child()
{
    selected_item=get_last_child();
    if (selected_item!=NULL)
	((PolicyListItem*)selected_item)->set_state(GTK_STATE_SELECTED);
}

void PolicyListElement::deselect_all()
{
    Gtk::Box_Helpers::BoxList::iterator bl_i;
    Gtk::Box_Helpers::Child            *ch;
    for (bl_i=(vbox->children()).begin(); 
	 bl_i!=(vbox->children()).end(); bl_i++) {
	ch=(*bl_i);
	if (ch!=NULL) 
	    ((PolicyListItem*)ch->get_widget())->set_state(GTK_STATE_PRELIGHT);
    }
    selected_item=NULL;
}

void PolicyListElement::deselect()
{
    deselect_all();
    set_state(GTK_STATE_PRELIGHT);
}


void PolicyListElement::request_focus()
{
    if (!has_focus()) {
	grab_focus();
	draw_focus();
    }
}

void PolicyListElement::drag_data_received_impl(GdkDragContext   *context,
						gint              x,
						gint              y,
						GtkSelectionData *data,
						guint             info,
						guint32           time)
{
    void           *ptr;
    FWObject       *obj;
    PolicyListItem *itm=NULL;
    char           obj_id[64];

    Gdk_DragContext gdc ( context );
    Gtk::Widget::drag_finish ( gdc , false, false, time );

    if ( data->length >= 0 ) {
/*
 *   ObjectTree sends FWObject* via drag&drop mechanism, while
 *   PolicyListItem sends itself (PolicyListItem*, that is).  We will
 *   differentiate them using data->format here. This allows for
 *   different actions to be performed depending on where dragged
 *   object has been taken from
 *
 *   switch (data->format) {
 *
 *     case 6:    moving the whole rule, ptr is PolicyListItem*
 *     case 7:    moving object, ptr is PolicyListItem*
 *     case 8:    moving object, ptr is char* ( its Id )
 *    
 */
	obj=NULL;

	if (data->format==7) {
	    memcpy(&ptr,data->data,data->length);
	    itm=(PolicyListItem*)ptr;
	    obj=itm->getObject();
	}

	if (data->format==8 && rel!=NULL) {
	    memcpy(obj_id,(char*)data->data,data->length);
	    obj=rel->getRoot()->getById( obj_id , true );
	}

	if (obj==NULL) return;
	
	if ( add_item_to_policy(obj) ) {
/* object has successfully been added */
	    if (itm) {
		PolicyListElement* pe=itm->getParentPolicyListElement();
		if (pe)
		    pe->remove_item_from_policy(itm);
	    }
	}
    }
}

gint PolicyListElement::key_press_event_impl(GdkEventKey* ev)
{
    PolicyList* pp=(PolicyList*)get_parent();

    if ( selected_item==NULL ) goto another_element;

    switch (ev->keyval) {

    case GDK_space:
	popup_menu( (PolicyListItem*)selected_item );
	break;

    case GDK_Up:
    {
	Gtk::Box_Helpers::BoxList::reverse_iterator i;
	i=vbox->children().find( *selected_item );
	if (i!=vbox->children().rend()) {
//	    i++;
	    if (i!=vbox->children().rend())  {
		deselect_all();
		PolicyListItem *itm=(PolicyListItem*)((*i)->get_widget());
		itm->set_state(GTK_STATE_SELECTED);
		selected_item=itm;
	    }
	    else goto another_element;
	}
	else goto another_element;
	break;
    }
    case GDK_Down:
    {
	Gtk::Box_Helpers::BoxList::iterator i;
	i=vbox->children().find( *selected_item );
	if (i!=vbox->children().end()) {
	    i++;
	    if ( i!=vbox->children().end())  {
		deselect_all();
		PolicyListItem *itm=(PolicyListItem*)((*i)->get_widget());
		itm->set_state(GTK_STATE_SELECTED);
		selected_item=itm;
	    }
	    else goto another_element;
	}
	else goto another_element;
	break;
    }
    case GDK_F5:
    {
	cerr << "------- PolicyListElement dump:\n";
	cerr << "row=" << get_row() << "  col=" << get_col() << endl;
	Gtk::Box_Helpers::BoxList::iterator bl_i;
	Gtk::Box_Helpers::Child    *ch;
	Gtk::Widget                *cc;
	for (bl_i=vbox->children().begin(); bl_i!=vbox->children().end(); ++bl_i) {
	    ch= *bl_i;
	    cc=ch->get_widget();
	    cerr << " child widget: " << cc << endl;
	}
	rel->dump(true,true);

	break;
    }

    default:
	
 another_element:

	switch (ev->keyval ) {
	case GDK_Up:     pp->move_focus(GTK_DIR_UP);     break;
	case GDK_Down:   pp->move_focus(GTK_DIR_DOWN);   break;
	case GDK_Left:   pp->move_focus(GTK_DIR_LEFT);   break;
	case GDK_Right:  pp->move_focus(GTK_DIR_RIGHT);  break;
	default:         return(false);
	}
    }

    return(true);
}

void PolicyListElement::popup_menu(PolicyListItem *itm)
{
    FWObject               *obj;
    gen_popup_menu         *gpm;
    PolicyList             *pl=(PolicyList*)get_user_data();

    g_assert(rel!=NULL);



    obj=itm->getObject();

    const char *menu_items0[] = 
    { "Edit","","Copy","Cut","Paste","","Negate",NULL } ;
    gpm=new gen_popup_menu( menu_items0 );

    if ( rel->isAny() ) {
	Gtk::Widget *menu_itm;
	menu_itm=find_widget("Edit",gpm);   
	if (menu_itm) menu_itm->set_sensitive(false);
	menu_itm=find_widget("Negate",gpm); 
	if (menu_itm) menu_itm->set_sensitive(false);
	menu_itm=find_widget("Copy",gpm);   
	if (menu_itm) menu_itm->set_sensitive(false);
	menu_itm=find_widget("Cut",gpm);    
	if (menu_itm) menu_itm->set_sensitive(false);
    }

    gpm->popup(0,0);
    gint menu_choice=gpm->run();
    delete gpm;

    deselect();

    switch (menu_choice) {
    case 0:    // edit
/*
 *  pass object via clipboard to avoid passing reference to the stack
 *
 *  do not edit "Any"
 */
	if ( obj!=NULL &&  ! rel->isAny() ) {
	    FWObjectClipboard::obj_clipboard->putObject( obj );
// open_item is a signal
	    pl->open_item( FWObjectClipboard::obj_clipboard->getObject() );
	}
	return;
    case 2:    // copy
	if ( obj!=NULL && ! rel->isAny() ) {
	    FWObjectClipboard::obj_clipboard->putObject( obj );
	}
	return;
    case 3:    // cut
	if ( obj!=NULL && ! rel->isAny() ) { 
	    FWObjectClipboard::obj_clipboard->putObject( obj );
	    remove_item_from_policy(itm);
	}
	return;
    case 4:    // paste
    {
	obj=FWObjectClipboard::obj_clipboard->getObject();
	if ( obj!=NULL ) {
	    add_item_to_policy(obj);
	    set_data_changed_flag(true);
	}
    }
    return;

    case 6:    // negate
	toggleNeg();   
	return;
    }
    select();
}

/*******************************************************************/

static int last_drag_motion_x=-1;


PolicyListRuleNum::PolicyListRuleNum(gint r,gint c,gint rule_n,bool neg) :
    PolicyListElement(r,c)
{
    PolicyListRuleNumItem    *itm;
    char  str[64];

    sprintf(str,"%02d",rule_n);
    itm= manage(new PolicyListRuleNumItem( str , neg ));
    vbox->pack_start( *itm , true , true , 0 );
    itm->show();

    selected_item=get_first_child();
    set_name("PolicyListRuleNum");

    last_drag_motion_x=-1;
}

gboolean PolicyListRuleNum::drag_motion_impl(GdkDragContext* context,
					     gint x,gint y,guint time)
{
    if (last_drag_motion_x == -1) last_drag_motion_x=x;
    
    if (last_drag_motion_x!=x) {

    }
    return false;
}

void PolicyListRuleNum::drag_data_received_impl(GdkDragContext   *context,
						gint              x,
						gint              y,
						GtkSelectionData *data,
						guint             info,
						guint32           time)
{
    PolicyListItem *itm=NULL;

    Gdk_DragContext gdc ( context );
    Gtk::Widget::drag_finish ( gdc , false, false, time );

    if ( data->length >= 0 ) {
/*
 *   ObjectTree sends FWObject* via drag&drop mechanism, while
 *   PolicyListItem sends itself (PolicyListItem*, that is).  We will
 *   differentiate them using data->format here. This allows for
 *   different actions to be performed depending on where dragged
 *   object has been taken from
 *
 *   switch (data->format) {
 *
 *     case 6:    moving the whole rule, ptr is PolicyListItem*
 *     case 7:    moving object, ptr is PolicyListItem*
 *     case 8:    moving object, ptr is FWObject*
 *    
 */
	void* ptr;
	memcpy(&ptr,data->data,data->length);

	if (data->format==6) {
	    itm=(PolicyListItem*)ptr;
	    PolicyListElement  *pe=itm->getParentPolicyListElement();
	    gint src_rule = PolicyList::get_rule_by_row( pe->get_row() );
	    gint dst_rule = PolicyList::get_rule_by_row( get_row() );

/* rule number src_rule moves right above rule dst_rule */
	    PolicyList         *pl=(PolicyList*)get_user_data();
	    RuleSet            *pol=pl->getRuleSet();
	    if ( pol->moveRule(src_rule,dst_rule) ) {
		pl->schedule_rebuild((src_rule<dst_rule)?dst_rule-1:dst_rule);
		set_data_changed_flag(true);
	    }
	    return;
	}
    }
}

      

void PolicyListRuleNum::popup_menu(PolicyListItem *itm)
{
    gen_popup_menu         *gpm;
    PolicyList             *pl=(PolicyList*)get_user_data();
    RuleSet                *rs=pl->getRuleSet();
    gint rule = PolicyList::get_rule_by_row( get_row() );

    const char *menu_items0[] = 
    { "Insert new rule above",
      "Add new rule below",
      "Remove this rule",
      "Move this rule up",
      "Move this rule down",
      "",
      "Copy",
      "Cut",
      "Paste above",
      "Paste below",
      "",
      NULL,
      NULL } ;

    if (pl->isRuleDisabled( rule ))  menu_items0[11]="Enable";
    else                             menu_items0[11]="Disable";

    gpm=new gen_popup_menu( menu_items0 );
    gpm->popup(0,0);
    gint menu_choice=gpm->run();
    delete gpm;

    deselect();

    switch (menu_choice) {
    case 0:
	pl->insertRuleBefore( rule );
	set_data_changed_flag(true);
	break;
    case 1:
	pl->appendRuleAfter( rule );
	set_data_changed_flag(true);
	break;
    case 2:
	pl->delRule( rule );
	set_data_changed_flag(true);
	break;
    case 3:
	pl->moveRuleUp( rule );
	set_data_changed_flag(true);
	break;
    case 4:
	pl->moveRuleDown( rule );
	set_data_changed_flag(true);
	break;

    case 6:  // copy
    {
	Rule *r=rs->getRuleByNum(rule);
	FWObjectClipboard::obj_clipboard->putObject( r );
    }
    break;

    case 7:  // cut
    {
	Rule *r=rs->getRuleByNum(rule);

	FWObjectClipboard::obj_clipboard->putObject( r );
	pl->delRule( rule );
	set_data_changed_flag(true);
    }
    break;

    case 8:  // paste above
    {
	FWObject *obj;
	if ( (obj=FWObjectClipboard::obj_clipboard->getObject())!=NULL &&
	     dynamic_cast<Rule*>(obj)!=NULL ) {
	    pl->insertRuleBefore( rule , dynamic_cast<Rule*>(obj) );
	    set_data_changed_flag(true);
	}
    }
    break;

    case 9:  // paste below
    {
	FWObject *obj;
	if ( (obj=FWObjectClipboard::obj_clipboard->getObject())!=NULL &&
	     dynamic_cast<Rule*>(obj)!=NULL ) {
	    pl->appendRuleAfter( rule , dynamic_cast<Rule*>(obj) );
	    set_data_changed_flag(true);
	}
    }
    break;

    case 11:
	if (pl->isRuleDisabled( rule )) pl->enableRule( rule );
	else                            pl->disableRule( rule );
	set_data_changed_flag(true);
    }
    select();
}

/*******************************************************************/


PolicyListRuleAction::PolicyListRuleAction(gint r,gint c, Rule *ru)
  :    PolicyListElement(r,c)
{
    rule=ru;
    rebuild();
    selected_item=get_first_child();
}
      
void PolicyListRuleAction::popup_menu(PolicyListItem *itm)
{
    gen_popup_menu         *gpm;

    const char *menu_choices[] = 
    { "Accept",
      "Deny",
      "Reject",
      NULL } ;
    gpm=new gen_popup_menu( menu_choices );
    gpm->popup(0,0);
    gint res=gpm->run();
    delete gpm;

    if (res>=0) {
	string s=menu_choices[res];
	rule->setStr("action",s);
	rebuild();

	set_data_changed_flag(true);
    }

}

void PolicyListRuleAction::rebuild() 
{
    bool o=false;

    deselect();

    PolicyListItem *itm;

    Gtk::Widget *w=get_first_child();
    if (w) { o=true; vbox->remove(*w); }

    string action=rule->getStr("action");
    if (action=="") action="Deny";

    string icn=	Resources::global_res->getIconPath(action);
    assert(icn!="");

    itm= manage(new PolicyListItem( icn,action ));
    vbox->pack_start( *itm , true , true , 0 );
    itm->show();

    if (o) itm->set_state(GTK_STATE_SELECTED);
}

/*******************************************************************/


PolicyListRuleLog::PolicyListRuleLog(gint r,gint c, Rule *ru) :
    PolicyListElement(r,c)
{
    rule=ru;
    rebuild();
    selected_item=get_first_child();
}

void PolicyListRuleLog::rebuild() 
{
    bool o=false;

    deselect();

    PolicyListItem *itm;

    Gtk::Widget *w=get_first_child();
    if (w) { o=true; vbox->remove(*w); }

    itm= manage(new PolicyListItem( "","" ));

    bool    log=rule->getBool("log");
    if (log) {
	string icn= Resources::global_res->getIconPath("Log");
	assert(icn!="");
	itm->addIcon(icn);
	itm->setQuickView("Logging enabled");
    }
    else {
	itm->setQuickView("Logging disabled");
    }
    vbox->pack_start( *itm , true , true , 0 );
    itm->show();

    if (o) itm->set_state(GTK_STATE_SELECTED);
}
      
void PolicyListRuleLog::popup_menu(PolicyListItem *itm)
{
    gen_popup_menu         *gpm;
    string icn;

    const char *menu_choices[] = 
    { "Log",
      "Do not log",
      NULL } ;
    gpm=new gen_popup_menu( menu_choices );
    gpm->popup(0,0);
    gint res=gpm->run();
    delete gpm;

    if (res>=0) {
	rule->setBool("log", (res==0) );
	rebuild();

	set_data_changed_flag(true);
    }

}
/*******************************************************************/


PolicyListRuleComment::PolicyListRuleComment(gint r,gint c, Rule *ru) : 

    PolicyListElement(r,c)
{
    rule=ru;
    rebuild();
    selected_item=get_first_child();
}

void PolicyListRuleComment::rebuild() 
{
    bool o=false;

    deselect();

    PolicyListItem *itm;

    Gtk::Widget *w=get_first_child();
    if (w) { o=true; vbox->remove(*w); }

    itm= manage(new PolicyListCommentItem( rule->getComment() ));
    vbox->pack_start( *itm , true , true , 0 );
    itm->show();

    if (o) itm->set_state(GTK_STATE_SELECTED);
}
      
void PolicyListRuleComment::popup_menu(PolicyListItem *itm)
{
    gen_popup_menu         *gpm;
    //PolicyList             *pl=(PolicyList*)get_user_data();

    const char *menu_choices[] = 
    { "Edit Comment",
      NULL } ;

    gpm=new gen_popup_menu( menu_choices );
    gpm->popup(0,0);
    int res=gpm->run();
    delete gpm;

    if (res==0) {
	CommentDialog *cmdlg=new CommentDialog( rule->getComment() );
	string str=cmdlg->run();
	delete cmdlg;
	rule->setComment(str);
	rebuild();
	set_data_changed_flag(true);
    }
}

/*******************************************************************/

PolicyListRuleDir::PolicyListRuleDir(gint r,gint c, Rule *ru) : 
    PolicyListElement(r,c)
{
    rule=ru;
    rebuild();
    selected_item=get_first_child();
}

void PolicyListRuleDir::rebuild() 
{
    bool o=false;

    deselect();

    PolicyListItem *itm;

    Gtk::Widget *w=get_first_child();
    if (w) { o=true; vbox->remove(*w); }

    itm= manage(new PolicyListItem( "","" ));

    string dir=rule->getStr("direction");
    string icn=	Resources::global_res->getIconPath(dir);

    itm->addIcon(icn);
    itm->addLabel(dir);

    vbox->pack_start( *itm , true , true , 0 );
    itm->show();

    if (o) itm->set_state(GTK_STATE_SELECTED);
}
      
void PolicyListRuleDir::popup_menu(PolicyListItem *itm)
{
    gen_popup_menu         *gpm;

    const char *menu_choices[] = 
    { "Inbound",
      "Outbound",
      "Both",
      NULL } ;

    gpm=new gen_popup_menu( menu_choices );
    gpm->popup(0,0);
    gint res=gpm->run();
    delete gpm;

    if (res>=0) {
	rule->setStr("direction", menu_choices[res] );
	rebuild();

	set_data_changed_flag(true);
    }
}

/*******************************************************************/


PolicyListRuleOpt::PolicyListRuleOpt(gint r,gint c, Rule *ru) : 
    PolicyListElement(r,c)
{
    rule=ru;
    rebuild();
    selected_item=get_first_child();
}

void PolicyListRuleOpt::rebuild() 
{
    bool o=false;

    deselect();

    PolicyListItem *itm;

    Gtk::Widget *w=get_first_child();
    if (w) { o=true; vbox->remove(*w); }

    itm= manage(new PolicyListItem( "","" ));

    string qv_text;
    string icn;

    itm->clear();

    bool    log=rule->getBool("log");
    if (log) {
	icn= Resources::global_res->getIconPath("Log");
	assert(icn!="");
	itm->addIcon(icn);
	qv_text+="Logging enabled";
    }
    else {
	qv_text+="Logging disabled";
    }

    if (! rule->getOptionsObject()->isDefault()) {
	icn=Resources::global_res->getIconPath("Options");
	assert(icn!="");
	itm->addIcon(icn);
	qv_text+="\nNon-default options";
    }

    itm->setQuickView( qv_text );

    vbox->pack_start( *itm , true , true , 0 );
    itm->show();

    if (o) itm->set_state(GTK_STATE_SELECTED);
}
      
void PolicyListRuleOpt::popup_menu(PolicyListItem *itm)
{
    gen_popup_menu         *gpm;
    string icn;

    const char *menu_choices[] = 
    { "Modify Options",
      "",
      "Turn logging ON",
      "Turn logging OFF",
      NULL } ;

    gpm=new gen_popup_menu( menu_choices );
    gpm->popup(0,0);
    gint res=gpm->run();
    delete gpm;

    if (res>=0) {
	switch (res) {
	case 0: 
	{
	    RuleOptionsDialog *dlg=new RuleOptionsDialog(rule->getOptionsObject());
	    dlg->run();
	    delete dlg;
	    rebuild();
	    break;
	}
	case 2:
	case 3:
	    rule->setBool("log", (res==2) );
	    rebuild();
	    break;
	}

	set_data_changed_flag(true);
    }
}










