/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyListItem.cc,v 1.42 2001/12/29 10:06:35 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include <gdk_imlib.h>

#include "ObjectQuickView.hh"

#include "glademm_support.hh"
#include "helpers.hh"
#include "fwbuilder/FWObjectDatabase.hh"
#include "BuiltinDialog.hh"
#include "gen_popup_menu.hh"

#include "fwbuilder/Host.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/Firewall.hh"

#include "PolicyListItem.hh"
#include "PolicyListElement.hh"
#include "PolicyList.hh"

#include "main_window.hh"

#define MAX_COMMENT_FIELD_WIDTH   32

using namespace libfwbuilder;


GdkImlibImage* PolicyListItem::neg_icon=NULL;


PolicyListItem::PolicyListItem(const string &pixfl, 
			       const string &label_text,
			       bool  neg_flag) 
{
    object=NULL;

    icon_w=0;
    icon_h=0;
    x=y=0;

    neg=neg_flag;
    label=NULL;

    quick_view_txt="";

    overlap=false;

    x=0;
    y=0;
    width=0;
    height=0;

    loadNegIcon();

    icon_size=
	Resources::global_res->getResourceInt("/FWBuilderResources/UI/SmallIconSize");

    set_name("PolicyListItem");
    set_flags( GTK_CAN_FOCUS );

    tbl=manage(new Gtk::Table(1,1));
    add(*tbl);
    tbl->show();

    if (pixfl!="") addIcon(pixfl);

    addLabel(label_text);

    show_all();

    button_press_event.connect(
	SigC::slot(this,&PolicyListItem::on_button_press_event));
    button_release_event.connect_after(
	SigC::slot(this,&PolicyListItem::on_button_release_event));

    enter_notify_event.connect( 
	SigC::slot(this,&PolicyListItem::on_enter_notify_event));
    leave_notify_event.connect( 
	SigC::slot(this,&PolicyListItem::on_leave_notify_event));

}

PolicyListItem::~PolicyListItem()
{
    for (unsigned i=0; i<pixmaps.size(); ++i) {
	gdk_imlib_kill_image( pixmaps[i] );
    }
    pixmaps.clear();
    n_pixmaps.clear();


    deactivateObjectQuickView();
}

PolicyListElement* PolicyListItem::getParentPolicyListElement()
{
    Gtk::Widget       *p=this;
    do { p=p->get_parent(); } 
    while ( dynamic_cast<PolicyListElement*>(p)==NULL );
    return (PolicyListElement*)p;
}


void PolicyListItem::activateObjectQuickView()
{
    if ( object!=NULL || quick_view_txt!="" ) {

	PolicyListElement *pl_el=getParentPolicyListElement();
	RuleElement       *rel=pl_el->get_rule_element();
	if (rel!=NULL && rel->isAny()) return;

	ObjectQuickView* ov=ObjectQuickView::getInstance();
	if (object) ov->setObject( object );
	else        ov->setText( quick_view_txt );
	ov->attachTo(this);
	ov->activate();
    }
}

void PolicyListItem::deactivateObjectQuickView()
{
    ObjectQuickView* ov=ObjectQuickView::getInstance();
    ov->deactivate();
}

gint PolicyListItem::on_enter_notify_event(GdkEventCrossing* ev)
{
    activateObjectQuickView();
    return(true);
}

gint PolicyListItem::on_leave_notify_event(GdkEventCrossing* ev)
{
    deactivateObjectQuickView();
    return(true);
}

gint PolicyListItem::key_press_event_impl(GdkEventKey* ev)
{
    return (getParentPolicyListElement()->key_press_event_impl(ev));
}

gint PolicyListItem::on_button_press_event(GdkEventButton *ev)
{
    PolicyListElement *pe=getParentPolicyListElement();

    deactivateObjectQuickView();

    if ( ev->type == GDK_BUTTON_PRESS) {
	pe->deselect_all();
	pe->select();
	pe->set_current_selected(this);
	set_state(GTK_STATE_SELECTED);
    }

    return( true );
}

gint PolicyListItem::on_button_release_event(GdkEventButton *ev)
{
    PolicyListElement *pe=getParentPolicyListElement();

    deactivateObjectQuickView();

    if ( ev->type == GDK_BUTTON_RELEASE && ev->button==3 ) {
	if (get_state()!=GTK_STATE_SELECTED) {
	    pe->deselect_all();
	    pe->select();
	    pe->set_current_selected(this);
	    set_state(GTK_STATE_SELECTED);
	}
	pe->popup_menu(this);

	gtk_signal_emit_stop_by_name( GTK_OBJECT( gtkobj() ),
				      "button_release_event" );

	return(true);
    }
    return( true );
}


void PolicyListItem::setQuickView(const string &txt)
{
    quick_view_txt=txt;
}

void PolicyListItem::state_changed_impl(GtkStateType p0)
{
    deactivateObjectQuickView();
    Gtk::EventBox::state_changed_impl(p0);
}


void PolicyListItem::addLabel(const string &txt)
{
    if (label!=NULL) {
	tbl->remove(*label);
	label=NULL;
    }
    label= manage(new Gtk::Label(txt));

    int  n=pixmaps.size();

    if (overlap) n--;

    tbl->resize(1,n+1);
    tbl->attach(*label, n,n+1,0,1, 0 );
    label->show();
}

string PolicyListItem::getLabel()
{
    return(label->get_text());
}

void  PolicyListItem::addIcon(const string &icn_file)
{
    GdkImlibImage         *im;

    int  n=pixmaps.size();

    if (! icn_file.empty() && 
	(im=gdk_imlib_load_image( (char*)(icn_file.c_str())))!=NULL ) {

	pixmaps[n]=im;
	Gtk::Pixmap* pm=combineWithNeg(im);
	n_pixmaps[n]=pm;

	tbl->resize(1,n+2);
	tbl->attach(*pm, n,n+1,0,1, 0 );
	pm->show();
    }
}

void PolicyListItem::clear()
{
// remove all the children

    tbl->hide();
    remove();
    tbl=manage(new Gtk::Table(1,1));
    add(*tbl);
    tbl->show();

    for (unsigned i=0; i<pixmaps.size(); ++i) {
	gdk_imlib_kill_image( pixmaps[i] );
//	Gdk_Pixmap p;
//	Gdk_Bitmap m;
//	n_pixmaps[i]->get(p,m);
//	gdk_imlib_free_pixmap( p );
    }
    pixmaps.clear();
    n_pixmaps.clear();

    label=NULL;
}

void PolicyListItem::loadNegIcon()
{
    if ( neg_icon==NULL ) {

	string  icn_file=
	    Resources::global_res->getResourceStr("/FWBuilderResources/Paths/Icndir");
	icn_file += "/";
	icn_file += 
	    Resources::global_res->getResourceStr("/FWBuilderResources/UI/Icons/neg");
	
	assert (! icn_file.empty() );

	neg_icon=gdk_imlib_load_image( (char*)(icn_file.c_str()));
    }
}

void PolicyListItem::setNeg(bool neg_flag)
{
    neg=neg_flag;

    for (unsigned i=0; i<pixmaps.size(); ++i) {
	Gtk::Pixmap *pm=n_pixmaps[i];
	tbl->remove( *pm );
	pm = combineWithNeg( pixmaps[i] );
	n_pixmaps[i]= pm;
	tbl->attach(*pm, i,i+1,0,1, 0 );
	pm->show();
    }
}

Gtk::Pixmap* PolicyListItem::combineWithNeg(GdkImlibImage* im)
{
    GdkPixmap *p;
    GdkBitmap *m;

    int icon_w=im->rgb_width;
    int icon_h=im->rgb_height;
    int res;
    Gtk::Pixmap  *npm;

    res=gdk_imlib_render( im , icon_w , icon_h );
    assert(res==1);

    if (neg) {

	int minw=min(im->rgb_width,neg_icon->rgb_width);
	int minh=min(im->rgb_height,neg_icon->rgb_height);
	int sx=(im->rgb_width-minw)/2;
	int sy=(im->rgb_height-minh)/2;
	GdkImlibImage *nim=gdk_imlib_crop_and_clone_image(im,sx,sy,minw,minh);
	assert(nim!=NULL);

	sx=(neg_icon->rgb_width-minw)/2;
	sy=(neg_icon->rgb_height-minh)/2;
	GdkImlibImage *nneg=gdk_imlib_crop_and_clone_image(neg_icon,sx,sy,minw,minh);
	assert(nim!=NULL);


	unsigned char *pixel, *negpixel;
	int            iw,ih;

	pixel    = nim->rgb_data; 
	negpixel = nneg->rgb_data;

	GdkImlibColor tr_color;
	gdk_imlib_get_image_shape(nneg,&tr_color);

	for (iw=0; iw<minw; iw++) {
	    if (iw>=minw) continue;
	    for (ih=0; ih<minh; ih++) {
		if (ih>=minh) continue;

		if ( *negpixel     == tr_color.r && 
 		     *(negpixel+1) == tr_color.g &&
 		     *(negpixel+2) == tr_color.b ) {
		    pixel   +=3;
		    negpixel+=3;
		    continue; // transparent pixel
		}

		for (int i=0; i<3; ++i) {
		    *pixel = *negpixel;
		    pixel++; negpixel++;
		}
	    }
	}

	gdk_imlib_render(nim,nim->rgb_width,nim->rgb_height);

	p=gdk_imlib_move_image(nim);
	m=gdk_imlib_move_mask(nim);
	npm= manage(new Gtk::Pixmap( p, m ) );

//	gdk_imlib_free_pixmap(nneg);
	gdk_imlib_destroy_image(nneg);

    } else {
	p=gdk_imlib_move_image(im);
	m=gdk_imlib_move_mask(im);
	npm= manage(new Gtk::Pixmap( p, m ) );
    }
    return npm;
}

void PolicyListItem::setOverlap(bool ovr)
{
    overlap=ovr;
}


/************************************************************************/

PolicyListObjectItem::PolicyListObjectItem(FWObject *o,
					   bool  neg_flag,
					   bool  change_any_2_orig) :
    PolicyListItem( Resources::getXPMIconFileName(o, ""),  o->getName(),  neg_flag  )
{
    object=o;

    drag_source_set ( 
	static_cast<GdkModifierType>(GDK_BUTTON1_MASK) ,
	target_table, n_targets, 
	static_cast<GdkDragAction>(GDK_ACTION_COPY) );


    drag_begin.connect(
	slot(this,&PolicyListObjectItem::source_drag_begin));
    drag_data_get.connect(
	slot(this,&PolicyListObjectItem::source_drag_data_get));

    if (change_any_2_orig &&  getLabel()=="Any")	addLabel("Original");

}

void PolicyListObjectItem::source_drag_begin(GdkDragContext     *context)
{
    GdkPixmap *pmap;
    GdkBitmap *mask;
    Gdk_Colormap cmap ( get_colormap () );

    string  icn=Resources::global_res->getIconPath("Drag");

    if (gdk_imlib_load_file_to_pixmap((char*)(icn.c_str()), &pmap, &mask)){
	Gdk_Pixmap drag_icon=Gdk_Pixmap(pmap);
	Gdk_Bitmap drag_mask=Gdk_Bitmap(mask);
	drag_source_set_icon(cmap, drag_icon, drag_mask);
	gdk_imlib_free_pixmap(pmap);
    }
}

void PolicyListObjectItem::source_drag_data_get  ( GdkDragContext     *context,
					   GtkSelectionData   *selection_data,
						   guint               info,
						   guint32             time )
{
    PolicyListObjectItem *plitm=this;

    gtk_selection_data_set (selection_data,
			    gdk_atom_intern("PolicyListObjectItem",TRUE),
			    7,
			    (const guchar*)(&plitm),
			    sizeof(PolicyListItem*) );
}

/************************************************************************/

PolicyListRuleNumItem::PolicyListRuleNumItem(const string& label,
					     bool neg_flag) :
    PolicyListItem(Resources::global_res->getIconPath("Blank"),
		   "",neg_flag)
{
    set_name("PolicyListRuleNumItem");

    setOverlap(true);

    addLabel(label);

    drag_source_set ( 
	static_cast<GdkModifierType>(GDK_BUTTON1_MASK) ,
	target_table, n_targets, 
	static_cast<GdkDragAction>(GDK_ACTION_COPY) );
    drag_begin.connect(
	slot(this,&PolicyListRuleNumItem::source_drag_begin));
    drag_data_get.connect(
	slot(this,&PolicyListRuleNumItem::source_drag_data_get));
}

void PolicyListRuleNumItem::source_drag_begin(GdkDragContext     *context)
{
    PolicyListElement *pl_el=getParentPolicyListElement();

    Gtk::Widget *pl = pl_el->get_parent(); /* PolicyList */
    Gtk::Widget *par;



    GdkImlibImage *img;
    GdkPixmap     *pmap;
    GdkBitmap     *mask;
    Gdk_Colormap  cmap ( get_colormap () );

/*
 *     Trick: determine physical size of one policy line in pixels,
 *     then grab portion of the screen in that area and use it as dragging
 *     icon. User is going to see that the whole rule is being dragged 
 */

    gint x,y,w,h;
    gint w1,h1;

    x=((PolicyList*)pl)->getX();
    y=((PolicyListElement*)pl_el)->getY();
    h=((PolicyListElement*)pl_el)->getHeight();
    w=((PolicyList*)pl)->getWidth();

    par=pl->get_parent();

/*
 *  PolicyList may be too large and can be clipped. Check the size of its
 *  parent and if it is smaller, use it.
 */
    gdk_window_get_size( par->gtkobj()->window , &w1, &h1);
    if (w1<w) w=w1;

    x-=2;
//    y-=1;
    h+=4;
    w+=4;

    img=gdk_imlib_create_image_from_drawable(
	par->gtkobj()->window,
	NULL,
	x, y, w, h);

    if (img) {
	unsigned char *pixel, r,g,b, bw;
	int      iw,ih;

	pixel=img->rgb_data; 
	for (iw=0; iw<img->rgb_width; iw++) {
	    for (ih=0; ih<img->rgb_height; ih++) {

		r= *pixel;
		g= *(pixel+1);
		b= *(pixel+2);
	    
		bw= (r+g+b)/3;

		*pixel++ = bw;
		*pixel++ = bw;
		*pixel++ = bw;
	    }
	}

	gdk_imlib_render(img,w,h);

	pmap=gdk_imlib_move_image(img);
	mask=gdk_imlib_move_mask(img);

	Gdk_Pixmap drag_icon=Gdk_Pixmap(pmap);
	Gdk_Bitmap drag_mask=Gdk_Bitmap(mask);

	drag_source_set_icon(cmap, drag_icon, drag_mask);
     
	gdk_imlib_free_pixmap(pmap);
    }
}

void PolicyListRuleNumItem::source_drag_data_get( GdkDragContext     *context,
					   GtkSelectionData   *selection_data,
						   guint               info,
						   guint32             time )
{
    PolicyListRuleNumItem *plitm=this;

    gtk_selection_data_set (selection_data,
			    gdk_atom_intern("PolicyListRuleNumItem",TRUE),
			    6,
			    (const guchar*)(&plitm),
			    sizeof(PolicyListItem*) );
}

/************************************************************************/




PolicyListCommentItem::PolicyListCommentItem(const string& text) :
    PolicyListItem("","")
{
    int            r;
    unsigned long  i,j,p;

    r=1;
    p=0;
    j=0;


    Gtk::Label *l=manage(new Gtk::Label(text));
    l->set_alignment(0, 0.5);

    tbl->resize(r,1);
    tbl->attach(*l, 0,1,r-1,r );
    l->show();

/*
    do {
	while ( (i=text.find(' ',j))!=string::npos && 
		i-p<MAX_COMMENT_FIELD_WIDTH) {
	    j=i+1;
	}
	string substring=(i==string::npos)?text.substr(p):text.substr(p,j-p);
	p=j;

	Gtk::Label *l=manage(new Gtk::Label(substring));
	l->set_alignment(0, 0.5);

	tbl->resize(r,1);
	tbl->attach(*l, 0,1,r-1,r );
	l->show();
	r++;
    } while (i!=string::npos);
*/
}

