/* 

                          Firewall Builder

                 Copyright (C) 2000,2001 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: nat.c,v 1.30 2001/12/06 03:55:25 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Puplic License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

 
#include "iptables.h"

#include <stdlib.h>
#include <string.h>

static int         snat, dnat,redir,src_netmap, dst_netmap;
static int         ofs,ofd,tfs,tfd;

static GSList     *ready_nat_subrules;


void NatRulePrologue(xmlNodePtr firewall,
		     xmlNodePtr rule,
		     int subrules_n)
{
    xmlNodePtr osrc,odst,osrv, tsrc,tdst,tsrv;
    int           num=getInt(rule,"position");
    const char* comm =getStr(rule,"comment");

    snat=dnat=redir=src_netmap=dst_netmap=0;
    
    osrc=getFirstChildXmlNode(rule,"OSrc");
    odst=getFirstChildXmlNode(rule,"ODst");
    osrv=getFirstChildXmlNode(rule,"OSrv");

    tsrc=getFirstChildXmlNode(rule,"TSrc");
    tdst=getFirstChildXmlNode(rule,"TDst");
    tsrv=getFirstChildXmlNode(rule,"TSrv");


    fprintf(ofile,"#\n");
    fprintf(ofile,"#   NAT Rule #%d\n" , num );
    fprintf(ofile,"#\n");
    if (comm) {
	fprintf(ofile,"#    %s\n",comm);
	fprintf(ofile,"#\n");
    }

    if ( getBool(tsrc,"neg") || getBool(tdst,"neg") || getBool(tsrv,"neg") ) {
	fprintf(stderr,"Negation is not supported in 'translated source' and 'translated destination' in rule #%d\n",num);
	exit(1);
    }

    
    if (getBool(osrc,"neg") || getBool(odst,"neg") || getBool(osrv,"neg") ) {
        sprintf(temp_n_chain,"T_N_RULE_%d"    ,num);
	fprintf(ofile,"iptables -t nat -N %s\n\n",temp_n_chain);
    }
    else 
	strcpy(temp_n_chain,"");

}

void NatRuleEpilogue(xmlNodePtr firewall,
		     xmlNodePtr rule,
		     elementaryNatRule *last_erule)
{
}

/*
 * this function generates final rule for the case with negation in any
 * rule element. This function is called _only_ when there is negation,
 * so we won't check for it here again
 */
void generateFinalRulesForNegation(xmlNodePtr firewall,
				   xmlNodePtr rule,
				   elementaryNatRule *erule)
{
    xmlNodePtr osrc,odst,osrv, tsrc,tdst,tsrv;
    xmlNodePtr c1;
    const char *ifaddr;
    elementaryNatRule *trule;

    osrc=getFirstChildXmlNode(rule,"OSrc");
    odst=getFirstChildXmlNode(rule,"ODst");
    osrv=getFirstChildXmlNode(rule,"OSrv");

    tsrc=getFirstChildXmlNode(rule,"TSrc");
    tdst=getFirstChildXmlNode(rule,"TDst");
    tsrv=getFirstChildXmlNode(rule,"TSrv");


    for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"Interface")) {
	    if ( ! isExternalInterface(c1) ) continue; 

	    trule=createElementaryNatRule();

	    copyElementaryNatRule(trule,erule);
	    if (getBool(osrc,"neg"))  {
		trule->osrc=NULL;
		trule->o->p_src=NULL;
	    }
	    if (getBool(odst,"neg"))  {
		trule->odst=NULL;
		trule->o->p_dst=NULL;
	    }
	    
	    trule->p_group = fwb_strdup(temp_n_chain);

	    ifaddr=getStr(c1,"address");
	    if (strcmp(ifaddr,"127.0.0.1")==SAME) continue;

/*	    trule->p_iface = getName(c1);

	    if (ofd && dnat) {
		if ( isDynamicInterface(c1) ) {
		    trule->p_action="DNAT";
		    trule->odst=NULL;
		} else {
		    trule->p_action="DNAT";
		    trule->odst=c1;
		}
*/
	    if (dnat) {
		trule->p_action="DNAT";
		trule->odst=NULL;
	    
		parseNatRule( rule,trule);
	    }
	    if (tfs && snat) {
		if ( isDynamicInterface(c1) ) {
		    trule->p_action="MASQUERADE";
		    trule->tsrc=NULL;
		} else {
		    trule->p_action="SNAT";
		    trule->tsrc=c1;
		}
		trule->odst=NULL;
		
		parseNatRule( rule,trule);
	    }
	    if (tsrc && redir) {
		trule->p_action  = "REDIRECT";
		trule->p_group   = fwb_strdup(temp_n_chain);

		parseNatRule( rule,trule);

		break;
	    }
	}
    }
}


void addARPEntries(xmlNodePtr firewall)
{
    GSList            *list_ptr1;
    elementaryNatRule *erule;
    
    for (list_ptr1=ready_nat_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {
	
	erule=(elementaryNatRule*)(list_ptr1->data);
	
	if (strcmp(erule->p_action,"DNAT")==SAME && 
	    cmpObjects(erule->odst , firewall)==SAME )

	    printARPEntryCommands(firewall, erule->odst, erule->tdst);
    }
}

void elementaryNatRulePrologue(xmlNodePtr firewall,
			       xmlNodePtr rule,
			       elementaryNatRule *erule)
{
    elementaryNatRule *trule;


    erule->p_group = "POSTROUTING";

    
    if (erule->osrc_neg || erule->odst_neg || erule->osrv_neg ) {
	if (src_netmap) {
	    trule=createElementaryNatRule();
	    trule->p_group = "POSTROUTING";
	    trule->p_action  = fwb_strdup(temp_n_chain);
	    storeNatRule(trule);
	}
	if (dst_netmap) {
	    trule=createElementaryNatRule();
	    trule->p_group = "PREROUTING";
	    trule->p_action  = fwb_strdup(temp_n_chain);
	    storeNatRule(trule);
	}
	if (snat) {
	    trule=createElementaryNatRule();
	    trule->p_group = "POSTROUTING";
	    trule->p_action  = fwb_strdup(temp_n_chain);
	    storeNatRule(trule);
	    if ( cmpObjects(erule->osrc , firewall)==SAME ) {
		trule=createElementaryNatRule();
		trule->p_group = "OUTPUT";
		trule->p_action  = fwb_strdup(temp_n_chain);
		storeNatRule(trule);
	    }
	}
	if (dnat || redir) {
	    trule=createElementaryNatRule();
	    trule->p_group = "PREROUTING";
	    trule->p_action  = fwb_strdup(temp_n_chain);
	    storeNatRule(trule);
	}
	erule->p_group = fwb_strdup(temp_n_chain);
	erule->p_action  = "RETURN";

    } else {
	if (src_netmap)
	    erule->p_group="POSTROUTING";

	if (dst_netmap)
	    erule->p_group="PREROUTING";
	
	if (snat) {
	    erule->p_group=
		(cmpObjects(erule->osrc , firewall)==SAME)?"OUTPUT":"POSTROUTING";
	}
	if (dnat || redir) {
	    erule->p_group="PREROUTING";
/*	    erule->p_action  = "DNAT"; */
	}
    }

}

void elementaryNatRuleEpilogue(xmlNodePtr firewall,
			       xmlNodePtr rule,
			       elementaryNatRule *erule)
{
}

void parseNatRule(xmlNodePtr rule,
		  elementaryNatRule *erule)
{

    processSrc(erule->osrc,erule->o);
    processDst(erule->odst,erule->o);
    
    processSrc(erule->tsrc,erule->t);
    processDst(erule->tdst,erule->t);
/*
    if (erule->osrv==NULL || isAny(erule->osrv) ) {
	storeNatRule(erule);
	return;
    }
*/    
    processSrv(erule->osrv,erule->o);
    processSrv(erule->tsrv,erule->t);

    /*
     *  TODO: sanity checking
     */

    storeNatRule(erule);
}


void processNATRule(xmlNodePtr firewall,
		    xmlNodePtr rule,
		    const char *comment,
		    GSList    *subrules)
{
    elementaryNatRule *erule, *trule, *last_erule;
    GSList            *list_ptr1;
    const char        *ifaddr;
    xmlNodePtr         c1;
    int                srn;
    const char        *snat_addr;
    xmlNodePtr         osrc,odst,osrv, tsrc,tdst,tsrv;
    int                seen_external;
    int                has_interfaces;
    
    osrc=getFirstChildXmlNode(rule,"OSrc");
    odst=getFirstChildXmlNode(rule,"ODst");
    osrv=getFirstChildXmlNode(rule,"OSrv");

    tsrc=getFirstChildXmlNode(rule,"TSrc");
    tdst=getFirstChildXmlNode(rule,"TDst");
    tsrv=getFirstChildXmlNode(rule,"TSrv");

    
    snat=dnat=0;
    ready_nat_subrules=NULL;
    
    seen_external=0;
    has_interfaces=0;
    
    for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"Interface")) {
	    has_interfaces=1;
	    if ( isExternalInterface(c1) ) {
		seen_external=1;
		break;
	    }
	}
    }
    if (!has_interfaces) {
	fprintf(stderr,
     "Firewall does not have any interfaces, can not configure NAT\n");
	exit(1);
    }
    if (!seen_external) {
	fprintf(stderr,
     "At least one interface must be marked as 'external'. Can not configure NAT\n");
	exit(1);
    }

    
    NatRulePrologue(firewall, rule, g_slist_length(subrules) );

    srn=0;
    for (list_ptr1=subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	erule=(elementaryNatRule*)(list_ptr1->data);
	
	if (erule->num==-1) continue;

	erule->subrule_no=srn;


	ofs=ofd=tfs=tfd=0;
	if ( cmpObjects(erule->osrc , firewall)==SAME ) ofs=1;
	if ( cmpObjects(erule->odst , firewall)==SAME ) ofd=1;
	if ( cmpObjects(erule->tsrc , firewall)==SAME ) tfs=1;
	if ( cmpObjects(erule->tdst , firewall)==SAME ) tfd=1;

	if ( isAny(erule->tdst) ) {
	    snat=1;
	    snat_addr = getStr(erule->tsrc,"address");
	    if (snat_addr && strcmp(snat_addr,"0.0.0.0")==SAME)
		erule->p_action="ACCEPT";  /* special case - no translation */
	    else {

		if (isElement(erule->osrc,"Network") &&
		    isElement(erule->tsrc,"Network")) {
		    if ( convertNetmask(erule->osrc) ==
			 convertNetmask(erule->tsrc) ) {

			src_netmap=1;
			erule->p_action="SRCNETMAP";

			printf(
"WARNING: Translation in rule #%d requires NETMAP
patch from patch-o-matic\n", erule->num);

		    } else {
			fprintf(stderr,
"Unsupported translation in rule #%d:
original source and translated source must be hosts
or networks of equal size\n",erule->num);
			exit(1);
		    }
		} else
		    erule->p_action = "SNAT";

	    }
	} else {
    
	    if ( tfd && isAny(erule->tsrc) ) {
		redir=1;
		erule->p_action  = "REDIRECT";
	    } else
    
		if (isElement(erule->odst,"Network") &&
		    isElement(erule->tdst,"Network")) {
		    if ( convertNetmask(erule->odst) ==
			 convertNetmask(erule->tdst) ) {

			dst_netmap=1;
			erule->p_action="DSTNETMAP";

			printf(
"WARNING: Translation in rule #%d requires NETMAP
patch from patch-o-matic\n", erule->num);

		    } else {
			fprintf(stderr,
"Unsupported translation in rule #%d:
original destination and translated destination must be hosts
or networks of equal size\n",erule->num);
			exit(1);
		    }
		} else {
/*
		    if ( ! isAny(erule->odst) &&
			 isAny(erule->tsrc) &&
			 ! isAny(erule->tdst) ) {
*/
		    if (   isAny(erule->tsrc) &&
			 ! isAny(erule->tdst) ) {
			dnat=1;
			erule->p_action  = "DNAT";
		    }
		}
	}


	if (snat==0 && dnat==0 && redir==0 && src_netmap==0 && dst_netmap==0) {
	    fprintf(stderr,"Unsupported translation in rule #%d\n",erule->num);
	    exit(1);
	}
    

	
	if (ofs || ofd || tfs) {
    
	    if (tfd && redir) {
		trule=createElementaryNatRule();
		copyElementaryNatRule(trule,erule);
		trule->tdst=NULL;
		
		elementaryNatRulePrologue(firewall,rule,trule);
		parseNatRule(rule,trule);
		elementaryNatRuleEpilogue(firewall,rule,trule);
		
		last_erule=trule;

	    } else {
		for(c1=firewall->xmlChildrenNode; c1; c1=c1->next) {
		    if ( xmlIsBlankNode(c1) ) continue;
		    if (isElement(c1,"Interface")) {
			if ( ! isExternalInterface(c1) ) continue; 

			ifaddr=getStr(c1,"address");
			if (strcmp(ifaddr,"127.0.0.1")==SAME) continue;

			trule=createElementaryNatRule();
			copyElementaryNatRule(trule,erule);

			if (ofd && dnat) {
			    if ( isDynamicInterface(c1) ) 
				trule->odst=NULL;
			    else
				trule->odst=c1;

			    trule->p_iface = getName(c1);

			    elementaryNatRulePrologue(firewall,rule,trule);
			    parseNatRule(rule,trule);
			    elementaryNatRuleEpilogue(firewall,rule,trule);
			}
			if (tfs && snat) {
			    if ( isDynamicInterface(c1) ) {
				trule->tsrc=NULL;
				trule->p_action = "MASQUERADE";
			    } else
				trule->tsrc=c1;

			    trule->p_iface = getName(c1);

			    elementaryNatRulePrologue(firewall,rule,trule);
			    parseNatRule(rule,trule);
			    elementaryNatRuleEpilogue(firewall,rule,trule);
			}
		    
			last_erule=trule;
		    }
		}
	    }

	} else {
	
	    if (tfd && redir) {
		trule=createElementaryNatRule();
		copyElementaryNatRule(trule,erule);
		trule->tdst=NULL;
		
		elementaryNatRulePrologue(firewall,rule,trule);
		parseNatRule(rule,trule);
		elementaryNatRuleEpilogue(firewall,rule,trule);
		
		last_erule=trule;

	    } else {
		elementaryNatRulePrologue(firewall,rule,erule);
		parseNatRule(rule,erule);
		elementaryNatRuleEpilogue(firewall,rule,erule);
/*
 *  printARPEntryCommands adds arp entry or ip alias address only
 *  once, checking for duplicates while processing individual NAT rule
 *  and between different NAT rules
 */
		if (global_manage_virtual_addr) {
		    if (snat && !tfs &&
			! src_netmap && ! dst_netmap &&
			! isAny(erule->tsrc) && ! isAny(erule->osrc)) {
			printARPEntryCommands(firewall,
					      erule->tsrc, erule->osrc);
		    }
		    
		    if (dnat &&	! src_netmap && ! dst_netmap ) {
			printARPEntryCommands(firewall,
					      erule->odst, erule->tdst);
		    }
		}
		last_erule=erule;
	    }
	}

	srn++;
    }

/*
 *  Now generate final rules if there is negation 
 */


    if (getBool(osrc,"neg") || getBool(odst,"neg") || getBool(osrv,"neg") ) {

	for (list_ptr1=subrules;
	     list_ptr1!=NULL;
	     list_ptr1=g_slist_next(list_ptr1)) {

	    erule=(elementaryNatRule*)(list_ptr1->data);
	
	    if (erule->num==-1) continue;
	    
	    erule->subrule_no=srn;

	    generateFinalRulesForNegation(firewall, rule, erule);
	    
	}
    }
    
/*    NatRuleEpilogue(firewall,rule,last_erule); */

    if (! global_no_optimisation && ! turn_off_optimisation)
	optimiseNatRules();
    
/*    if (global_manage_arp) 
	addARPEntries(firewall);
*/ 
    printNatRule();
    
    return;
}

void storeNatRule(elementaryNatRule *erule)
{
    if (strcmp(erule->p_action,"RETURN")==SAME ) 	{
	erule->p_iface=NULL;
	erule->t->p_dst=NULL;
	erule->t->p_src=NULL;
    }
    ready_nat_subrules=g_slist_append(ready_nat_subrules , erule);
}

void optimiseNatRules()
{
    elementaryNatRule     *nrptr1, *nrptr2;
    GSList                *list_ptr1, *list_ptr2;

    /*
     *  remove rules with the same network or host in both source
     *  and destination
     *
     *  Is this necessary ? ( removed because of bug #480410 )
     *

loop1:
    for (list_ptr1=ready_nat_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	nrptr1=(elementaryNatRule*)(list_ptr1->data);
	if (nrptr1->num==-1) continue;

	if (! isTripletEmpty(nrptr1->o) && ! isTripletEmpty(nrptr1->t) &&
	    cmpParameter(nrptr1->o->p_src, nrptr1->o->p_dst)==SAME     &&
	    cmpParameter(nrptr1->o->p_sprt,nrptr1->o->p_dprt)==SAME ) {

	    ready_nat_subrules=g_slist_remove(ready_nat_subrules, nrptr1);
	    goto loop1;
	}
	
    }
    */
 loop2:
    for (list_ptr1=ready_nat_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {

	nrptr1=(elementaryNatRule*)(list_ptr1->data);
	if (nrptr1->num==-1) continue;
	    
	for (list_ptr2=g_slist_next(list_ptr1);
	     list_ptr2!=NULL;
	     list_ptr2=g_slist_next(list_ptr2)) {

	    nrptr2=(elementaryNatRule*)(list_ptr2->data);

	    if (cmpPNatRules( nrptr1,nrptr2 )==SAME) {
		ready_nat_subrules=g_slist_remove(ready_nat_subrules, nrptr2);
		goto loop2;
	    }
	}
    }
}

void printNatRule()
{
    elementaryNatRule     *erule;
    GSList                *list_ptr1;
    char                  *c;

    
    for (list_ptr1=ready_nat_subrules;
	 list_ptr1!=NULL;
	 list_ptr1=g_slist_next(list_ptr1)) {
	
	erule=(elementaryNatRule*)(list_ptr1->data);
	    
	fprintf(ofile,"iptables -t nat -A %s ",erule->p_group);

	if (erule->p_iface) {
	    if (strcmp(erule->p_action,"SNAT")==SAME ||
		strcmp(erule->p_action,"MASQUERADE")==SAME)
		fprintf(ofile," -o %s",erule->p_iface);
	    if (strcmp(erule->p_action,"DNAT")==SAME )
		fprintf(ofile," -i %s",erule->p_iface);
	}

	if ( erule->o->p_proto ) {
	    fprintf(ofile," -p %s",erule->o->p_proto);
	} else
	    if ( erule->t->p_proto ) {
		fprintf(ofile," -p %s",erule->t->p_proto);
	    }
    
	if (strcmp(erule->p_action,"DSTNETMAP")!=SAME && 
	    erule->o->p_src!=NULL) fprintf(ofile," -s %s",erule->o->p_src);

	if ( erule->o->p_sprt && strlen(erule->o->p_sprt)!=0)
	    fprintf(ofile," --source-port %s",erule->o->p_sprt);
	
	if (strcmp(erule->p_action,"SRCNETMAP")!=SAME && 
	   erule->o->p_dst!=NULL) fprintf(ofile," -d %s",erule->o->p_dst);
    
	if (erule->o->p_dprt && strlen(erule->o->p_dprt)!=0)
	    fprintf(ofile," --destination-port %s",erule->o->p_dprt);
	
	if ( erule->o->p_icmp_type )
	    fprintf(ofile," --icmp-type %s", erule->o->p_icmp_type);
	
	if (erule->p_action!=NULL) {
	    if (strcmp(erule->p_action,"SRCNETMAP")==SAME ||
		strcmp(erule->p_action,"DSTNETMAP")==SAME )
		fprintf(ofile," -j NETMAP");
	    else
		fprintf(ofile," -j %s",erule->p_action);
	}

	if(strcmp(erule->p_action,"SRCNETMAP")==SAME && erule->t->p_src!=NULL){
	    fprintf(ofile," --to %s", erule->t->p_src);
	}
	
	if(strcmp(erule->p_action,"DSTNETMAP")==SAME && erule->t->p_dst!=NULL){
	    fprintf(ofile," --to %s", erule->t->p_dst);
	}
	
	if (strcmp(erule->p_action,"SNAT")==SAME && erule->t->p_src!=NULL) {
	    fprintf(ofile," --to-source %s", erule->t->p_src);

	    if ( erule->o->p_proto!=NULL &&
		 ( strcmp(erule->o->p_proto,"tcp")==SAME ||
		   strcmp(erule->o->p_proto,"udp")==SAME ) && 
		 erule->t->p_sprt!=NULL ) {
/* kludge: in SNAT/DNAT rules port range is specified as port1-port2, unlike
 * in source/destination where range is specified as port1:port2
 */
		if ( (c=strchr(erule->t->p_sprt,':'))!=NULL) *c='-';
		fprintf(ofile,":%s", erule->t->p_sprt);
	    }
	}

	if (strcmp(erule->p_action,"DNAT")==SAME && erule->t->p_dst!=NULL) {
	    fprintf(ofile," --to-destination %s", erule->t->p_dst);

	    if ( erule->o->p_proto!=NULL &&
		 ( strcmp(erule->o->p_proto,"tcp")==SAME ||
		   strcmp(erule->o->p_proto,"udp")==SAME ) && 
		 erule->t->p_dprt!=NULL ) {
		if ( (c=strchr(erule->t->p_dprt,':'))!=NULL) *c='-';
		fprintf(ofile,":%s", erule->t->p_dprt);
	    }
	}

	if (strcmp(erule->p_action,"REDIRECT")==SAME && erule->t->p_dprt!=NULL) {
	    fprintf(ofile," --to-ports %s", erule->t->p_dprt);
	}

	fprintf(ofile,"\n");
    }

}

