/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				dihedral_angles.c

Purpose:
	Calculate dihedral angles phi and psi and set cis-trans flag. The
	phi angle is the angle of right-handed rotation around N-CA bond,
	the value being zero if CA-C bond is trans to  N-H bond.  The psi
	angle is the  angle  of right-handed  rotation around  CA-C bond,
	the value  being zero if  N-CA bond is trans to C-O bond.  Range:
	-180 to +180 for  both angles. Use cis-trans flag to distinguish
	cis and trans residues, as well as bad residues. Flag values are:
	0 = bad/undefined,  1 = trans (the most common),  2 = cis (rare).

Input:
	(1) Pointer to MolComplexS structure, with macromolecular data.

Output:
	(1) Dihedral angles calculated, cis-trans flag set.
	(2) Return value.

Return value:
	(1) Positive on success.
	(2) Zero, if complex contains no atoms.
	(3) Negative on failure.

Notes:
	(1) This should help you to understand phi and psi definitions:
	    ........./................
	    | H * - * N              |
	    |        \    phi1 = 180 |
	    |         * CA           |
	    |        /    psi1 = 180 |
	    | O * - * C              |
	    |........\...............|
	    |       N * - * H        |
	    |        /    phi2 = 180 |
	    |    CA *                |
	    |        \    psi2 = 180 |
	    |       C * - * O        |
	    |......../...............|

	(2) By default,  the peptide group has undefined conformation. To
	    define the peptide group, the coordinates of four atoms are
	    required.

========includes:============================================================*/

#include <stdio.h>

#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

int		ExtractNCACO_ (VectorS *, VectorS *, VectorS *, VectorS *,
			       AtomS *, size_t, size_t);
int		CountNCBonds_ (AtomS *, size_t, size_t);
double		CalculatePhi_ (VectorS *, VectorS *, VectorS *, VectorS *);
double		CalculatePsi_ (VectorS *, VectorS *, VectorS *, VectorS *);
int		CisTrans_ (VectorS *, VectorS *, VectorS *, VectorS *);

/*======calculate dihedral angles and set cis-trans flag:====================*/

int DihedralAngles_ (MolComplexS *mol_complexSP)
{
size_t			residuesN, residueI;
size_t			struct_size;
ResidueS		*curr_residueSP;
size_t			atom_startI, atom_endI;
int			n;
static VectorS		N_vectorS, CA_vectorS, C_vectorS, O_vectorS;
static VectorS		C_old_vectorS, O_old_vectorS;
int			phi_availableF;
int			cis_transF;

/* Return zero if there are no atoms: */
if (mol_complexSP->atomsN == 0) return 0;

/* Prepare and check the number of residues - at least is required: */
residuesN = mol_complexSP->residuesN;
if (residuesN < 1) return -1;

/* The size of VectorS structure: */
struct_size = sizeof (VectorS);

/* Scan residues: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Pointer to the current residue: */
	curr_residueSP = mol_complexSP->residueSP + residueI;

	/* Initialize dihedral angles with values which are out of range: */
	curr_residueSP->phi = BADDIHEDANGLE;
	curr_residueSP->psi = BADDIHEDANGLE;

	/* Prepare the atomic index range: */
	atom_startI = curr_residueSP->residue_startI;
	atom_endI   = curr_residueSP->residue_endI;

	/* Extract N, CA, C and O coordinates: */
	n = ExtractNCACO_ (&N_vectorS, &CA_vectorS, &C_vectorS, &O_vectorS,
			   mol_complexSP->atomSP, atom_startI, atom_endI);

	/* Normally, four vectors should be extracted: */
	if (n != 4)
		{
		/* Backup the coordinates of  C  atom, */
		/* it will be reused in the next step: */
		memcpy (&C_old_vectorS, &C_vectorS, struct_size);

		/* Take the next residue: */
		continue;
		}

	/* Check is N atom bound to C atom of the previous residue: */
	n = CountNCBonds_ (mol_complexSP->atomSP, atom_startI, atom_endI);
	if (n < 2)
		{
		phi_availableF = 0;
		}
	else phi_availableF = 1;

	/* Calculate phi (not available for the first residue): */
	if ((residueI > 0) && (phi_availableF))
		{
		curr_residueSP->phi = CalculatePhi_ (&C_old_vectorS,
						     &N_vectorS,
						     &CA_vectorS,
						     &C_vectorS);
		}

	/* Calculate psi: */
	curr_residueSP->psi = CalculatePsi_ (&N_vectorS,
					     &CA_vectorS,
					     &C_vectorS,
					     &O_vectorS);

	/* Backup the coordinates of  C  atom, */
	/* it will be reused in the next step: */
	memcpy (&C_old_vectorS, &C_vectorS, struct_size);
	}

/* Scan residues once more, to distinguish cis, trans and bad peptides: */
for (residueI = 0; residueI < residuesN; residueI++)
	{
	/* Pointer to the current residue: */
	curr_residueSP = mol_complexSP->residueSP + residueI;

	/* By default, peptide group has undefined conformation: */
	curr_residueSP->cis_transF = 0;

	/* Prepare the atomic index range: */
	atom_startI = curr_residueSP->residue_startI;
	atom_endI   = curr_residueSP->residue_endI;

	/* Extract N, CA, C and O coordinates: */
	n = ExtractNCACO_ (&N_vectorS, &CA_vectorS, &C_vectorS, &O_vectorS,
			   mol_complexSP->atomSP, atom_startI, atom_endI);

	/* If some coordinates are missing, backup C and O and continue: */
	if (n < 4)
		{
		memcpy (&C_old_vectorS, &C_vectorS, struct_size);
		memcpy (&O_old_vectorS, &O_vectorS, struct_size);
		continue;
		}

	/* Prepare the cis-trans flag (but not for the first residue): */
	if (residueI != 0)
		{
		cis_transF = CisTrans_ (&N_vectorS, &CA_vectorS,
					&C_old_vectorS, &O_old_vectorS);
		if (cis_transF != 0)
			{
			curr_residueSP->cis_transF =
						(unsigned char) cis_transF;
			}
		}

	/* Backup the coordinates of C and O, for the next step: */
	memcpy (&C_old_vectorS, &C_vectorS, struct_size);
	memcpy (&O_old_vectorS, &O_vectorS, struct_size);
	}

/* If this point is reached, return positive value (success indicator): */
return 1;
}

/*===========================================================================*/


