#! /usr/bin/env gosh
;;;
;;; glstate.scm - state variable table
;;;
;;;  Copyright(C) 2001-2002 by Shiro Kawai (shiro@acm.org)
;;;
;;;  Permission to use, copy, modify, distribute this software and
;;;  accompanying documentation for any purpose is hereby granted,
;;;  provided that existing copyright notices are retained in all
;;;  copies and that this notice is included verbatim in all
;;;  distributions.
;;;  This software is provided as is, without express or implied
;;;  warranty.  In no circumstances the author(s) shall be liable
;;;  for any damages arising out of the use of this software.
;;;
;;;  $Id: glstate.scm,v 1.6 2003/10/05 05:05:25 shirok Exp $
;;;

;;
;; Generate several C source code that includes information about
;; dimensions of state variables
;;

(use srfi-13)
(use gauche.parseopt)

;; Data
;;   (name dimension type [version ...])
;;   type: 'getf - available via glGetFloatv
;;         'geti - available via glGetIntegerv
;;         'getb - available via glGetBooleanv
;;         'texp - available via glGetTextureParameter
;;         'texl - available via glGetTexLevelParameter
;;         'texe - available via glGetTexEnv
;;         'texg - available via glGetTexGen
;;         'colp - available via glGetColroTableParameter
;;         'conp - available via glGetConvolutionParameter
;;         'hisp - available via glGetHistogramParameter
;;         #f    - others
;;   version:  (none) - OpenGL v1.1 and v1.2
;;             'v1.1  - OpenGL v1.1 only
;;             'v1.2  - OpenGL v1.2 only
;;             'no-cygwin - Cygwin OpenGL (1.1.0-6) doesn't have this symbol

;; Source of this list is the table is the Appendix B: State Variables
;; of OpenGL Programming Guide.

(define *state*
  '(;; Current Values and Associated Data
    (gl_current_color                     4   getf)
    (gl_current_index                     1   geti)
    (gl_current_texture_coords            4   getf)
    (gl_current_normal                    3   getf)
    (gl_current_raster_position           4   getf)
    (gl_current_raster_distance           1   getf)
    (gl_current_raster_color              4   getf)
    (gl_current_raster_index              1   geti)
    (gl_current_raster_texture_coords     4   getf)
    (gl_current_raster_position_valid     1   getb)
    (gl_edge_flag                         1   getb)
    ;; Vertex Array
    (gl_vertex_array                      1   getb)
    (gl_vertex_array_size                 1   geti)
    (gl_vertex_array_type                 1   geti)
    (gl_vertex_array_stride               1   geti)
    (gl_vertex_array_pointer              #f  getp)
    (gl_normal_array                      1   getb)
    (gl_normal_array_type                 1   geti)
    (gl_normal_array_stride               1   geti)
    (gl_normal_array_pointer              #f  getp)
    (gl_color_array                       1   getb)
    (gl_color_array_size                  1   geti)
    (gl_color_array_type                  1   geti)
    (gl_color_array_stride                1   geti)
    (gl_color_array_pointer               #f  getp)
    (gl_index_array                       1   getb)
    (gl_index_array_type                  1   geti)
    (gl_index_array_stride                1   geti)
    (gl_index_array_pointer               #f  getp)
    (gl_texture_coord_array               1   getb)
    (gl_texture_coord_array_size          1   geti)
    (gl_texture_coord_array_type          1   geti)
    (gl_texture_coord_array_stride        1   geti)
    (gl_texture_coord_array_pointer       #f  getp)
    (gl_edge_flag_array                   1   getb)
    (gl_edge_flag_array_stride            1   geti)
    (gl_edge_flag_array_pointer           1   getp)
    (gl_client_active_texture_arb         1   geti v1.2)
    ;; Transformation
    (gl_color_matrix                     16   getf v1.2)
    (gl_modelview_matrix                 16   getf)
    (gl_projection_matrix                16   getf)
    (gl_texture_matrix                   16   getf)
    (gl_viewport                          4   geti)
    (gl_depth_range                       2   getf)
    ;(gl_color_stack_depth                 1   geti v1.2) ; Mesa missing this
    (gl_modelview_stack_depth             1   geti)
    (gl_projection_stack_depth            1   geti)
    (gl_texture_stack_depth               1   geti)
    (gl_matrix_mode                       1   geti)
    (gl_normalize                         1   getb)
    (gl_rescale_normal                    1   getb no-cygwin)
    (gl_clip_plane0                       1   getb)
    (gl_clip_plane1                       1   getb)
    (gl_clip_plane2                       1   getb)
    (gl_clip_plane3                       1   getb)
    (gl_clip_plane4                       1   getb)
    (gl_clip_plane5                       1   getb)
    ;; Coloring
    (gl_fog_color                         4   getf)
    (gl_fog_index                         1   getf)
    (gl_fog_density                       1   getf)
    (gl_fog_start                         1   getf)
    (gl_fog_end                           1   getf)
    (gl_fog_mode                          1   geti)
    (gl_fog                               1   getb)
    (gl_shade_model                       1   geti)
    ;; Lighting
    (gl_lighting                          1   getb)
    (gl_color_material                    1   getb)
    (gl_color_material_parameter          1   geti)
    (gl_color_material_face               1   geti)
    (gl_light_model_ambient               4   getf)
    (gl_light_model_local_viewer          1   getb)
    (gl_light_model_two_side              1   getb)
    (gl_light_model_color_control         1   geti no-cygwin)
    (gl_light0                            1   getb)
    (gl_light1                            1   getb)
    (gl_light2                            1   getb)
    (gl_light3                            1   getb)
    (gl_light4                            1   getb)
    (gl_light5                            1   getb)
    (gl_light6                            1   getb)
    (gl_light7                            1   getb)
    ;; Rasterization
    (gl_point_size                        1   getf)
    (gl_point_smooth                      1   getb)
    (gl_line_width                        1   getf)
    (gl_line_smooth                       1   getb)
    (gl_line_stipple_pattern              1   geti)
    (gl_line_stipple_repeat               1   geti)
    (gl_line_stipple                      1   getb)
    (gl_cull_face                         1   getb)
    (gl_cull_face_mode                    1   geti)
    (gl_front_face                        1   geti)
    (gl_polygon_smooth                    1   getb)
    (gl_polygon_mode                      1   geti)
    (gl_polygon_offset_factor             1   getf)
    ;(gl_polygon_offset_bias               1   getf v1.2)  ; Mesa missing this
    (gl_polygon_offset_point              1   getb)
    (gl_polygon_offset_line               1   getb)
    (gl_polygon_offset_fill               1   getb v1.2)
    (gl_polygon_stipple                   1   getb)
    ;; Texturing
    (gl_texture_1d                        1   getb)
    (gl_texture_2d                        1   getb)
    (gl_texture_3d                        1   getb no-cygwin)
    (gl_texture_binding_1d                1   geti)
    (gl_texture_binding_2d                1   geti)
    (gl_texture_binding_3d                1   geti v1.2)
    (gl_texture_width                     1   texl)
    (gl_texture_height                    1   texl)
    (gl_texture_depth                     1   texl)
    (gl_texture_border                    1   texl)
    (gl_texture_internal_format           1   texl)
    (gl_texture_red_size                  1   texl)
    (gl_texture_green_size                1   texl)
    (gl_texture_blue_size                 1   texl)
    (gl_texture_alpha_size                1   texl)
    (gl_texture_luminance_size            1   texl)
    (gl_texture_intensity_size            1   texl)
    (gl_texture_border_color              4   texp)
    (gl_texture_min_filter                1   texp)
    (gl_texture_mag_filter                1   texp)
    (gl_texture_wrap_s                    1   texp)
    (gl_texture_wrap_t                    1   texp)
    (gl_texture_wrap_u                    1   texp)
    (gl_texture_priority                  1   texp)
    (gl_texture_resident                  1   texp)
    (gl_texture_min_lod                   1   texp)
    (gl_texture_max_lod                   1   texp)
    (gl_texture_base_level                1   texp)
    (gl_texture_max_level                 1   texp)
    (gl_texture_env_mode                  1   texe)
    (gl_texture_env_color                 4   texe)
    (gl_texture_gen_s                     1   getb)
    (gl_texture_gen_t                     1   getb)
    (gl_texture_gen_r                     1   getb)
    (gl_texture_gen_q                     1   getb)
    (gl_eye_plane                         4   texg)
    (gl_object_plane                      4   texg)
    (gl_texture_gen_mode                  1   texg)
    (gl_active_texture_arb                1   geti v1.2)
    ;; Pixel Operations
    (gl_scissor_test                      1   getb)
    (gl_scissor_box                       4   geti)
    (gl_alpha_test                        1   getb)
    (gl_alpha_test_func                   1   geti)
    (gl_stencil_test                      1   getb)
    (gl_stencil_func                      1   geti)
    (gl_stencil_value_mask                1   geti)
    (gl_stencil_ref                       1   geti)
    (gl_stencil_fail                      1   geti)
    (gl_stencil_pass_depth_fail           1   geti)
    (gl_stencil_pass_depth_pass           1   geti)
    (gl_depth_test                        1   getb)
    (gl_depth_func                        1   geti)
    (gl_blend                             1   getb)
    (gl_blend_src                         1   geti)
    (gl_blend_dst                         1   geti)
    (gl_blend_equation                    1   geti v1.2)
    (gl_blend_equation_ext                1   geti v1.1 no-cygwin)
    (gl_blend_color                       4   getf v1.2)
    (gl_blend_color_ext                   4   getf v1.1 no-cygwin)
    (gl_dither                            1   getb)
    (gl_index_logic_op                    1   getb)
    (gl_color_logic_op                    1   getb)
    (gl_logic_op_mode                     1   geti)
    ;; Framebuffer Control
    (gl_draw_buffer                       1   geti)
    (gl_index_writemask                   1   geti)
    (gl_color_writemask                   4   getb)
    (gl_depth_writemask                   1   getb)
    (gl_stencil_writemask                 1   geti)
    (gl_color_clear_value                 4   getf)
    (gl_index_clear_value                 1   getf)
    (gl_depth_clear_value                 1   geti)
    (gl_stencil_clear_value               1   geti)
    (gl_accum_clear_value                 1   getf)
    ;; Pixels
    (gl_unpack_swap_bytes                 1   getb)
    (gl_unpack_lsb_first                  1   getb)
    (gl_unpack_image_height               1   geti no-cygwin)
    (gl_unpack_skip_images                1   geti no-cygwin)
    (gl_unpack_row_length                 1   geti)
    (gl_unpack_skip_rows                  1   geti)
    (gl_unpack_skip_pixels                1   geti)
    (gl_unpack_alignment                  1   geti)
    (gl_pack_swap_bytes                   1   getb)
    (gl_pack_lsb_first                    1   getb)
    (gl_pack_image_height                 1   geti no-cygwin)
    (gl_pack_skip_images                  1   geti no-cygwin)
    (gl_pack_row_length                   1   geti)
    (gl_pack_skip_rows                    1   geti)
    (gl_pack_skip_pixels                  1   geti)
    (gl_pack_alignment                    1   geti)
    (gl_map_color                         1   getb)
    (gl_map_stencil                       1   getb)
    (gl_index_shift                       1   geti)
    (gl_index_offset                      1   geti)
    (gl_red_scale                         1   getf)
    (gl_green_scale                       1   getf)
    (gl_blue_scale                        1   getf)
    (gl_alpha_scale                       1   getf)
    (gl_depth_scale                       1   getf)
    (gl_red_bias                          1   getf)
    (gl_green_bias                        1   getf)
    (gl_blue_bias                         1   getf)
    (gl_alpha_bias                        1   getf)
    (gl_depth_bias                        1   getf)
    (gl_color_table                       1   getb v1.2)
    (gl_post_convolution_color_table      1   getb v1.2)
    (gl_post_color_matrix_color_table     1   getb v1.2)
    (gl_color_table_format                1   colp)
    (gl_color_table_width                 1   colp)
    (gl_color_table_red_size              1   colp)
    (gl_color_table_green_size            1   colp)
    (gl_color_table_blue_size             1   colp)
    (gl_color_table_alpha_size            1   colp)
    (gl_color_table_luminance_size        1   colp)
    (gl_color_table_intensity_size        1   colp)
    (gl_color_table_scale                 4   colp)
    (gl_color_table_bias                  4   colp)
    (gl_convolution_1d                    1   getb v1.2)
    (gl_convolution_2d                    1   getb v1.2)
    (gl_separable_2d                      1   getb v1.2)
    (gl_convolution_border_color          4   conp)
    (gl_convolution_border_mode           1   conp)
    (gl_convolution_filter_scale          4   conp)
    (gl_convolution_filter_bias           4   conp)
    (gl_convolution_format                1   conp)
    (gl_convolution_width                 1   conp)
    (gl_convolution_height                1   conp)
    (gl_post_convolution_red_scale        1   conp)
    (gl_post_convolution_green_scale      1   conp)
    (gl_post_convolution_blue_scale       1   conp)
    (gl_post_convolution_alpha_scale      1   conp)
    (gl_post_convolution_red_bias         1   conp)
    (gl_post_convolution_green_bias       1   conp)
    (gl_post_convolution_blue_bias        1   conp)
    (gl_post_convolution_alpha_bias       1   conp)
    (gl_post_color_matrix_red_scale       1   conp)
    (gl_post_color_matrix_green_scale     1   conp)
    (gl_post_color_matrix_blue_scale      1   conp)
    (gl_post_color_matrix_alpha_scale     1   conp)
    (gl_post_color_matrix_red_bias        1   conp)
    (gl_post_color_matrix_green_bias      1   conp)
    (gl_post_color_matrix_blue_bias       1   conp)
    (gl_post_color_matrix_alpha_bias      1   conp)
    (gl_histogram                         1   getb v1.2)
    (gl_histogram_width                   1   hisp)
    (gl_histogram_format                  1   hisp)
    (gl_histogram_red_size                1   hisp)
    (gl_histogram_green_size              1   hisp)
    (gl_histogram_blue_size               1   hisp)
    (gl_histogram_alpha_size              1   hisp)
    (gl_histogram_luminance_size          1   hisp)
    (gl_histogram_sink                    1   hisp)
    (gl_minmax                            1   getb v1.2)
    (gl_zoom_x                            1   getf)
    (gl_zoom_y                            1   getf)
    (gl_pixel_map_i_to_i_size             1   geti)
    (gl_pixel_map_s_to_s_size             1   geti)
    (gl_pixel_map_i_to_r_size             1   geti)
    (gl_pixel_map_i_to_g_size             1   geti)
    (gl_pixel_map_i_to_b_size             1   geti)
    (gl_pixel_map_i_to_a_size             1   geti)
    (gl_pixel_map_r_to_r_size             1   geti)
    (gl_pixel_map_g_to_g_size             1   geti)
    (gl_pixel_map_b_to_b_size             1   geti)
    (gl_pixel_map_a_to_a_size             1   geti)
    (gl_read_buffer                       1   geti)
    ;; Evaluators
    (gl_map1_vertex_3                     1   getb)
    (gl_map1_vertex_4                     1   getb)
    (gl_map1_index                        1   getb)
    (gl_map1_color_4                      1   getb)
    (gl_map1_normal                       1   getb)
    (gl_map1_texture_coord_1              1   getb)
    (gl_map1_texture_coord_2              1   getb)
    (gl_map1_texture_coord_3              1   getb)
    (gl_map1_texture_coord_4              1   getb)
    (gl_map2_vertex_3                     1   getb)
    (gl_map2_vertex_4                     1   getb)
    (gl_map2_index                        1   getb)
    (gl_map2_color_4                      1   getb)
    (gl_map2_normal                       1   getb)
    (gl_map2_texture_coord_1              1   getb)
    (gl_map2_texture_coord_2              1   getb)
    (gl_map2_texture_coord_3              1   getb)
    (gl_map2_texture_coord_4              1   getb)
    (gl_map1_grid_domain                  2   getf)
    (gl_map2_grid_domain                  4   getf)
    (gl_map1_grid_segments                1   getf)
    (gl_map2_grid_segments                2   getf)
    (gl_auto_normal                       1   getb)
    ;; Hints
    (gl_perspective_correction_hint       1   geti)
    (gl_point_smooth_hint                 1   geti)
    (gl_line_smooth_hint                  1   geti)
    (gl_polygon_smooth_hint               1   geti)
    (gl_fog_hint                          1   geti)
    ;; Implementation-Dependent Values
    (gl_max_lights                        1   geti)
    (gl_max_clip_planes                   1   geti)
    (gl_max_modelview_stack_depth         1   geti)
    (gl_max_projection_stack_depth        1   geti)
    (gl_max_texture_stack_depth           1   geti)
    (gl_subpixel_bits                     1   geti)
    (gl_max_3d_texture_size               1   geti no-cygwin)
    (gl_max_texture_size                  1   geti)
    (gl_max_pixel_map_table               1   geti)
    (gl_max_name_stack_depth              1   geti)
    (gl_max_list_nesting                  1   geti)
    (gl_max_eval_order                    1   geti)
    (gl_max_viewport_dims                 1   geti)
    (gl_max_attrib_stack_depth            1   geti)
    (gl_max_client_attrib_stack_depth     1   geti)
    (gl_aux_buffers                       1   geti)
    (gl_rgba_mode                         1   getb)
    (gl_indeX_mode                        1   getb)
    (gl_doublebuffer                      1   getb)
    (gl_stereo                            1   getb)
    (gl_point_size_range                  2   getf v1.1)
    (gl_point_size_granularity            1   getf v1.1)
    (gl_aliased_point_size_range          2   getf v1.2)
    (gl_smooth_point_size_range           2   getf v1.2)
    (gl_smooth_point_size_granularity     1   getf v1.2)
    (gl_aliased_line_width_range          2   getf v1.2)
    (gl_smooth_line_width_range           2   getf v1.2)
    (gl_smooth_line_width_granularity     1   getf v1.2)
    (gl_max_convolution_width             1   conp)
    (gl_max_convolution_height            1   conp)
    (gl_max_elements_indices              1   geti no-cygwin)
    (gl_max_elements_vertices             1   geti no-cygwin)
    (gl_max_texture_units_arb             1   geti v1.2)
    ;; Implementation-Dependent Pixel Depths
    (gl_red_bits                          1   geti)
    (gl_green_bits                        1   geti)
    (gl_blue_bits                         1   geti)
    (gl_alpha_bits                        1   geti)
    (gl_index_bits                        1   geti)
    (gl_depth_bits                        1   geti)
    (gl_stencil_bits                      1   geti)
    (gl_accum_red_bits                    1   geti)
    (gl_accum_green_bits                  1   geti)
    (gl_accum_blue_bits                   1   geti)
    (gl_accum_alpha_bits                  1   geti)
    ;; Miscellaneous
    (gl_list_base                         1   geti)
    (gl_list_index                        1   geti)
    (gl_list_mode                         1   geti)
    (gl_attrib_stack_depth                1   geti)
    (gl_client_attrib_stack_depth         1   geti)
    (gl_name_stack_depth                  1   geti)
    (gl_render_mode                       1   geti)
    (gl_selection_buffer_pointer          1   getp)
    (gl_selection_buffer_size             1   geti)
    (gl_feedback_buffer_pointer           1   getp)
    (gl_feedback_buffer_size              1   geti)
    (gl_feedback_buffer_type              1   geti)
    )
  )

;; generate case statements for size of values of getTYPEv calls
(define (gen-getTYPEv-size-table)
  (let ((tab '()))
    (for-each (lambda (e)
                (when (memq (caddr e) '(geti getf getb))
                  (cond ((assv (cadr e) tab)
                         => (lambda (p) (push! (cdr p) e)))
                        (else
                         (push! tab (list (cadr e) e))))))
              *state*)
    (for-each (lambda (n)
                (for-each
                 (lambda (e)
		   (gen-case-label (string-upcase (symbol->string (car e)))
				   (cdddr e)))
                 (reverse (cdr n)))
                (print #`"    return ,(car n);"))
              (reverse tab))))

;; output case label, optionally wrapping by #if-#endif
(define (gen-case-label name versions)
  (if (null? versions)
      (print #`"  case ,|name|:;")
      (let ((condition
	     (case (car versions)
	       ((v1.1) "defined(GL_VERSION_1_1) && !defined(GL_VERSION_1_2)")
	       ((v1.2) "defined(GL_VERSION_1_2)")
	       ((no-cygwin) "!defined(__CYGWIN__)")
	       (else (error "unknown version identifier:" (car versions))))))
	(print #`"#if ,condition")
	(gen-case-label name (cdr versions))
	(print "#endif"))))

;;-----------------------------

(define *outfile* #f)

(define (usage)
  (format (current-error-port)
          "Usage: gosh glstate.scm -- [-o outfile] function\n")
  (format (current-error-port)
          "  possible function : gettype\n")
  (exit 1))

(define (main args)
  (let ((arg (parse-options (cdr args)
               (("o=s" (file) (set! *outfile* file))
                (else _ (usage))))))
    (unless (= (length arg) 1) (usage))
    (if *outfile*
        (with-output-to-file *outfile* (lambda () (dispatch (car arg))))
        (dispatch (car arg))))
  0)

(define (dispatch function)
  (cond ((equal? function "gettype") (gen-getTYPEv-size-table))
        (else (usage))))
