/*
 * Copyright (C) 2007 Jordi Mas i Hernàndez <jmas@softcatala.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

using System;
using Gtk;
using Mono.Unix;
using System.Collections;

using gbrainy.Core.Main;
using gbrainy.Core.Libraries;

namespace gbrainy.Clients.Classical
{
	public class CustomGameDialog : BuilderDialog
	{
		static ListStore games_store;
		[GtkBeans.Builder.Object] Gtk.TreeView treeview;
		[GtkBeans.Builder.Object] Box preview_vbox;
		[GtkBeans.Builder.Object] Gtk.Box question_vbox;
		CairoPreview drawing_area;
		SimpleLabel question_label;
		GameManager manager;
		int ngames, npos;
		Type [] custom_games;

		const int COL_ENABLED = 2;
		const int COL_OBJECT = 3;

		public CustomGameDialog (GameManager manager) : base ("CustomGameDialog.ui", "customgame")
		{
			Game game;
			Type[] games;
			GameManager gm;

			this.manager = manager;
			gm = new GameManager ();
			gm.GameType = GameSession.Types.AllGames;
			games = gm.CustomGames;

			drawing_area = new CairoPreview ();
			preview_vbox.Add (drawing_area);
			drawing_area.Visible = true;

			question_label = new SimpleLabel ();
			question_label.HeightMargin = 2;
			question_label.Visible = true;
			question_vbox.Add (question_label);

			// Define columns
			TreeViewColumn name_column = new TreeViewColumn (Catalog.GetString("Game Name"), 
				new CellRendererText(), "text", 0);

			name_column.Expand = true;
			treeview.AppendColumn (name_column);

			TreeViewColumn type_column = new TreeViewColumn (Catalog.GetString("Type"), 
				new CellRendererText(), "text", 1);

			type_column.Expand = true;
			treeview.AppendColumn (type_column);

			CellRendererToggle toggle_cell = new CellRendererToggle();
			TreeViewColumn toggle_column = new TreeViewColumn(Catalog.GetString("Enabled"), 
				toggle_cell, "active", COL_ENABLED);
			toggle_cell.Activatable = true;
			toggle_cell.Toggled += OnActiveToggled;
			toggle_column.Expand = false;
			treeview.CursorChanged += OnCursorChanged;
			treeview.AppendColumn (toggle_column);

			if (games_store == null) {
				games_store = new ListStore (typeof(string), typeof (string), typeof(bool), typeof (Game));
					 
				// Data
				string type;
				for (int i = 0; i < games.Length; i++)
				{	
					game =  (Game) Activator.CreateInstance (games [i], true);
					type = Game.GetGameTypeDescription (game.Type);
					games_store.AppendValues (game.Name, type, true, game);
				}
			}

			treeview.Model = games_store;
			game =  (Game) Activator.CreateInstance (games [0], true);
			game.Initialize ();
			drawing_area.puzzle = game;
			question_label.Text = game.Question;
			treeview.ColumnsAutosize ();
		}

		public int NumOfGames {
			get { return ngames;}
		}

		private void OnCursorChanged (object o, EventArgs args) 
		{
			TreeIter iter;
		    
		    	if (!treeview.Selection.GetSelected (out iter)) {
				return;
		    	}

			Game game = games_store.GetValue (iter, COL_OBJECT) as Game;

			// We should not be using IsPreviewMode to know if Initialize has been called
			if (game.IsPreviewMode == false) 
			{
				game.IsPreviewMode = true;
				game.Initialize ();
			}

			question_label.Text = game.Question;
			drawing_area.puzzle = game;
			drawing_area.QueueDraw ();
		}

		private void OnActiveToggled (object o, ToggledArgs args) 
		{
			TreeIter iter;

			if (!games_store.GetIter (out iter, new TreePath (args.Path)))
				return;

			bool enabled = !(bool) games_store.GetValue (iter, COL_ENABLED);
			games_store.SetValue (iter, COL_ENABLED, enabled);
		}

		void OnSelectAll (object sender, EventArgs args)
		{
			games_store.Foreach (delegate (TreeModel model, TreePath path, TreeIter iter)  {
				games_store.SetValue (iter, COL_ENABLED, true);
				return false;
			});
		}

		void OnUnSelectAll (object sender, EventArgs args)
		{
			games_store.Foreach (delegate (TreeModel model, TreePath path, TreeIter iter)  {
				games_store.SetValue (iter, COL_ENABLED, false);
				return false;
			});
		}

		void OnOK (object sender, EventArgs args)
		{
			ngames = 0;
			npos = 0;

			games_store.Foreach (delegate (TreeModel model, TreePath path, TreeIter iter)  {
				if ((bool) games_store.GetValue (iter, COL_ENABLED) == true)
					ngames++;

				return false;
			});

			if (ngames == 0)
				return;

			custom_games = new Type [ngames];
			games_store.Foreach (delegate (TreeModel model, TreePath path, TreeIter iter)  {
				Game game = games_store.GetValue (iter, COL_OBJECT) as Game;
				bool enabled = (bool) games_store.GetValue (iter, COL_ENABLED);

				if (enabled == true) {
					custom_games[npos] = game.GetType ();
					npos++;
				}
				return false;
			});

	
			manager.CustomGames = custom_games;
		}

		public class CairoPreview : DrawingArea 
		{
			public Game puzzle;

			protected override bool OnExposeEvent (Gdk.EventExpose args)
			{
				if(!IsRealized)
					return false;

				int w, h, nw, nh;
				double x = 0, y = 0;
				Cairo.Context cc = Gdk.CairoHelper.Create (args.Window);
				CairoContextEx cr = new CairoContextEx (cc.Handle, this);
				args.Window.GetSize (out w, out h);

				nh = nw = Math.Min (w, h);

				if (nw < w) {
					x = (w - nw) / 2;
				}

				if (nh < h) {
					y = (h - nh) / 2;
				}

				cr.Translate (x, y);
				puzzle.DrawPreview (cr, nw, nh, Direction == Gtk.TextDirection.Rtl);
				((IDisposable)cc).Dispose();
				((IDisposable)cr).Dispose();
	   			return base.OnExposeEvent(args);
			}
		}	
	}
}
