/*
 * GChangepass -- GTK+ Frontend to passwd
 * Copyright (C) 2005 Guilherme de Siqueira Pastore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <errno.h>
#include <gtk/gtk.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <termios.h>
#include <pty.h>
#include <stdio.h>
#include <sys/wait.h>
#include "gchangepass.h"
#include "main.h"
#include "misc.h"

static void wait_in_ms (int fd, int ms)
{
  fd_set fds;
  struct timeval tv;

  tv.tv_sec = 0; 
  tv.tv_usec = 1000*ms;

  FD_ZERO (&fds);
  FD_SET (fd, &fds);

  select (fd+1, &fds, 0L, 0L, &tv);
}

static void wait_to_write (char *path)
{
  gint slave = open (path, O_RDWR);
  struct termios tio;

  while (TRUE) 
    {
      if (tcgetattr (slave, &tio) < 0)
        {
	  gchangepass_dialog (GTK_MESSAGE_ERROR, _("tcgetattr failed: %s"),
			      strerror(errno));
	  close (slave);
	  gchangepass_exit (-1);
        }

      if (tio.c_lflag & ECHO) 
        {
	  wait_in_ms (slave, 100);
	  continue;
        }
      break;
    }

  close (slave);
}

static gboolean is_prompt (char *str, char *prefix)
{
  return (g_str_has_suffix (str, ": ") && g_strrstr (str, prefix));
}

static gint run_passwd (void)
{
  _internals->pid = forkpty (&_internals->master,
			     _internals->slave,
			     NULL, NULL);

  if (_internals->pid == 0)
    {
      setenv ("LANG", "C", TRUE);

      if (execlp ("passwd", "passwd", _userdata->user, NULL) == -1)
	{
	  gchangepass_dialog (GTK_MESSAGE_ERROR,
			      _("Unable to run passwd: %s"), strerror(errno));
	  gchangepass_exit (-1);
	}
    }
  else if (_internals->pid == -1)
    {
      gchangepass_dialog (GTK_MESSAGE_ERROR,
			  _("Unable to fork: %s"), strerror(errno));
      gchangepass_exit (-1);
    }
  else
    return 0;
}

gint gchangepass_magic (void)
{
  if (!run_passwd ())
    {
      fd_set rfds;
      gint status;
      struct timeval tv;

      FD_ZERO (&rfds);
      FD_SET (_internals->master, &rfds);

      tv.tv_sec = 0;
      tv.tv_usec = 100;

      while (!waitpid (_internals->pid, &status, WNOHANG))
	{
	  if (select (_internals->master+1, &rfds, NULL, NULL, &tv) < 0)
	    {
	      gchangepass_dialog (GTK_MESSAGE_ERROR,
				  _("Could not read from the pipe with "
				    "the child: %s"), strerror(errno));
	      gchangepass_exit (-1);
	    }

	  if (FD_ISSET (_internals->master, &rfds))
	    {
	      gchar buf[256];

	      while (TRUE)
		{
		  bzero (buf, 256);
		  if (read (_internals->master, buf, 255) < 1)
		    break;

		  if (is_prompt (buf, "current") || is_prompt (buf, "Old"))
		    {
		      wait_to_write (_internals->slave);

		      write (_internals->master,
			     _userdata->current_password,
			     strlen (_userdata->current_password));
		      write (_internals->master, "\n", 1);
		      continue;
		    }

		  if (is_prompt (buf, "Re"))
		    {
		      wait_to_write (_internals->slave);

		      write (_internals->master,
			     _userdata->new_password,
			     strlen (_userdata->new_password));
		      write (_internals->master, "\n", 1);
		      continue;
		    }
			
		  if (is_prompt (buf, "new") || is_prompt (buf, "New"))
		    {
		      wait_to_write (_internals->slave);

		      write (_internals->master,
			     _userdata->new_password,
			     strlen (_userdata->new_password));
		      write (_internals->master, "\n", 1);
		      continue;
		    }

		  if (g_strrstr (buf, "success") || g_strrstr (buf, "done"))
		    return 0;

		  if (g_strrstr (buf, "fail")
		      || g_strrstr (buf, "incorrect")
		      || g_strrstr (buf, "wrong"))
		    return -2;
		}
	    }
	  else
	    FD_SET (_internals->master, &rfds);
	}
    }
}
