/*
 * GChangepass -- GTK+ Frontend to passwd
 * Copyright (C) 2005 Guilherme de Siqueira Pastore
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <pwd.h>
#include <string.h>
#include "gchangepass.h"
#include "main.h"
#include "misc.h"
#include "passwd.h"
#include "ui.h"

static gboolean exit_cb (GtkWidget *window, GdkEvent *event)
{
  gtk_main_quit ();
  return FALSE;
}

static void root_own_cb (GtkWidget *button, gpointer data)
{
  GtkWidget *win = (GtkWidget *) data;

  gtk_main_quit ();
  gtk_widget_destroy (win);
  gchangepass_query (NEW);
}

static void root_other_cb (GtkWidget *button, gpointer data)
{
  GtkWidget *win = (GtkWidget *) data;

  gtk_main_quit ();
  gtk_widget_destroy (win);
  gchangepass_query (USER);
}

static void entry_ok_cb (GtkWidget *widget, gpointer data)
{
  GtkWidget *dialog = (GtkWidget *) data;

  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}

void gchangepass_ask_root (void)
{
  GtkWidget
    *askroot_win,
    *vbox,
    *hbox,
    *hbox2,
    *question,
    *image,
    *butleft,
    *butright;
  gchar *t;
  GtkTooltips *tips;

  askroot_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (askroot_win),
			_("Whose password to change?"));
  gtk_window_set_position (GTK_WINDOW (askroot_win), GTK_WIN_POS_CENTER);
  gtk_window_set_resizable (GTK_WINDOW (askroot_win), FALSE);
  g_signal_connect (G_OBJECT (askroot_win), "delete-event",
		    G_CALLBACK (exit_cb), NULL);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (askroot_win), vbox);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_set_spacing (GTK_BOX (hbox), 10);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  image = gtk_image_new_from_file (DATA_DIR "/gchangepass/padlock.png");
  gtk_box_pack_start (GTK_BOX (hbox), image, TRUE, TRUE, 0);

  question = gtk_label_new (NULL);
  t = g_strdup_printf (_("The system administrator (%s) can change <i>any "
			 "other user</i>'s password, in addition to its "
			 "own. Whose password do you want to change?"),
		       _userdata->user);
  gtk_label_set_markup (GTK_LABEL (question), t);
  g_free (t);
  gtk_box_pack_start (GTK_BOX (hbox), question, FALSE, FALSE, 0);
  gtk_label_set_line_wrap (GTK_LABEL (question), TRUE);
  
  hbox2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);

  t = g_strdup_printf (_("_Mine (%s's)"), _userdata->user);
  butleft = gtk_button_new_with_mnemonic (t);
  g_free (t);
  gtk_box_pack_start (GTK_BOX (hbox2), butleft, TRUE, TRUE, 0);
  g_signal_connect (G_OBJECT (butleft), "clicked", G_CALLBACK (root_own_cb),
		    askroot_win);

  butright = gtk_button_new_with_mnemonic (_("_Another user's"));
  gtk_box_pack_start (GTK_BOX (hbox2), butright, TRUE, TRUE, 0);
  g_signal_connect (G_OBJECT (butright), "clicked",
		    G_CALLBACK (root_other_cb), askroot_win);

  gtk_window_set_icon_from_file (GTK_WINDOW (askroot_win),
				 DATA_DIR "/pixmaps/gchangepass-icon.png",
				 NULL);

  tips = gtk_tooltips_new ();
  gtk_tooltips_set_tip (tips, butleft,
			_("Click here to change your own password"),
			NULL);
  gtk_tooltips_set_tip (tips, butright,
			_("Click here to choose a user "
			  "and change their password"),
			NULL);

  gtk_widget_show_all (askroot_win);
}

void gchangepass_query (GPdtype dtype)
{
  GtkWidget
    *dialog,
    *vbox,
    *hbox,
    *label,
    *entry;
  gint response;
  gchar *t, *query, *title = NULL;

  switch (dtype)
    {
    case USER:
      t = g_strdup (_("Username"));
      title = g_strdup_printf (_("%s changing someone's password"),
			       _userdata->user);
      break;
    case CURRENT:
      t = g_strdup (_("Current password"));
      break;
    case NEW:
      t = g_strdup (_("New password"));
      break;
    default: /* case CONFIRM: */
      t = g_strdup (_("Confirm new password"));
      break;
    }

  query = g_strdup_printf ("<b>%s:</b> ", t);
  g_free (t);

  if (!title)
    title = g_strdup_printf (_("Changing %s's password"), _userdata->user);

  dialog = gtk_dialog_new_with_buttons (title, NULL, 0,
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_CANCEL,
					GTK_STOCK_OK,
					GTK_RESPONSE_OK,
					NULL);

  g_free (title);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

  vbox = GTK_DIALOG (dialog)->vbox;

  hbox = gtk_hbox_new (FALSE, 3);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 10);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

  label = gtk_label_new (query);
  g_free (query);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  if (dtype == USER)
    {
      GList *users = get_user_list (_internals->uid);
      entry = gtk_combo_new ();
      gtk_combo_set_popdown_strings (GTK_COMBO (entry), users);
      free_user_list (users);
    }
  else
    {
      entry = gtk_entry_new ();
      gtk_entry_set_visibility (GTK_ENTRY (entry), FALSE);
      gtk_signal_connect (GTK_OBJECT (entry), "activate",
			  GTK_SIGNAL_FUNC (entry_ok_cb),
			  dialog);
    }

  gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);

  gtk_window_set_icon_from_file (GTK_WINDOW (dialog),
				 DATA_DIR "/pixmaps/gchangepass-icon.png",
				 NULL);

  gtk_widget_show_all (dialog);
  response = gtk_dialog_run (GTK_DIALOG (dialog));

  switch (response)
    {
    case GTK_RESPONSE_OK:
      /* Get input from the user from combo or entry, depending on the dtype */
      t = g_strdup (gtk_editable_get_chars (GTK_EDITABLE((dtype == USER) ? GTK_COMBO(entry)->entry : entry), 0, -1));
      gtk_widget_destroy (dialog);

      if (!strcmp (t, ""))
	{
	  g_free (t);
	  gchangepass_query (dtype);
	  break;
	}

      if (dtype == USER)
	{
	  struct passwd *pwd = getpwnam (t);

	  if (!pwd)
	    {
	      gchangepass_dialog (GTK_MESSAGE_ERROR,
				  _("User '%s' does not exist!"), t);
	      g_free (t);
	      gchangepass_query (USER);
	      break;
	    }

	  g_free (_userdata->user);
	  _userdata->user = g_strdup (t);
	  g_free (t);
	  gchangepass_query (NEW);
	  break;
	}

      if (dtype == CURRENT)
	{
	  secure_free (_userdata->current_password);
	  _userdata->current_password = g_strdup (t);
	  secure_free (t);
	  gchangepass_query (NEW);
	  break;
	}

      if (dtype == NEW)
	{
	  secure_free (_userdata->new_password);
	  _userdata->new_password = g_strdup (t);
	  secure_free (t);
	  gchangepass_query (CONFIRM);
	  break;
	}

      if (dtype == CONFIRM)
	{
	  gint
	    new_len = strlen (_userdata->new_password),
	    conf_len = strlen (t);

	  if (new_len != conf_len ||
	      strncmp (_userdata->new_password, t, new_len))
	    {
	      gchangepass_dialog (GTK_MESSAGE_ERROR,
				  _("New password and confirmation "
				    "do not match!"));
	      gchangepass_query (NEW);
	      break;
	    }
	  if (_userdata->current_password &&
	      !strncmp (_userdata->current_password, 
			_userdata->new_password,
			new_len))
	    {
	      gchangepass_dialog (GTK_MESSAGE_ERROR,
				  _("Old and new password are the same!"));
	      gchangepass_query (CURRENT);
	      break;
	    }

	  switch (gchangepass_magic ())
	    {
	    case 0:
	      gchangepass_dialog (GTK_MESSAGE_INFO,
				  _("Password changed successfully!"));
	      break;
	    case -2:
	      gchangepass_dialog (GTK_MESSAGE_ERROR,
				  _("Authentication failure: password incorrect"));
	    }
	}
      break;

    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
      gtk_widget_destroy (dialog);
      break;
    default:
      break;
    }
  return;
}
