% $Id: reorder.Rd 1435 2010-05-02 06:11:26Z warnes $

\name{reorder.factor}
\alias{reorder.factor}
\title{Reorder the Levels of a Factor}
\description{
  Reorder the levels of a factor
}
\usage{
\method{reorder}{factor}(x,
                         X,
                         FUN,
                         ...,
                         order=is.ordered(x),
                         new.order,
                         sort=mixedsort)
}
\arguments{
  \item{x}{factor}
  \item{X}{auxillary data vector}
  \item{FUN}{function to be applied to subsets of \code{X} determined by
    \code{x}, to determine factor order}
  \item{...}{optional parameters to \code{FUN}}
  \item{order}{logical value indicating whether the returned
    object should be an \code{\link{ordered}} factor}
  \item{new.order}{a vector of indexes or a vector of label names giving
    the order of the new factor levels}
  \item{sort}{function to use to sort the factor level names, used only
    when \code{new.order} is missing}
}
\details{
  This function changes the order of the levels of a factor. It can do
  so via three different mechanisms, depending on whether, \code{X}
  \emph{and} \code{FUN}, \code{new.order} or \code{sort} are provided.

  If \code{X} \emph{and} \code{Fun} are provided: The data in \code{X}
  is grouped by the levels of \code{x} and \code{FUN} is applied.
  The groups are then sorted by this value, and the resulting order is
  used for the new factor level names.

  If \code{new.order} is provided: For a numeric vector, the new factor
  level names are constructed by reordering the factor levels according
  to the numeric values. For vectors, \code{new.order} gives the list of
  new factor level names. In either case levels omitted from
  \code{new.order} will become missing (\code{NA}) values.

  If \code{sort} is provided (as it is by default): The new
  factor level names are generated by applying the supplied function
  to the existing factor level names. With \code{sort=mixedsort} the
  factor levels are sorted so that combined numeric and character
  strings are sorted in according to character rules on the character
  sections (including ignoring case), and the numeric rules for the
  numeric sections. See \code{\link[gtools]{mixedsort}} for details.
}
\value{
  A new factor with reordered levels
}

\author{Gregory R. Warnes \email{greg@warnes.net}}

\seealso{\code{\link{factor}} and \code{\link[stats]{reorder}}}

\examples{
   # Create a 4 level example factor
   trt <- factor( sample( c("PLACEBO", "300 MG", "600 MG", "1200 MG"),
                  100, replace=TRUE ) )
   summary(trt)
   # Note that the levels are not in a meaningful order.

   # Change the order to something useful
   # default "mixedsort" ordering
   trt2 <- reorder(trt)
   summary(trt2)
   # using indexes:
   trt3 <- reorder(trt, new.order=c(4, 2, 3, 1))
   summary(trt3)
   # using label names:
   trt4 <- reorder(trt, new.order=c("PLACEBO", "300 MG", "600 MG", "1200 MG"))
   summary(trt4)
   # using frequency
   trt5 <- reorder(trt, X=as.numeric(trt), FUN=length)
   summary(trt5)

   # drop out the '300 MG' level
   trt6 <- reorder(trt, new.order=c("PLACEBO", "600 MG", "1200 MG"))
   summary(trt6)
}
\keyword{ manip }
