#
# Rhythmlet - gDesklets sensor for Rhythmbox
# Copyright (c) 2004 Alex Revo
# ----------------------------------------------------------------------
#  AudioScrobbler.py - AudioScrobbler class for Rhythmlet
#  For specifications see:
#	- http://wiki.audioscrobbler.com/index.php/Protocol1
#   - http://wiki.audioscrobbler.com/index.php/Protocol1.1
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
########################################################################

#
# TODO:
#	split out from Rhythmlet into Gnome app.
#	improve queue file
#	save last song between instances to avoid submitting twice


# Import the required (system) modules.
import md5
import os
import re
import socket
import string
import time
import types
import urllib

# Import Rhythmlet-specific modules.
import RLBase


# constants:
MAX_QUEUE_SIZE			= 1000
CLIENT_ID				= "rbx"
CLIENT_VERSION			= "0.1"
SCROBBLER_URL			= "http://post.audioscrobbler.com/"
SCROBBLER_VERSION		= "1.1"



class AudioScrobbler(RLBase.RLBase):
#-----------------------------------------------------------------------
	def __init__(self, *args):
	#-------------------------------------------------------------------
		"""Constructor for AudioScrobbler class."""

		# Initialise base class.
		RLBase.RLBase.__init__(self)

		self.dPrint("AudioScrobbler class instantiated.")

		# Initialise variables.
		self._username				= ""
		self._password				= ""
		self._failures				= 0
		self._handshake				= 0
		self._handshakeNext			= 0
		self._md5Challenge			= ""
		self._queue					= []
		self._runQueue				= 0
		self._submitNext			= 0

		self._scrobblerSubmitUrl	= "http://post.audioscrobbler.com/v1.1.php"

		# Load saved queue.
		self.loadQueue()
	#-------------------------------------------------------------------


	def setUsername(self, newUsername):
	#-------------------------------------------------------------------
		self.scrobPrint("Setting username to: " + newUsername)
		self._username = newUsername
	#-------------------------------------------------------------------


	def setPassword(self, newPassword):
	#-------------------------------------------------------------------
		self._password = newPassword
	#-------------------------------------------------------------------


	def doHandshake(self):
	#-------------------------------------------------------------------
		"""Perform handshake with AudioScrobbler.com server."""

		# Check for current handshake
		if (self._handshake):
			return(0)

		# Don't hammer the AS servers
		if (int(time.time()) < self._handshakeNext):
			return(0)

		self.scrobPrint("Performing handshake with AudioScrobbler server...")

		# Build URL (GET):
		url = "%s?hs=true&p=%s&c=%s&v=%s&u=%s" % (
			SCROBBLER_URL,
			SCROBBLER_VERSION,
			CLIENT_ID,
			CLIENT_VERSION,
			urllib.quote_plus(self._username)
		)

		# Wait at least 30 minutes between handshakes
		self._handshakeNext = int(time.time()) + 1800

		# parse handshake response
		if (self.parseResponse(url)):
			self.scrobPrint("  Handshake was successful.")

			# Force queue run and reset failure count.
			self._handshake = 1
			self._runQueue	= 1
			self._failures = 0

			return(1)
		else:
			self.scrobPrint("  Handshake failed (retrying in 30 minutes.)")

			self._runQueue = 0
		# end if

		return(0)
	#-------------------------------------------------------------------


	def addToQueue(self, data):
	#-------------------------------------------------------------------
		"""Add data to local queue."""

		try:
			self.scrobPrint("Adding entry to queue: %s" % data)

			# clean queue
			while (len(self._queue) > MAX_QUEUE_SIZE):
				self._queue.pop()
			# end while

			# Check for duplicate data.
			if (len(self._queue) > 0):
				prevEntry= re.sub("\[\d+?\]", "", self._queue[0])
				newEntry = re.sub("\[\d+?\]", "", data)

				# Ignore time difference (last 26 chars)
				if (prevEntry[0:-26] == newEntry[0:-26]):
					self.scrobPrint("  Duplicate entry detected; not adding to queue.")

					return(0)
				# end if
			# end if

			self._queue.insert(0, data)

			# Save queue
			self.saveQueue()

			return(data)
		except Exception, error:
			self.handleException("AudioScrobbler::addToQueue()", error)
	#-------------------------------------------------------------------


	def runQueue(self):
	#-------------------------------------------------------------------
		"""Submit queue to AudioScrobbler server"""

		try:
			# return if we have no reason to run the queue
			if (not self._runQueue):
				return(1)
			if (not len(self._queue)):
				return(1)
			if (int(time.time()) < self._submitNext):
				return(1)
			if (not self._handshake):
				return(1)
			# end if

			self.scrobPrint("Running queue.")

			# If updating has failed 3/+ times, renegotiate handshake and
			# stop running queue until new handshake.
			if (self._failures >= 3):
				self.scrobPrint("  [!!] Queue run has failed 3+ times; update has been cached locally.")

				self._runQueue = 0
				self._handshake = 0

				return(1)
			# end if


			# Build POST data:
			newPostData = ""
			numQEntries = 0

			# Loop through queue, combine all entries into one dict.
			for i in self._queue:
				newPostData += "%s&" % i
				numQEntries += 1
			# end for

			# Add username and md5 response.
			web_username= urllib.quote_plus(self.utf8Enc(self._username))
			web_md5resp = urllib.quote_plus(self.utf8Enc(self.hashPassword(self._password)))
			newPostData += "u=%s&s=%s" % (web_username, web_md5resp)

			# parse response
			resp = self.parseResponse(self._scrobblerSubmitUrl, newPostData)
			if (resp):
				self.scrobPrint("  Success. %s song(s) submitted." % numQEntries)

				# clear queue & reset failure count
				self._queue = []
				self._failures = 0

				# Save new (empty) queue.
				self.saveQueue()
			else:
				# FIXME: spam protection triggered
				#  urllib times out but the track is still submitted (sometimes)
				self._failures += 1
				self._runQueue = 0

				self.scrobPrint("  Queue run has failed %s time(s)." % self._failures)
			# end if
		except Exception, error:
			self.handleException("AudioScrobbler:runQueue()", error)
		# end try

		return(0)
	#-------------------------------------------------------------------


	def parseResponse(self, url, postData = ""):
	#-------------------------------------------------------------------
		"""Parse response from AudioScrobbler server"""

		retVal = 0

		try:
			resp = urllib.urlopen(url, postData)
		except Exception, error:
			self.scrobPrint("  Error: %s" % error)
			return(0)
		# end if


		respType = resp.readline().rstrip()
		if (respType[0:8] == "UPTODATE"):
			self._md5Challenge = resp.readline().rstrip()
			self._scrobblerSubmitUrl = resp.readline().rstrip()
			retVal = 1
		if (respType[0:6] == "UPDATE"):
			self.scrobPrint("  UPDATE: An updated version is available: <" + respType[7:] + ">")
			self._md5Challenge = resp.readline().rstrip()
			self._scrobblerSubmitUrl = resp.readline().rstrip()
			retVal = 1
		if (respType[0:2] == "OK"):
			retVal = 1
		if (respType[0:6] == "FAILED"):
			self.scrobPrint("  [!!] FAILED: " + respType[7:])
		if (respType[0:7] == "BADUSER"):
			self.scrobPrint("  [!!] BADUSER: Invalid username.")
		if (respType[0:7] == "BADAUTH"):
			self.scrobPrint("  [!!] BADAUTH: Invalid username and/or password.")
		# end if

		intervalLine = resp.readline().rstrip()
		if (intervalLine[0:8] == "INTERVAL"):
			if (intervalLine[9:]):
				#self.scrobPrint("  INTERVAL: " + intervalLine[9:])
				self._submitNext = time.time() + int(intervalLine[9:])
			# end if
		# end if

		return(retVal)
	#-------------------------------------------------------------------


	def loadQueue(self):
	#-------------------------------------------------------------------
		"""Load saved queue from ~/.rlscrobbler.queue."""

		self.scrobPrint("Loading AudioScrobbler queue...")

		try:
			filePath = os.path.join(os.environ['HOME'], ".rlscrobbler.queue")
			numEntries = 0

			if (os.path.isfile(filePath)):
				fileCache = open(filePath, "r")
				for i in fileCache.readlines():
					self._queue.append(re.sub("\n$", "", i))
					numEntries += 1
				# end for
				fileCache.close()

				self.scrobPrint("  %s entries loaded." % numEntries)

				return(numEntries)
			# end if
		except Exception, error:
			self.handleException("AudioScrobbler::loadQueue()", error)
		# end try

		return(0)
	#-------------------------------------------------------------------


	def saveQueue(self):
	#-------------------------------------------------------------------
		"""Save self._queue to ~/.rlscrobbler.queue."""

		try:
			filePath = os.path.join(os.environ['HOME'], ".rlscrobbler.queue")

			fileCache = open(filePath, "w")
			for i in self._queue:
				fileCache.write("%s\n" % i)
			# end for
			fileCache.close()

			return(1)
		except Exception, error:
			self.handleException("AudioScrobbler::saveQueue()", error)
		# end try

		return(0)
	#-------------------------------------------------------------------


	def hashPassword(self, strPassword):
	#-------------------------------------------------------------------
		"""Generate MD5 response from user's password."""

		# The MD5 response is md5(md5(password) + challenge), where MD5
		# is the ascii-encoded MD5 representation, and + represents
		# concatenation.

		tmp = md5.new(strPassword).hexdigest()
		md5_response = md5.new(tmp + self._md5Challenge).hexdigest()

		return(md5_response)
	#-------------------------------------------------------------------


	def utf8Enc(self, strInput):
	#-------------------------------------------------------------------
		"""Return unicode type from input."""

		if (type(strInput) == types.UnicodeType):
			return(strInput)
		# end if

		return(unicode(strInput, 'iso-8859-1'))
	#-------------------------------------------------------------------


	def scrobPrint(self, message):
	#-------------------------------------------------------------------
		if (self._DEBUG):
			print("\x1b[33;01m[SCROB]\x1b[0m %s" % message)
		# end if
	#-------------------------------------------------------------------
#-----------------------------------------------------------------------
