/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include <gdk/gdk.h>

#include "gdis.h"

#define DEBUG 0

/* externals  - put in sysenv? */
extern GtkWidget *window;
extern GdkPixmap *pixmap;

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/*******************************************/
/* general call to expose the drawing area */
/*******************************************/
#define DEBUG_EXPOSE_CANVAS 0
void expose_canvas()
{
GdkRectangle update_rect;

/* this is a general update, ie redraws gtk-gdk or opengl canvas */
update_rect.x = sysenv.x; 
update_rect.y = sysenv.y + 50;          /* YTF is this needed??? */
update_rect.width = sysenv.width;
update_rect.height = sysenv.height;

#if DEBUG_EXPOSE_CANVAS
printf("    origin: (%d,%d)\n",update_rect.x,update_rect.y);
printf("dimensions:  %dx%d\n",update_rect.width,update_rect.height);
#endif

gtk_widget_draw (window, &update_rect);
}

/****************************/
/* Ensure a model is viewed */
/****************************/
void force_view(gint model)
{
gint i, j, na, nb, np;

/* nothing to do if the model is already displayed */
for (i=0 ; i<sysenv.num_displayed ; i++)
  if (sysenv.displayed[i] == model)
    return;

/* determine where to display the model */
np = model;                    /* available models previous */
na = sysenv.num_models-np-1;   /* available models after */

/* can we completely forward fill? */
if (na >= sysenv.num_displayed-1)
  {
  j = model;
  for (i=0 ; i<sysenv.num_displayed ; i++)
    sysenv.displayed[i] = j++;
  }
else
  {
/* number of required back fill models */
  i = sysenv.num_displayed-na-1;
/* assign number of possible back fill models */
  nb = (np >= i) ? i : np;
/* fill as much as possible */
  i=model-nb;
  j=0;
  while (j<sysenv.num_displayed)
    {
    if (i<sysenv.num_models)
      sysenv.displayed[j++]=i++;
    else
      sysenv.displayed[j++]=-1;
    }
  }
}

/******************************/
/* Configure screen splitting */
/******************************/
gint mod_screen(gint wmod, gint hmod)
/* wmod - num horizontal frames */
/* hmod - num vertical frames */
{
gint i, j, k, mod;
struct model_pak *data;

/* dimension check */
if (wmod*hmod > MAX_DISPLAYED)
  {
  show_text("Too many subwindows requested, sorry!");
  return(1);
  }

/* calculate sub-window dimensions */
sysenv.num_displayed = wmod*hmod;
sysenv.subx = wmod;
sysenv.suby = hmod;
mod = (wmod>hmod) ? wmod : hmod;
/* subwin scaling (want it uniform in x & y directions) */
sysenv.fsubwidth = 1.0/(gfloat) mod;
sysenv.fsubheight =  1.0/(gfloat) mod; 
/* actual subwin dimensions */
sysenv.subwidth = (gint) ((gfloat) sysenv.width / (gfloat) wmod);
sysenv.subheight = (gint) ((gfloat) sysenv.height / (gfloat) hmod);
/* subwin scaling (want it uniform in x & y directions) */
sysenv.subscale = (sysenv.subwidth > sysenv.subheight) ? 
                   sysenv.subheight : sysenv.subwidth;
/* calculate pixel centres */
if (sysenv.num_displayed > 0)
  {
  i=0;
  for (k=0 ; k<hmod ; k++)
    {
    for (j=0 ; j<wmod ; j++)
      {
/* determine subwin centres */
      sysenv.subcenx[i] = (gint) (sysenv.width*(0.5 + (gfloat) j)/wmod);
      sysenv.subceny[i] = (gint) (sysenv.height*(0.5 + (gfloat) k)/hmod);
#if DEBUG
printf("[%d] : centre (%d,%d) : frac %fx%f : real %dx%d\n",i,
                              sysenv.subcenx[i],sysenv.subceny[i],
                              sysenv.fsubwidth,sysenv.fsubheight,
                              sysenv.subwidth,sysenv.subheight);
#endif
      i++;
      }
    }
  }

/* changed the layout - update (ie re-scale) coords */
/* update all coords (even if not displayed - as they may be later) */
for (i=0 ; i<sysenv.num_models ; i++)
  {
  data = model_ptr(i, RECALL);
  if (data)
    calc_coords(REFRESH, data);
  }

/* clean all display slots */
for (i=0 ; i<sysenv.num_displayed ; i++)
  sysenv.displayed[i] = -1;
/* calculate the display order */
force_view(sysenv.active);

return(0);
}

/**************************/
/* default system colours */
/**************************/
void set_colour(GdkGC *gc, gint name)
{
GdkColor colour;

/* determine the required RGB code */
switch(name)
  {
  case BLACK:
    colour.red = 0;
    colour.green = 0;
    colour.blue = 0;
    break;
  case RED:
    colour.red = 65535;
    colour.green = 0;
    colour.blue = 0;
    break;
  case GREEN:
    colour.red = 0;
    colour.green = 65535;
    colour.blue = 0;
    break;
  case BLUE:
    colour.red = 0;
    colour.green = 0;
    colour.blue = 65535;
    break;
  case YELLOW:
    colour.red = 65535;
    colour.green = 65535;
    colour.blue = 0;
    break;
  case ORANGE:
  case GOLD:
    colour.red = 50000;
    colour.green = 50000;
    colour.blue = 10000;
    break;
  case SEA_GREEN:
    colour.red = 11150;
    colour.green = 55705;
    colour.blue = 40632;
    break;
  case AQUAMARINE:
    colour.red = 10000;
    colour.green = 65535;
    colour.blue = 50000;
    break;
  case NAVY_BLUE:
    colour.red = 0;
    colour.green = 32000;
    colour.blue = 65535;
    break;
  case LIGHT_TAN:
    colour.red = 65535;
    colour.green = 65535;
    colour.blue = 53739;
    break;
  case LEMON:
    colour.red = 64890;
    colour.green = 63569;
    colour.blue = 29491;
    break;
  case PURPLE:
    colour.red = 64000;
    colour.green = 30000;
    colour.blue = 64000;
    break;
  case WHITE:
  default:
    colour.red = 65535;
    colour.green = 65535;
    colour.blue = 65535;
    break;
  }

/* TODO - default (safe) colour if fails? */
/* get closest colour in colourmap */
if (!gdk_color_alloc(sysenv.colourmap, &colour))
  printf("Warning: colour assignment failed.\n");

gdk_gc_set_foreground(gc, &colour);
}

/****************************/
/* Colour interface for RGB */
/****************************/
void set_rgb_colour(GdkGC *gc, gint *rgb)
{
GdkColor colour;

/* get the colour map */
colour.red = *rgb;
colour.green = *(rgb+1);
colour.blue = *(rgb+2);

/* TODO - default (safe) colour if fails? */
if (!gdk_color_alloc(sysenv.colourmap, &colour))
  printf("Warning: colour assignment failed.\n");

gdk_gc_set_foreground(gc, &colour);
}

/************************/
/* GTK/GDK drawing code */
/************************/
#define DEBUG_GDK_DRAW 0
/* TODO - instead of putting if statements inside the */
/* loops over atoms & bonds (which is expensive) for the */
/* special highlighting/label objects, put them afterwards */
void gdk_draw(gint action)
{
gint model,a,b,c,i,j,k,l,n,m,mlen;
gint px1, py1, px2, py2, pmx, pmy, ox, oy, type, ghost;
gint ac, bc;  /* atom/bond count */
gfloat dx, dy, lx, ly, t1, t2, ta, tmp, rad;
gfloat arm1[2],arm2[2],vec[2],len1,len2;
gchar facet_label[10], label[ELEM_LABEL_SIZE], *mode_label, *txt;
struct model_pak *data;
struct vec_pak *axes;
struct plane_pak *plane;
GdkRectangle update_rect;
GdkGC *gc1;
GSList *bond, *plist;

/* no canvas, no draw */
if (pixmap == NULL)
  return;

#if DEBUG_GDK_DRAW
printf("gdk_draw() start\n");
#endif
#ifdef TIMER
start_timer("gdk_draw");
#endif

/* create GC */
gc1 = gdk_gc_new(pixmap);

/* sysenv.active should always be visible */
force_view(sysenv.active);

/* wipe the drawing area clean */
if (action == ALL) 
  {
  set_colour(gc1,BLACK);
  gdk_draw_rectangle (pixmap, gc1, TRUE, 0, 0, sysenv.width, sysenv.height);
  }

m=0;
while (m < sysenv.num_displayed)
  {
/* NEW - do we do this subwin? */
  if (action == SINGLE && sysenv.displayed[m] != sysenv.active)
    {
    m++;
    continue;
    }
/* calculate the subwin's drawing area (including frame) */
  px1 = sysenv.subcenx[m] - sysenv.subwidth/2.0;
  py1 = sysenv.subceny[m] - sysenv.subheight/2.0;
  px2 = sysenv.subwidth;
  py2 = sysenv.subheight;
/* wipe the drawing area clean */
  if (action == SINGLE)
    {
    set_colour(gc1,BLACK);
    gdk_draw_rectangle (pixmap, gc1, TRUE, px1, py1, px2, py2);
    }
/* calculate the frame coords */
  px1 += sysenv.fsubwidth*W_BORDER;
  py1 += sysenv.fsubheight*H_BORDER;
  px2 -= 2*sysenv.fsubwidth*W_BORDER;
  py2 -= 2*sysenv.fsubheight*H_BORDER;

/* NEW - gc clipping for this subwindow's FRAME */
/* NB: this must be done in order for the frame to be displayed; as */
/* clipping from the previous subwindow will prevent it being drawn */
  update_rect.x = px1;
  update_rect.y = py1;
  update_rect.width = px2+1;
  update_rect.height = py2+1;
  gdk_gc_set_clip_rectangle(gc1, &update_rect);

/* set the frame colour */
  if (sysenv.displayed[m] == sysenv.active)
    set_colour(gc1, LEMON);
  else
    set_colour(gc1, WHITE);
/* draw frame */
  gdk_draw_rectangle (pixmap, gc1, FALSE, px1, py1, px2, py2);

/* gc clipping for this subwindow's model drawing */
  update_rect.x = px1+2;
  update_rect.y = py1+2;
  update_rect.width = px2-3;
  update_rect.height = py2-3;
  gdk_gc_set_clip_rectangle(gc1, &update_rect);

/* nothing else to do? */
  model = sysenv.displayed[m];
  data = model_ptr(model, RECALL);

/* CURRENT - loop while data exists */
  ghost=0;
  while(data)
    {
#if DEBUG_GDK_DRAW
printf("Displaying model %d : address %p\n",model,data);
#endif
/* NB: can't loop over molecules - as some atoms */
/* may be deleted, and the molecule list is as yet */
/* not updated when atoms are deleted */

/* loop over atoms in model */
  ac=0;
  for(n=data->num_atoms ; n-- ; )
    {
/* isolated atoms */
    if (!(data->atoms+n)->atom_bonds)
      {
/* non-deleted atoms */
      if ((data->atoms+n)->status & (DELETED | HIDDEN))
        continue;
/* get pixel coordinates */
      px1 = (data->atoms+n)->px + (data->atoms+n)->offx;
      py1 = (data->atoms+n)->py + (data->atoms+n)->offy;
/* translation - offset */
      px1 += data->offset[0];
      py1 += data->offset[1];
/* draw cross */
      set_rgb_colour(gc1, (data->atoms+n)->colour);
      SUBLINE(pixmap, gc1, m, px1-5, py1, px1+5, py1);
      SUBLINE(pixmap, gc1, m, px1, py1-5, px1, py1+5);
      ac++;
/* atom labels */
      if (data->atom_labels)
        SUBTEXT(pixmap, gc1, m, px1+7, py1, (data->atoms+n)->element);
/* square highlight mode */
      if ((data->atoms+n)->status & (SQUARE_HL | SELECT_HL))
        {
        set_colour(gc1, GOLD);
        for (a=-1 ; a<2 ; a+= 2)
          for (b=-1 ; b<2 ; b+= 2)
            SUBLINE(pixmap, gc1, m, px1+a*5, py1+a*5, px1+b*5, py1-b*5);
        }
      }
    }

#if DEBUG_GDK_DRAW
printf("Drew %d isolated atoms.\n",ac);
#endif

/*
#define GET_BOND_MEMBER(b,m) {((struct bond_pak *)b->data)->m};
*/

/* loop over bonds */
  bc=0;
  bond = data->bonds;
  while(bond != NULL)
    {
/* the two atoms */
/* FIXME - there's got to be a better way to do this */
    i = ((struct bond_pak *) bond->data)->atom1;
    j = ((struct bond_pak *) bond->data)->atom2;

    if ((data->atoms+i)->status & DELETED)
      continue;
    if ((data->atoms+j)->status & DELETED)
      continue;

/* get pixel coordinates */
    px1 = (data->atoms+i)->px + (data->atoms+i)->offx;
    py1 = (data->atoms+i)->py + (data->atoms+i)->offy;
    px2 = (data->atoms+j)->px + (data->atoms+j)->offx;
    py2 = (data->atoms+j)->py + (data->atoms+j)->offy;
/* bond midpoint */
    pmx = 0.5 * (px1 + px2);
    pmy = 0.5 * (py1 + py2);
    ((struct bond_pak *) bond->data)->px = pmx;
    ((struct bond_pak *) bond->data)->py = pmy;
/* translation via offset */
    px1 += data->offset[0];
    py1 += data->offset[1];
    px2 += data->offset[0];
    py2 += data->offset[1];
    pmx += data->offset[0];
    pmy += data->offset[1];
/* crop draw */
    bc++;
/* draw half the bond using element i's colour from the database */
    if (!((data->atoms+i)->status & HIDDEN))
      {
      set_rgb_colour(gc1, (data->atoms+i)->colour);
      SUBLINE(pixmap, gc1, m, px1, py1, pmx, pmy);
/* square highlight */
      if ((data->atoms+i)->status & (SQUARE_HL | SELECT_HL))
        {
        set_colour(gc1, GOLD);
        for (a=-1 ; a<2 ; a+= 2)
          for (b=-1 ; b<2 ; b+= 2)
            SUBLINE(pixmap, gc1, m, px1+a*5, py1+a*5, px1+b*5, py1-b*5);
        }
/* atom label */
      if (data->atom_labels)
        {
        set_colour(gc1, SEA_GREEN);
        SUBTEXT(pixmap, gc1, m, px1+3, py1, (data->atoms+i)->element);
        }
      }
/* draw half the bond using element j's colour from the database */
    if (!((data->atoms+j)->status & HIDDEN))
      {
      set_rgb_colour(gc1, (data->atoms+j)->colour);
      SUBLINE(pixmap, gc1, m, px2, py2, pmx, pmy);
/* square highlight */
      if ((data->atoms+j)->status & (SQUARE_HL | SELECT_HL))
        {
        set_colour(gc1, GOLD);
        for (a=-1 ; a<2 ; a+= 2)
          for (b=-1 ; b<2 ; b+= 2)
            SUBLINE(pixmap, gc1, m, px2+a*5, py2+a*5, px2+b*5, py2-b*5);
        }
/* atom label */
      if (data->atom_labels)
        {
        set_colour(gc1, SEA_GREEN);
        SUBTEXT(pixmap, gc1, m, px2+3, py2, (data->atoms+j)->element);
        }
      }
    bond = g_slist_next(bond);
    }

#if DEBUG_GDK_DRAW
printf("Drew %d bonds\n",bc);
#endif

/* NEW - user bonds */
for (n=data->num_ubonds ; n-- ; )
  {
/* bond type dependence */
  type = 0;
  switch((data->ubonds+n)->type)
    {
    case SINGLE:
      continue;
    case TRIPLE:
      type++;
    case DOUBLE:
      type++;
      break;
    }
/* the two atoms */
  i = (data->ubonds+n)->atom1;
  j = (data->ubonds+n)->atom2;
/* get pixel coordinates */
  px1 = (data->atoms+i)->px + (data->atoms+i)->offx;
  py1 = (data->atoms+i)->py + (data->atoms+i)->offy;
  px2 = (data->atoms+j)->px + (data->atoms+j)->offx;
  py2 = (data->atoms+j)->py + (data->atoms+j)->offy;
/* bond midpoint */
  pmx = (px1 + px2)/2;
  pmy = (py1 + py2)/2;
/* translation via offset */
  px1 += data->offset[0];
  py1 += data->offset[1];
  px2 += data->offset[0];
  py2 += data->offset[1];
  pmx += data->offset[0];
  pmy += data->offset[1];
/* orientation dependent offsets */
  dy = (gfloat) py2 - py1;
  dx = (gfloat) px2 - px1;
  if (dx)
    ta = PI/2.0 - atan(dy/dx);
  else
    ta = 0.0;
/* pixel offsets */
  ox = (gint) (5.0*cos(ta));
  oy = (gint) (-5.0*sin(ta));  /* inv screen coord */
/* crop draw */
  for (k=-1 ; k<type ; k+=2)
    {
/* draw half the bond using element i's colour from the database */
    set_rgb_colour(gc1, (data->atoms+i)->colour);
    SUBLINE(pixmap, gc1, m, px1-k*ox, py1-k*oy, pmx-k*ox, pmy-k*oy);
/* draw half the bond using element j's colour from the database */
    set_rgb_colour(gc1, (data->atoms+j)->colour);
    SUBLINE(pixmap, gc1, m, px2-k*ox, py2-k*oy, pmx-k*ox, pmy-k*oy);
    }
  }

#if DEBUG_GDK_DRAW
printf("Drew %d user bonds\n",data->num_ubonds);
#endif

/* loop over shells */
if (data->show_shells)
  {
  for (n=data->num_shells ; n-- ; )
    {
/* non-deleted shells */
    if ((data->shells+n)->status & (DELETED | HIDDEN))
      continue;
/* get pixel coordinates */
    px1 = (data->shells+n)->px + (data->shells+n)->offx;
    py1 = (data->shells+n)->py + (data->shells+n)->offy;
/* translation - offset */
    px1 += data->offset[0];
    py1 += data->offset[1];
/* get shell's colour */
    if ((data->shells+n)->status & (SQUARE_HL | SELECT_HL))
      set_colour(gc1, GOLD);
    else
      set_rgb_colour(gc1, (data->shells+n)->colour);
/* NEW - draw a square (distinguish from core) */
    for (a=-1 ; a<2 ; a+= 2)
      for (b=-1 ; b<2 ; b+= 2)
        SUBLINE(pixmap, gc1, m, px1+a*5, py1+a*5, px1+b*5, py1-b*5);
    }
  }

/*************************/
/* SPECIAL OBJECT - AXES */
/*************************/
  if (data->axes_on)
    { 
    if (data->axes_type == CARTESIAN)
      {
      strcpy(label, "x");
      axes = data->axes;
      }
    else
      {
      strcpy(label, "a");
      axes = data->xlat;
      }
/* draw the lines and labels */
    for (k=0; k<3 ; k++)
      {
/* axis */
      px1 = 35 + axes[k].px[0];
      py1 = 40 + axes[k].px[1];
      px2 = 35;
      py2 = 40;
/* subwin scaling */
      px1 += sysenv.subcenx[m]-0.5*sysenv.subwidth;
      py1 += sysenv.subceny[m]-0.5*sysenv.subheight;
      px2 += sysenv.subcenx[m]-0.5*sysenv.subwidth;
      py2 += sysenv.subceny[m]-0.5*sysenv.subheight;
      set_colour(gc1, WHITE);
      gdk_draw_line (pixmap, gc1, px1, py1, px2, py2);
/* axis label */
      lx = 35 + axes[k+3].px[0];
      ly = 40 + axes[k+3].px[1];
      lx += sysenv.subcenx[m]-0.5*sysenv.subwidth;
      ly += sysenv.subceny[m]-0.5*sysenv.subheight;
      set_colour(gc1, SEA_GREEN);
      gdk_draw_text(pixmap, sysenv.dfont, gc1, lx, ly, label, 1);
      label[0]++;
      }
    }

/*************************/
/* SPECIAL OBJECT - CELL */
/*************************/
  if (data->cell_on)
    {
    set_colour(gc1, WHITE);
/* draw the opposite ends of the frame */
    for(k=8 ; k-- ; )
      {
/* part 1 */
      i = 2*(k/2) + 1;
      j = 4*(k/4);
/* retrieve coordinates */
      px1 = data->cell[i].px[0] + data->offset[0];
      py1 = data->cell[i].px[1] + data->offset[1];
      px2 = data->cell[j].px[0] + data->offset[0];
      py2 = data->cell[j].px[1] + data->offset[1];
/* draw */
      SUBLINE(pixmap, gc1, m, px1, py1, px2, py2);
/* part2 */
      px2 = data->cell[j+2].px[0] + data->offset[0];
      py2 = data->cell[j+2].px[1] + data->offset[1];
      SUBLINE(pixmap, gc1, m, px1, py1, px2, py2);
      }
/* draw the sides of the frame */
    for (k=4 ; k-- ; )
      {
      l = k+4;
/* retrieve coordinates */
      px1 = data->cell[k].px[0] + data->offset[0];
      py1 = data->cell[k].px[1] + data->offset[1];
      px2 = data->cell[l].px[0] + data->offset[0];
      py2 = data->cell[l].px[1] + data->offset[1];
/* draw */
      SUBLINE(pixmap, gc1, m, px1, py1, px2, py2);
      }
    }

/*************************/
/* SPECIAL OBJECT - MESH */
/*************************/
/* TODO - user can change spacing */
  if (data->mesh_on)
    {
/* compute on-screen spacing */
    lx = (sysenv.subwidth - 2*sysenv.fsubwidth*W_BORDER)/data->mesh.px;
    ly = (sysenv.subheight - 2*sysenv.fsubheight*W_BORDER)/data->mesh.py;
/* compute start points */
    px1 = sysenv.subcenx[m] - 0.5*sysenv.subwidth + sysenv.fsubwidth*W_BORDER;
    py1 = sysenv.subceny[m] - 0.5*sysenv.subheight + sysenv.fsubheight*H_BORDER;
    py2 = sysenv.subceny[m] + 0.5*sysenv.subheight - sysenv.fsubheight*H_BORDER;
/* draw mesh */
    set_colour(gc1, LIGHT_TAN);
    for (i=1 ; i<data->mesh.px ; i++)
      {
/* lx must be a float to get the correct spacing here */
      px2 = px1 + i*lx;
      gdk_draw_line (pixmap, gc1, px2, py1, px2, py2);
      }
    px2 = sysenv.subcenx[m] + 0.5*sysenv.subwidth - sysenv.fsubwidth*W_BORDER;
    for (j=1 ; j<data->mesh.py ; j++)
      {
/* ly must be a float to get the correct spacing here */
      py2 = py1 + j*ly;
      gdk_draw_line (pixmap, gc1, px1, py2, px2, py2);
      }
    }

/**********************************/
/* SPECIAL OBJECT - BOX SELECTION */
/**********************************/
  if (data->box_on)
    {
/* new box selection style */
    gdk_gc_set_line_attributes(gc1, 1, GDK_LINE_DOUBLE_DASH,
                               GDK_CAP_NOT_LAST, GDK_JOIN_MITER);
    px1 = data->box_select[0].px;
    py1 = data->box_select[0].py;
    px2 = data->box_select[1].px;
    py2 = data->box_select[1].py;
/* draw box */
    set_colour(gc1, ORANGE);
    gdk_draw_line (pixmap, gc1, px1, py1, px1, py2);
    gdk_draw_line (pixmap, gc1, px1, py2, px2, py2);
    gdk_draw_line (pixmap, gc1, px2, py2, px2, py1);
    gdk_draw_line (pixmap, gc1, px2, py1, px1, py1);
/* reset */
    gdk_gc_set_line_attributes(gc1, 1, GDK_LINE_SOLID,
                               GDK_CAP_NOT_LAST, GDK_JOIN_ROUND);
    }

/********************************/
/* SPECIAL OBJECT - GEOM LABELS */
/*******************************/
  if (data->num_geom)
    {
    set_colour(gc1, YELLOW);
    for (n=data->num_geom ; n-- ; )
      {
/* default */
      gdk_gc_set_line_attributes(gc1, 1, GDK_LINE_DOUBLE_DASH,
                                 GDK_CAP_NOT_LAST, GDK_JOIN_ROUND);
      switch((data->geom+n)->type)
        {
        case BOND:
          gdk_gc_set_line_attributes(gc1, 1, GDK_LINE_SOLID,
                                     GDK_CAP_NOT_LAST, GDK_JOIN_ROUND);
        case DIST:
/* get the coordinates + offset */
          i = *((data->geom+n)->list);
          j = *((data->geom+n)->list+1);
          px1 = data->offset[0] + (data->atoms+i)->px; 
          py1 = data->offset[1] + (data->atoms+i)->py; 
          px2 = data->offset[0] + (data->atoms+j)->px; 
          py2 = data->offset[1] + (data->atoms+j)->py; 
/* draw the line */
          SUBLINE(pixmap, gc1, m, px1, py1, px2, py2);
/* draw the label */
          SUBTEXT(pixmap, gc1, m, 5+(px1+px2)/2, (py1+py2)/2, 
                                       (data->geom+n)->text);
          break;
        case ANGLE:
          gdk_gc_set_line_attributes(gc1, 1, GDK_LINE_SOLID,
                                     GDK_CAP_NOT_LAST, GDK_JOIN_ROUND);
/* draw a circular arc */
/* TODO - nice (but harder) elipsoidal arcs */
/* get the coordinates + offset */
          i = *((data->geom+n)->list);
          j = *((data->geom+n)->list+1);
          k = *((data->geom+n)->list+2);
/* mid atom */
          pmx = (data->atoms+j)->px;
          pmy = (data->atoms+j)->py;
/* vectors */
          arm1[0] = (data->atoms+i)->px - (data->atoms+j)->px;
          arm1[1] = (data->atoms+i)->py - (data->atoms+j)->py;
          arm2[0] = (data->atoms+k)->px - (data->atoms+j)->px;
          arm2[1] = (data->atoms+k)->py - (data->atoms+j)->py;
/* get the minimum length */
          len1 = sqrt(arm1[0]*arm1[0] + arm1[1]*arm1[1]);
          len2 = sqrt(arm2[0]*arm2[0] + arm2[1]*arm2[1]);
          rad = (len1 < len2) ? len1 : len2;
/* set to fixed value, unless too small */
          if (rad > MIN_ARC_RAD)
            rad = MIN_ARC_RAD;
/* 3 o'clock position */
          vec[0] = 1;
          vec[1] = 0;
/* find 1st angle with the 3 o'clock position  - relative to center! */
          t1 = via(vec,arm1,2);
          if (arm1[1] > 0.0)
            {
            t1 -= 2.0*PI;
            t1 *= -1.0;
            }
/* find 2nd angle with the 3 o'clock position */
          t2 = via(vec,arm2,2);
          if (arm2[1] > 0.0)
            {
            t2 -= 2.0*PI;
            t2 *= -1.0;
            }
/* force t1 < t2 */
          if (t1 > t2)
            {
            tmp = t1;
            t1 = t2;
            t2 = tmp;
            }
/*
printf("angles: %f & %f\n",180.0*t1/PI,180.0*t2/PI);
*/
/* determine which angle should be 1st - given counterclock rotation */
          if (t2-t1 > PI)
            t1 = t2;
          t2 = via(arm1,arm2,2);
/*
printf("angles: %f & %f\n",180.0*t1/PI,180.0*t2/PI);
*/
/* draw the arc */
          SUBARC(pixmap, gc1, m, 
                 data->offset[0]+pmx - (gint) rad,
                 data->offset[1]+pmy - (gint) rad, 
                 (gint) (2.0*rad), (gint) (2.0*rad),
                 (gint) (64*180*t1/PI), (gint) (64*180*t2/PI));

/* compute text angle */
          ta = (t1+t2/2.0);
/* write text */
          SUBTEXT(pixmap, gc1, m, data->offset[0]+pmx+2*rad*cos(ta),
                  data->offset[1]+pmy-2*rad*sin(ta), (data->geom+n)->text);
          break;
        case DIHEDRAL:
/* combo - line along the axis + ANGLE type arc */
/* get the coordinates + offset */
          i = *((data->geom+n)->list);
          j = *((data->geom+n)->list+1);
          k = *((data->geom+n)->list+2);
          l = *((data->geom+n)->list+3);
/* do the line part */
          px1 = data->offset[0] + (data->atoms+j)->px; 
          py1 = data->offset[1] + (data->atoms+j)->py; 
          px2 = data->offset[0] + (data->atoms+k)->px; 
          py2 = data->offset[1] + (data->atoms+k)->py; 
/* draw */
          SUBLINE(pixmap, gc1, m, px1, py1, px2, py2);

/* do the angle part */
/* mid pt */
          pmx = ((data->atoms+j)->px + (data->atoms+k)->px)/2;
          pmy = ((data->atoms+j)->py + (data->atoms+k)->py)/2;
/* vectors */
          arm1[0] = (data->atoms+i)->px - (data->atoms+j)->px;
          arm1[1] = (data->atoms+i)->py - (data->atoms+j)->py;
          arm2[0] = (data->atoms+l)->px - (data->atoms+k)->px;
          arm2[1] = (data->atoms+l)->py - (data->atoms+k)->py;
/* get the minimum length */
          len1 = sqrt(arm1[0]*arm1[0] + arm1[1]*arm1[1]);
          len2 = sqrt(arm2[0]*arm2[0] + arm2[1]*arm2[1]);
          rad = (len1 < len2) ? len1 : len2;
/* set to fixed value, unless too small */
          if (rad > MIN_ARC_RAD)
            rad = MIN_ARC_RAD;
/* 3 o'clock position */
          vec[0] = 1;
          vec[1] = 0;
/* find 1st angle with the 3 o'clock position  - relative to center! */
          t1 = via(vec,arm1,2);
          if (arm1[1] > 0.0)
            {
            t1 -= 2.0*PI;
            t1 *= -1.0;
            }
/* find 2nd angle with the 3 o'clock position */
          t2 = via(vec,arm2,2);
          if (arm2[1] > 0.0)
            {
            t2 -= 2.0*PI;
            t2 *= -1.0;
            }
/* force t1 < t2 */
          if (t1 > t2)
            {
            tmp = t1;
            t1 = t2;
            t2 = tmp;
            }
/*
printf("angles: %f & %f\n",180.0*t1/PI,180.0*t2/PI);
*/
/* determine which angle should be 1st - given counterclock rotation */
          if (t2-t1 > PI)
            t1 = t2;
          t2 = via(arm1,arm2,2);
/*
printf("angles: %f & %f\n",180.0*t1/PI,180.0*t2/PI);
*/
/* draw a filled arc */
          FSUBARC(pixmap, gc1, m, 
                  data->offset[0]+pmx - (gint) rad,
                  data->offset[1]+pmy - (gint) rad, 
                  (gint) (2.0*rad), (gint) (2.0*rad),
                  (gint) (64*180*t1/PI), (gint) (64*180*t2/PI));

/* compute text angle */
          ta = (t1+t2/2.0);
/* write text */
          SUBTEXT(pixmap, gc1, m, data->offset[0]+pmx+2*rad*cos(ta),
                  data->offset[1]+pmy-2*rad*sin(ta), (data->geom+n)->text);
          break;
        }
      }
    }

/***************************/
/* SPECIAL OBJECT - MYSURF */
/***************************/
if (data->solsurf_on)
  {
  set_colour(gc1, YELLOW);
  for (i=0 ; i<data->mysurf[SOLSURF].num_points ; i++)
    {
    px1 = data->mysurf[SOLSURF].px[i];
    py1 = data->mysurf[SOLSURF].py[i];
/* translation - offset */
    px1 += data->offset[0];
    py1 += data->offset[1];
/* draw small line */
    SUBLINE(pixmap, gc1, m, px1, py1, px1, py1);
    }
  }
if (data->molsurf_on)
  {
  for (i=0 ; i<data->mysurf[MOLSURF].num_points ; i++)
    {
    j = data->mysurf[MOLSURF].touch[i]; 
    set_rgb_colour(gc1, (data->atoms+j)->colour);  /* contact */

    px1 = data->mysurf[MOLSURF].px[i];
    py1 = data->mysurf[MOLSURF].py[i];
/* translation - offset */
    px1 += data->offset[0];
    py1 += data->offset[1];
/* draw small line */
    SUBLINE(pixmap, gc1, m, px1, py1, px1, py1);
    }
  }

/**************************************/
/* SPECIAL OBJECT - CONNOLLY SURFACE */
/*************************************/
  if (data->csurf_on)
    {
    for(n=0 ; n<data->csurf.grid*data->csurf.grid ; n++)
      {
/* get atom touched (if any) */
      j = *(data->csurf.touch+n); 
/* don't draw true holes */
      if (j == HOLE)
        continue;
/* set colour according to atom touched */
/* TODO - user choice, diff colours/not plotted */
      if (j < 0)
        {
        if (sysenv.depth < 16)
          set_colour(gc1, YELLOW);  /* stupid SG's CURRENT - no longer nec? */
        else
          set_colour(gc1, PURPLE);                     /* re-entrant */
        }
/* NB - if assoc atom was deleted (& mem_shrink called) we can */
/* no longer get the touched atom's colour from data->atoms */
/* ie there should be no dependence on data->atoms in this drawing code */
      else
        set_rgb_colour(gc1, elements[*(data->csurf.code+n)].colour);
/* periodic surface */
      for (a=-data->image_limit[0] ; a<data->image_limit[1] ; a++)
        {
      for (b=-data->image_limit[2] ; b<data->image_limit[3] ; b++)
        {
      for (c=-data->image_limit[4] ; c<data->image_limit[5] ; c++)
        {
        if (!data->periodic)
          if (a || b || c)
            continue;
        px1 = *(data->csurf.px+n);
        py1 = *(data->csurf.py+n);
/* NEW - pbc translation */
        px1 += a*data->piv_pix[0];
        py1 += a*data->piv_pix[1];
        px1 += b*data->piv_pix[2];
        py1 += b*data->piv_pix[3];
        px1 += c*data->piv_pix[4];
        py1 += c*data->piv_pix[5];
/* translation - offset */
        px1 += data->offset[0];
        py1 += data->offset[1];
/* draw small line */
        SUBLINE(pixmap, gc1, m, px1, py1, px1, py1+1);
        }
        }
        }
      }
    }

/*******************************/
/* SPECIAL OBJECT - MORPHOLOGY */
/*******************************/
  if (data->num_vertices)
    {
/* draw facet labels */
    set_colour(gc1, WHITE);
    plist = data->planes;
    while (plist != NULL)
      {
      plane = (struct plane_pak *) plist->data;
      if (plane->present && plane->visible && data->morph_label)
        {
        g_snprintf(facet_label,7,"%2d%2d%2d",plane->index[0]
                                            ,plane->index[1]
                                            ,plane->index[2]);
        SUBTEXT(pixmap, gc1, m, 
               plane->px + data->offset[0],
               plane->py + data->offset[1],
               facet_label);
        }
      plist = g_slist_next(plist);
      }

/* FIXME - back-ridges can overwrite front-ridges causing small gaps
in front. Even better morphologies are drawn if first all the back faces
are drawn and then the rest. However, we'd have to loop twice, slowing
us down a lot */

/* draw all ridges */
    for (i=data->num_vertices ; i-- ; )
      {
      for (j=(data->vertices+i)->num_adj ; j-- ; )
        {
        k = *((data->vertices+i)->adj+j);
/* hidden line removal */
        if (!ridge_visible(data,i,k) && data->hpr)
          continue;
        px1 = (data->vertices+i)->px + data->offset[0];
        py1 = (data->vertices+i)->py + data->offset[1];
        px2 = (data->vertices+k)->px + data->offset[0];
        py2 = (data->vertices+k)->py + data->offset[1];

/* (sxm) alterations to line style (front - thick, back - dashed) */
        if (ridge_visible(data,i,k))
          gdk_gc_set_line_attributes(gc1, 3, GDK_LINE_SOLID,
                                    GDK_CAP_NOT_LAST, GDK_JOIN_MITER);
        else
          gdk_gc_set_line_attributes(gc1, 2, GDK_LINE_DOUBLE_DASH,
                                    GDK_CAP_NOT_LAST, GDK_JOIN_MITER);
        SUBLINE(pixmap, gc1, m, px1, py1, px2, py2);
        }
      }
    }

/* use thicker line style always? (maybe...) */
/* reset line style */
gdk_gc_set_line_attributes(gc1, 1, GDK_LINE_SOLID,
                           GDK_CAP_NOT_LAST, GDK_JOIN_MITER);

#if DEBUG_GDK_DRAW
printf("Done special objects\n");
#endif

/* write the current mode label - should this be removed? */
/* ie put everything in the tray to free up model display space */
if (!ghost)
  {
  switch(data->mode)
    {
    case FREE:
      mode_label = g_strdup("normal");
      break;
    case ATOM_ADD:
      mode_label = g_strdup("add atoms");
      break;
    case ATOM_MOVE:
      mode_label = g_strdup("move atoms");
      break;
    case ATOM_DELETE:
      mode_label = g_strdup("delete atoms");
      break;
    case ATOM_CHANGE_TYPE:
      mode_label = g_strdup("change atom type");
      break;
    case BOND_SINGLE:
      mode_label = g_strdup("single bonds");
      break;
    case BOND_DOUBLE:
      mode_label = g_strdup("double bonds");
      break;
    case BOND_TRIPLE:
      mode_label = g_strdup("triple bonds");
      break;
    case MOL_MOVE:
      mode_label = g_strdup("move mol");
      break;
    case MOL_DELETE:
      mode_label = g_strdup("delete mols");
      break;
    case BOND_INFO:
      mode_label = g_strdup("bonds");
      break;
    case DIST_INFO:
      mode_label = g_strdup("distances");
      break;
    case ANGLE_INFO:
      mode_label = g_strdup("angles");
      break;
    case DIHEDRAL_INFO:
      mode_label = g_strdup("dihedral angles");
      break;
    case ALTER_SELECT:
      mode_label = g_strdup("alter selection");
      break;
    case REGION_LITE:
      mode_label = g_strdup("region lighting");
      break;
    default:
      mode_label = g_strdup("undefined");
      break;
    }
  mlen = strlen(mode_label);
  lx = sysenv.subcenx[m]+0.5*sysenv.subwidth-7*mlen-10;
  ly = sysenv.subceny[m]+0.5*sysenv.subheight-15;
  set_colour(gc1, SEA_GREEN);
  gdk_draw_text(pixmap, sysenv.dfont, gc1, lx, ly, mode_label, mlen);
  }

/* write the basename (ie no path) */
  if (sysenv.show_names && !ghost)
    {
/* fresh models (ie MDI, creator) have no preceding path until saved */
    txt = g_strdup(data->basename);
    i = strlen(txt);
    lx = sysenv.subcenx[m]+0.5*sysenv.subwidth-7*i-10;
    ly = sysenv.subceny[m]-0.5*sysenv.subheight+20;
    gdk_draw_text(pixmap, sysenv.dfont, gc1, lx, ly, txt, i);
    }

/* write the energy */
  if (sysenv.show_energy && !ghost)
    {
/* Eatt & Esurf if 2D, otherwise the normal/lattice energy */
    if (data->periodic ==2)
      {
      txt = g_strdup_printf("Ea = %f %s",data->gulp.eatt,
                                         data->gulp.eatt_units);
      i = strlen(txt);
      lx = sysenv.subcenx[m]-0.5*sysenv.subwidth+15;
      ly = sysenv.subceny[m]+0.5*sysenv.subheight-15-15;
      gdk_draw_text(pixmap, sysenv.dfont, gc1, lx, ly, txt, i);
      txt = g_strdup_printf("Es = %f %s",data->gulp.esurf,
                                         data->gulp.esurf_units);
      }
    else
      txt = g_strdup_printf("E0 = %f eV",data->gulp.energy);
    i = strlen(txt);
    lx = sysenv.subcenx[m]-0.5*sysenv.subwidth+15;
    ly = sysenv.subceny[m]+0.5*sysenv.subheight-15;
    gdk_draw_text(pixmap, sysenv.dfont, gc1, lx, ly, txt, i);
    }

/* CURRENT - next ghost (if any) */
  if (ghost < data->num_ghosts)
    {
    model = *(data->ghosts+ghost);
    data = model_ptr(model, RECALL);
    ghost++;
    }
  else
    data = NULL;

  }  /* while ghost models (data != NULL) */

/* do the next subwin */
  m++;
  }

/* expose the drawing area */
expose_canvas();

gdk_gc_destroy (gc1);

#ifdef TIMER
stop_timer("gdk_draw");
#endif
}

