/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef __sun
#include <sys/dirent.h>
#else
#include <sys/dir.h>
#endif

#include "gdis.h"

#define DEBUG_MORE 0
#define MAX_KEYS 15

/* main structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/***************/
/* CIF parsing */
/***************/
#define DEBUG_LOAD_CIF 0
gint load_cif(FILE *fp, gint model, gchar *inp_file)
{
gint i, n, new;
gint *keyword;
gchar **buff, *tmp, line[LINELEN];
struct model_pak *data;

data = model_ptr(model, ASSIGN);
if (data == NULL)
  return(1);

keyword = get_keyword(line, MAX_KEYS);
/* atom counter */
n=0;
new = model-1;
for(;;)
  {
/* search for start of a data block */
  while (!*keyword)
    {
    if (fgetline(fp, line))
      goto cif_done;
    g_free(keyword);
    keyword = get_keyword(line, MAX_KEYS);
    }

/* clear for the next scan (iff nothing else subsequently read in) */
  *keyword=0;
  switch(*(keyword+1))
    {
/* model labels */
    case CIF_CHEMICAL_NAME:
      tmp = g_strdup(get_token_pos(line,1));
      for (i=0 ; i<strlen(tmp) ; i++)
        if (*(tmp+i) == '\'')
          *(tmp+i) = ' ';
      g_free(data->basename);
      data->basename = g_strdup(g_strstrip(tmp));
      g_free(tmp);
      break;
    case CIF_MINERAL_NAME:
      tmp = g_strdup(get_token_pos(line,1));
      for (i=0 ; i<strlen(tmp) ; i++)
        if (*(tmp+i) == '\'')
          *(tmp+i) = ' ';
      g_free(data->basename);
      data->basename = g_strdup(g_strstrip(tmp));
      g_free(tmp);
      break;
/* new model trigger */
    case CIF_DATA_START:
      new++;
/* 1st time exception (as we've already alloc'd a data pointer for safety) */
      if (new > model)
        {
#if DEBUG_LOAD_CIF
printf("Found %d atoms.\n",n);
#endif
        data->num_asym = data->num_orig = data->num_atoms = n;
        data->num_shells = 0;
/* alloc new pointer */
        data = model_ptr(new, ASSIGN);
        if (data == NULL)
          goto cif_done;
        }
/* init */
      data->id = CIF;
      data->mode = FREE;
      g_free(data->basename);
      data->basename = strdup_no_extension(g_basename(inp_file));
      strcpy(data->filename, data->basename);
/* atom counter */
      n=0;
#if DEBUG_LOAD_CIF
printf("Start of new model %d: %s.\n",new,data->basename);
#endif
      break;
/* model specific data - the *data ptr MUST be allocated */
    case CIF_CELL_A:
      buff = get_tokens(line, 3);
      data->pbc[0] = str_to_float(*(buff+1));
      g_strfreev(buff);
      break;
    case CIF_CELL_B:
      buff = get_tokens(line, 3);
      data->pbc[1] = str_to_float(*(buff+1));
      g_strfreev(buff);
      break;
    case CIF_CELL_C:
      buff = get_tokens(line, 3);
      data->pbc[2] = str_to_float(*(buff+1));
      g_strfreev(buff);
      break;
    case CIF_CELL_ALPHA:
      buff = get_tokens(line, 3);
      data->pbc[3] = PI*str_to_float(*(buff+1))/180.0;
      g_strfreev(buff);
      break;
    case CIF_CELL_BETA:
      buff = get_tokens(line, 3);
      data->pbc[4] = PI*str_to_float(*(buff+1))/180.0;
      break;
    case CIF_CELL_GAMMA:
      buff = get_tokens(line, 3);
      data->pbc[5] = PI*str_to_float(*(buff+1))/180.0;
      g_strfreev(buff);
      break;
    case CIF_SPACE_NAME:
/* remove the enclosing ' characters */
      tmp = g_strdup(get_token_pos(line,1));
      for (i=0 ; i<strlen(tmp) ; i++)
        if (*(tmp+i) == '\'')
          *(tmp+i) = ' ';
/* store the name, stripping spaces */
      g_free(data->sginfo.spacename);
      data->sginfo.spacename = g_strdup(g_strstrip(tmp));
/* indicate that name should used in lookup */
      data->sginfo.spacenum = -1;
#if DEBUG_LOAD_CIF
printf("[%s]\n",data->sginfo.spacename);
#endif
      g_free(tmp);
      break;
    case CIF_SPACE_NUM:
      buff = get_tokens(line, 3);
      g_strfreev(buff);
      break;
/* FIXME - how flexible is the cif format here? */
/* also, can we have cartesian here? (back transform if so) */
    case CIF_FRAC_DATA:
      data->fractional = TRUE;
      data->periodic = 3;
/* search for atom data until new keyword or EOF */
      *keyword=0;
      while (!*keyword)
        {
        buff = get_tokens(line, 10);
/* while item #1 (really #2) is an element */
        if (elem_type(*(buff+1)))
          {
          if (n >= data->atom_limit)
            mem_grow(data, ATOM, 10);
/* setup */
          (data->atoms+n)->status = NORMAL;
          (data->atoms+n)->primary = TRUE;
          (data->atoms+n)->orig = TRUE;
          (data->atoms+n)->region = REGION1A;
          (data->atoms+n)->offx = 0;
          (data->atoms+n)->offy = 0;
          sscanf(*(buff+1),"%s",(data->atoms+n)->element);
          sscanf(*buff,"%s",(data->atoms+n)->label);
          (data->atoms+n)->x = str_to_float(*(buff+4));
          (data->atoms+n)->y = str_to_float(*(buff+5));
          (data->atoms+n)->z = str_to_float(*(buff+6));
          (data->atoms+n)->sof = str_to_float(*(buff+7));

/* spurious SOF's - why do CIF's have these? */
if ((data->atoms+n)->sof < 0.0 || (data->atoms+n)->sof > 1.0)
  (data->atoms+n)->sof = 1.0;

/* enable colour by SOF option */
          if ((data->atoms+n)->sof < 1.0)
            data->has_sof = TRUE;
/* these are the primary coords (for the purpose of display) */
/* FIXME - do this at the end in case pbc's were not yet found */
          elem_seek(n, ATOM, data);
          n++;
          }
/* get next line */
        g_strfreev(buff);
        if (fgetline(fp, line))
          goto cif_done;
        g_free(keyword);
        keyword = get_keyword(line, MAX_KEYS);
        }
      break;
    } 
  }
cif_done:;
#if DEBUG_LOAD_CIF
printf("Found %d atoms.\n",n);
#endif
/* last model */
data->num_asym = data->num_orig = data->num_atoms = n;
data->num_shells = 0;

#if DEBUG_LOAD_CIF
printf("Found %d models.\n",new-model+1);
#endif

/* setup for display */
for (i=0 ; i<new-model+1 ; i++)
  {
  data = model_ptr(i, RECALL);
  data->axes_on = TRUE;
  if (data->periodic)
    {
    data->cell_on = TRUE;
    data->axes_type = OTHER;
/* space group init here */
    update_shells(data);
    if (!data->sginfo.spacenum)
      {
#if DEBUG_LOAD_CIF
printf("No Space Group found: assuming P1.\n");
#endif
      data->sginfo.spacenum=1;
      }
    if (genpos(data))
      printf("Error in Space Group lookup.\n");
    }
/* main coord setup */
  init_objs(INIT_COORDS, data);
  save_cart(data);
  calc_bonds(data);
  calc_mols(data);
  new_tree_item(data->number, APPEND);
  }

/* clean up & exit */
g_free(keyword);
return(0);
}

