/*
Copyright (C) 2001 by Brock Robert Reynolds 

brock@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include "rdf.h"

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

extern GtkWidget *window;

/* Global variables used for runtime access to gtk objects */
GtkWidget *scale_spin, *points_spin, *filename_text, *firstelement, *secondelement;
GtkWidget *atomselect1, *atomselect2, *atomselect3, *calcmethod1, *calcmethod2;
GtkWidget *countmethod1, *countmethod2, *outputtype1, *outputtype2, *smoothpoints;
gint firstelement_index[ELEMENT_MAX], secondelement_index[ELEMENT_MAX];






/* Interface functions... */

void on_atomselect1_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
	gtk_widget_set_sensitive(firstelement, FALSE);
	gtk_widget_set_sensitive(secondelement, FALSE);
}

void on_atomselect2_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
	gtk_widget_set_sensitive(firstelement, TRUE);
	gtk_widget_set_sensitive(secondelement, TRUE);
}

void on_atomselect3_toggled (GtkToggleButton *togglebutton, gpointer user_data)
{
	gtk_widget_set_sensitive(firstelement, FALSE);
	gtk_widget_set_sensitive(secondelement, FALSE);
}

void create_rdfwindow(void)
{
  gint id, i, j;
  gboolean elements_added[ELEMENT_MAX];
  GtkWidget *vbox1, *vbox2, *vbox3, *vbox4, *vbox5,
            *vbox6, *vbox7, *vbox8, *vbox9, *vbox10;
  GtkWidget *hbox1, *hbox2, *hbox3;
  GtkWidget *frame1, *frame2, *frame3, *frame4,
            *frame5, *frame6, *frame7;	
  GtkWidget *label, *button, *firstelement_menu,
            *secondelement_menu;
  GSList *atomselect_group = NULL, *calcmethod_group = NULL,
         *countmethod_group = NULL, *outputtype_group = NULL;
  GtkAdjustment *adj;
  GtkStyle *style;
  struct dialog_pak *rdf_dlg;	
  struct model_pak *model;
		
  /* Scrub variable arrays */
  j = 0;
  for (i = 0; i < ELEMENT_MAX; i++)
    elements_added[i] = FALSE;
  for (i = 0; i < ELEMENT_MAX; i++)
    firstelement_index[i] = 0;
  for (i = 0; i < ELEMENT_MAX; i++)
    secondelement_index[i] = 0;
	
  /* Retrieve a suitable dialog */
  if((id = request_dialog(-1,RDF))<0)
    return;
  rdf_dlg = &sysenv.dialog[id];
	
  model = model_ptr(sysenv.active, RECALL);
  g_return_if_fail(model != NULL);

  /* Create window and link destroy function */	
  rdf_dlg->win = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(rdf_dlg->win), "Radial Distribution Function");
  gtk_signal_connect(GTK_OBJECT(rdf_dlg->win), "destroy",
                     GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
	
  /* Get window style */
  style = gtk_widget_get_style(rdf_dlg->win);
	
  /* Create and add vboxes and hboxes for alignment */
  hbox1 = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(hbox1), 2);
  gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(rdf_dlg->win)->vbox)), hbox1);

  vbox1 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox1), 2);
  gtk_container_add(GTK_CONTAINER(GTK_BOX(GTK_DIALOG(rdf_dlg->win)->vbox)), vbox1);

  vbox2 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox2), 2);
  gtk_box_pack_start(GTK_BOX(hbox1), vbox2, TRUE, TRUE, 0);
	
  vbox3 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox3), 2);
  gtk_box_pack_start(GTK_BOX(hbox1), vbox3, TRUE, TRUE, 0);

  frame1 = gtk_frame_new("Output Filename");
  gtk_box_pack_start (GTK_BOX(vbox1), frame1, TRUE, TRUE, 0);
	
  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(hbox2), 2);
  gtk_box_pack_start(GTK_BOX(vbox1), hbox2, TRUE, TRUE, 0);
	
  frame2 = gtk_frame_new("Atom Selection");
  gtk_box_pack_start(GTK_BOX(vbox2), frame2, TRUE, TRUE, 0);
	
  vbox4 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox4), 2);
  gtk_container_add(GTK_CONTAINER(frame2), vbox4);	

  frame3 = gtk_frame_new("Atoms");
  gtk_box_pack_start(GTK_BOX(vbox3), frame3, TRUE, TRUE, 0);

  vbox5 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox5), 2);
  gtk_container_add(GTK_CONTAINER(frame3), vbox5);

  frame4 = gtk_frame_new("Calculation Method");
  gtk_box_pack_start(GTK_BOX(vbox2), frame4, TRUE, TRUE, 0);	

  vbox6 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox6), 2);
  gtk_container_add(GTK_CONTAINER(frame4), vbox6);

  frame6 = gtk_frame_new("Counting Method");
  gtk_box_pack_start(GTK_BOX(vbox2), frame6, TRUE, TRUE, 0);

  vbox9 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox9), 2);
  gtk_container_add(GTK_CONTAINER(frame6), vbox9);
	
  frame5 = gtk_frame_new("Data Characteristics");
  gtk_box_pack_start(GTK_BOX(vbox3), frame5, TRUE, TRUE, 0);
	
  hbox3 = gtk_hbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(hbox3), 2);
  gtk_container_add(GTK_CONTAINER(frame5), hbox3);
	
  vbox7 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox7), 2);
  gtk_box_pack_start(GTK_BOX(hbox3), vbox7, TRUE, TRUE, 0);
	
  vbox8 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox8), 2);
  gtk_box_pack_start(GTK_BOX(hbox3), vbox8, TRUE, TRUE, 0);	
	
  frame7 = gtk_frame_new("Output Type");
  gtk_box_pack_start(GTK_BOX(vbox3), frame7, TRUE, TRUE, 0);
	
  vbox10 = gtk_vbox_new(FALSE, 0);
  gtk_container_border_width(GTK_CONTAINER(vbox10), 2);
  gtk_container_add(GTK_CONTAINER(frame7), vbox10);

  /* Add elements to frames and vboxes */
	
  /* Add filename box */
  filename_text = gtk_entry_new();
  gtk_container_add(GTK_CONTAINER(frame1), filename_text);
  gtk_entry_set_text(GTK_ENTRY(filename_text),
                     g_strconcat(model->basename, "_rdf", NULL));
  gtk_editable_set_editable(GTK_EDITABLE(filename_text), TRUE);
	
  /* Add atom selection radio buttons (3) */
  atomselect1 = gtk_radio_button_new_with_label(atomselect_group, "All Atoms");
  atomselect_group = gtk_radio_button_group (GTK_RADIO_BUTTON (atomselect1));
  gtk_box_pack_start(GTK_BOX(vbox4), atomselect1, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(atomselect1), "toggled",
                     GTK_SIGNAL_FUNC(on_atomselect1_toggled), NULL);
	
  atomselect2 = gtk_radio_button_new_with_label(atomselect_group, "Two Elements");
  atomselect_group = gtk_radio_button_group (GTK_RADIO_BUTTON (atomselect2));
  gtk_box_pack_start(GTK_BOX(vbox4), atomselect2, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(atomselect2), "toggled",
                     GTK_SIGNAL_FUNC(on_atomselect2_toggled), NULL);
	
  atomselect3 = gtk_radio_button_new_with_label(atomselect_group, "Selected Atoms");
  atomselect_group = gtk_radio_button_group (GTK_RADIO_BUTTON (atomselect3));
  gtk_box_pack_start(GTK_BOX(vbox4), atomselect3, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(atomselect3), "toggled",
                     GTK_SIGNAL_FUNC(on_atomselect3_toggled), NULL);
	
  /* Add calculation method radio buttons (2) */
  calcmethod1 = gtk_radio_button_new_with_label(calcmethod_group, "Radial Distribution Function");
  calcmethod_group = gtk_radio_button_group(GTK_RADIO_BUTTON(calcmethod1));
  gtk_box_pack_start(GTK_BOX(vbox6), calcmethod1, TRUE, FALSE, 0);
	
  calcmethod2 = gtk_radio_button_new_with_label(calcmethod_group, "Pair Counting");
  calcmethod_group = gtk_radio_button_group(GTK_RADIO_BUTTON(calcmethod2));
  gtk_box_pack_start(GTK_BOX(vbox6), calcmethod2, TRUE, FALSE, 0);

  /* Add counting method radio buttons (2) */
  countmethod1 = gtk_radio_button_new_with_label(countmethod_group, "Count All Atoms");
  countmethod_group = gtk_radio_button_group(GTK_RADIO_BUTTON(countmethod1));
  gtk_box_pack_start(GTK_BOX(vbox9), countmethod1, TRUE, FALSE, 0);
	
  countmethod2 = gtk_radio_button_new_with_label(countmethod_group, "Count Closest Atoms");
  countmethod_group = gtk_radio_button_group(GTK_RADIO_BUTTON(countmethod2));
  gtk_box_pack_start(GTK_BOX(vbox9), countmethod2, TRUE, FALSE, 0);

  /* Add output type radio buttons (2) */
  outputtype1 = gtk_radio_button_new_with_label(outputtype_group, "Grace Format");
  outputtype_group = gtk_radio_button_group(GTK_RADIO_BUTTON(outputtype1));
  gtk_box_pack_start(GTK_BOX(vbox10), outputtype1, TRUE, FALSE, 0);
	
  outputtype2 = gtk_radio_button_new_with_label(outputtype_group, "Space Delimited ASCII");
  outputtype_group = gtk_radio_button_group(GTK_RADIO_BUTTON(outputtype2));
  gtk_box_pack_start(GTK_BOX(vbox10), outputtype2, TRUE, FALSE, 0);
	
  smoothpoints = gtk_check_button_new_with_label("Smooth Points");
  gtk_box_pack_start(GTK_BOX(vbox10), smoothpoints, TRUE, FALSE, 0);

  /* Add data characterisation spin boxes and labels */
  adj = (GtkAdjustment*) gtk_adjustment_new(0.2, 0.05, 2., 0.05, 0.05, 0.);
  scale_spin = gtk_spin_button_new(adj, 0., 2);
  gtk_box_pack_start (GTK_BOX(vbox7), scale_spin, TRUE, FALSE, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(scale_spin), FALSE);
	
  label = gtk_label_new("Data Step Size");
  gtk_box_pack_start(GTK_BOX(vbox8), label, TRUE, TRUE, 0);

  adj = (GtkAdjustment*) gtk_adjustment_new(100, MIN_RDF_POINTS, MAX_RDF_POINTS, 1, 10, 0.);
  points_spin = gtk_spin_button_new(adj, 0., 0);
  gtk_box_pack_start (GTK_BOX(vbox7), points_spin, TRUE, FALSE, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(points_spin), FALSE);
	
  label = gtk_label_new("Number of Points");
  gtk_box_pack_start(GTK_BOX(vbox8), label, TRUE, TRUE, 0);
	
  /* Add command buttons */
  button = gtk_button_new_with_label("Run");
  gtk_widget_set_style(button, style);
  gtk_box_pack_start(GTK_BOX(hbox2), button, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(init_rdf_task), (gpointer) id);

  button = gtk_button_new_with_label ("Close");
  gtk_widget_set_style (button, style);
  gtk_box_pack_start (GTK_BOX(hbox2), button, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT(button), "clicked",
                      GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
	
  /* Add element selection option menus */
  firstelement = gtk_option_menu_new();
  secondelement = gtk_option_menu_new();
  firstelement_menu = gtk_menu_new();
  secondelement_menu = gtk_menu_new();
  for (i = 0; i < model->num_atoms; i++)
  {
    elements_added[(model->atoms+i)->atom_code] = TRUE;
  } 
  for(i = 0; i < ELEMENT_MAX; i++)
  {
    if (elements_added[i])
    {
      gtk_menu_append(GTK_MENU(firstelement_menu),
                      gtk_menu_item_new_with_label(elements[i].name));
      gtk_menu_append(GTK_MENU(secondelement_menu),
                      gtk_menu_item_new_with_label(elements[i].name));
      firstelement_index[j] = i;
      secondelement_index[j] = i;
      j++;
    }
  }	
  gtk_option_menu_set_menu(GTK_OPTION_MENU(firstelement), firstelement_menu);	
  gtk_option_menu_set_menu(GTK_OPTION_MENU(secondelement), secondelement_menu);	
  gtk_box_pack_start(GTK_BOX(vbox5), firstelement, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(vbox5), secondelement, TRUE, TRUE, 0);
	
  gtk_widget_set_sensitive(firstelement, FALSE);
  gtk_widget_set_sensitive(secondelement, FALSE);
	
  gtk_widget_show_all(rdf_dlg->win);
}






/* Queing system interface function... */

void init_rdf_task(void)
{
  struct task_pak task;

  task.label = g_strdup("rdf");
  task.primary = &exec_rdf_task;
  task.ptr1 = (gpointer *) NULL;
  task.cleanup = NULL;
  task.ptr2 = (gpointer *) NULL;

  start_task(&task);
}






/* Main process loop... */






void exec_rdf_task(gpointer *ptr)
{
  /* Taken from dialog box */
  gint count_method, calc_method, atom_select, output_type;
  gboolean smooth_points;
  
  gint i, flag, distances[MAX_RDF_POINTS], hist_points, active_index;
  gdouble scale, density, nid_const, nid_rlower, nid_rupper, nid, nreal, volume;
  gdouble plotdata[MAX_RDF_POINTS];
  GString *filename, *title, *yaxis;
  GtkWidget *menu, *active_item;
  pid_t pid;
  gchar *argv[4];
  FILE *fp;
	
  /* Initialise rdf_pak object / structure */
  struct model_pak *model;
  struct rdf_pak *rdf;

#ifdef DEBUG_RDF_MEMORY
  printf("exec_rdf_task(): Initialising model_pak structure...\n");
#endif
  model = model_ptr(sysenv.active, RECALL);
  g_return_if_fail(model != NULL);
#ifdef DEBUG_RDF_MEMORY
  printf("exec_rdf_task(): Allocated %i bytes for model from 0x%x to 0x%x...\n",
          sizeof (struct model_pak), (gint) model,
	  (gint) model + sizeof (struct model_pak));
#endif

  rdf = rdf_init(model->num_frames);

  /* Retrieve values from dialog box */
	
  /* Get data characteristics from spin boxes */
  hist_points = (gint) gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(points_spin));
  scale = (gfloat) gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(scale_spin));

  /* Get filename from text box */
  filename = g_string_new(gtk_editable_get_chars(GTK_EDITABLE(filename_text), 0, -1));

  /* Get selection method from toggle button */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(atomselect1)))
    atom_select = RDF_ALL_ATOMS;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(atomselect2)))
    atom_select = RDF_TWO_ELEMENTS;
  else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(atomselect3)))
    atom_select = RDF_SELECTED_ATOMS;
	
  /* Get calculation method from toggle button */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(calcmethod1)))
    calc_method = CALCULATE_RDF;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(calcmethod2)))
    calc_method = CALCULATE_PAIRS;
	
  /* Get counting method from toggle button */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(countmethod1)))
    count_method = COUNT_ALL_SECONDARY;
  else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(countmethod2)))
    count_method = COUNT_CLOSEST_SECONDARY;
	
  /* Get output type from toggle button */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(outputtype1)))
    output_type = OUTPUT_GRACE;
  else if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(outputtype2)))
    output_type = OUTPUT_ASCII;
	
  /* Get point smoothing status from check button */
  smooth_points = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(smoothpoints));
	
  /* Get atom id of target elements from menu lists */
  switch (atom_select)
  {
    case  RDF_TWO_ELEMENTS:
    menu = GTK_OPTION_MENU (firstelement)->menu;
    active_item = gtk_menu_get_active(GTK_MENU(menu));
    active_index = g_list_index (GTK_MENU_SHELL (menu)->children, active_item);
    rdf->atom_element[0] = firstelement_index[active_index];
    menu = GTK_OPTION_MENU (secondelement)->menu;
    active_item = gtk_menu_get_active(GTK_MENU(menu));
    active_index = g_list_index (GTK_MENU_SHELL (menu)->children, active_item);
    rdf->atom_element[1] = secondelement_index[active_index];
    break;
  }
  
  for (i=0; i < 2; i++)
  {
    rdf->frame_atoms[i] = rdf_frame_atoms(model, atom_select, rdf->atom_element[i]);
  }
 
  /* Scrub some variables and arrays*/	
  flag = 0;
	
  for (i=0; i < hist_points; i++)
  {
    plotdata[i] = 0.;
    distances[i] = 0;
  }
	
#ifdef DEBUG_RDF
  printf("calculate_rdf(): a = %f b = %f c = %f alpha = %f beta = %f gamma = %f\n",
            model->pbc[0], model->pbc[1], model->pbc[2],
            model->pbc[3], model->pbc[4], model->pbc[5]);
  printf("calculate_rdf(): latmat[0] = %f latmat[1] = %f latmat[2] = %f
          \ncalculate_rdf(): latmat[3] = %f latmat[4] = %f latmat[5] = %f
          \ncalculate_rdf(): latmat[6] = %f latmat[7] = %f latmat[8] = %f\n",
            model->latmat[0], model->latmat[1], model->latmat[2],
            model->latmat[3], model->latmat[4], model->latmat[5],
            model->latmat[6], model->latmat[7], model->latmat[8]);
#endif
	
  /* Cycle through, collecting atoms from each frame */
  flag = get_rdf_data(rdf, model, atom_select);

  if (flag != 1)
  {
    show_text("get_rdf_data(): Error in obtaining RDF data from file.");
    return;
  }
	
  /* Calculate distances between each atom pair, and assign */
  /* each distance to the array for the graph.              */
  switch (atom_select)
  {
    case RDF_TWO_ELEMENTS:
    distances_elements(rdf, model, distances, hist_points, scale, count_method);
    break;

    case RDF_ALL_ATOMS:
    case RDF_SELECTED_ATOMS:
    distances_atoms(rdf, model, distances, hist_points, scale, count_method);
    break;
  }
	
  /* Calculation of g(r) - radial dist. ftn */
  /* g(r) = n(b) / nid(b)                   */
	
  /* Cell volume for parallelopiped */
  /* V=abc(1 - cos^2 alpha - cos^2 beta - cos^2 gamma)  */
  /*        + 2(cos alpha * cos beta * cos gamma)^0.5   */
  volume = model->pbc[0] * model->pbc[1] * model->pbc[2]
	     * (1 - pow(cos(model->pbc[3]), 2)
             - pow(cos(model->pbc[4]), 2)
             - pow(cos(model->pbc[5]), 2))
             + 2 * pow(pow(cos(model->pbc[3])
             * cos(model->pbc[4])
             * cos(model->pbc[5]), 2), 0.25); 

#ifdef DEBUG_RDF	
  printf("rdf_calculate(): Volume = %f\n", volume);
#endif

  /* Calculate density for rdf probability... */
  density = 0.;
	
  switch (calc_method)
  {
    case CALCULATE_RDF:			
    switch (atom_select)
    {
      case RDF_ALL_ATOMS:
      density = (gdouble) (model->num_atoms) / volume;
      break;

      case RDF_SELECTED_ATOMS:
      density = (gdouble) ((rdf->frame+0)->atom_count[0]) / volume;
      break;

      case RDF_TWO_ELEMENTS:
      density = (gdouble) ((rdf->frame+0)->atom_count[1]) / volume;
    }
    nid_const = 4.0 * PI * density / 3.0;
    for (i=0; i < hist_points; i++)
    {
      nid_rlower = ((gdouble) i * scale) * pow(1., -10);
      nid_rupper = nid_rlower + scale * pow(1., -10);
      nid = nid_const * (pow(nid_rupper, 3) - pow(nid_rlower, 3));
      nreal = (gdouble) distances[i] /  (gdouble) rdf->atom_total[0];
      plotdata[i] = nreal / nid;
    }
    break;
    
    case CALCULATE_PAIRS:
    for (i=0; i < hist_points; i++)
    {
      plotdata[i] = distances[i];
    }
    break;
  }		

#ifdef DEBUG_RDF	
  printf("rdf_calculate(): Density = %f\n", density);
#endif

  if (smooth_points)
    smooth_data(plotdata, hist_points);
	
  /* Open file for output - overwrite existing file */
  fp = fopen(filename->str, "w");
	
  /* Print header for Grace plot file */
  if (output_type == OUTPUT_GRACE)
  {
    switch (calc_method)
    {
      case CALCULATE_RDF:
      title = g_string_new("Radial Distribution Function");
      yaxis = g_string_new("g(r)");
      break;

      case CALCULATE_PAIRS:
      title = g_string_new("Pair Counting");
      yaxis = g_string_new("Pairs");
      break;
    }
    
    switch (atom_select)
    {
      case RDF_TWO_ELEMENTS:
      fprintf(fp, "@ title %c%s - %s %s%c\n", 34,
                elements[rdf->atom_element[0]].name,
                elements[rdf->atom_element[1]].name, title->str, 34);
      break;

      case RDF_ALL_ATOMS:
      fprintf(fp, "@ title %cAll Atoms - %s%c\n", 34, title->str, 34);
      break;

      case RDF_SELECTED_ATOMS:
      fprintf(fp, "@ title %cSelected Atoms - %s%c\n", 34, title->str, 34);
      break;
    }
    
    fprintf(fp, "@ xaxis label %cDistance (A)%c\n", 34, 34);
    fprintf(fp, "@ yaxis label %c%s%c\n", 34, yaxis->str, 34);
  }
	
  /* Print data to file */
  for (i=0; i < hist_points; i++)
  {
    fprintf(fp, "%0.2f %f\n", ((gdouble)i) * scale + scale, plotdata[i]);
  }
	
  /* Closing data file */
  fclose(fp);
	
  /* Run grace if grace output is selected */
  if (output_type == OUTPUT_GRACE)
  {
    g_string_free(title, TRUE);
    g_string_free(yaxis, TRUE);
    pid = fork();
    switch (pid)
    {
      case -1:
#ifdef DEBUG_RDF
  printf("rdf_calculate(): Fork failed, unable to open grace...\n");
#endif
	show_text("Fork failed, unable to open grace...");	
      break;

      case 0:
      argv[0] = g_strdup("sh");
      argv[1] = g_strdup("-c");
      argv[2] = g_strconcat("xmgrace ", filename->str, NULL);
      argv[3] = NULL;
      execvp("sh", argv);
      break;
    }		
  }
  g_string_free(filename, TRUE);
  rdf_cleanup(rdf);
  return;
}






/* Memory allocation / deallocation / scrubbing functions... */

struct rdf_pak *rdf_init(gint num_frames)
{
  gint i;
  struct rdf_pak *rdf;
  
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_init(): Initializing memory for rdf_pak...\n");
#endif
  rdf = g_malloc(sizeof (struct rdf_pak));
  if (!rdf)
  {
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_init(): Failed to initialize memory for rdf_pak...\n");
#endif
    return 0;
  }
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_init(): Allocated %i bytes for rdf_pak from 0x%x to 0x%x...\n",
          sizeof (struct rdf_pak), (gint) rdf, (gint) rdf + sizeof (struct rdf_pak));
#endif
  
  for (i=0; i < 2; i++)
  {
    rdf->atom_element[i] = 0;
    rdf->atom_total[i] = 0;
    rdf->frame_atoms[i] = 0;
  }
  rdf->frames_count = 0;

#ifdef DEBUG_RDF_MEMORY
  printf("rdf_init(): Initialising memory for frames...\n");
#endif
  rdf->frame = g_malloc(num_frames * sizeof (struct rdf_frame_pak));
  if (!rdf->frame)
  {
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_init(): Failed to initialize memory for frames...\n");
#endif
    return 0;
  }
#ifdef DEBUG_RDF_MEMORY
  printf("rdf-Init(): Allocated %i bytes for frames from 0x%x to 0x%x...\n",
          num_frames * sizeof (struct rdf_frame_pak), (gint) rdf->frame,
	  (gint) rdf->frame + num_frames * sizeof (struct rdf_frame_pak));
#endif

  return rdf;
}

gint rdf_cleanup(struct rdf_pak *rdf)
{
  gint i;
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_cleanup(): Cleaning up allocated memory...\n");
#endif
  for (i=0; i < rdf->frames_count; i++)
  {
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_cleanup(): Cleaning up frame %i first atom coordinates from 0x%x...\n",
          i, (gint) (rdf->frame+i)->firstatom);
#endif
    g_free ((rdf->frame+i)->firstatom);
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_cleanup(): Cleaning up frame %i second atom coordinates from 0x%x...\n",
          i, (gint) (rdf->frame+i)->secondatom);
#endif
    g_free ((rdf->frame+i)->secondatom);
  }
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_cleanup(): Cleaning up frame data from 0x%x...\n", (gint) rdf->frame);
#endif
  g_free (rdf->frame);
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_cleanup(): Cleaning up rdf data from 0x%x...\n", (gint) rdf);
#endif
  g_free (rdf);
  
  return 1;
}

gint rdf_add_frame(struct rdf_pak *rdf, gint frames_total)
{
  (rdf->frame+(rdf->frames_count))->atom_count[0] = 0;
  (rdf->frame+(rdf->frames_count))->atom_count[1] = 0;

#ifdef DEBUG_RDF_MEMORY
  printf("rdf_add_frame(): Initializing memory for first atom coordinates for frame %i...\n",
          rdf->frames_count);
#endif
  (rdf->frame+(rdf->frames_count))->firstatom
    = g_malloc(rdf->frame_atoms[0] * sizeof (struct rdf_coords_pak));
  if (!(rdf->frame+(rdf->frames_count))->firstatom)
  {
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_add_frame(): Failure to initialise memory for first atom coordinates...\n");
#endif
    return 0;
  }
  
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_add_frame(): Allocated %i bytes for first atom coordinates from 0x%x to 0x%x...\n",
          rdf->frame_atoms[0] * sizeof (struct rdf_coords_pak),
	  (gint) (rdf->frame+(rdf->frames_count))->firstatom,
	  (gint) (rdf->frame+(rdf->frames_count))->firstatom +
	   rdf->frame_atoms[0] * sizeof (struct rdf_coords_pak));
  printf("rdf_add_frame(): Initializing memory for second atom coordinates for frame %i...\n",
          rdf->frames_count);
#endif

  (rdf->frame+(rdf->frames_count))->secondatom
    = g_malloc(rdf->frame_atoms[1] * sizeof (struct rdf_coords_pak));
  if (!(rdf->frame+(rdf->frames_count))->secondatom)
  {
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_add_frame(): Failure to initialize memory for second atom coordinates...\n");
#endif
    return 0;
  }
  
#ifdef DEBUG_RDF_MEMORY
  printf("rdf_add_frame(): Allocated %i bytes for second atom coordinates from 0x%x to 0x%x...\n",
          rdf->frame_atoms[1] * sizeof (struct rdf_coords_pak),
	  (gint) (rdf->frame+(rdf->frames_count))->secondatom,
	  (gint) (rdf->frame+(rdf->frames_count))->secondatom +
	   rdf->frame_atoms[1] * sizeof (struct rdf_coords_pak));
#endif

  rdf->frames_count ++;

  return 1;
}

gint rdf_add_coords(struct rdf_frame_pak *frame, gint atom)
{
  switch (atom)
  {
    case 1:
    frame->atom_count[0] ++;
    (frame->firstatom+(frame->atom_count[0] - 1))->x = 0.;
    (frame->firstatom+(frame->atom_count[0] - 1))->y = 0.;
    (frame->firstatom+(frame->atom_count[0] - 1))->z = 0.;
    break;
  
    case 2:    
    frame->atom_count[1] ++;
    (frame->secondatom+(frame->atom_count[1] - 1))->x = 0.;
    (frame->secondatom+(frame->atom_count[1] - 1))->y = 0.;
    (frame->secondatom+(frame->atom_count[1] - 1))->z = 0.;
    break;
  }

  return 1;
}






/* Data sorting / retrieval functions... */

gint rdf_frame_atoms(struct model_pak *model, gint atom_select, gint atom_element)
{
  gint i, frame_atoms;

  frame_atoms = 0;

  switch (atom_select)
  {
    case RDF_ALL_ATOMS:
    frame_atoms = model->num_atoms; 
    break;

    case RDF_SELECTED_ATOMS:
    frame_atoms = model->select_size;
    break;

    case RDF_TWO_ELEMENTS:
    for (i=0; i < model->num_atoms; i++)
      if ((model->atoms+i)->atom_code == atom_element)
        frame_atoms ++;
    break;
  }

  return frame_atoms;
}

gint get_rdf_data(struct rdf_pak *rdf, struct model_pak *model, gint atom_select)
{
  gint flag, i, frame_num;
  gdouble latmatx, latmaty, latmatz;
 
  flag = 0;
  frame_num = 0;

#ifdef DEBUG_RDF_MEMORY
  printf("get_rdf_data(): Getting frame %i...\n", frame_num);
#endif
  flag = get_frame(model, TRUE);
#ifdef DEBUG_RDF_MEMORY
  printf("get_rdf_data(): Successfully got frame...\n");
#endif

  while (!flag)
  {
    if (model->periodic)
    {
      model->cell_on = TRUE;
      pbc_constrain_mols(model);
      init_objs(REDO_COORDS, model);
      calc_bonds(model);
      calc_mols(model);
    }
    update_geom_info();

    /* Insert per-frame calculations here */

    latmatx = model->latmat[0] + model->latmat[3]
                + model->latmat[6];
    latmaty = model->latmat[1] + model->latmat[4]
                + model->latmat[7];
    latmatz = model->latmat[2] + model->latmat[5]
                + model->latmat[8];

    rdf_add_frame(rdf, model->num_frames);

    switch (atom_select)
    {
      case RDF_ALL_ATOMS:
      for (i=0; i < model->num_atoms; i++)
      {
	rdf_add_coords(rdf->frame+frame_num, 1);

	((rdf->frame+frame_num)->firstatom+i)->x
	  = (model->atoms+i)->x * latmatx;
	((rdf->frame+frame_num)->firstatom+i)->y
 	  = (model->atoms+i)->y * latmaty;
	((rdf->frame+frame_num)->firstatom+i)->z
	  = (model->atoms+i)->z * latmatz;

	rdf->atom_total[0] ++;
      }
      break;
      
      case RDF_SELECTED_ATOMS:
      for (i=0; i < model->select_size; i++)
      {
	rdf_add_coords(rdf->frame+frame_num, 1);

	((rdf->frame+frame_num)->firstatom+i)->x
	  = (model->atoms+(model->select_list[i]))->x * latmatx;
	((rdf->frame+frame_num)->firstatom+i)->y
	  = (model->atoms+(model->select_list[i]))->y * latmaty;
	((rdf->frame+frame_num)->firstatom+i)->z
	  = (model->atoms+(model->select_list[i]))->z * latmatz;

	rdf->atom_total[0] ++;
      }
      break;

      case RDF_TWO_ELEMENTS:	
      
      for (i=0; i < model->num_atoms; i++)
      {

        if ((model->atoms+i)->atom_code == rdf->atom_element[0])
	{
	  rdf_add_coords(rdf->frame+frame_num, 1);	  
	
	  ((rdf->frame+frame_num)->firstatom+((rdf->frame+frame_num)->atom_count[0]-1))->x
	    = (model->atoms+i)->x * latmatx;
	  ((rdf->frame+frame_num)->firstatom+((rdf->frame+frame_num)->atom_count[0]-1))->y
	    = (model->atoms+i)->y * latmaty;
	  ((rdf->frame+frame_num)->firstatom+((rdf->frame+frame_num)->atom_count[0]-1))->z
	    = (model->atoms+i)->z * latmatz;

	  rdf->atom_total[0] ++;
	}
	
	if ((model->atoms+i)->atom_code == rdf->atom_element[1]
              && (model->atoms+i)->atom_code != rdf->atom_element[0])
	{
	  rdf_add_coords(rdf->frame+frame_num, 2);

	  ((rdf->frame+frame_num)->secondatom+((rdf->frame+frame_num)->atom_count[1]-1))->x
	    = (model->atoms+i)->x * latmatx;
	  ((rdf->frame+frame_num)->secondatom+((rdf->frame+frame_num)->atom_count[1]-1))->y
	    = (model->atoms+i)->y * latmaty;
	  ((rdf->frame+frame_num)->secondatom+((rdf->frame+frame_num)->atom_count[1]-1))->z
	    = (model->atoms+i)->z * latmatz;

	  rdf->atom_total[1] ++;
	}
      }
      break;
    }

    frame_num ++;
    
    if (frame_num == model->num_frames)
      break;
#ifdef DEBUG_RDF
  printf("get_rdf_data(): Getting frame %i...\n", frame_num);
#endif
    flag = get_frame(model, FALSE);
#ifdef DEBUG_RDF
  printf("get_rdf_data(): Successfully got frame...\n");
#endif
  }

  switch (atom_select)
  {
    case RDF_ALL_ATOMS:
    case RDF_SELECTED_ATOMS:
    rdf->atom_total[1] = rdf->atom_total[0];
    (rdf->frame+0)->atom_count[1] = (rdf->frame+0)->atom_count[0];
    break;

    case RDF_TWO_ELEMENTS:
    if (rdf->atom_element[0] == rdf->atom_element[1])
    {
      rdf->atom_total[1] = rdf->atom_total[0];
      (rdf->frame+0)->atom_count[1] = (rdf->frame+0)->atom_count[0];
    }
    break;
  }
  return 1;
}






/* Calculation functions... */

gint distances_atoms(struct rdf_pak *rdf, struct model_pak *model, gint *distances,
		       gint hist_points, gdouble scale, gint count_method)
{
  gint i, j, k, ia, ib, ic, cube_size;
  gdouble distance, distance_new, xnew, ynew, znew;
  gdouble latmatx, latmaty, latmatz;

  latmatx = model->latmat[0] + model->latmat[3] + model->latmat[6];
  latmaty = model->latmat[1] + model->latmat[4] + model->latmat[7];
  latmatz = model->latmat[2] + model->latmat[5] + model->latmat[8];

  switch (count_method)
  {
    case COUNT_CLOSEST_SECONDARY:
    cube_size = 1;
    break;

    case COUNT_ALL_SECONDARY:
    cube_size = (gint) ((gdouble) hist_points * scale / model->pbc[0]) + 1;
    break;
  }

  for (k=0; k < rdf->frames_count; k++)
  {
    for (i=0; i < (rdf->frame+k)->atom_count[0]; i++)
    {
      for (j = i + 1; j < (rdf->frame+k)->atom_count[0]; j++)
      {
	distance = 10000;
	for (ia=-(cube_size); ia<=cube_size; ia++)
	{
	  for (ib=-(cube_size); ib<=cube_size;ib++)
	  {
	    for (ic=-(cube_size);ic<=cube_size;ic++)
	    {
	      xnew = ((rdf->frame+k)->firstatom+j)->x + ia * latmatx;
	      ynew = ((rdf->frame+k)->firstatom+j)->y + ib * latmaty;
	      znew = ((rdf->frame+k)->firstatom+j)->z + ic * latmatz;
	      distance_new = (gdouble) sqrt(
			       pow(((rdf->frame+k)->firstatom+i)->x - xnew, 2)
			       + pow(((rdf->frame+k)->firstatom+i)->y - ynew, 2)
			       + pow(((rdf->frame+k)->firstatom+i)->z - znew, 2));	      

	      switch (count_method)
	      {
		case COUNT_CLOSEST_SECONDARY:
		if (distance_new < distance)
		  distance = distance_new;
		break;
	
		case COUNT_ALL_SECONDARY:
		if ((distance_new / scale) < MAX_RDF_POINTS)
		{
		  distance = distance_new;
		  distances[(gint) floor(distance / scale)] += 2; 
		}
		break;
		
              }
	    }
	  }
	}
	if (count_method == COUNT_CLOSEST_SECONDARY &&
             ((distance / scale) < MAX_RDF_POINTS))
	  distances[(gint) floor(distance / scale)] += 2;
      }
    }
  }
  return 1;  
}

gint distances_elements(struct rdf_pak *rdf, struct model_pak *model, int *distances,
			  gint hist_points, gdouble scale, gint count_method)
{
  gint k, i, j, ia, ib, ic, cube_size;
  gdouble distance, xnew, ynew, znew, distance_new;
  gdouble latmatx, latmaty, latmatz, biggest_side;
  
  latmatx = model->latmat[0] + model->latmat[3] + model->latmat[6];
  latmaty = model->latmat[1] + model->latmat[4] + model->latmat[7];
  latmatz = model->latmat[2] + model->latmat[5] + model->latmat[8];
  biggest_side = model->pbc[0];
  for (i = 1; i <= 2; i++)
  {
    if (biggest_side < model->pbc[i])
      biggest_side = model->pbc[i];
  }  

  /* Calculate number of periodic images required to span data points */
  switch (count_method)
  {
    case COUNT_CLOSEST_SECONDARY:
      cube_size = 1;
      break;
    case COUNT_ALL_SECONDARY:
      cube_size = (gint) ((gdouble) hist_points * scale / biggest_side) + 1;
      break;
  }

#ifdef DEBUG_RDF
  printf("distances_elements(): Cube Size = %i\n", cube_size);
#endif

  /* For each frame in the model... */
  for (k=0; k < rdf->frames_count; k++)
  {
    /* For each primary atom in the frame... */
    for (i=0; i < (rdf->frame+k)->atom_count[0]; i++)
    {
      if (rdf->atom_element[0] != rdf->atom_element[1])
      {
        /* For each secondary atom in the frame... */
	for (j=0; j < (rdf->frame+k)->atom_count[1]; j++)
	{
	  distance = 10000.;
	  /* Loop through periodic images - x, y and z axes */
	  for (ia=-(cube_size); ia<=cube_size; ia++)
	  {
	    for (ib=-(cube_size); ib<=cube_size;ib++)
	    {
	      for (ic=-(cube_size);ic<=cube_size;ic++)
	      {
	        xnew = ((rdf->frame+k)->secondatom+j)->x + ia * latmatx;
		ynew = ((rdf->frame+k)->secondatom+j)->y + ib * latmaty;
		znew = ((rdf->frame+k)->secondatom+j)->z + ic * latmatz;
		distance_new = (gdouble) sqrt(
				 pow(((rdf->frame+k)->firstatom+i)->x - xnew, 2)
				 + pow(((rdf->frame+k)->firstatom+i)->y - ynew, 2)
				 + pow(((rdf->frame+k)->firstatom+i)->z - znew, 2));
	 	
		switch (count_method)
		{
		  case COUNT_CLOSEST_SECONDARY:
		  if (distance_new < distance)
		    distance = distance_new;
		  break;
		  
                  case COUNT_ALL_SECONDARY:
		  distance = distance_new;
		  if ((distance / scale) < MAX_RDF_POINTS)
		    distances[(gint) floor(distance / scale)] ++;
		  break;
		}		
              }
            }
          }
          if ((count_method == COUNT_CLOSEST_SECONDARY) && ((distance / scale) < MAX_RDF_POINTS))
	    distances[(gint) floor(distance / scale)] ++;
	}
      }
      else
      {
	for (j=0; j < (rdf->frame+k)->atom_count[0]; j++)
	{
	  if (i != j)
	  {
	    distance = 10000.;
	    for(ia=-(cube_size); ia<=cube_size;ia++)
	    {
	      for(ib=-(cube_size);ib<=cube_size;ib++)
	      {
		for(ic=-(cube_size);ic<=cube_size;ic++)
		{
		  xnew = ((rdf->frame+k)->firstatom+j)->x + ia * latmatx;
		  ynew = ((rdf->frame+k)->firstatom+j)->y + ib * latmaty;
		  znew = ((rdf->frame+k)->firstatom+j)->z + ic * latmatz;
		  distance_new = (gdouble) sqrt(
				   pow(((rdf->frame+k)->firstatom+i)->x - xnew, 2)
				   + pow(((rdf->frame+k)->firstatom+i)->y - ynew, 2)
				   + pow(((rdf->frame+k)->firstatom+i)->z - znew, 2));
		  
		  switch (count_method)
		  {
		    case COUNT_CLOSEST_SECONDARY:
		      if (distance_new > 0.001 && distance_new < distance)
			distance = distance_new;
		      break;
		    case COUNT_ALL_SECONDARY:
		      if (distance_new > 0.001 && (distance_new / scale) < MAX_RDF_POINTS)
		      {
			distance = distance_new;
			distances[(gint) floor(distance / scale)]++; 
		      }
		      break;
		  }
		}
	      }
	    }
	  if (count_method == COUNT_CLOSEST_SECONDARY && ((distance / scale) < MAX_RDF_POINTS))
	    distances[(gint) floor(distance / scale)] ++;
	  }
        }
      }
    }
  }
  return 1;
}

gint smooth_data(gdouble *plotdata, gint hist_points)
{
  gint i;
  gdouble original[MAX_RDF_POINTS];
	
  for (i = 0; i < hist_points; i++)
    original[i] = plotdata[i];
  
  if (hist_points > 4)
  {
    plotdata[0] = (69 * original[0] + 4 * original[1] - 6 * original[2]
                      + 4 * original[3] - original[4]) / 70;
    plotdata[1] = (2 * original[0] + 27 * original[1] + 12 * original[2]
                      - 8 * original[3] + 2 * original[4]) / 35;
    plotdata[hist_points - 1] = (2 * original[hist_points] + 27 * original[hist_points - 1]
                                 + 12 * original[hist_points - 2] - 8 * original[hist_points - 3]
                                 + 2 * original[hist_points - 4]) / 35;
    plotdata[hist_points] = (69 * original[hist_points] + 4 * original[hist_points - 1]
                             - 6 * original[hist_points - 2] + 4 * original[hist_points - 3]
                             - original[hist_points - 4]) / 70;
    for (i=2; i < hist_points - 2; i++)
    {
      plotdata[i] = (-3 * original[i - 2] + 12 * original[i - 1] + 17 * original[i]
                    + 12 * original[i + 1] - 3 * original[i + 2]) / 35;
    }
  }
  return 1;
}

