/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "interface.h"
#include "matrix.h"
#include "measure.h"
#include "spatial.h"
#include "surface.h"
#include "numeric.h"
#include "morph.h"
#include "opengl.h"
#include "select.h"
#include "zone.h"

/* data structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/**********************/
/* debugging routines */
/**********************/
void dump_links(GSList *list)
{
GSList *item;
struct core_pak *core;
struct shel_pak *shell;

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

  printf("core: %p, shell: %p ", core, core->shell);
  P3VEC(" : ", core->x);
  }
for (item=list ; item ; item=g_slist_next(item))
  {
  shell = (struct shel_pak *) item->data;

  printf("shell: %p, core: %p\n", shell, shell->core);
  P3VEC(" : ", shell->x);
  }
}
void dump_cores(GSList *list)
{
gint i=0;
GSList *item;
struct core_pak *core;
struct shel_pak *shell;

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/*
if (i==4)
*/
  {
  printf("[%4s] %1d %1d c %10.4f %10.4f %10.4f : %6.2f\n",
          core->label, core->primary, core->orig,
          core->x[0], core->x[1], core->x[2], core->charge);

  if (core->shell)
    {
    shell = (struct shel_pak *) core->shell;

    printf("[%4s] %1d %1d s %10.4f %10.4f %10.4f : %6.2f\n",
            shell->label, shell->primary, shell->orig,
            shell->x[0], shell->x[1], shell->x[2], shell->charge);
    }
  }
  i++;
  }
}

/****************/
/* SPECIAL OBJS */
/****************/
#define DEBUG_INIT 0
void init_objs(gint type, struct model_pak *data)
{
GSList *list;
struct core_pak *core;
struct shel_pak *shel;

g_return_if_fail(data != NULL);

switch(type)
  {
/* set all atoms back to element colour */
  case REFRESH_COLOUR:
    model_colour_scheme(data->colour_scheme, data);
    for (list=data->shels ; list ; list=g_slist_next(list))
      {
      shel = (struct shel_pak *) list->data;
      ARR3SET(shel->colour, elements[shel->atom_code].colour);
      }
    break;

/* scale all atom colours by their sof */
  case SOF_COLOUR:
    if (data->has_sof)
      for (list=data->cores ; list ; list=g_slist_next(list))
        {
        core = (struct core_pak *) list->data;
        VEC3MUL(core->colour, core->sof);
        }
    break;

  case REFRESH_BONDS:
    for (list=data->cores ; list ; list=g_slist_next(list))
      {
      core = (struct core_pak *) list->data;
      init_elem(core, type, data);
      }
    break; 

/* atomic coordinates processing */
  case INIT_COORDS:
/* find unique elements */
    g_slist_free(data->unique_atom_list);
    data->unique_atom_list = find_unique(ELEMENT, data);
/* init the charges */
    init_model_charges(data);
/* update electrostatic info */
    calc_emp(data);
/* repetition here, since there is interdependence */
/* between centroid (cent_coords) and latmat (calc_coords) calc */
    cent_coords(1, data);
    calc_coords(INITIAL, data);
  case CENT_COORDS:
    cent_coords(1, data);
  case REDO_COORDS:
    calc_coords(REFRESH, data);
    break;

  default:
    printf("Unknown object type requested!\n");
  }
}

/********/
/* AXES */
/********/
void make_axes(struct model_pak *data)
{
VEC3SET(data->axes[0].x, 1.0, 0.0, 0.0);
VEC3SET(data->axes[1].x, 0.0, 1.0, 0.0);
VEC3SET(data->axes[2].x, 0.0, 0.0, 1.0);
}

/********/
/* CELL */
/********/
void make_cell(struct model_pak *data)
{
gdouble b, c;

if (!data->periodic)
  return;

/* periodicity hacks */
b = (data->periodic < 2) ? 0.0 : 1.0;
c = (data->periodic < 3) ? 0.0 : 1.0;

/* end face 1 */
VEC3SET(data->cell[0].x, 0.0, 0.0, 0.0);
VEC3SET(data->cell[1].x, 0.0, b, 0.0);
VEC3SET(data->cell[2].x, 1.0, b, 0.0);
VEC3SET(data->cell[3].x, 1.0, 0.0, 0.0);

/* end face 2 */
VEC3SET(data->cell[4].x, 0.0, 0.0, c);
VEC3SET(data->cell[5].x, 0.0, b, c);
VEC3SET(data->cell[6].x, 1.0, b, c);
VEC3SET(data->cell[7].x, 1.0, 0.0, c);
}


/*********************************/
/* construct core-shell linkages */
/*********************************/
#define DEBUG_SHELLS 0
#define MAX_SHELL_DIST 0.6
void shell_make_links(struct model_pak *model)
{
gint i, match, total;
gchar *text;
gdouble min, sep, vec[3];
GSList *list1, *list2, *locality;
struct core_pak *core;
struct shel_pak *shel;

/* checks */
g_assert(model != NULL);
g_assert(model->zones != NULL);

/* enumerate all shells */
total = 0;
for (list1=model->shels ; list1 ; list1=g_slist_next(list1))
  {
  shel = (struct shel_pak *) list1->data;

/* get neighbourhood core list for the shell */
  i = zone_index(shel->x, model);
  locality = zone_area_cores(1, model->zones[i], model);

/* upper bound for core-shell link */
  min = MAX_SHELL_DIST*MAX_SHELL_DIST;

/* enumerate cores in the shell's vicinity */
  match = 0;
  for (list2=locality ; list2 ; list2=g_slist_next(list2))
    {
    core = (struct core_pak *) list2->data;

    if (shel->atom_code == core->atom_code)
      {
/* get the minimum fractional separation */
      ARR3SET(vec, core->x);
      ARR3SUB(vec, shel->x);
      fractional_minsq(vec, model->periodic);
/* convert to cartesian & compare with cutoff */
      vecmat(model->latmat, vec);
      sep = VEC3MAGSQ(vec);
      if (sep < min)
        {
/* FIXME - what to do with previous matches? eg set (core->shell)->shell = NULL ? */
/* reference each other */
        core->shell = shel;
        shel->core = core;
        match++;
/* keep track of the minimum, so only the closest pair is recorded */
        min = sep;
        }
      }
    }
/* record any shells with no linkages */
  if (!match)
    total++;
/* free the constructed locality */
  g_slist_free(locality);
  }

/* found some floating shells? */
if (total)
  {
  text = g_strdup_printf("Warning: found %d shell(s) with no cores.\n", total);
  show_text(WARNING, text);
  g_free(text);
  }
}

/**********/
/* SHELLS */
/**********/
void update_shells(struct model_pak *data)
{
gint tot_match, match;
gdouble min, sep, vec[3];
GSList *core_list, *shel_list;
struct core_pak *core;
struct shel_pak *shel;


/* CURRENT */
shell_make_links(data);
return;



/* checks */
g_assert(data != NULL);

/* find core-shell connections */
tot_match = 0;
for (core_list=data->cores ; core_list ; core_list=g_slist_next(core_list))
  {
  core = (struct core_pak *) core_list->data;
  core->shell = NULL;

/* the closest separation found */
  min = MAX_SHELL_DIST*MAX_SHELL_DIST;
  match = 0;
  for (shel_list=data->shels ; shel_list ; shel_list=g_slist_next(shel_list))
    {
    shel = (struct shel_pak *) shel_list->data;

    if (shel->atom_code == core->atom_code)
      {
/* get the absolute fractional separation */
      ARR3SET(vec, core->x);
      ARR3SUB(vec, shel->x);

/* NEW - get the minimum fractional separation */
      fractional_minsq(vec, data->periodic);

/* convert to cartesian & compare with cutoff */
      vecmat(data->latmat, vec);
      sep = VEC3MAGSQ(vec);
      if (sep < min)
        {
/* FIXME - what to do with previous matches? eg set (core->shell)->shell = NULL ? */
/* reference each other */
        core->shell = shel;
        shel->core = core;
        match++;
/* keep track of the minimum, so only the closest pair is recorded */
        min = sep;
        }
      }
    }
  if (match)
    tot_match++;
  }

#if DEBUG_SHELLS
printf("core-shell connections: %d/%d\n", tot_match, g_slist_length(data->shels));
#endif

/* do some shells float down here? */
if (tot_match != g_slist_length(data->shels))
  {
/* TODO - somehow only print this once for a model */
  show_text(WARNING, "Warning: missing core-shell connection(s).\n");
#if DEBUG_SHELLS
for (shel_list=data->shels ; shel_list ; shel_list=g_slist_next(shel_list))
  {
  shel = (struct shel_pak *) shel_list->data;
  if (!shel->core)
    {
/* NEW - also calculate cartesian values */
    ARR3SET(shel->rx, shel->x);
    vecmat(data->latmat, shel->rx);
    ARR3ADD(shel->rx, data->centroid);
    print_shell(shel); 
    }
  }
#endif
  }
}

/*********************************/
/* coordinate matching routine 1 */
/*********************************/
/* done via 3D world coordinate matching */
#define DEBUG_SEEK_COORD3D 0
struct core_pak *seek_coord3d(gdouble *x, struct model_pak *data)
{
gdouble d2, tol, vec[3], piv[3];
GSList *list=NULL, *ilist=NULL;
struct core_pak *core, *best=NULL;
struct image_pak *image;

/* tolerance (angstroms squared)  */
tol = data->rmax * 20.0*data->scale/sysenv.size;

#if DEBUG_SEEK_COORD3D
printf(" size = %d\n", sysenv.size);
printf("scale = %f\n", data->scale);
printf(" rmax = %f\n", data->rmax);
printf("  tol = %f\n", tol);
#endif

do
  {
  if (ilist)
    {
/* retrieve periodic image vector */
    image = (struct image_pak *) ilist->data;
    ARR3SET(piv, image->rx);
    ilist = g_slist_next(ilist);
    }
  else
    {
    VEC3SET(piv, 0.0, 0.0, 0.0);
    ilist = data->images;
    }

/* attempt to match core coords */
  for (list=data->cores ; list ; list=g_slist_next(list))
    {
    core = (struct core_pak *) list->data;

    ARR3SET(vec, core->rx);
    ARR3ADD(vec, piv);
    VEC3MUL(vec, data->scale);
    ARR3SUB(vec, x);
    d2 = vec[0]*vec[0] + vec[1]*vec[1];
    if (d2 < tol)
      {
      best = core;
      tol = d2;
      }
    }
  }
while (ilist);

return(best);
}

/*********************************/
/* coordinate matching routine 2 */
/*********************************/
/* done via 2D window coordinate matching */
#define DEBUG_SEEK_COORD2D 0
struct core_pak *seek_coord2d(gint x, gint y, struct model_pak *data)
{
gint tol, d2, w[2];
gdouble vec[3];
GSList *list=NULL;
struct core_pak *core, *best=NULL;

/* tolerance (pixels squared) */
tol = 49;

#if DEBUG_SEEK_COORD2D
printf("search: [%d,%d]\n", x, y);
#endif

/* TODO - periodic images */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  ARR3SET(vec, core->rx);
  VEC3MUL(vec, data->scale);

  gl_get_window_coords(vec, w);

#if DEBUG_SEEK_COORD2D
printf("  (%d, %d)\n", w[0], w[1]);
#endif

  w[0] -= x;
  w[1] -= y;

  d2 = w[0]*w[0] + w[1]*w[1];
  if (d2 < tol)
    {
    best = core;
    tol = d2;
    }
  }
return(best);
}

/******************************/
/* vertex coordinate extremes */
/******************************/
void vertex_limits(gdouble *min, gdouble *max, struct model_pak *model)
{
gint i;
gdouble x[3];
GSList *list;
struct vertex_pak *v;

g_assert(model != NULL);

/* failsafe limit init */
VEC3SET(min, 0.0, 0.0, 0.0);
VEC3SET(max, 0.0, 0.0, 0.0);
list = model->vertices;
if (!list)
  return;

/* failsafe limit init */
v = list->data;
ARR3SET(x, v->x);
ARR3SET(min, x);
ARR3SET(max, x);

/* scan list and update limits */
while (list)
  {
  v = list->data;

  ARR3SET(x, v->x);

  for (i=3 ; i-- ; )
    {
    if (x[i] < min[i])
      min[i] = x[i];
    if (x[i] > max[i])
      max[i] = x[i];
    }
  list = g_slist_next(list);
  }
}

/**********/
/* COORDS */
/**********/
#define DEBUG_CENT 0
gint cent_coords(gint global, struct model_pak *data)
{
gint i, n;
gdouble min[3], max[3], vec[3], r, r2, rmax;
GSList *list=NULL, *ilist, *clist;
struct image_pak *image;
struct spatial_pak *spatial;
struct core_pak *core;
struct vertex_pak *v;
struct vec_pak *p1;

/* initialize */
VEC3SET(data->offset, 0, 0, 0);
VEC3SET(data->centroid, 0.0, 0.0, 0.0);
r2 = 0.0;

/* core list is (non-empty) selection for non global scaling */
clist = data->cores;
if (!global && g_slist_length(data->selection))
  clist = data->selection;

/* centroid calc */
n=0;
for (list=clist ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

  ARR3ADD(data->centroid, core->x);
  n++;
  }
if (n)
  {
  VEC3MUL(data->centroid, 1.0 / (gdouble) n);
  }
else
  {
/* if we have no atoms in a periodic model - make pbc/2 the centroid */
  if (data->periodic)
    {
    VEC3SET(data->centroid, 0.5, 0.5, 0.5);
    }
  else
    {
/* morphology centroid (NB: won't be the origin if non-centrosymetric) */
    vertex_limits(min, max, data);
    ARR3SET(data->centroid, min);
    ARR3ADD(data->centroid, max);
    VEC3MUL(data->centroid, 0.5);
    }
  }

/* if centroid is substantially within repeat cell, make pbc/2 the centroid */
if (data->periodic)
  {
  for (i=data->periodic ; i-- ; )
    if (fabs(data->centroid[i] - 0.5) < 0.45)
      data->centroid[i] = 0.5; 
  }

/* adjust for periodic images */
n=1;
VEC3SET(vec, 0.0, 0.0, 0.0);
for (list=data->images ; list ; list=g_slist_next(list))
  {
  image = (struct image_pak *) list->data;

  ARR3ADD(vec, image->pic);
  n++;
  }
VEC3MUL(vec, 1.0 / (gdouble) n);
ARR3ADD(data->centroid, vec);

/* get distance of furtherest displayed item from centroid */
for (list=clist ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

  ilist=NULL;
  do
    {
    ARR3SET(vec, core->x);
    if (ilist)
      {
      image = (struct image_pak *) ilist->data;
/* image */
      ARR3ADD(vec, image->pic);
      ilist = g_slist_next(ilist);
      }
    else
      {
      ilist = data->images;
      }

    ARR3SUB(vec, data->centroid);
    vecmat(data->latmat,vec);

/* dist squared */
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  while (ilist);
  }

/* check cell vertices */
if (data->periodic)
  {
  for (i=8 ; i-- ; )
    {
    ARR3SET(vec, data->cell[i].x);
/* centroid removal */
    ARR3SUB(vec, data->centroid);
/* transform */
    vecmat(data->latmat,vec);  
/* dist sq. */
    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  }

/* check vertices */
for (list=data->vertices ; list ; list=g_slist_next(list))
  {
  v = list->data;
  ARR3SET(vec, v->x);
  ARR3SUB(vec, data->centroid);
  vecmat(data->latmat, vec);
  r = VEC3MAGSQ(vec);
  if (r > r2)
    r2 = r; 
  }

/* check spatial objects */
for (list=data->spatial ; list ; list=g_slist_next(list))
  {
  spatial = list->data;

/* molecular surface contribution */
  if (spatial->type == SPATIAL_MOLSURF)
    {
    p1 = g_slist_nth_data(spatial->data, 0);

    ARR3SET(vec, p1->x);
    ARR3SUB(vec, data->centroid);
    vecmat(data->latmat, vec);  

    r = VEC3MAGSQ(vec);
    if (r > r2)
      r2 = r; 
    }
  }
rmax = sqrt(r2);

/* assign the calculated value, unless 0 */
if (rmax)
  data->rmax = rmax;
else
  data->rmax = RMAX_FUDGE;
data->zoom = data->rmax/data->scale;

#if DEBUG_CENT
P3VEC("centroid = ", data->centroid);
printf("rmax = %f\n", data->rmax);
printf("zoom = %f\n", data->zoom);
printf("cent_coords() done.\n");
#endif

return(0);
}

/******************************************/
/* sort coords by z - mainly for surfaces */
/******************************************/
gint sort_cores(struct core_pak *core1, struct core_pak *core2)
{
if (core1->x[2] < core2->x[2])
  return(1);
if (core1->x[2] > core2->x[2])
  return(-1);
return(0);
}
gint sort_shels(struct shel_pak *shel1, struct shel_pak *shel2)
{
if (shel1->x[2] < shel2->x[2])
  return(1);
if (shel1->x[2] > shel2->x[2])
  return(-1);
return(0);
}
void sort_coords(struct model_pak *data)
{
data->cores = g_slist_sort(data->cores, (gpointer) sort_cores);
data->shels = g_slist_sort(data->shels, (gpointer) sort_shels);
}

/*************************/
/* get coordinate limits */
/*************************/
void cor_calc_xlimits(gdouble *min, gdouble *max, GSList *list)
{
gint i;
GSList *item;
struct core_pak *core;

/* no cores? */
if (!list)
  return;

/* init */
core = list->data;
for (i=3 ; i-- ; )
  min[i] = max[i] = core->x[i];

/* NB: even tho' we check the "fractional" part - the values */
/* will still be cartesian for isolated (or the appropriate */
/* mixed frac/cart values for 1D/2D) models */
/* loop to get limits */
for (item=list ; item ; item=g_slist_next(item))
  {
  core = item->data;

  for (i=3 ; i-- ; )
    {
    if (core->x[i] < min[i])
      min[i] = core->x[i];
    if (core->x[i] > max[i])
      max[i] = core->x[i];
    }
  }
}

/**********************/
/* CALC SCREEN COORDS */
/**********************/
#define DEBUG_UPDATE_COORDS 0
void calc_coords(gint mode, struct model_pak *data)
{
gint n, ghost, model;
gdouble scale;
gdouble rot[9], vec[3];
gdouble tmp4[16], mat4[16], vec4[4];
GSList *list=NULL, *list1, *list2;
struct core_pak *core;
struct shel_pak *shell;
struct model_pak *orig;
struct vec_pak *vector;
struct vertex_pak *v;
struct plane_pak *plane;
struct ribbon_pak *ribbon;
struct spatial_pak *spatial;
struct object_pak *odata;
struct image_pak *image;
GSList *plist, *glist=NULL, *olist;

#if DEBUG_UPDATE_COORDS
printf("calc_coords() start.\n");
#endif

g_return_if_fail(data != NULL);

/* update model & any overlayed (ghost) models */
ghost=0;
orig = data;
while(data)
  {
/* get current incremental rotation matrix */
  make_rotmat(rot, orig->da, mode);
  switch(mode)
    {
    case INITIAL:
      init_rotmat(data->rotmat);
      break;
/* set and constrain (0-360) angles */
    case YAW:
      data->angle[0] += R2D*orig->da;
      ca_deg(&data->angle[0]);
      break;
    case PITCH:
      data->angle[1] += R2D*orig->da;
      ca_deg(&data->angle[1]);
      break;
    case ROLL:
      data->angle[2] += R2D*orig->da;
      ca_deg(&data->angle[2]);
      break;
    }

#if DEBUG_UPDATE_COORDS
P3MAT("instantaneous rot matrix", rot);
#endif

/* update cummulative rotation matrix */
matmat(rot, data->rotmat);
/* compute inverse, needed for correct selection handling */
ARR3SET(&data->irotmat[0], &data->rotmat[0]);
ARR3SET(&data->irotmat[3], &data->rotmat[3]);
ARR3SET(&data->irotmat[6], &data->rotmat[6]);
invmat(data->irotmat);

#if DEBUG_UPDATE_COORDS
P3MAT("cummulative rot matrix", data->rotmat);
P3MAT("           irot matrix", data->irotmat);
#endif

/* TODO - eliminate scale? (use camera position) */
/* precalc scale */
scale = data->scale * sysenv.subscale * 0.5 / data->rmax;

/* precalc matrix products */
ARR3SET(&mat4[0], &data->latmat[0]);
ARR3SET(&mat4[4], &data->latmat[3]);
ARR3SET(&mat4[8], &data->latmat[6]);
ARR3SET(vec, data->centroid);
vecmat(data->latmat, vec);
mat4[3] = -vec[0];
mat4[7] = -vec[1];
mat4[11] = -vec[2];
VEC4SET(&mat4[12], 0.0, 0.0, 0.0, 1.0);

ARR3SET(&tmp4[0], &data->rotmat[0]);
ARR3SET(&tmp4[4], &data->rotmat[3]);
ARR3SET(&tmp4[8], &data->rotmat[6]);
tmp4[3] = 0.0;
tmp4[7] = 0.0;
tmp4[11] = 0.0;
VEC4SET(&tmp4[12], 0.0, 0.0, 0.0, 1.0);
mat4mat(tmp4, mat4);

#if DEBUG_UPDATE_COORDS
P4MAT("mat4:", mat4);
#endif

memcpy(data->display_lattice, mat4, 16*sizeof(gdouble));

/* update image translation vectors */
for (list=data->images ; list ; list=g_slist_next(list))
  {
  image = (struct image_pak *) list->data;

  ARR3SET(vec4, image->pic);
/* NB: is a vector */
  vec4[3] = 0.0;
  vec4mat(mat4, vec4);
  ARR3SET(image->rx, vec4);
  }

/* update atoms */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & DELETED)
    continue;

/* calc coords */
  ARR3SET(vec4, core->x);

ARR3ADD(vec4, core->offset);

  vec4[3] = 1.0;
  vec4mat(mat4, vec4);
  ARR3SET(core->rx, vec4);
  }

/* rotate shells (if any ) */
for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shell = (struct shel_pak *) list->data;
  if (shell->status & DELETED)
    continue;

/* get coordinates */
  ARR3SET(vec4, shell->x);

ARR3ADD(vec4, shell->offset);

  vec4[3] = 1.0;
  vec4mat(mat4, vec4);
  ARR3SET(shell->rx, vec4);
  }

/* CURRENT */
measure_update_global(data);

/********/
/* CELL */
/********/
for (n=8 ; n-- ; )
  {
  ARR3SET(vec4, data->cell[n].x);
  vec4[3] = 1.0;
  vec4mat(mat4, vec4);
  ARR3SET(data->cell[n].rx, vec4);
  }

/********/
/* AXES */
/********/
for (n=6 ; n-- ; )
  {
  ARR3SET(vec4, data->axes[n].x);
  vec4[3] = 0.0;

/* cartesian or unit cell frame */
  if (data->axes_type == CARTESIAN)
    vecmat(data->rotmat, vec4);
  else
    {
    vec4mat(mat4, vec4);
    normalize(vec4, 3);
    }

  ARR3SET(data->axes[n].rx, vec4);
/* about a 16th of rmax */
  VEC3MUL(data->axes[n].rx, 0.06*data->rmax);
  }

/******************/
/* ribbon updates */
/******************/
olist = data->ribbons;
while (olist)
  {
  odata = (struct object_pak *) olist->data;

  switch (odata->type)
    {
    case RIBBON:
      plist = (GSList *) odata->data;
      while (plist != NULL)
        {
        ribbon = (struct ribbon_pak *) plist->data;

/* end point 1 */
        ARR3SET(vec4, ribbon->x1);
        vec4[3] = 1.0;
        vec4mat(mat4, vec4); 
        ARR3SET(ribbon->r1, vec4);

/* end point 2 */
        ARR3SET(vec4, ribbon->x2);
        vec4[3] = 1.0;
        vec4mat(mat4, vec4); 
        ARR3SET(ribbon->r2, vec4);

/* normal 1 */
        ARR3SET(vec4, ribbon->u1);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->n1, vec4);

/* normal 2 */
        ARR3SET(vec4, ribbon->u2);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->n2, vec4);

/* orientation vector 1 */
        ARR3SET(vec4, ribbon->v1);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->o1, vec4);

/* orientation vector 2 */
        ARR3SET(vec4, ribbon->v2);
        vec4[3] = 0.0;
        vec4mat(mat4, vec4);
        ARR3SET(ribbon->o2, vec4);

        plist = g_slist_next(plist);
        }
    break;
    }
  olist = g_slist_next(olist);
  }

/* spatials */
for (olist=data->spatial ; olist ; olist=g_slist_next(olist))
  {
  spatial = (struct spatial_pak *) olist->data;

  plist = (GSList *) spatial->data;
  while (plist)
    {
    vector = (struct vec_pak *) plist->data;

/* rotate each point */
    ARR3SET(vec4, vector->x);
    vec4[3] = 1.0;
    vec4mat(mat4, vec4);
    ARR3SET(vector->rx, vec4);

/* rotate the normal */
    ARR3SET(vec4, vector->n);
    vec4[3] = 0.0;
    vec4mat(mat4, vec4);
    normalize(vec4, 3);
    ARR3SET(vector->rn, vec4);

    plist = g_slist_next(plist);
    }
  }

/**************/
/* morphology */
/**************/
if (data->id == MORPH)
  {
  for (list1=data->vertices ; list1 ; list1=g_slist_next(list1))
    {
    v = (struct vertex_pak *) list1->data;

    ARR3SET(vec4, v->x);
    vec4[3] = 1.0;
    vec4mat(mat4, vec4);
    ARR3SET(v->rx, vec4);
    }

/* facet center calc. */
  for (list1=data->planes ; list1 ; list1=g_slist_next(list1))
    {
    plane = (struct plane_pak *) list1->data;
    if (plane->present)
      {
/* set visible flag */
      if (facet_visible(data, plane))
        plane->visible = TRUE;
      else
        plane->visible = FALSE;

      VEC3SET(vec, 0.0, 0.0, 0.0);
      n = 0; 
      for (list2=plane->vertices ; list2 ; list2=g_slist_next(list2))
        {
        v = (struct vertex_pak *) list2->data;
        ARR3ADD(vec, v->rx);
        n++;
        }
/* assign */
/* NB: n=0 can sometimes occur with a bad (ie non-closed) polyhedron */
      if (n)
        VEC3MUL(vec, 1.0/(gdouble) n);
      ARR3SET(plane->rx, vec);
      }
    }
  }

/*********************/
/* ghost processsing */
/*********************/
  if (!ghost)
    {
    glist = data->ghosts;
    ghost++;
    }
  else
    glist = g_slist_next(glist);
/* get the ghost model's pointer */
  if (glist)
    {
    model = GPOINTER_TO_INT(glist->data);
    data = model_ptr(model, RECALL);
    }
  else
    data = NULL;
  }

#if DEBUG_UPDATE_COORDS
printf("calc_coords() done.\n");
#endif
}

/***************************************/
/* unified pixel to coordinate routine */
/***************************************/
#define DEBUG_PCM 0
/* deprec. - use gl_get_world)coords() */
void pixel_coord_map(gint px, gint py, gdouble *r, struct model_pak *data)
{
/* checks */
g_return_if_fail(data != NULL);

/* the model should be displayed for this to make sense */
VEC3SET(r, 0.0, 0.0, 0.0);
gl_get_world_coords(px, py, r);

return;
}

/***********************************/
/* print atom coord data primitive */
/***********************************/
void print_core(struct core_pak *core)
{
gchar *txt;

g_assert(core != NULL);

txt = g_strdup_printf("%4s core at (%9.4f,%9.4f,%9.4f).\n",
                      core->label, core->x[0], core->x[1], core->x[2]);
show_text(STANDARD, txt);
g_free(txt);
}

/************************************/
/* print shell coord data primitive */
/************************************/
void print_shell(struct shel_pak *shell)
{
gchar *txt;

g_assert(shell != NULL);

txt = g_strdup_printf("%4s shel at (%9.4f,%9.4f,%9.4f) [%9.4f, %9.4f, %9.4f]\n",
                      shell->label, shell->x[0], shell->x[1], shell->x[2],
                      shell->rx[0], shell->rx[1], shell->rx[2]);
show_text(STANDARD, txt);
g_free(txt);
}

void print_cores(struct model_pak *data)
{
GSList *list;
struct core_pak *core;

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  printf("%4s core at (%9.4f,%9.4f,%9.4f).\n",
         core->label, core->x[0], core->x[1], core->x[2]);
  }
}

void print_shells(struct model_pak *data)
{
GSList *list;
struct shel_pak *shel;

for (list=data->shels ; list ; list=g_slist_next(list))
  {
  shel = (struct shel_pak *) list->data;

  printf("%4s shel at (%9.4f,%9.4f,%9.4f).\n",
         shel->label, shel->x[0], shel->x[1], shel->x[2]);
  }
}

/****************************************/
/* print all connected cores and shells */
/****************************************/
void print_core_shell(struct model_pak *data)
{
gint n=0;
GSList *list;
struct core_pak *core;
struct shel_pak *shel;

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->shell)
    {
    shel = (struct shel_pak *) core->shell;

if (n == 7 || n == 15)
  {
    printf("(%p) %4s core at (%9.4f,%9.4f,%9.4f).\n",
           core, core->label, core->x[0], core->x[1], core->x[2]);
    printf("(%p) %4s shel at (%9.4f,%9.4f,%9.4f).\n",
           shel, shel->label, shel->x[0], shel->x[1], shel->x[2]);
  }

    n++;
    }
  }
}

/***********************/
/* core free primitive */
/***********************/
void core_free(gpointer data)
{
struct core_pak *core = data;

g_free(core->atom_label);
g_free(core->atom_type);
g_free(core->res_name);
g_slist_free(core->bonds);

free_slist(core->vibx_list);
free_slist(core->viby_list);
free_slist(core->vibz_list);

g_free(core);
}

/****************************************************/
/* free all lists that reference cores or core data */
/****************************************************/
void free_core_list(struct model_pak *data)
{
GSList *list;

g_assert(data != NULL);

/* free cores */
for (list=data->cores ; list ; list=g_slist_next(list))
  core_free(list->data);
g_slist_free(data->cores);

/* free shells */
free_slist(data->shels);

/* free connectivity */
free_slist(data->bonds);
free_slist(data->ubonds);
free_mol_list(data);

/* others */
g_slist_free(data->selection);
/* NB: don't free data */
g_slist_free(data->unique_atom_list);

/* enforce empty lists */
data->cores = NULL;
data->shels = NULL;
data->bonds = NULL;
data->ubonds = NULL;
data->moles = NULL;
data->selection = NULL;
data->unique_atom_list = NULL;
}

