/*
Copyright (C) 1993 by David H. Gay and Andrew L. Rohl 

dgay@ricx.royal-institution.ac.uk
andrew@ricx.royal-institution.ac.uk

Modified 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "numeric.h"
#include "vector.h"
#include "space.h"
#include "surface.h"
#include "zone.h"
#include "interface.h"

typedef gchar boolean;



/********************************************/
/* allocate for new shift & init for safety */
/********************************************/
gpointer shift_new(gdouble shift)
{
struct shift_pak *sdata;

/* alloc */
sdata = g_malloc(sizeof(struct shift_pak));

sdata->locked = FALSE;
sdata->shift = shift;
sdata->dipole_computed = FALSE;
sdata->dipole = 99.9;
sdata->region[0] = 1;
sdata->region[1] = 1;
sdata->esurf[0] = 0.0;
sdata->esurf[1] = 0.0;
sdata->eatt[0] = 0.0;
sdata->eatt[1] = 0.0;
sdata->gnorm = -1.0;
sdata->procfile = NULL;

return(sdata);
}

/********************************************/
/* allocate for new plane & init for safety */
/********************************************/
#define DEBUG_CREATE_PLANE 0
gpointer plane_new(gdouble *m, struct model_pak *model)
{
gint h, k, l, n;
gdouble vec[3];
struct plane_pak *plane=NULL;

/* checks */
g_return_val_if_fail(model != NULL, NULL);

plane = g_malloc(sizeof(struct plane_pak));

#if DEBUG_CREATE_PLANE
printf("creating: %d %d %d -> ", (gint) m[0], (gint) m[1], (gint) m[2]);
#endif

/* save orig */
h = m[0];
k = m[1];
l = m[2];
n = 2;

/* NB: check for my new absence testing function */
/*
printf("%d %d %d : [%d ?= %d]\n", h, k, l,
IsSysAbsent_hkl((T_SgInfo *) model->sginfo.raw, h, k , l, NULL),
surf_sysabs(model, h, k, l));
*/

/* check for reflection planes */
/*
while (IsSysAbsent_hkl((T_SgInfo *) model->sginfo.raw, h, k , l, NULL) && n<100)
*/

/* my absence testing function */
while (surf_sysabs(model, h, k, l) && n<100)
  {
  h = n*m[0];
  k = n*m[1];
  l = n*m[2];
  n++;
  }

if (n == 100)
  {
  printf("WARNING: screwed up systematic absence check.\n");
  h /= 99;
  k /= 99;
  l /= 99;
  n = 2;
  }

#if DEBUG_CREATE_PLANE
printf("%d %d %d  (x %d)\n", h, k, l, n-1);
#endif

/* init */
VEC3SET(plane->m, (gdouble) h, (gdouble) k, (gdouble) l);
VEC3SET(plane->norm, (gdouble) h, (gdouble) k, (gdouble) l);
VEC3SET(plane->index, h, k, l);
plane->shifts = NULL;
plane->vertices = NULL;

/* calc Dhkl */
VEC3SET(vec, h, k, l);
vecmat(model->rlatmat, vec);
plane->dhkl = 1.0/VEC3MAG(vec);
plane->esurf[0] = 0.0;
plane->esurf[1] = 0.0;
plane->eatt[0] = 0.0;
plane->eatt[1] = 0.0;
plane->f[0] = 0.0;
plane->f[1] = 0.0;
/* calc? */
plane->multiplicity = 1;
plane->present = TRUE;
plane->visible = FALSE;
plane->primary = TRUE;
VEC3SET(plane->rx, 0.0, 0.0, 0.0);

return(plane);
}

/************************************************************/
/* determine if a particular plane has already been created */
/************************************************************/
gpointer plane_find(gdouble *hkl, struct model_pak *model)
{
GSList *list;
struct plane_pak *comp, *plane;

/* get corrected (sys absent!) miller index */
comp = plane_new(hkl, model);
g_return_val_if_fail(comp != NULL, NULL);

for (list=model->planes ; list ; list=g_slist_next(list))
  {
  plane = (struct plane_pak *) list->data;
  if (facet_equiv(model, comp->index, plane->index))
    return(plane);
  }

g_free(comp);
return(NULL);
}

/***********************/
/* free a single shift */
/***********************/
void shift_free(gpointer data)
{
struct shift_pak *shift = data;

g_assert(shift != NULL);

/* FIXME - this may create a memory leak, but it'll be */
/* the users fault */
if (shift->locked)
  {
  printf("ERROR: shift is locked.\n");
  return;
  }
g_free(shift);
}

/*************************************/
/* free the data in a list of shifts */
/*************************************/
void shift_data_free(GSList *shifts)
{
GSList *list;
struct shift_pak *shift;

/* free a list of shift pak structures */
list = shifts;
while (list)
  {
  shift = list->data;
  list = g_slist_next(list);

  shift_free(shift);
  }
}

/***********************/
/* free a single plane */
/***********************/
void plane_free(gpointer data)
{
struct plane_pak *plane = data;

shift_data_free(plane->shifts);
g_slist_free(plane->shifts);

/* FIXME - is this good enough? */
g_slist_free(plane->vertices);
g_free(plane);
}

/*************************************/
/* free the data in a list of planes */
/*************************************/
void plane_data_free(GSList *planes)
{
GSList *list;
struct plane_pak *plane;

/* free a list of plane pak structures */
list = planes;
while (list)
  {
  plane = list->data;
  list = g_slist_next(list);

  plane_free(plane);
  }
}

/****************************************/
/* uses marvin code to create a surface */
/****************************************/
#define DEBUG_GENSURF 0
gint generate_surface(struct model_pak *src, struct model_pak *dest)
{
vector normal;
vector lattice_vector[3];    /* lattice vectors */
vector t_mat[3];             /* transformation matrix */
vector work_lat[3];          /* transformed lattice vectors */
vector rec_work_lat[3];      /* reciprocal transformed lattice vectors */
vector s[3];                 /* space we are trying to fill */
vector rec_s[2];             /* reciprocal of s[0] and s[1] */
vector tempvec;
vector *sv;
vector a, *v_a=NULL, *v_b=NULL;
boolean s2_found = FALSE;
gint c, i, j, h, k, l, flag;
gint ia, ib, ic;
gint amin, amax, bmin, bmax, cmin, cmax;
gint GCD(gint, gint);
gint vector_compare(vector *, vector *);
gdouble inv_denom;
gdouble shift, depth, depth_1, depth_2, depth_min;
gdouble gcd, cd, x;
gdouble z1_max, z1_min, z2_max, z2_min;
gdouble tempfloat;
gdouble tmat[9], norm[3], va[3];
gdouble wlat[9], temp[9], lpm[9];
gdouble sign, vec[3], vec2[3], dn[3];
GSList *list1, *list2, *sv_list1, *sv_list2;
GSList *list, *clist, *slist, *mlist;
struct core_pak *core;
struct shel_pak *shel;
struct mol_pak *mol;

/* NB: we now APPEND - so ensure we have no old crap in the lists */
g_assert(dest->cores == NULL);
g_assert(dest->shels == NULL);
g_assert(dest->bonds == NULL);
g_assert(dest->moles == NULL);

/* find surface normal from the miller index and lattice vectors*/
V_ZERO(normal);

/* setup */
h = dest->surface.miller[0];
k = dest->surface.miller[1];
l = dest->surface.miller[2];

/* acquire lattice vectors */
V_X(lattice_vector[0]) = src->latmat[0];
V_Y(lattice_vector[0]) = src->latmat[3];
V_Z(lattice_vector[0]) = src->latmat[6];
V_X(lattice_vector[1]) = src->latmat[1];
V_Y(lattice_vector[1]) = src->latmat[4];
V_Z(lattice_vector[1]) = src->latmat[7];
V_X(lattice_vector[2]) = src->latmat[2];
V_Y(lattice_vector[2]) = src->latmat[5];
V_Z(lattice_vector[2]) = src->latmat[8];

V_CROSS(tempvec, lattice_vector[1], lattice_vector[2]);
V_QADD(normal, normal, +h*, tempvec);
V_CROSS(tempvec, lattice_vector[2], lattice_vector[0]);
V_QADD(normal, normal, +k*, tempvec);
V_CROSS(tempvec, lattice_vector[0], lattice_vector[1]);
V_QADD(normal, normal, +l*, tempvec);

sv_list1 = NULL;
 
c = (float) GCD(h, k);
cd = c ? c : 1.0;
V_2_ASSIGN(a, = k/cd * ,lattice_vector[0], - h/cd *, lattice_vector[1]);
if (V_MAGSQ(a) > EPSILON) 
  {
  sv = g_malloc(sizeof(vector));
  V_EQUATE(*sv, a);
  sv_list1 = g_slist_prepend(sv_list1, sv);
  }
  
c = (float) GCD(h, l);
cd = c ? c : 1.0;
V_2_ASSIGN(a, = l/cd * ,lattice_vector[0], - h/cd *, lattice_vector[2]);
if (V_MAGSQ(a) > EPSILON) 
  {
  sv = g_malloc(sizeof(vector));
  V_EQUATE(*sv, a);
  sv_list1 = g_slist_prepend(sv_list1, sv);
  }
  
c = (float) GCD(k, l);
cd = c ? c : 1.0;
V_2_ASSIGN(a, = l/cd * ,lattice_vector[1], - k/cd *, lattice_vector[2]);
if (V_MAGSQ(a) > EPSILON) 
  {
  sv = g_malloc(sizeof(vector));
  V_EQUATE(*sv, a);
  sv_list1 = g_slist_prepend(sv_list1, sv);
  }

sv_list2 = NULL;
list1 = sv_list1;
for (i=0 ; i<g_slist_length(sv_list1)-1 ; i++)
  {
  v_a = (vector *) list1->data;
  list1 = list2 = g_slist_next(list1);
  for (j=i+1 ; j<g_slist_length(sv_list1) ; j++)
    {
    v_b = (vector *) list2->data;
    list2 = g_slist_next(list2);


    V_2_ASSIGN(a, = , *v_a, + , *v_b);
    if (V_MAGSQ(a) > EPSILON)
      {
      sv = g_malloc(sizeof(vector));
      V_EQUATE(*sv, a);
      sv_list2 = g_slist_prepend(sv_list2, sv);
      }
    V_2_ASSIGN(a, = , *v_a, - , *v_b);
    if (V_MAGSQ(a) > EPSILON)
      {
      sv = g_malloc(sizeof(vector));
      V_EQUATE(*sv, a);
      sv_list2 = g_slist_prepend(sv_list2, sv);
      }
    }
  }
sv_list1 = g_slist_concat(sv_list1, sv_list2);
sv_list1 = g_slist_sort(sv_list1, (gpointer) vector_compare);

/* set v_a to first (shortest now sorted) vector */
list1 = sv_list1;
v_a = (vector *) list1->data;
list1 = g_slist_next(list1);
/* loop over remaining vectors - find shortest w/ orthogonal component */
while (list1)
  {
  v_b = (vector *) list1->data;
  
  V_CROSS(a, *v_a, *v_b);
  x = V_MAGSQ(a);
  if (x > EPSILON) 
    {
    s2_found = TRUE;
    break;
    }
  list1 = g_slist_next(list1);
  }

/* checks */
if (!s2_found) 
  {
  show_text(ERROR, "Failed to find surface vectors.\n");
  return(1);
  }
g_assert(v_a != NULL);
g_assert(v_b != NULL);

/* calculate transformation matrix */
V_SCALER(t_mat[2], (1.0/V_MAG(normal)), normal);
V_SCALER(t_mat[0], (1.0/V_MAG(*v_a)), *v_a);
V_CROSS(t_mat[1], t_mat[2], t_mat[0]);

ARR3SET(norm, normal.element);
ARR3SET(va, v_a->element);
normalize(norm, 3);
normalize(va, 3);
ARR3SET(&tmat[6], norm);
ARR3SET(&tmat[0], va);
crossprod(&tmat[3], norm, va);

/* calculate transformed lattice vectors */
for (j = 0; j < 3; j++)
  for (i = 0; i < 3; i++)
    V_E(work_lat[j], i) = V_DOT(t_mat[i], lattice_vector[j]);

/* calculate reciprocal transformed lattice vectors */
V_CROSS(tempvec, work_lat[1], work_lat[2]);
inv_denom = 1.0 / V_DOT(tempvec, work_lat[0]);
V_CROSS(tempvec, work_lat[1], work_lat[2]);
V_SCALER(rec_work_lat[0], inv_denom, tempvec);
V_CROSS(tempvec, work_lat[2], work_lat[0]);
V_SCALER(rec_work_lat[1], inv_denom, tempvec);
V_CROSS(tempvec, work_lat[0], work_lat[1]);
V_SCALER(rec_work_lat[2], inv_denom, tempvec);

/* calculate transformed surface vectors */
for (i = 0; i < 3; i++)
  V_E(s[0], i) = V_DOT(t_mat[i], *v_a);
for (i = 0; i < 3; i++)
  V_E(s[1], i) = V_DOT(t_mat[i], *v_b);

/* return transformed surface vectors data in dest->latmat */
  /* NB: latmat[0..8] is a 3x3 matrix so, */
  /*
    | a b 0 |
    | c d 0 |
    | 0 0 0 |

    goes into latmat rows first ie a,b,0,c,d,0,0,0,0
  */
  
/* set s[2] = to depths */
/* depth = 1.0e10; */

/* surface specification */
gcd = GCD(GCD(h, k), GCD(k, l));
VEC3SET(vec, h, k, l);
vecmat(src->rlatmat, vec);
 
/* calculate the Dhkl */
/* NB: we want the correct dspacing wrt hkl, ie DON'T remove the GCD */
dest->surface.dspacing = 1.0/VEC3MAG(vec);

/* actual cut depth */
/* NB: done wrt the FULL depth (ie remove the gcd) */
depth = gcd/VEC3MAG(vec);
dest->surface.depth = depth;

/* round off the shift */
shift = decimal_round(dest->surface.shift, 4);

#if DEBUG_GENSURF
printf("\n----------------------------\n");
printf("    hkl: %d %d %d\n", h, k, l);
printf("    gcd: %f \n", gcd);
printf("  shift: %.20f\n", shift);
printf("regions: %d %d\n", (gint) dest->surface.region[0], (gint) dest->surface.region[1]);
printf("   Dhkl: %f\n", dest->surface.dspacing);
printf("  depth: %f\n", depth);
printf("----------------------------\n");
#endif

/* NB: depth_1 is region[0], but depth_2 is region[0]+region[1] */
depth_1 = dest->surface.region[0] * depth;
depth_2 = depth_1 + dest->surface.region[1] * depth;

V_ZERO(s[2]);
V_Z(s[2]) = -(depth_2);

/* transfer surface vectors */
VEC3SET(&dest->latmat[0], V_X(s[0]), V_X(s[1]), 0.0);
VEC3SET(&dest->latmat[3], V_Y(s[0]), V_Y(s[1]), 0.0);
VEC3SET(&dest->latmat[6], 0.0, 0.0, 1.0);

/* calculate reciprocal of two surface vectors */
inv_denom = 1.0 / (V_X(s[0])*V_Y(s[1]) - V_Y(s[0])*V_X(s[1]));
V_X(rec_s[0]) =  V_Y(s[1])*inv_denom;
V_Y(rec_s[0]) = -V_X(s[1])*inv_denom;
V_Z(rec_s[0]) =  0.0;
V_X(rec_s[1]) = -V_Y(s[0])*inv_denom;
V_Y(rec_s[1]) =  V_X(s[0])*inv_denom;
V_Z(rec_s[1]) =  0.0;

z2_min = z1_min = 0.00;
z2_max = V_Z(s[2]);
z1_max = depth_1 * (z2_max < 0 ? -1.0: +1.0);

if (z2_max < z2_min) 
  {
  tempfloat = z2_max;
  z2_max = z2_min;
  z2_min = tempfloat;
  }
if (z1_max < z1_min) 
  {
  tempfloat = z1_max;
  z1_max = z1_min;
  z1_min = tempfloat;
  }

/* display name */
g_free(dest->basename);
dest->basename = g_strdup_printf("%s_%-1d%-1d%-1d_%6.4f",
                 g_strstrip(src->basename), h, k, l, shift);

/* store the work lattice */
for (i=0 ; i<3 ; i++)
  {
  wlat[3*i+0] = V_E(work_lat[0], i);
  wlat[3*i+1] = V_E(work_lat[1], i);
  wlat[3*i+2] = V_E(work_lat[2], i);
  }

#if DEBUG_GENSURF
P3MAT(" src lat: ", src->latmat);
#endif

/* construct the depth vector */
/* compute the surface normal */
VEC3SET(vec, dest->latmat[0], dest->latmat[3], dest->latmat[6]);
VEC3SET(vec2, dest->latmat[1], dest->latmat[4], dest->latmat[7]);
crossprod(dn, vec, vec2);

/* search for a vector with the smallest */
/* orthogonal component to the surface vectors */
flag=0;
depth_min=0;
for (i=0 ; i<3 ; i++)
  {
/* ignore zero indices, which yield 0 orthogonal component */
  if (dest->surface.miller[i])
    {
/* dotproduct of vector with normal */
    ARR3SET(vec, dn);
    vec[0] *= wlat[i];
    vec[1] *= wlat[i+3];
    vec[2] *= wlat[i+6];
    depth = fabs(vec[0] + vec[1] + vec[2]);

#if DEBUG_GENSURF
printf("[i=%d, depth=%f]", i, depth);
#endif

   if (flag)
     {
     if (depth > depth_min)
       continue;
     }
   else
     flag++;

    depth_min = depth;

/* ensure the depth vector is pointing in the +ve z direction */
    if (wlat[i+6] < 0.0)
      sign = -1.0;
    else
      sign = 1.0;

    dest->surface.depth_vec[0] = sign*wlat[i];
    dest->surface.depth_vec[1] = sign*wlat[i+3];
    dest->surface.depth_vec[2] = sign*wlat[i+6];
    dest->latmat[2] = sign*wlat[i];
    dest->latmat[5] = sign*wlat[i+3];
    dest->latmat[8] = sign*wlat[i+6];
    }
  }
#if DEBUG_GENSURF
printf(" : Minimum = %f\n", depth_min);
P3VEC("depth vec:", dest->surface.depth_vec);
#endif

g_assert(flag != 0);

/* generate lattice matrix inverse */
memcpy(dest->ilatmat, dest->latmat, 9*sizeof(gdouble));
invmat(dest->ilatmat);

/* locate the position of the new cell's repeat vectors */
/* in terms of the lattice space of the source model */
/* this gives us the lattice point matrix (lpm) which */
/* is used to determine how many source unit cells */
/* will be required to fill out the new surface cell */
memcpy(lpm, dest->latmat, 9*sizeof(gdouble));
memcpy(temp, tmat, 9*sizeof(gdouble));
invmat(temp);
matmat(temp, lpm);
matmat(src->ilatmat, lpm);

#if DEBUG_GENSURF
P3MAT("tmat: ", tmat);
P3MAT("dest latmat: ", dest->latmat);
P3MAT("dest ilatmat: ", dest->ilatmat);
P3MAT("lpm: ", lpm);
#endif

/* setup repeats required to fill the new cell */
amin=bmin=cmin=0;
amax=bmax=cmax=0;

/* required number of a cells */
for (i=0 ; i<3 ; i++)
  {
  ia = rint(lpm[i]);
  if (ia < amin)
    amin = ia;
  if (ia > amax)
    amax = ia;
  }
/* required number of b cells */
for (i=3 ; i<6 ; i++)
  {
  ib = rint(lpm[i]);
  if (ib < bmin)
    bmin = ib;
  if (ib > bmax)
    bmax = ib;
  }
/* required number of c cells */
for (i=6 ; i<9 ; i++)
  {
  ic = rint(lpm[i]);

  if (ic < cmin)
    cmin = ic;
  if (ic > cmax)
    cmax = ic;
  }

/* one cell buffer */
amax+=2;
bmax+=2;
cmax+=2;
amin--;
bmin--;
cmin--;

#if DEBUG_GENSURF
printf("a: %d - %d\n", amin, amax);
printf("b: %d - %d\n", bmin, bmax);
printf("c: %d - %d\n", cmin, cmax);
#endif

/* compute transformation pipeline */
memcpy(temp, src->latmat, 9*sizeof(gdouble));
matmat(tmat, temp);
matmat(dest->ilatmat, temp);

/* create a single transformed unit cell */
/* full loop required to cover one cell in the transformed coordinates */
for (ic=cmin ; ic<cmax ; ic++)
  {
  for (ib=bmin ; ib<bmax ; ib++)
    {
    for (ia=amin ; ia<amax ; ia++)
      {
/* current source cell translation */
      VEC3SET(vec, ia, ib, ic);

/* loop over all molecules */
      for (mlist=src->moles ; mlist ; mlist=g_slist_next(mlist))
        {
        mol = (struct mol_pak *) mlist->data;

/* transform the centroid */
        ARR3SET(va, mol->centroid);
        ARR3ADD(va, vec);
        vecmat(temp, va);
        va[2] += shift;

/* ignore molecules outside the transformed cell */
/* NB: loose tolerances to ensure we don't */
/* lose some molecules at the boundaries */
#define EPS 0.0001
        if (va[0] < -EPS || va[0] > 1.0+EPS)
          continue;
        if (va[1] < -EPS || va[1] > 1.0+EPS)
          continue;
        if (va[2] < -EPS || va[2] > 1.0+EPS)
          continue;

/* loop over all atoms in molecule */
        for (clist=mol->cores ; clist ; clist=g_slist_next(clist))
          {
/* dup_core() will duplicate an attached shell as well */
          core = dup_core(clist->data);
          dest->cores = g_slist_prepend(dest->cores, core);

/* transform */
          ARR3ADD(core->x, vec);
          vecmat(temp, core->x);
          core->x[2] += shift;

/* init flags */
          core->primary = TRUE;
          core->primary_core = NULL;
          core->orig = TRUE;

/* shell */
          if (core->shell)
            {
            shel = core->shell;
            dest->shels = g_slist_prepend(dest->shels, shel);
/* transform */
            ARR3ADD(shel->x, vec);
            vecmat(temp, shel->x);
            shel->x[2] += shift;
/* init flags */
            shel->primary = TRUE;
            shel->primary_shell = NULL;
            shel->orig = TRUE;
            }
          }
        }
      }
    }
  }

dest->periodic = 3;
dest->fractional = FALSE;
dest->axes_type = CARTESIAN;
dest->construct_pbc = TRUE;

/* NB: can't leave the (expensive) remove_duplicates() to the */
/* final prep_model() as that treats the model as 2D periodic */
/* and won't clip the top & bottom of the surface*/
make_latmat(dest);
remove_duplicates(dest);

/* build surface core and shell lists from transformed cell */
clist = slist = NULL;

/* region 1 cores and shells */
amax = dest->surface.region[0] + 1;
for (i=1 ; i<amax ; i++)
  {
  for (list=dest->cores ; list ; list=g_slist_next(list))
    {
    core = dup_core(list->data);
    clist = g_slist_prepend(clist, core);

    core->x[2] -= i;

/*
printf("[%s] c ", core->label);
P3VEC(" : ", core->x);
*/

    vecmat(dest->latmat, core->x);
    core->region = REGION1A;

    if (core->shell)
      {
      shel = core->shell;
      slist = g_slist_prepend(slist, shel);
      shel->x[2] -= i;

/*
printf("[%s] s ", shel->label);
P3VEC(" : ", shel->x);
*/

      vecmat(dest->latmat, shel->x);
      shel->region = REGION1A;
      }
    }
  }
/* region 2 cores and shells */
bmax = amax + dest->surface.region[1];
for (i=amax ; i<bmax ; i++)
  {
  for (list=dest->cores ; list ; list=g_slist_next(list))
    {
    core = dup_core(list->data);
    clist = g_slist_prepend(clist, core);
    core->x[2] -= i;
    vecmat(dest->latmat, core->x);
    core->region = REGION2A;
    if (core->shell)
      {
      shel = core->shell;
      slist = g_slist_prepend(slist, shel);
      shel->x[2] -= i;
      vecmat(dest->latmat, shel->x);
      shel->region = REGION2A;
      }
    }
  }

/* replace old structure (transformed cell) with the new surface */
free_core_list(dest);
dest->cores = g_slist_reverse(clist);
dest->shels = g_slist_reverse(slist);

/* adjust depth translation vector to account for region sizes */
dest->latmat[2] *= dest->surface.region[0]+dest->surface.region[1];
dest->latmat[5] *= dest->surface.region[0]+dest->surface.region[1];
dest->latmat[8] *= dest->surface.region[0]+dest->surface.region[1];

/* surface cell init */
dest->periodic = 2;
dest->sginfo.spacenum = 1;
dest->fractional = FALSE;
dest->axes_type = CARTESIAN;
dest->construct_pbc = TRUE;
dest->surface.keep_atom_order = src->surface.keep_atom_order;

/* surface cell type */
if (src->surface.true_cell)
  dest->surface.true_cell = TRUE;
else
  {
  dest->surface.true_cell = FALSE;

/* want to force c to be // to z */
/* NB: vectors are in columns */
  dest->latmat[2] = 0.0;
  dest->latmat[5] = 0.0;
  }
free_slist(sv_list1);

#if DEBUG_GENSURF
printf("gensurf cores: %d\n", g_slist_length(dest->cores));
printf("gensurf shels: %d\n", g_slist_length(dest->shels));
#endif

/* build mode follows source structure */
dest->build_molecules = src->build_molecules;

/* last init */
prep_model(dest);
calc_emp(dest);

return(0);
}

/******************************************************************************
 * GCD
 *      Greatest common denominator
 ******************************************************************************/
gint GCD(gint p, gint q)
{
#if defined(__MWERKS__)  /* fix for compiler bug! */
  gint  i;  
  
  if (q == 0) {
    i = abs(p);
    return(i);
  }
#else
  if (q == 0)
    return(abs(p));
#endif
  else
    return(GCD(q, p%q));
}

/******************************************************************************
 * vector_compare
 *      compare two vectors returning their difference as an integer.  This
 *      routine is to be used in conjuction with SORT().
 ******************************************************************************/
gint vector_compare(vector *a, vector *b)
{
int i;
double  diff;
  
diff = V_MAGSQ(*a) - V_MAGSQ(*b);
if (diff < -EPSILON)
  return(-1);
else if (diff > EPSILON)
  return(1);
else
  {
/* magnitude of a & b is the same, so sort based on element magnitudes */ 
  for (i=0 ; i<3 ; i++)
    {
    diff = V_E(*a,i) - V_E(*b,i);
    if (diff < -EPSILON)
      return(1);
    else if (diff > EPSILON)
      return(-1);
    }
  }
/* a & b are identical */
return(0);
}

