/* gdkxft - Provide Xft font support in gdk
 * Copyright (C) 2001 Josh Parsons
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <gnome.h>

#include "gdkxftcfg.h"

/* the lists */
GtkCList *good_font_clist,*bad_font_clist,*bad_app_clist;

#define clist_get_srow(cl) (GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(cl)))-1)
#define clist_set_srow(cl,r) gtk_object_set_user_data(GTK_OBJECT(cl),GINT_TO_POINTER((r+1)))

void widget_sensitise(GtkWidget *widget)
{
  gtk_widget_set_sensitive(widget,TRUE);
}
void widget_desensitise(GtkWidget *widget)
{
  gtk_widget_set_sensitive(widget,FALSE);
}

/* connected clist has selected a row */
void entry_clist_select_row(GtkEntry *entry,
			    gint row,
			    gint column,
			    GdkEventButton *event,
			    GtkCList *clist)
{
  gchar *text;
  gtk_clist_get_text(clist,row,column,&text);
  gtk_entry_set_text(entry,text);
  clist_set_srow(clist,row);
}

/* clist has unselected a row */
void entry_clist_unselect_row (GtkEntry *entry,
			       gint row,
			       gint column,
			       GdkEventButton *event,
			       GtkCList *clist)
{
  clist_set_srow(clist,-1);
  gtk_entry_set_text(entry,"");
}

/* connected entry has changed */
void clist_entry_changed(GtkCList *clist,GtkEntry *entry)
{
  gchar *text;
  gint row;

  row=clist_get_srow(clist);
  if(row<0) return;
  
  text=gtk_editable_get_chars(GTK_EDITABLE(entry),0,-1);
  gtk_clist_set_text(clist,row,0,text);
  g_free(text);

  changed();
}

/* font picker used to choose a font */
void entry_font_set(GtkEntry *entry,
		    gchar *font_name,
		    GnomeFontPicker *picker)
{
  entry = GTK_ENTRY(entry);
  picker = GNOME_FONT_PICKER(picker);

  gtk_entry_set_text(entry,font_name);
}

/* delete current row, if any */
void clist_button_delete(GtkCList *clist,
			 gpointer user_data)
{
  gint row;

  row=clist_get_srow(clist);
  if(row<0) return;
  
  gtk_clist_remove(clist,row);

  changed();
}

/* insert a new row */
void clist_button_add(GtkCList *clist,
		      GtkButton *button)
{
  GtkEditable *editable;
  gint row;
  char *text="";

  editable=GTK_EDITABLE(gtk_object_get_user_data(GTK_OBJECT(button)));
  if(editable)
    text=gtk_editable_get_chars(editable,0,-1);

  row=gtk_clist_append(clist,&text);
  gtk_clist_select_row(clist,row,0);

  if(editable && text) 
    g_free(text);

  changed();
}

/* read a configuration file */
static gboolean
read_cfg_file(const gchar *fname)
{
  FILE *fin=fopen(fname,"r");
  if(fin==NULL) return FALSE;

#ifdef DEBUG_CONFIG
  g_message("gdkxft-capplet: reading %s",fname);
#endif

  /* parse config file */
  while(!feof(fin)) {
    GtkCList *clist;
    char buf[512],*s;
    char *text[1]={NULL};
    if(fgets(buf,512,fin)==NULL) break;
    /* eat comments and newlines */
    s=strchr(buf,'#');
    if(s) *s=0;
    s=strchr(buf,'\n');
    if(s) *s=0;
    /* decide which list it's in */
    s=buf;
    if(*s=='!') {clist=bad_font_clist;s++;}
    else if(*s=='&') {clist=bad_app_clist;s++;}
    else clist=good_font_clist;
    /* ignore blanks */
    if(!*s) continue;
    /* add to clist */
    text[0]=s;
    gtk_clist_append(clist,text);
  };

  /* clean up and return */
  fclose(fin);
  return TRUE;
}

/* output from a clist to a file */
static void
write_list(FILE *fout,GtkCList *clist,const gchar *prefix)
{
  gint row=0;
  gchar *text=NULL;
  while(gtk_clist_get_text(clist,row++,0,&text))
    fprintf(fout,"%s%s\n",prefix,text);
}

/* output config to a file */
static gboolean
write_cfg_file(const char *fname)
{
  FILE *fout=fopen(fname,"w");
  if(fout==NULL) return FALSE;

  fprintf(fout,"# generated by gdkxft-capplet\n");

  fprintf(fout,"\n# Fonts to render using core Xlib font support\n");
  write_list(fout,bad_font_clist,"!");

  fprintf(fout,"\n# Fonts to render using Xft (anti-aliased)\n");
  write_list(fout,good_font_clist,"");

  fprintf(fout,"\n# Ill-behaved applications\n");
  write_list(fout,bad_app_clist,"&");

  /* clean up and return */
  fclose(fout);
  return TRUE;
}

/* search for and read config files */
void
read_cfg(void)
{
  gchar *env_home,*cfg;

  /* wipe all the clists */
  gtk_clist_clear(good_font_clist);
  gtk_clist_clear(bad_font_clist);
  gtk_clist_clear(bad_app_clist);

  /* find a config file */
  env_home=getenv("HOME");
  cfg=g_strconcat(env_home?env_home:"","/.gdkxft",NULL);

  /* parse it */
  if(!access(cfg,R_OK)) 
    read_cfg_file(cfg);
  else if(!access(SYSCONFDIR "/gdkxft.conf",R_OK)) 
    read_cfg_file(SYSCONFDIR "/gdkxft.conf");
  else 
    read_cfg_file(DATADIR "/gdkxft.conf");

  /* clean up */
  g_free(cfg);
};

/* write personal config file */
void
write_cfg(void)
{
  gchar *env_home,*cfg;

  /* find config file */
  env_home=getenv("HOME");
  cfg=g_strconcat(env_home?env_home:"","/.gdkxft",NULL);

  /* parse it */
  write_cfg_file(cfg);

  /* clean up */
  g_free(cfg);
};

