/* gdkxft - Provide Xft font support in gdk
 * Copyright (C) 2001 Josh Parsons
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <regex.h>

#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xft/Xft.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkprivate.h>

#include <iconv.h>

#include "realfuncs.h"

/* uncomment to switch on debugging output */
/*#define DEBUG_INIT*/
/*#define DEBUG_CONFIG*/
/*#define DEBUG_EXTENTS*/
/*#define DEBUG_FONTS*/
/*#define DEBUG_DRAW*/

/* uncomment to force synchronous X protocol usage (for debugging) */
/* #define FORCE_SYNC */

/* uncomment to trap X errors (rather than crashing) - costly */
/* #define TRAP_DRAWING_ERRORS */

/* uncomment to disable anti-aliasing of multibyte fonts */
/* #define NO_MULTIBYTE_FONTS */

/**
 * Convert the String the from the original encoding to assigned encoding by iconv.
 * It will extract the encoding from the encoding come from "C" function search_font_encoding in this file.
 * But it is not completed yet. Since it will remove string start from the last "-" from the xlfd.
 * if the encoding name is iso8859-1, it will become iso8859. Since it is not
 * multi-byte font. it has no problem.
 * Also, it will remove string that start from ".". e.g. gb2312.1980-1 --> gb2312
 */
static int code_conversion(wchar_t* output, char* input, int in_buf_len, char *src_encoding) {
  iconv_t handle;
  int out_buf_len = (in_buf_len + 1) * sizeof(wchar_t);
  size_t result;
  size_t original = out_buf_len;
  char *value = NULL;

  if (in_buf_len == 0 || output == NULL) return -1;

  if (strstr(src_encoding, ".") != NULL) {
    int dot_index = 0;

    while (src_encoding[strlen(src_encoding) - dot_index - 1] != '.') {
      dot_index++;
    }

    value = g_malloc(strlen(src_encoding));
    strncpy(value, src_encoding, strlen(src_encoding) - dot_index - 1);
    value[strlen(src_encoding) - dot_index - 1] = 0;
  } else if (strstr(src_encoding, "-0") != NULL) {
    value = g_malloc(strlen(src_encoding));
    strncpy(value, src_encoding, strlen(src_encoding) - 2);
    value[strlen(src_encoding) - 2] = 0;
  }

  if (value != NULL) {
    handle = iconv_open("UTF-8", value);
  } else {
    char *cur_locale, *encode;
    int index = 0;

    // Extract Encoding Method from Current Locale.
    // If the encoding method cannot be found, assume it to be iso8859-1.

    cur_locale = getenv("LC_ALL");

    if (cur_locale == NULL) {
      handle = iconv_open("UTF-8", "ISO8859-1");
    } else {
      if (strstr(cur_locale, ".") == NULL) {
	handle = iconv_open("UTF-8", "ISO8859-1");
      } else {
	encode = g_malloc(strlen(cur_locale) + 10);
	index = strlen(cur_locale) - 1;
	while (cur_locale[index] != '.') {
	  index--;
	}
	strncpy(encode, cur_locale + index + 1, strlen(cur_locale) - index - 1);
	encode[strlen(cur_locale) - index - 1] = 0;
	handle = iconv_open("UTF-8", encode);
	g_free(encode);
      }
    }
  }

  if (value != NULL) {
    g_free(value);
  }

  if (handle != (iconv_t) (-1)) {
    result = iconv(handle, (char**) &input, &in_buf_len, (char**) &output, &out_buf_len);
    iconv_close(handle);
  } else {
    return -1;
  }

  return (original - out_buf_len);
}


/**
 * extract the font encoding from XLFD.
 * it will extract the encoding string that start from second last "-" character.
 * e.g. -arphic-ar pl mingti2l big5-medium-r-normal-*-*-140-*-*-m-*-big5-0 -> big5-0
 */
static char* search_font_encoding(const char* font) {
  int len = strlen(font);
  int last = len - 1;
  char *encoding = NULL;

  while (font[last] != '-') {
    last--;
  }

  last--;

  while (font[last] != '-') {
    last--;
  }

  encoding = g_malloc(len - last);
  strncpy(encoding, font + last + 1, len - last);
  encoding[len - last] = 0;

  return encoding;
}

/*** a hash table mapping gdk fonts to xft fonts ***/

static GHashTable *xftfont_hash = NULL;
static GHashTable *xftfont_encoding = NULL;

static XftFont *xftfont_hash_lookup(GdkFont *font)
{
  return g_hash_table_lookup(xftfont_hash, font);
}

static gboolean xftfont_hash_check(GdkFont *font)
{
  gpointer orig_key,value;
  return g_hash_table_lookup_extended(xftfont_hash,
				      font,
				      &orig_key,
				      &value);
}

static void xftfont_hash_insert(GdkFont *font,XftFont *xftfont)
{
  g_hash_table_insert(xftfont_hash,font,xftfont);
}

static void xftfont_hash_remove(GdkFont *font)
{
  XftFont *xftfont;

  if(!xftfont_hash_check(font)) return;

  xftfont=xftfont_hash_lookup(font);
  if(xftfont) XftFontClose(gdk_display,xftfont);

  g_hash_table_remove(xftfont_hash,font);
}


/*** a hash table mapping gdk GCs to Xlib clipping regions ***/

static GHashTable *gc_hash=NULL;
static Region gc_hash_lookup(GdkGC *gc)
{
  return g_hash_table_lookup(gc_hash,gc);
}
static gboolean gc_hash_check(GdkGC *gc)
{
  gpointer orig_key,value;
  return g_hash_table_lookup_extended(gc_hash,
				      gc,
				      &orig_key,
				      &value);
}
static void gc_hash_insert(GdkGC *gc,Region xregion)
{
  g_hash_table_insert(gc_hash,gc,xregion);
}
static void gc_hash_remove(GdkGC *gc)
{
  Region xregion;

  if(!gc_hash_check(gc)) return;

  xregion = gc_hash_lookup(gc);
  if(xregion!=None) XDestroyRegion(xregion);

  g_hash_table_remove(gc_hash,gc);
}


/*** support functions ***/

/* TRUE if we should never try to use Xft fonts */
static gboolean gdkxft_disabled = FALSE;
static gboolean mozilla_app = FALSE;

/* read a configuration file */
static gboolean
read_cfg_file(const gchar *fname,GString *xfre,GString *ftre,GString *apre)
{
  FILE *fin=fopen(fname,"r");
  if(fin==NULL) return FALSE;

#ifdef DEBUG_CONFIG
  g_message(PACKAGE ": reading %s",fname);
#endif

  /* parse config file */
  while(!feof(fin)) {
    char buf[512],*s;
    GString *re;
    if(fgets(buf,512,fin)==NULL) break;
    /* eat comments and newlines */
    s=strchr(buf,'#');
    if(s) *s=0;
    s=strchr(buf,'\n');
    if(s) *s=0;
    /* decide which list it's in */
    s=buf;
    if(*s=='!') {re=xfre;s++;}
    else if(*s=='&') {re=apre;s++;}
    else re=ftre;
    /* ignore blanks */
    if(!*s) continue;
    /* convert font spec into a eregexp */
    if(*(re->str)) g_string_append_c(re,'|');
    g_string_append_c(re,'(');
    while(*s) {
      switch(*s) {
      case('*'): g_string_append(re,".*"); break;
      case('.'): g_string_append(re,"[.]"); break;
      default: g_string_append_c(re,*s); break;
      };
      s++;
    };
    g_string_append_c(re,')');
  };

  /* clean up and return */
  fclose(fin);
  return TRUE;
}

/* regexes for matching font names */
static regex_t xfrt[1],ftrt[1],aprt[1];

/* search for and read config files */
static void
read_cfg(void)
{
  gchar *env_home,*cfg;
  GString *ftre;
  GString *xfre;
  GString *apre;

  ftre=g_string_sized_new(1);
  xfre=g_string_sized_new(1);
  apre=g_string_sized_new(1);

  /* find a config file */
  env_home=getenv("HOME");
  cfg=g_strconcat(env_home?env_home:"","/.gdkxft",NULL);

  /* parse it */
  if(!access(cfg,R_OK)) 
    read_cfg_file(cfg,xfre,ftre,apre);
  else if(!access(SYSCONFDIR "/gdkxft.conf",R_OK)) 
    read_cfg_file(SYSCONFDIR "/gdkxft.conf",xfre,ftre,apre);
  else 
    read_cfg_file(DATADIR "/gdkxft.conf",xfre,ftre,apre);

  /* compile regexps */
  g_string_append_c(xfre,'$');
  g_string_append_c(ftre,'$');
  g_string_append_c(apre,'$');
  g_string_prepend_c(xfre,'^');
  g_string_prepend_c(ftre,'^');
  g_string_prepend_c(apre,'^');
#ifdef DEBUG_CONFIG
  g_message(PACKAGE ": XF-regexp='%s'",xfre->str);
  g_message(PACKAGE ": FT-regexp='%s'",ftre->str);
  g_message(PACKAGE ": app-regexp='%s'",apre->str);
#endif
  regcomp(xfrt,xfre->str,REG_EXTENDED|REG_NOSUB);
  regcomp(ftrt,ftre->str,REG_EXTENDED|REG_NOSUB);
  regcomp(aprt,apre->str,REG_EXTENDED|REG_NOSUB);

  /* tidy up */
  g_free(cfg);
  g_string_free(xfre,1);
  g_string_free(ftre,1);
  g_string_free(apre,1);
};

/* test a font name to see if we support xft for it */
static gboolean
want_xft_for(const gchar *font_name)
{
  if(!regexec(ftrt,font_name,0,NULL,0)) return TRUE;
  if(!regexec(xfrt,font_name,0,NULL,0)) return FALSE;
  return TRUE;
}


/*** replacements for gdk functions ***/

MY_GDK_INIT_CHECK
{
  const gchar *app_name = NULL;

#ifdef DEBUG_INIT
  g_message(PACKAGE ": " VERSION);
#endif

  /* check gtk version */
  if(gtk_major_version!=1 || gtk_minor_version!=2)
    g_error("Gdkxft requires gtk+ version 1.2 - this appears to be %d.%d",
	    gtk_major_version,gtk_minor_version);

  /* setup hash tables */
  xftfont_hash = g_hash_table_new(g_direct_hash, g_direct_equal);
  xftfont_encoding = g_hash_table_new(g_direct_hash, g_direct_equal);
  gc_hash = g_hash_table_new(g_direct_hash, g_direct_equal);

  /* read config */
  read_cfg();

  /* check application name */
  if (argc!=NULL && (*argc)>0) app_name = (*argv)[0];

  if (strstr(app_name, "mozilla") != NULL) {
    mozilla_app = TRUE;
  }

  if (app_name==NULL || !regexec(aprt,app_name,0,NULL,0)) {
    gdkxft_disabled=TRUE;
#if 1 /*def DEBUG_INIT*/
    g_message(PACKAGE ": bad app '%s' - disabling gdkxft",app_name);
#endif
  }
  else {
    gdkxft_disabled=FALSE;
#ifdef DEBUG_INIT
    g_message(PACKAGE ": good app '%s'",app_name);
#endif
  };

#ifdef FORCE_SYNC /* for silly apps that offer no way to pass args to gdk */
  {
    static gint argc=2;
    static gchar *_argv[128]={"gdkxft","--sync"};
    static gchar **argv=_argv;;
    return real_gdk_init_check(&argc,&argv);
  };
#else
  return REAL_GDK_INIT_CHECK;
#endif
}

/*** replacements for gdkgc functions ***/

/* 
   This is all necessary because Xft has its own clipping mechanism,
   which is independent of the regular X graphics contexts.  Gdk will
   be using graphics contexts to clip text, so we need to track which
   GC has which clipping rectangle so we can tell Xft how to clip the
   text.  This is a kludge and there are cases for which it will not
   work.  (For example, if someone tries to clip text to a
   non-rectangular shape). 
*/

MY_GDK_GC_UNREF
{
  GdkGCPrivate *private = (GdkGCPrivate*) gc;
  
  g_return_if_fail (gc != NULL);
  g_return_if_fail (private->ref_count > 0);

  /* clean up region if GC about to go away */
  if(private->ref_count==1)
    gc_hash_remove(gc);

  REAL_GDK_GC_UNREF;
}

MY_GDK_GC_SET_CLIP_MASK
{
  if(gc) gc_hash_remove(gc);
  REAL_GDK_GC_SET_CLIP_MASK;
}

MY_GDK_GC_SET_CLIP_RECTANGLE
{
  GdkGCPrivate *private;

  g_return_if_fail (gc != NULL);

  private = (GdkGCPrivate*) gc;

  gc_hash_remove(gc);

  if (rectangle)
    {
      XRectangle xrectangle;
      Region xregion = XCreateRegion();

      xrectangle.x = rectangle->x; 
      xrectangle.y = rectangle->y;
      xrectangle.width = rectangle->width;
      xrectangle.height = rectangle->height;
      
      XUnionRectWithRegion(&xrectangle,
                           xregion,
                           xregion);

      gc_hash_insert(gc,xregion);
    }

  REAL_GDK_GC_SET_CLIP_RECTANGLE;
}

MY_GDK_GC_SET_CLIP_REGION
{
  GdkGCPrivate *private;

  g_return_if_fail (gc != NULL);

  private = (GdkGCPrivate*) gc;

  gc_hash_remove(gc);

  if (region)
    {
      GdkRegionPrivate *region_private = (GdkRegionPrivate*) region;
      Region xregion = XCreateRegion();
      XUnionRegion(region_private->xregion,
                   xregion,
                   xregion);
      gc_hash_insert(gc,xregion);
    }

  REAL_GDK_GC_SET_CLIP_REGION;
}


/*** replacements for gdkdraw functions ***/

MY_GDK_DRAW_STRING
{
  g_return_if_fail(font!=NULL);
  g_return_if_fail(string!=NULL);

  /* fall back if not xftfont */
  if(xftfont_hash_lookup(font)==NULL)
    { REAL_GDK_DRAW_STRING; return; };

  /* otherwise, pipe it through text version */
  gdk_draw_text(drawable,font,gc,x,y,string,strlen(string));
}

MY_GDK_DRAW_TEXT
{
  GdkWindowPrivate *drawable_private; 
  GdkFontPrivate *font_private;
  GdkGCPrivate *gc_private;       
  XftDraw *xftdraw;
  XftColor xftcolor;
  XColor xcolor;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGCValues values;
  GdkColormap *gdkcmap; 
  GdkColormapPrivate *gdkcmap_private;
  Colormap colmap;
  Region xregion;
  wchar_t *_text;
  int len;
  gchar *_text1;

  Window d_root;
  unsigned int d_x,d_y,d_border_width;
  unsigned int d_width,d_height,d_depth;

  /* some sanity checks */
  g_return_if_fail(drawable!=NULL);
  g_return_if_fail(font!=NULL);
  g_return_if_fail(gc!=NULL);
  g_return_if_fail(text!=NULL);

  if (text_length == 0) return;

  drawable_private = (GdkWindowPrivate *) drawable;
  if(drawable_private->destroyed) return;
  font_private = (GdkFontPrivate *) font;
  gc_private = (GdkGCPrivate *) gc;
  xfont = font_private -> xfont;

  /* fall back if not xftfont */
  xftfont = xftfont_hash_lookup(font);
  if (xftfont == NULL) { REAL_GDK_DRAW_TEXT; return; };

  /* fall back if drawable is a bitmap */
  XGetGeometry(drawable_private->xdisplay,
	       drawable_private->xwindow,
	       &d_root,
	       &d_y,&d_x,
	       &d_width,&d_height,
	       &d_border_width,
	       &d_depth);
  if(d_depth<=1) { REAL_GDK_DRAW_TEXT; return; };

#ifdef TRAP_DRAWING_ERRORS
  gdk_error_trap_push();
#endif

  /* sort out the drawing colour */
  XGetGCValues(drawable_private->xdisplay,
	       gc_private->xgc,
	       GCForeground | GCBackground,
	       &values);
       
  memset(&xcolor,0,sizeof(xcolor));
  xcolor.pixel = values.foreground;
       
  if(drawable_private->colormap != NULL) {
    gdkcmap = drawable_private->colormap;
    gdkcmap_private = (GdkColormapPrivate *) gdkcmap;
    colmap = gdkcmap_private->xcolormap;
  }
  else { 
    colmap = DefaultColormap(gc_private->xdisplay,
			     DefaultScreen(gc_private->xdisplay) );
  }
       
  XQueryColor(drawable_private->xdisplay,
	      colmap, 
	      &xcolor);

  /* ready to draw */
  xftdraw = XftDrawCreate(gc_private->xdisplay,
			  (Drawable) drawable_private->xwindow,
			  DefaultVisual(gc_private->xdisplay,
					DefaultScreen(gc_private->xdisplay)),
			  DefaultColormap(gc_private->xdisplay,
					  DefaultScreen(gc_private->xdisplay))
			  );

  if(xftdraw==NULL) {
    g_warning(PACKAGE ": could not create an XftDraw");
    REAL_GDK_DRAW_TEXT; 
    return;
  };

  xftcolor.color.red = xcolor.red;
  xftcolor.color.green = xcolor.green;
  xftcolor.color.blue = xcolor.blue;
  xftcolor.color.alpha =  0xffff;
  xftcolor.pixel = values.foreground;
  
  /* set up clipping */
  xregion = gc_hash_lookup(gc);
  if (xregion != None)
    XftDrawSetClip(xftdraw,xregion);

  /* do it */
  if ((xfont->min_byte1 == 0) && (xfont->max_byte1 == 0))
    XftDrawString8(xftdraw, 
		   &xftcolor, 
		   xftfont, 
		   x, 
		   y, 
		   (XftChar8 *) text,
		   text_length);
  else {
    int mbs = 0;
    char *encoding = g_hash_table_lookup(xftfont_encoding, font);

    _text1 = g_malloc(text_length + 10);
    strncpy(_text1, text, text_length);
    _text1[text_length] = 0;

    if (mozilla_app && (!(xfont->max_byte1 & 0x80))) {
      int counter = 0;

      for (counter = 0; counter < text_length; counter++) {
	_text1[counter] |= 0x80;
      }
    }

    _text = (wchar_t *) g_malloc((text_length + 1) * sizeof(wchar_t));

    len = -1;

    if (encoding != NULL) {
      len = code_conversion(_text, _text1, text_length, encoding);
      mbs = 0;
    }

    if (len < 0) {
      len = mbstowcs(_text, _text1, text_length);
      mbs = -1;
    }

    g_free(_text1);

    if (len > 0) {
      if (mbs) {
	XftDrawString32(xftdraw, 
			&xftcolor, 
			xftfont, 
			x, 
			y, 
			(XftChar32 *) _text,
			len);
      } else {
	XftDrawStringUtf8(xftdraw, 
			  &xftcolor, 
			  xftfont, 
			  x, 
			  y, 
			  (XftChar8 *) _text,
			  len);	
      }
    } else {
      XftDrawString16(xftdraw,
		      &xftcolor,
		      xftfont,
		      x,
		      y,
		      (XftChar16 *) text,
		      text_length / 2);
    }
    g_free(_text);
  }

  /* tidy up */
  XftDrawDestroy(xftdraw);

#ifdef TRAP_DRAWING_ERRORS
  gdk_flush();
  if(gdk_error_trap_pop()) {
    static gboolean warned=0;
    if(!warned)
      g_warning(PACKAGE ": X error. Falling back to core fonts.");
    warned=1;
    REAL_GDK_DRAW_TEXT;
  }
#endif
}

MY_GDK_DRAW_TEXT_WC
{
  GdkWindowPrivate *drawable_private; 
  GdkFontPrivate *font_private;
  GdkGCPrivate *gc_private;       
  XftDraw *xftdraw;
  XftColor xftcolor;
  XColor xcolor;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGCValues values;
  GdkColormap *gdkcmap; 
  GdkColormapPrivate *gdkcmap_private;
  Colormap colmap;
  Region xregion;

  Window d_root;
  unsigned int d_x,d_y,d_border_width;
  unsigned int d_width,d_height,d_depth;

  /* some sanity checks */
  g_return_if_fail(drawable!=NULL);
  g_return_if_fail(font!=NULL);
  g_return_if_fail(gc!=NULL);
  g_return_if_fail(text!=NULL);

  drawable_private = (GdkWindowPrivate *) drawable;
  if(drawable_private->destroyed) return;
  font_private = (GdkFontPrivate *) font;
  gc_private = (GdkGCPrivate *) gc;
  xfont = font_private -> xfont;

  /* fall back if not xftfont */
  xftfont = xftfont_hash_lookup(font);
  if (xftfont == NULL) { REAL_GDK_DRAW_TEXT_WC; return; };

  /* fall back if drawable is a bitmap */
  XGetGeometry(drawable_private->xdisplay,
	       drawable_private->xwindow,
	       &d_root,
	       &d_y,&d_x,
	       &d_width,&d_height,
	       &d_border_width,
	       &d_depth);
  if(d_depth<=1) { REAL_GDK_DRAW_TEXT_WC; return; };

#ifdef TRAP_DRAWING_ERRORS
  gdk_error_trap_push();
#endif

  /* sort out the drawing colour */
  XGetGCValues(drawable_private->xdisplay,
	       gc_private->xgc,
	       GCForeground | GCBackground,
	       &values);
       
  memset(&xcolor,0,sizeof(xcolor));
  xcolor.pixel = values.foreground;
       
  if(drawable_private->colormap != NULL) {
    gdkcmap = drawable_private->colormap;
    gdkcmap_private = (GdkColormapPrivate *) gdkcmap;
    colmap = gdkcmap_private->xcolormap;
  }
  else { 
    colmap = DefaultColormap(gc_private->xdisplay,
			     DefaultScreen(gc_private->xdisplay) );
  }
       
  XQueryColor(drawable_private->xdisplay,
	      colmap, 
	      &xcolor);

  /* ready to draw */
  xftdraw = XftDrawCreate(gc_private->xdisplay,
			  (Drawable) drawable_private->xwindow,
			  DefaultVisual(gc_private->xdisplay,
					DefaultScreen(gc_private->xdisplay)),
			  DefaultColormap(gc_private->xdisplay,
					  DefaultScreen(gc_private->xdisplay))
			  );

  if(xftdraw==NULL) {
    g_warning(PACKAGE ": could not create an XftDraw");
    REAL_GDK_DRAW_TEXT_WC; 
    return;
  };

  xftcolor.color.red = xcolor.red;
  xftcolor.color.green = xcolor.green;
  xftcolor.color.blue = xcolor.blue;
  xftcolor.color.alpha =  0xffff;
  xftcolor.pixel = values.foreground;
  
  /* set up clipping */
  xregion=gc_hash_lookup(gc);
  if(xregion!=None)
    XftDrawSetClip(xftdraw,xregion);
 
  /* do it */
  XftDrawString32(xftdraw, 
		  &xftcolor, 
		  xftfont, 
		  x, 
		  y, 
		  (XftChar32 *) text,
		  text_length);
  /* tidy up */
  XftDrawDestroy(xftdraw);

#ifdef TRAP_DRAWING_ERRORS
  gdk_flush();
  if(gdk_error_trap_pop()) {
    static gboolean warned=0;
    if(!warned)
      g_warning(PACKAGE ": X error. Falling back to core fonts.");
    warned=1;
    REAL_GDK_DRAW_TEXT_WC;
  }
#endif
}

/*** replacements for gdkfont functions ***/

MY_GDK_TEXT_EXTENTS
{
  GdkFontPrivate *private;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGlyphInfo glyph_info;
  wchar_t *_text;
  int len;
  gchar *_text1;

  g_return_if_fail (font != NULL);
  g_return_if_fail (text != NULL);

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  /* fall back if not xftfont */
  xftfont=xftfont_hash_lookup(font);
  if(xftfont==NULL) {
    REAL_GDK_TEXT_EXTENTS;
    return;
  };

#ifdef DEBUG_EXTENTS
  {
    gchar *str = g_strndup(text,text_length);
    g_message(PACKAGE ": gdk_text_extents('%s')",str);
    g_free(str);
  };
#endif

  /* find out extents */
  if((xfont->min_byte1 == 0) && (xfont->max_byte1 == 0))
    XftTextExtents8(gdk_display,
		    xftfont,
		    (XftChar8 *)text,
		    text_length,
		    &glyph_info);
  else {
    int mbs = 0;
    char *encoding = g_hash_table_lookup(xftfont_encoding, font);

    _text1 = g_malloc(text_length + 10);
    strncpy(_text1, text, text_length);
    _text1[text_length] = 0;

    if (mozilla_app && (!(xfont->max_byte1 & 0x80))) {
      int counter = 0;

      for (counter = 0; counter < text_length; counter++) {
	_text1[counter] |= 0x80;
      }
    }

    _text = (wchar_t *) g_malloc((text_length + 1) * sizeof(wchar_t));

    len = -1;

    if (encoding != NULL) {
      len = code_conversion(_text, _text1, text_length, encoding);
      mbs = 0;
    }

    if (len < 0) {
      len = mbstowcs(_text, _text1, text_length);
      mbs = -1;
    }

    g_free(_text1);

    if (len > 0) {
      if (mbs) {
	XftTextExtents32(gdk_display,
			 xftfont,
			 (XftChar32 *)_text,
			 len,
			 &glyph_info);
      } else {
	XftTextExtentsUtf8(gdk_display,
			   xftfont,
			   (XftChar8 *)_text,
			   len,
			   &glyph_info);
      }
    } else {
      XftTextExtents16(gdk_display,
		      xftfont,
		      (XftChar16 *) text,
		      text_length / 2,
		      &glyph_info);
    }

    g_free(_text);
  }
  
#ifdef DEBUG_EXTENTS
  REAL_GDK_TEXT_EXTENTS;
  
  g_message(PACKAGE ": old lb=%d rb=%d w=%d a=%d d=%d",
	    lbearing?*lbearing:-1234,
	    rbearing?*rbearing:-1234,
	    width?*width:-1234,
	    ascent?*ascent:-1234,
	    descent?*descent:-1234);
#endif

  if (lbearing)
    *lbearing = -glyph_info.x;
  if (rbearing)
    *rbearing = glyph_info.width;
  if (width)
    *width = glyph_info.xOff;
  if (ascent)
    *ascent = glyph_info.y;
  if (descent)
    *descent = glyph_info.height - glyph_info.y;

#ifdef DEBUG_EXTENTS
  g_message("new lb=%d rb=%d w=%d a=%d d=%d",
	    lbearing?*lbearing:-1234,
	    rbearing?*rbearing:-1234,
	    width?*width:-1234,
	    ascent?*ascent:-1234,
	    descent?*descent:-1234);
#endif

}

MY_GDK_TEXT_EXTENTS_WC
{
  GdkFontPrivate *private;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGlyphInfo glyph_info;

  g_return_if_fail (font != NULL);
  g_return_if_fail (text != NULL);

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  xftfont=xftfont_hash_lookup(font);
  if(xftfont==NULL) {
    REAL_GDK_TEXT_EXTENTS_WC;
    return;
  };

#ifdef DEBUG_EXTENTS
  {
    gchar *str = g_strndup(text,text_length);
    g_message(PACKAGE ": gdk_text_extents('%s')",str);
    g_free(str);
  };
#endif

  XftTextExtents32(gdk_display,
		   xftfont,
		   (XftChar32 *)text,
		   text_length,
		   &glyph_info);
  
#ifdef DEBUG_EXTENTS
  REAL_GDK_TEXT_EXTENTS_WC;
  
  g_message(PACKAGE ": old lb=%d rb=%d w=%d a=%d d=%d",
	    lbearing?*lbearing:-1234,
	    rbearing?*rbearing:-1234,
	    width?*width:-1234,
	    ascent?*ascent:-1234,
	    descent?*descent:-1234);
#endif

  if (lbearing)
    *lbearing = -glyph_info.x;
  if (rbearing)
    *rbearing = glyph_info.width;
  if (width)
    *width = glyph_info.xOff;
  if (ascent)
    *ascent = glyph_info.y;
  if (descent)
    *descent = glyph_info.height - glyph_info.y;

#ifdef DEBUG_EXTENTS
  g_message("new lb=%d rb=%d w=%d a=%d d=%d",
	    lbearing?*lbearing:-1234,
	    rbearing?*rbearing:-1234,
	    width?*width:-1234,
	    ascent?*ascent:-1234,
	    descent?*descent:-1234);
#endif

}


MY_GDK_CHAR_WIDTH
{
  g_return_val_if_fail(font!=NULL,-1);

  /* fall back if not xftfont */
  if(xftfont_hash_lookup(font)==NULL)
    return REAL_GDK_CHAR_WIDTH;

  /* otherwise, pipe it through text version */
  return gdk_text_width(font,&character,1);
}

MY_GDK_STRING_WIDTH
{
  g_return_val_if_fail(font!=NULL,-1);
  g_return_val_if_fail(string!=NULL,-1);

  /* fall back if not xftfont */
  if(xftfont_hash_lookup(font)==NULL)
    return REAL_GDK_STRING_WIDTH;

  /* otherwise, pipe it through text version */
  return gdk_text_width(font,string,strlen(string));
}

/* gdk's "width" == Xrender's "xOff" == 
   the difference between the origin of a block of text 
   and the origin of an imaginary character directly following it */
MY_GDK_TEXT_WIDTH
{
  GdkFontPrivate *private;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGlyphInfo glyph_info;
  wchar_t *_text;
  int len;
  gchar *_text1;

  g_return_val_if_fail(font!=NULL,-1);
  g_return_val_if_fail(text!=NULL,-1);

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  /* fall back if not xftfont */
  xftfont=xftfont_hash_lookup(font);
  if(xftfont==NULL) return REAL_GDK_TEXT_WIDTH;

  /* find out extents */
  if((xfont->min_byte1 == 0) && (xfont->max_byte1 == 0))
    XftTextExtents8(gdk_display,
		    xftfont,
		    (XftChar8 *)text,
		    text_length,
		    &glyph_info);
  else {
    int mbs = 0;
    char *encoding = g_hash_table_lookup(xftfont_encoding, font);

    _text1 = g_malloc(text_length + 10);
    strncpy(_text1, text, text_length);
    _text1[text_length] = 0;

    if (mozilla_app && (!(xfont->max_byte1 & 0x80))) {
      int counter = 0;

      for (counter = 0; counter < text_length; counter++) {
	_text1[counter] |= 0x80;
      }
    }

    _text = (wchar_t *) g_malloc((text_length + 1) * sizeof(wchar_t));

    len = -1;

    if (encoding != NULL) {
      len = code_conversion(_text, _text1, text_length, encoding);
      mbs = 0;
    }

    if (len < 0) {
      len = mbstowcs(_text, _text1, text_length);
      mbs = -1;
    }

    g_free(_text1);

    if (len > 0) {
      if (mbs) {
	XftTextExtents32(gdk_display,
			 xftfont,
			 (XftChar32 *)_text,
			 len,
			 &glyph_info);
      } else {
	XftTextExtentsUtf8(gdk_display,
			   xftfont,
			   (XftChar8 *)_text,
			   len,
			   &glyph_info);
      }
    } else {
      XftTextExtents16(gdk_display,
		      xftfont,
		      (XftChar16 *)text,
		      text_length / 2,
		      &glyph_info);
    }
    g_free(_text);
  }
  
#ifdef DEBUG_EXTENTS
  {
    gchar *str = g_strndup(text,text_length);
    g_message(PACKAGE ": gdk_text_width('%s') -> %d (%d)",
	      str,
	      glyph_info.xOff,
	      REAL_GDK_TEXT_WIDTH);
    g_free(str);
  };
#endif

  return glyph_info.xOff;
}

MY_GDK_TEXT_WIDTH_WC
{
  GdkFontPrivate *private;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGlyphInfo glyph_info;

  g_return_val_if_fail(font!=NULL,-1);
  g_return_val_if_fail(text!=NULL,-1);

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  xftfont=xftfont_hash_lookup(font);
  if(xftfont==NULL) return REAL_GDK_TEXT_WIDTH_WC;

#ifdef DEBUG_EXTENTS
  {
    gchar *str = g_strndup(text,text_length);
    g_message(PACKAGE ": gdk_text_width('%s')",str);
    g_free(str);
  };
#endif
  
  XftTextExtents32(gdk_display,
		   xftfont,
		   (XftChar32 *) text,
		   text_length,
		   &glyph_info);
  
		   //  return glyph_info.xOff;
  return glyph_info.xOff;
}

/* gdk's "measure" ==
   the difference between the left most pixel and right most
   pixel of a block of text */
MY_GDK_TEXT_MEASURE { 
  GdkFontPrivate *private;
  XFontStruct *xfont;
  XftFont *xftfont; 
  XGlyphInfo glyph_info;
  wchar_t *_text;
  int len;
  gchar *_text1;

  g_return_val_if_fail(font!=NULL,-1);
  g_return_val_if_fail(text!=NULL,-1);

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  /* fall back if not xftfont */
  xftfont=xftfont_hash_lookup(font);
  if(xftfont==NULL) return REAL_GDK_TEXT_MEASURE;

  /* find out extents */
  if((xfont->min_byte1 == 0) && (xfont->max_byte1 == 0))
    XftTextExtents8(gdk_display,
		    xftfont,
		    (XftChar8 *)text,
		    text_length,
		    &glyph_info);
  else {
    int mbs = 0;
    char *encoding = g_hash_table_lookup(xftfont_encoding, font);

    _text1 = g_malloc(text_length + 10);
    strncpy(_text1, text, text_length);
    _text1[text_length] = 0;

    if (mozilla_app && (!(xfont->max_byte1 & 0x80))) {
      int counter = 0;

      for (counter = 0; counter < text_length; counter++) {
	_text1[counter] |= 0x80;
      }
    }

    _text = (wchar_t *) g_malloc((text_length + 1) * sizeof(wchar_t));

    len = -1;

    if (encoding != NULL) {
      len = code_conversion(_text, _text1, text_length, encoding);
      mbs = 0;
    }

    if (len < 0) {
      len = mbstowcs(_text, _text1, text_length);
      mbs = -1;
    }

    g_free(_text1);

    if (len > 0) {
      if (mbs) {
	XftTextExtents32(gdk_display,
			 xftfont,
			 (XftChar32 *)_text,
			 len,
			 &glyph_info);
      } else {
	XftTextExtentsUtf8(gdk_display,
			   xftfont,
			   (XftChar8 *)_text,
			   len,
			   &glyph_info);
      }
    } else {
      XftTextExtents16(gdk_display,
		      xftfont,
		      (XftChar16 *) text,
		      text_length / 2,
		      &glyph_info);
    }
    g_free(_text);
  }

#ifdef DEBUG_EXTENTS
  {
    gchar *str = g_strndup(text,text_length);
    g_message(PACKAGE ": gdk_text_measure('%s') -> %d (%d) x=%d",
	      str,
	      glyph_info.width,
	      REAL_GDK_TEXT_MEASURE,
	      glyph_info.x);
    g_free(str);
  };
#endif

  return glyph_info.width-glyph_info.x;
}

MY_GDK_STRING_HEIGHT
{
  g_return_val_if_fail(font!=NULL,-1);
  g_return_val_if_fail(string!=NULL,-1);

  /* fall back if not xftfont */
  if(xftfont_hash_lookup(font)==NULL)
    return REAL_GDK_STRING_HEIGHT;

  /* otherwise, pipe it through text version */
  return gdk_text_height(font,string,strlen(string));
}

MY_GDK_TEXT_HEIGHT
{
  GdkFontPrivate *private;
  XFontStruct *xfont;
  XftFont *xftfont;
  XGlyphInfo glyph_info;
  wchar_t *_text;
  int len;
  gchar *_text1;

  g_return_val_if_fail(font!=NULL,-1);
  g_return_val_if_fail(text!=NULL,-1);

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  /* fall back if not xftfont */
  xftfont=xftfont_hash_lookup(font);
  if(xftfont==NULL) return REAL_GDK_TEXT_HEIGHT;

  /* find out height */
  if((xfont->min_byte1 == 0) && (xfont->max_byte1 == 0))
    XftTextExtents8(gdk_display,
		    xftfont,
		    (XftChar8 *)text,
		    text_length,
		    &glyph_info);
  else {
    int mbs = 0;
    char *encoding = g_hash_table_lookup(xftfont_encoding, font);

    _text1 = g_malloc(text_length + 10);
    strncpy(_text1, text, text_length);
    _text1[text_length] = 0;

    if (mozilla_app && (!(xfont->max_byte1 & 0x80))) {
      int counter = 0;

      for (counter = 0; counter < text_length; counter++) {
	_text1[counter] |= 0x80;
      }
    }

    _text = (wchar_t *) g_malloc((text_length + 1) * sizeof(wchar_t));

    len = -1;

    if (encoding != NULL) {
      len = code_conversion(_text, _text1, text_length, encoding);
      mbs = 0;
    }

    if (len < 0) {
      len = mbstowcs(_text, _text1, text_length);
      mbs = -1;
    }

    g_free(_text1);

    if (len > 0) {
      if (mbs) {
	XftTextExtents32(gdk_display,
			 xftfont,
			 (XftChar32 *)_text,
			 len,
			 &glyph_info);
      } else {
	XftTextExtentsUtf8(gdk_display,
			   xftfont,
			   (XftChar8 *)_text,
			   len,
			   &glyph_info);
      }
    } else {
      XftTextExtents16(gdk_display,
		      xftfont,
		      (XftChar16 *) text,
		      text_length / 2,
		      &glyph_info);
    }
    g_free(_text);
  }

  return glyph_info.height;
}

MY_GDK_FONT_UNREF
{
  GdkFontPrivate *private = (GdkFontPrivate*) font;
  
  g_return_if_fail (font != NULL);
  g_return_if_fail (private->ref_count > 0);

  /* clean up xft font if about to go away */
  if(private->ref_count==1) {
#ifdef DEBUG_FONTS
    g_message(PACKAGE ": removing font %p",font);
#endif
    xftfont_hash_remove(font);
  };

  REAL_GDK_FONT_UNREF;
}

MY_GDK_FONT_LOAD
{
  XftFont *xftfont=NULL;
  GdkFont *font=REAL_GDK_FONT_LOAD;

  GdkFontPrivate *private;
  XFontStruct *xfont;
  gchar *name;

  /* nothing to do if real gdk_font_load failed */
  if (font == NULL) return font;

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  /* nothing to do if we are keeping a low profile */
  if (gdkxft_disabled) return font;

  /* nothing to do if this font should be handled by X */
  if (!want_xft_for(font_name)) return font;

  /* nothing to do if font already loaded */
  if (xftfont_hash_check(font))
    return font;

  /**
   * If Multi-Byte Fonts, force the encoding to iso10646-1 (Unicode Standard)
   */
  if ((xfont->min_byte1 != 0) || (xfont->max_byte1 != 0)) {
    char *encoding = search_font_encoding(font_name);

    if (strstr(font_name, encoding) != NULL) {
      name = g_malloc(strlen(font_name) - strlen(encoding) + 20);
      
      strncpy(name, font_name, strlen(font_name) - (strlen(encoding)));
      
      name[strlen(font_name) - strlen(encoding)] = 0;
      
      name = strcat(name, "iso10646-1");
      
      xftfont = XftFontOpenXlfd(gdk_display,
				DefaultScreen(gdk_display),
				name);

      /**
       * Store the Encoding from XLFD into Hashtable.
       * Use it to convert the String into UTF-8 Encoded String.
       */
      g_hash_table_insert(xftfont_encoding, font, encoding);

      g_free(name);
    }
  }
    
  /**
   * If No Font is loaded, load the original font.
   */
  if (xftfont == NULL) {
    xftfont = XftFontOpenXlfd(gdk_display,
			      DefaultScreen(gdk_display),
			      font_name);
  }

#ifdef DEBUG_FONTS
  g_message(PACKAGE ": loading '%s' -> %p",font_name,font);
  if ((xfont->min_byte1 != 0) || (xfont->max_byte1 != 0))
    g_message(PACKAGE ": '%s' is a 16 bit font",font_name);
#endif

  xftfont_hash_insert(font,xftfont);

  return font;
}

MY_GDK_FONTSET_LOAD
{
  XftFont *xftfont=NULL;
  GdkFont *font = REAL_GDK_FONTSET_LOAD;

  GdkFontPrivate *private;
  XFontStruct *xfont;
  gchar *name;

  /* nothing to do if real gdk_font_load failed */
  if (font == NULL) return font;

  private = (GdkFontPrivate*) font;
  xfont = private->xfont;

  /* nothing to do if we are keeping a low profile */
  if (gdkxft_disabled) return font;

  /* nothing to do if this font should be handled by X */
  if (!want_xft_for(fontset_name)) return font;

  /* nothing to do if font already loaded */
  if (xftfont_hash_check(font))
    return font;

  if ((xfont->min_byte1 != 0) || (xfont->max_byte1 != 0)) {
    char *encoding = search_font_encoding(fontset_name);

    if (strstr(fontset_name, encoding) != NULL) {
      name = g_malloc(strlen(fontset_name) - strlen(encoding) + 20);
      
      strncpy(name, fontset_name, strlen(fontset_name) - (strlen(encoding)));
      
      name[strlen(fontset_name) - strlen(encoding)] = 0;
      
      name = strcat(name, "iso10646-1");
      
      xftfont = XftFontOpenXlfd(gdk_display,
				DefaultScreen(gdk_display),
				name);

      g_hash_table_insert(xftfont_encoding, font, encoding);
      
      g_free(name);
    }
  }

  if (xftfont == NULL) {
    xftfont = XftFontOpenXlfd(gdk_display,
			      DefaultScreen(gdk_display),
			      fontset_name);
  }

#ifdef DEBUG_FONTS
  g_message(PACKAGE ": loading '%s' -> %p",fontset_name,font);
  if ((xfont->min_byte1 != 0) || (xfont->max_byte1 != 0))
    g_message(PACKAGE ": '%s' is a 16 bit font",fontset_name);
#endif

  xftfont_hash_insert(font,xftfont);

  return font;
}
