/* 
   $RCSfile: gcdparser.c,v $

   This file is part of:
   
       GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   $Id: gcdparser.c,v 1.43 2001/08/24 19:36:50 ntiffin Exp $
   
*/

/** \file gcdparser.c
 *  \brief This contains functions used inside the GCD parser.
 */

#include <glib.h>
#include <assert.h>

#include "config.h"
#include "gcdparser.h"
#include "classdata.h"
#include "lparser.h"

int yyerrorcount = 0;
int yywarncount = 0;

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
struct _odl_tree *
alloc_odl_tree (void)
{
  struct _odl_tree *t = g_new0 (struct _odl_tree, 1);
  g_assert (t != NULL);
  if (t)
    {
      t->root = alloc_odl_container ();
      g_assert (t->root != NULL);
      if (t->root)
        {
          /* root module stores all other modules */
          t->root->base.type = IT_module;
          t->root->base.name = g_strdup ("root");
          g_assert (t->root->base.name != NULL);
          if (!t->root->base.name)
            {
              free_odl_tree (t);
              return (NULL);
            }
        }
      else
        {
          /* error allocating root */
          free_odl_tree (t);
          t = NULL;
        }
    }
  return (t);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_tree (struct _odl_tree *tree)
{
#ifdef DEBUG
  assert (tree != NULL);
#endif
  if (tree)
    {
      if (tree->root)
        {
          free_odl_container (tree->root);
        }
      g_free (tree);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
struct _odl_container *
alloc_odl_container ()
{
  struct _odl_container *o = g_new0 (struct _odl_container, 1);

#ifdef DEBUG
  assert (o != NULL);
#endif
  if (o)
    {
      init_odl_base ((struct _odl_base *) o);

      o->contents = NULL;
      o->istype = FALSE;
      o->parents = NULL;
      o->indexes = NULL;
      o->orderby = NULL;
      o->filename = NULL;
      o->desc = FALSE;
    }
  return (o);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_container (struct _odl_container *c)
{
#ifdef DEBUG
  assert (c != NULL);
#endif
  if (c)
    {
      free_odl_base ((struct _odl_base *) c);
      if (c->contents)
        {
          /* free contained objects */
        }
      if (c->orderby)
        {
          g_free (c->orderby);
        }
      if (c->parents)
        {
          /* free links to parent classes */
          g_list_free (c->parents);
        }
      if (c->indexes)
        {
          /* free index data */
        }
      g_free (c);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
struct _odl_item *
alloc_odl_item ()
{
  struct _odl_item *i = g_new0 (struct _odl_item, 1);
#ifdef DEBUG
  assert (i != NULL);
#endif
  if (i)
    {
      init_odl_base ((struct _odl_base *) i);
      i->fieldtype = FT_unknown;
      i->datatype = DT_unknown;
      i->datatypeclass = NULL;
      i->properties = ODL_PROP_NONE;
      i->format = NULL;
      i->defaultval = NULL;
      i->tmpdefault = NULL;
      i->sourcefield = NULL;
      i->sourceclass = NULL;
      i->this_fields = NULL;
      i->source_fields = NULL;
      i->arguments = NULL;
      i->calculation = NULL;
      i->elements = NULL;
      i->bounds = -1;
    }
  return (i);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_item (struct _odl_item *item)
{
#ifdef DEBUG
  assert (item != NULL);
#endif
  if (item)
    {
      free_odl_base ((struct _odl_base *) item);

      /* TODO: free everything else */

      g_free (item);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
struct _odl_argument *
alloc_odl_argument ()
{
  struct _odl_argument *a = g_new0 (struct _odl_argument, 1);
#ifdef DEBUG
  assert (a != NULL);
#endif
  if (a)
    {
      a->method = NULL;
      a->name = NULL;
      a->datatype = DT_unknown;
      a->classdatatype = NULL;
    }
  return (a);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_argument (struct _odl_argument *argument)
{
#ifdef DEBUG
  assert (argument != NULL);
#endif
  /* TODO - there must be something to do here, neilt */
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
init_odl_base (struct _odl_base *base)
{
#ifdef DEBUG
  assert (base != NULL);
#endif
  base->name = NULL;
  base->fullname = NULL;
  base->mangledname = NULL;
  base->type = IT_unknown;
  base->access = ODL_ACCESS_PUBLIC;
  base->triggers = ODL_TRIG_NONE;
  base->parent = NULL;
  base->tree = NULL;
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_base (struct _odl_base *base)
{
#ifdef DEBUG
  assert (base != NULL);
  assert (base->name != NULL);
#endif
  if (base->name)
    {
      g_free (base->name);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
struct _odl_container *
odl_tree_get_root (struct _odl_tree *tree)
{
#ifdef DEBUG
  assert (tree != NULL);
#endif
  if (tree)
    {
      return (tree->root);
    }
  else
    {
      return (NULL);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_container_insert_container (struct _odl_container *parent,
                                struct _odl_container *child)
{
#ifdef DEBUG
  assert (parent != NULL);
  assert (child != NULL);
#endif
  parent->contents = g_list_append (parent->contents, child);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_module *
odl_new_module (unsigned long int access, const char *name)
{
  odl_module *m = alloc_odl_container ();
#ifdef DEBUG
  assert (name != NULL);
  assert (m != NULL);
#endif
  if (m)
    {
      m->base.type = IT_module;
      m->base.access = access;
      m->base.name = g_strdup (name);
    }
  return (m);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_class *
odl_new_class (unsigned long int access, const char *name)
{
  odl_field *f;
  GList *fields;
  _odl_index *i;
  odl_class *c = alloc_odl_container ();
#ifdef DEBUG
  assert (name != NULL);
  assert (c != NULL);
#endif
  if (c)
    {
      c->base.type = IT_class;
      c->base.access = access;
      c->base.name = g_strdup (name);

      /* add the objectid field */
      f = alloc_odl_item ();
      g_assert( f != NULL );
      f->base.parent = (odl_base *) c;
      f->base.access = ODL_ACCESS_SYSTEM;
      f->base.name = g_strdup ("objectid");
      f->base.type = IT_field;
      f->properties = ODL_PROP_NOTNULL | ODL_PROP_READONLY;
      f->fieldtype = FT_basic;
      f->datatype = DT_object;

      /* add it to the class */
      c->contents = g_list_append (c->contents, f);

      /* add indexes */
      fields = g_list_append (NULL, g_strdup ("objectid"));
      i = odl_make_index (c, TRUE, fields);
      i->primary = TRUE;
    }
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_item *
odl_new_enum (unsigned long int access, const char *name)
{
  odl_item *c = alloc_odl_item ();
#ifdef DEBUG
  assert (name != NULL);
  assert (c != NULL);
#endif
  if (c)
    {
      c->base.type = IT_enum;
      c->base.access = access;
      c->base.name = g_strdup (name);
    }
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_enum_add_element (odl_item * e, char *element)
{
  GList *l;
#ifdef DEBUG
  assert (element != NULL);
  assert (e != NULL);
  /* assert (e->elements); */
#endif
  l = e->elements;
  while (l)
    {
      if (g_strcasecmp (l->data, element) == 0)
        {
          g_free (element);
          return;
        }
      l = g_list_next (l);
    }
  e->elements = g_list_append (e->elements, g_strdup (element));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
_odl_datatype *
alloc_odl_datatype (gboolean unsignedval, char *name,
                    unsigned long int bound, enum odl_datatype dt)
{
  _odl_datatype *d = g_new0 (_odl_datatype, 1);

#ifdef DEBUG
  assert (name != NULL);
  assert (d != NULL);
#endif
  if (d)
    {
      if (g_strcasecmp (name, "void") == 0)
        {
          d->unsignedval = FALSE;
          d->name = name;
          d->bound = 0;
          d->dt = DT_void;
        }
      else
        {
          d->unsignedval = unsignedval;
          d->name = name;
          d->bound = bound;
          d->dt = dt;
        }
    }
  return (d);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_datatype (_odl_datatype * d)
{
#ifdef DEBUG
  assert (d != NULL);
#endif
  if (d)
    {
      if (d->name)
        {
          g_free (d->name);
        }
      g_free (d);
    }
}


/* ------------------------------------------------------------------------- *\
 * For implicit list and reference fields (i.e. classname * fieldname or
 * classname [] fieldname) makes
 * actual fields that are used to implement the requested fields.
 *
 * For all field types store the datatype.
 *
 * For TYPE fields set-up for future processing.
\* ------------------------------------------------------------------------- */
GList *
odl_reprocess_fields (_odl_datatype * datatype, GList * fields)
{
  GList *append = NULL;         /* saves new fields until after field list is */
  /* processed. */
  GList *f;
  odl_item *i;
  g_assert (datatype != NULL);
  g_assert (fields != NULL);

  f = fields;
  while (f)
    {
      /* get the field */
      i = (odl_item *) f->data;

      /* store the datatype in this field data */
      i->datatype = datatype->dt;
      i->bounds = datatype->bound;
      if ((datatype->unsignedval) && (i->datatype == DT_int))
        {
          i->datatype = DT_unsignedint;
        }
      /* set-up 'type' created fields for later processing */
      if (datatype->dt == DT_type)
        {
          i->base.type = IT_field;
          i->base.access = i->base.access;
          i->datatype = DT_type;
          i->datatypeclass = g_strdup (datatype->name);
        }
      if ((datatype->dt == DT_class || datatype->dt == DT_enum))
        {
          if (i->fieldtype == FT_basic && i->datatype == DT_class)
            {
              i->datatypeclass = g_strdup (datatype->name);
              i->datatype = datatype->dt;
              /* if -1 turn it into a reference  */
              /* else 0 or positive a list class */
              /* else < -1 invalid               */
              g_assert (i->bounds > -2);
              if (i->bounds == (-1))
                {
                  /* convert the existing field to a reference field */
                  /* odl_resolve_implicit_reference (i, append, datatype); */
                }
              else
                {
                  /* list, delayed until full tree is built */
                  if (current_pass == 2)
                    {
                      /* odl_resolve_implicit_list (i, datatype); */
                    }
                  else
                    {
                      /* ((odl_base *) i)->type = IT_ignore; */
                    }
                }
            }
          else
            {
              i->datatypeclass = g_strdup (datatype->name);
            }
        }
      /* next */
      f = g_list_next (f);
    }
  if (append)
    {
      fields = g_list_concat (fields, append);
    }
  free_odl_datatype (datatype);
  return (fields);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_item *
set_method_datatype (_odl_datatype * datatype, odl_item * method)
{
#ifdef DEBUG
  assert (method != NULL);
  assert (datatype != NULL);
#endif
  method->datatype = datatype->dt;
  return (method);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
_odl_datasource *
alloc_odl_datasource ()
{
  _odl_datasource *d = g_new (_odl_datasource, 1);

#ifdef DEBUG
  assert (d != NULL);
#endif
  if (d)
    {
      d->fields = NULL;
      d->classname = NULL;
      d->field = NULL;
    }
  return (d);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_odl_datasource (_odl_datasource * d)
{
#ifdef DEBUG
  assert (d != NULL);
#endif
  if (d)
    {
      if (d->classname)
        {
          g_free (d->classname);
        }
      if (d->field)
        {
          g_free (d->field);
        }
      g_free (d);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_get_base_full_name (odl_base * b)
{
  odl_base *tmp;
  GList *l = NULL;
  GList *l2;
  GString *buf;
  char *retval;

#ifdef DEBUG
  assert (b != NULL);
#endif
  if (b->fullname)
    {
      return (b->fullname);
    }
  buf = g_string_new ("");
  tmp = b;
  while (b)
    {
      if (b->name)
        {
          l = g_list_prepend (l, b->name);
        }
      b = b->parent;
    }
  l2 = l;
  while (l2)
    {
      g_string_append (buf, l2->data);
      g_string_append (buf, "::");
      l2 = g_list_next (l2);
    }
  if (l)
    {
      g_list_free (l);
    }
  retval = buf->str;
  while (retval[strlen (retval) - 1] == ':')
    {
      retval[strlen (retval) - 1] = '\0';
    }
  buf->str = NULL;
  g_string_free (buf, TRUE);

  tmp->fullname = retval;
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_standardise_parents (odl_class * c)
{
  /* char *buf; */
  GList *l;
  odl_base *ptr;
  odl_base *item = NULL;
  GList *parents = NULL;

#ifdef DEBUG
  assert (c != NULL);
#endif
  /* no parents, so is very quick */
  if (c->parents == NULL)
    {
      return;
    }
  l = c->parents;
  while (l)
    {
      ptr = c->base.parent;
      /* look for it in immediate parent, and keep moving up until */
      /* there's nowhere to go */
      while (ptr)
        {
          item = odl_find_relative (ptr, l->data);
          if (item)
            {
              ptr = NULL;
            }
          else
            {
              ptr = ptr->parent;
            }
        }
      if (item)
        {
          char *tmp = g_strdup (odl_item_get_full_name (item));
          parents = g_list_append (parents, tmp);
        }
      l = g_list_next (l);
    }
  odl_namelist_free (c->parents);
  c->parents = parents;
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
_odl_index *
odl_make_index (odl_class * c, gboolean unique, GList * fieldnames)
{
  _odl_index *i = g_new0 (_odl_index, 1);

#ifdef DEBUG
  assert (c != NULL);
  assert (fieldnames != NULL);
  assert (i != NULL);
#endif
  if (i)
    {
      fieldnames =
        g_list_sort (fieldnames, (GCompareFunc) odl_sort_string_name);
      i->primary = FALSE;
      i->unique = unique;
      i->fields = fieldnames;
      c->indexes = g_list_append (c->indexes, i);
    }
  return (i);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
odl_sort_string_name (gpointer a, gpointer b)
{
  return (g_strcasecmp (a, b));
}
