/*
   GEAS configuration management library

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: configuration.c,v 1.14 2001/06/08 20:06:52 reinhard Exp $
*/

/* ------------------------------------------------------------------------- *\
 * FIXME:
 * Use a prefix for all symbols in this library
 * Load configuration dynamically so that changes in the configuration file
   become valid without restarting the server (not sure if this is good)
 * Use GHashTable instead of GList for key/value pairs
 * Make configuration_data external and use that instead of configuration -
   this will obsolete several typecasts
\* ------------------------------------------------------------------------- */

/*
 * This library is an API for GEAS to use to read configuration details.
 * the internals are subject to change at any time, and will be what
 * defines how the config data is stored
 * 
 * 
 * when adjusting this module, load_configuration_file() may be rewritten
 * but simply adjusting read_shadow_password() should be sufficient.
 * these functions perform the main work, calling add_section_option()
 * and add_database_option() to enter options into the system.
 * 
 *   read_shadow_password()
 *   read_main_configuration()
 * 
 * This function sets default properties. If the options have been
 * set already, they are ignored. This should only be modified if the
 * defaults are changed
 * 
 * enter_default_properties()
 * 
 */

#include "config.h"
#include "configuration.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <glib.h>

/* ========================================================================= *\
 * Private structures
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * A line in the configuration file
\* ------------------------------------------------------------------------- */
struct _config_item
{
  char *key;
  char *value;
};

typedef struct _config_item config_item;

/* ------------------------------------------------------------------------- *\
 * The complete configuration
\* ------------------------------------------------------------------------- */

struct _configuration_data
{
  char *name;
  GList *database;                      /* contains a configuration_data
                                           structure for each database */
  GList *globals;                       /* contains the global key/value
                                           pairs */
};

typedef struct _configuration_data configuration_data;

/* ========================================================================= *\
 * Global variables
\* ========================================================================= */

configuration configdata = NO_CONFIGURATION;

static int config_failed = 0;

/* ========================================================================= *\
 * Private functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Allocate a new configuration item in memory
\* ------------------------------------------------------------------------- */
static configuration_data *
alloc_configuration_data (const char *name)
{
  configuration_data *c;

  g_assert (name);

  c = g_new0 (configuration_data, 1);

  c->name = g_strdup (name);
  c->database = NULL;
  c->globals = NULL;

  return (c);
}

/* ------------------------------------------------------------------------- *\
 * Find the configuration for a database
\* ------------------------------------------------------------------------- */
static configuration_data *
find_database (configuration_data *c, const char *name)
{
  configuration_data *i;
  GList *l;

  g_assert (c);
  g_assert (name);
  
  l = c->database;
  while (l)
    {
      i = (configuration_data *) l->data;
      if (g_strcasecmp (i->name, name) == 0)
	return (i);
      l = g_list_next (l);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * Free a configuration item in memory
\* ------------------------------------------------------------------------- */
static void
free_config_item (config_item *item)
{
  if (item)
    {
      g_free (item->key);
      g_free (item->value);
      g_free (item);
    }
}

/* ------------------------------------------------------------------------- *\
 * Allocate a configuration item in memory
\* ------------------------------------------------------------------------- */
static config_item *
alloc_config_item (const char *key, const char *value)
{
  config_item *i;
  char *p;
  char tmp = '\0';

  g_assert (key);
  g_assert (value);

  i = g_new0 (config_item, 1);

  i->key = g_strdup (key);
  g_strdown (i->key);

  /* if there's an extra space, truncate line */
  p = strchr (value, ' ');
  if (p && isspace (*p))
    {
      fprintf (stderr,
               "warning: option '%s' has been truncated from '%s' ",
               key, value);
      tmp = *p;
      *p = '\0';
      fprintf (stderr, "to '%s'\n", value);
    }
  i->value = g_strdup (value);
  if (p)
    *p = tmp;

  return (i);
}

/* ------------------------------------------------------------------------- *\
 * Add a key/value pair to a section
\* ------------------------------------------------------------------------- */
static void
add_section_option (configuration_data *c, const char *key, const char *value)
{
  config_item *i;

  g_assert (c);
  g_assert (key);
  g_assert (value);

  if (!get_global_option ((configuration) c, key))  /* ignore duplicates */
    {
      i = alloc_config_item (key, value);
      c->globals = g_list_append (c->globals, i);
    }
}

/* ------------------------------------------------------------------------- *\
 * Add a database to the configuration
\* ------------------------------------------------------------------------- */
static void
add_database (configuration_data *c, const char *name)
{
  configuration_data *i;

  g_assert (c);
  g_assert (name);

  if (!find_database ((configuration) c, name))  /* ignore duplicates */
    {
      i = alloc_configuration_data (name);
      c->database = g_list_append (c->database, i);
    }
}

/* ------------------------------------------------------------------------- *\
 * Add a key/value pair to a database configuration
\* ------------------------------------------------------------------------- */
static void
add_database_option (configuration_data *c, const char *name, const char *key,
                     const char *value)
{
  configuration_data *i;

  g_assert (c);
  g_assert (name);
  g_assert (key);
  g_assert (value);

  /* add database to config if not yet there */
  i = find_database ((configuration) c, key);
  if (!i)
    {
      add_database (c, name);
      i = find_database ((configuration) c, name);
      g_assert (i);
    }

  /* allocate and store new data */
  add_section_option (i, key, value);
}

/* ------------------------------------------------------------------------- *\
 * Read the password for the databases and the admin password from the
 * shadowpwfile and insert the values into the configuration structs in
 * memory as if they were in the normal config file
\* ------------------------------------------------------------------------- */
static void
read_shadow_password (configuration_data * c, const char *shadowpwfile)
{
  const char *adminuser;
  int errdone = 0;
  char *p, *q;
  char buf[256];
  FILE *fp;

  g_assert (c);
  g_assert (shadowpwfile);

  adminuser = get_global_option (c, "adminuser");

  fp = fopen (shadowpwfile, "r");
  if (!fp)
    return;
  buf[255] = '\0';
  while (!feof (fp))
    {
      if (fgets (buf, 255, fp) != NULL)
	{
	  /* strip comments */
	  p = strchr (buf, '#');
	  if (p)
	    *p = '\0';
	  for (p = buf; *p != '\0'; p++)
	    if (!isspace (*p))
	      break;
	  if (*p == '\0')
	    continue;		/* blank line or comment only line
				 */
	  q = strchr (p, ':');
	  if (!q)
	    {
	      config_failed = 1;
	      if (!errdone)
		fprintf (stderr, "Shadow password file error.");
	      errdone = 1;
	      continue;
	    }
	  *q = '\0';

	  q++;
	  /* trim whitespace from left side */
	  while (isspace (*p))
	    p++;
	  while (isspace (p[strlen (p) - 1]))
	    p[strlen (p) - 1] = '\0';
	  /* trim whitespace from right side */
	  while (isspace (*q))
	    q++;
	  while (isspace (q[strlen (q) - 1]))
	    q[strlen (q) - 1] = '\0';
	  if (adminuser && strcmp (p, "adminuser") == 0)
	    add_section_option (c, "adminpassword", q);
	  else
	    add_database_option (c, p, "password", q);
	}
    }
  fclose (fp);
}

/* ------------------------------------------------------------------------- *\
 * Print the current configuration (for debugging purposes)
\* ------------------------------------------------------------------------- */
static void
real_show_configuration (configuration config, int indent,
                         gboolean show_passwords)
{
  configuration_data *c = (configuration_data *) config;
  GList *l;
  int loop;
  config_item *i;

  g_return_if_fail (config);

  if (c)
    {
      for (loop = 0; loop < indent; loop++)
	printf (" ");
      if (c->name)
	{
	  printf ("Section: %s\n", c->name);
	  for (loop = 0; loop < indent; loop++)
	    printf (" ");
	  printf ("--------\n");
	}
      else
	{
	  printf ("Section: <unknown>\n");
	  for (loop = 0; loop < indent; loop++)
	    printf (" ");
	  printf ("--------\n");
	}
      l = c->globals;
      while (l)
	{
	  for (loop = 0; loop < indent; loop++)
	    printf (" ");
	  i = (config_item *) l->data;
	  if (show_passwords == FALSE && strstr (i->key, "password") != NULL)
	    printf ("%s \t: <hidden for security>\n", i->key);
	  else
	    printf ("%s \t: %s\n", i->key, i->value);
	  l = g_list_next (l);
	}

      l = c->database;
      while (l)
	{
	  printf ("\n");
	  real_show_configuration ((configuration) l->data, indent + 4,
				   show_passwords);
	  l = g_list_next (l);
	}
      printf ("\n");
    }
}

/* ------------------------------------------------------------------------- *\
 * Read a line from the config file
\* ------------------------------------------------------------------------- */
static char *
read_line (FILE *fp)
{
  char buf[1024];
  char *p, *q, *retval;

  g_assert (fp);

  buf[1023] = '\0';
  while (1)
    {
      if (fgets (buf, 1022, fp) != NULL)
	{
	  while (buf[strlen (buf) - 1] == '\r'
		 || buf[strlen (buf) - 1] ==
		 '\n') buf[strlen (buf) - 1] = '\0';
	  /* strip comments */
	  p = strchr (buf, '#');
	  if (p)
	    *p = '\0';
	  /* strip whitespace from ends */
	  p = buf;
	  while (isspace (*p))
	    p++;
	  if (*p == '\0')
	    continue;
	  while (isspace (p[strlen (p) - 1]))
	    p[strlen (p) - 1] = '\0';
	  /* skip empty lines/only comment lines */
	  if (*p == '\0')
	    continue;
	  /* strip double spaces */
	  retval = p;
	  q = p;
	  while (*p != '\0')
	    {
	      p = strchr (p, ' ');
	      if (!p)
		break;
	      p++;
	      q = p;
	      while (isspace (*q))
		q++;
	      memmove (p, q, strlen (q) + 1);
	      p++;
	    }

	  /* got a valid line */
	  return (retval);
	}

      /* EOF return NULL */
      if (feof (fp))
	return (NULL);
    }
}

/* ------------------------------------------------------------------------- *\
 * What part of the config file are we in?
\* ------------------------------------------------------------------------- */
#define MODE_UNKNOWN  0
#define MODE_DATABASE 1
#define MODE_GLOBAL   2

/* ------------------------------------------------------------------------- *\
 * Read the main configuration file, which contains all configuration data
 * except the passwords
\* ------------------------------------------------------------------------- */
static void
read_main_configuration (configuration_data *c, const char *filename)
{
  int mode = MODE_GLOBAL;
  char *p, *q, *tmp;
  FILE *fp;
  int done = 0;
  char *dbname = NULL;

  g_assert (c);
  g_assert (filename);

  fp = fopen (filename, "r");
  if (!fp)
    {
      add_section_option (c, "configfile", "unknown");
      return;
    }
  add_section_option (c, "configfile", filename);

  while (!done)
    {
      p = read_line (fp);
      if (!p)
	break;
      if (p[0] == '[' && p[strlen (p) - 1] == ']')
	{
	  mode = MODE_UNKNOWN;
	  /* strip [ ] braces */
	  if (p[1] == ' ')
	    p += 2;
	  else
	    p += 1;
	  p[strlen (p) - 1] = '\0';
	  if (isspace (p[strlen (p) - 1]))
	    p[strlen (p) - 1] = '\0';
	  q = strchr (p, ' ');
	  if (q)
	    *q++ = '\0';

	  if (g_strcasecmp (p, "database") == 0)
	    {
	      if (q && *q != '\0')
		{
		  mode = MODE_DATABASE;
		  if (dbname)
		    g_free (dbname);
		  dbname = g_strdup (q);
		  add_database (c, q);
		}
	      else
		{
		  mode = MODE_UNKNOWN;
		  fprintf (stderr, "error: database name not given.\n");
		  config_failed = 1;
		}
	    }
	  else if (g_strcasecmp (p, "global") == 0)
	    {
	      mode = MODE_GLOBAL;
	    }
	  else
	    {
	      config_failed = 1;
	      fprintf (stderr, "error: unknown section '%s'\n", p);
	    }
	}
      else
	{
	  q = strchr (p, ' ');
	  if (q)
	    *q++ = '\0';
	  if (!q)
	    {
	      config_failed = 1;
	      fprintf (stderr,
		       "error: option %s does not have a value.\n", p);
	      continue;
	    }
	  /* key to lower case */
	  tmp = p;
	  while (p != NULL && *p != '\0')
	    {
	      *p = tolower (*p);
	      p++;
	    }
	  p = tmp;

	  switch (mode)
	    {
	    case MODE_UNKNOWN:
	      break;
	    case MODE_DATABASE:
	      /* printf( "database %s : %s = %s\n" , dbname , p , q
	         ); */
	      add_database_option (c, dbname, p, q);
	      break;
	    case MODE_GLOBAL:
	      /* printf( "global: %s = %s\n" , p , q ); */
	      if (strstr (p, "password") != NULL)
		{
		  fprintf (stderr,
			   "error: passwords must not be included in the main configuration file.\n");
		  fprintf (stderr,
			   "       the shadow password file should be used, instead.\n");
		  config_failed = 1;
		}
	      add_section_option (c, p, q);
	      break;
	    }
	}
    }
  fclose (fp);
  if (dbname)
    g_free (dbname);
}

/* ------------------------------------------------------------------------- *\
 * to add default options, copy the line below, replacing key and value with
 * appropriate strings:
   add_section_option (c, "key", "value"); 
 * eg:
 * to set a default debug level of 5, use:
   add_section_option (c, "debuglevel", "5");
 * note that command line options override the configuration file settings.
\* ------------------------------------------------------------------------- */
static void
enter_default_properties (configuration_data * c)
{
  /* none defined yet */
}

/* ------------------------------------------------------------------------- *\
 * load the whole configuration
\* ------------------------------------------------------------------------- */
configuration
load_configuration_file (const char *filename, const char *shadowpwfile)
{
  configuration_data *c = alloc_configuration_data ("global options");

  /* read main configuration file */
  if (filename != NULL)
    {
      read_main_configuration (c, filename);
    }

  /* set defaults: function ignores if already set */
  enter_default_properties (c);

  /* read shadow password file */
  if (shadowpwfile != NULL)
    {
      read_shadow_password (c, shadowpwfile);
    }

  /* if anything failed, return nothing */
  if (config_failed != 0)
    {
      free_configuration (c);
      return (NULL);
    }

  return ((configuration) c);
}

/* ------------------------------------------------------------------------- *\
 * free configuration data in memory
\* ------------------------------------------------------------------------- */
void
free_configuration (configuration config)
{
  configuration_data *c = (configuration_data *) config;
  GList *l;
  config_item *i;

  if (c)
    {
      /* free name */
      if (c->name)
	g_free (c->name);

      /* free options in this section */
      l = c->globals;
      while (l)
	{
	  i = (config_item *) l->data;
	  free_config_item (i);
	  l = g_list_next (l);
	}
      g_list_free (c->globals);

      /* free contained sections */
      l = c->database;
      while (l)
	{
	  free_configuration ((configuration) l->data);
	  l = g_list_next (l);
	}
      g_list_free (c->database);

      /* free data */
      g_free (c);
    }
}

/* ------------------------------------------------------------------------- *\
 * Print out configuration for debugging purposes
\* ------------------------------------------------------------------------- */
void
show_configuration (configuration config, gboolean show_passwords)
{
  printf ("\nConfiguration:\n\n");
  real_show_configuration (config, 0, show_passwords);
}

/* ------------------------------------------------------------------------- *\
 * count databases in the file
\* ------------------------------------------------------------------------- */
int
count_configured_databases (configuration config)
{
  g_return_val_if_fail (config, 0);

  return (g_list_length (((configuration_data *) config)->database));
}

/* ------------------------------------------------------------------------- *\
 * get the name of a specific database
\* ------------------------------------------------------------------------- */
const char *
get_database_name (configuration config, int index)
{
  configuration_data *c = (configuration_data *) config;

  g_return_val_if_fail (config, NULL);

  /* validate index */
  if (index < 0 || index >= count_configured_databases (config))
    return (NULL);

  /* find entry */
  c = (configuration_data *) g_list_nth_data (c->database, index);
  if (!c)
    return (NULL);

  /* return its name */
  return (c->name);
}

/* ------------------------------------------------------------------------- *\
 * Global option - general
\* ------------------------------------------------------------------------- */
const char *
get_global_option (configuration config, const char *key)
{
  config_item *i = NULL;
  GList *l;

  g_return_val_if_fail (config, NULL);
  g_return_val_if_fail (key, NULL);

  l = ((configuration_data *) config)->globals;

  while (l)
    {
      i = (config_item *) l->data;
      g_assert (i);
      if (g_strcasecmp (i->key, key) == 0)
	{
	  return (i->value);
	}
      l = g_list_next (l);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * Global option - Integer
\* ------------------------------------------------------------------------- */
int
get_global_option_int (configuration config, const char *key, int defaultval)
{
  const char *o = get_global_option (config, key);
  if (!o)
    return (defaultval);
  return (atoi (o));
}

/* ------------------------------------------------------------------------- *\
 * Global option - Boolean
\* ------------------------------------------------------------------------- */
gboolean
get_global_option_bool (configuration config, const char *key,
			gboolean defaultval)
{
  const char *o = get_global_option (config, key);
  if (!o)
    return (defaultval);

  if (g_strcasecmp (o, "1") == 0)
    return (TRUE);
  if (g_strcasecmp (o, "yes") == 0)
    return (TRUE);
  if (g_strcasecmp (o, "true") == 0)
    return (TRUE);

  if (g_strcasecmp (o, "1") == 0)
    return (FALSE);
  if (g_strcasecmp (o, "no") == 0)
    return (FALSE);
  if (g_strcasecmp (o, "false") == 0)
    return (FALSE);

  return (defaultval);
}

/* ------------------------------------------------------------------------- *\
 * Global option - String
\* ------------------------------------------------------------------------- */
const char *
get_global_option_str (configuration config, const char *key,
		       const char *defaultval)
{
  const char *o = get_global_option (config, key);
  if (o)
    return (o);
  else
    return (defaultval);
}

/* ------------------------------------------------------------------------- *\
 * Database option - general
\* ------------------------------------------------------------------------- */
const char *
get_database_option (configuration tree, const char *database,
		     const char *key)
{
  configuration_data *c;

  g_return_val_if_fail (tree, NULL);
  g_return_val_if_fail (database, NULL);
  g_return_val_if_fail (key, NULL);

  c = find_database ((configuration_data *) tree, database);

  g_return_val_if_fail (c, NULL);

  return (get_global_option (c, key));
}

/* ------------------------------------------------------------------------- *\
 * Database option - String
\* ------------------------------------------------------------------------- */
const char *
get_database_option_str (configuration tree, const char *databasename,
			 const char *key, const char *defaultvalue)
{
  const char *o = get_database_option (tree, databasename, key);
  if (o)
    return (o);
  else
    return (defaultvalue);
}

/* ------------------------------------------------------------------------- *\
 * Database option - Boolean
\* ------------------------------------------------------------------------- */
gboolean
get_database_option_bool (configuration tree, const char *databasename,
			  const char *key, gboolean defaultval)
{
  const char *o = get_database_option (tree, databasename, key);
  if (!o)
    return (defaultval);

  if (g_strcasecmp (o, "1") == 0)
    return (TRUE);
  if (g_strcasecmp (o, "yes") == 0)
    return (TRUE);
  if (g_strcasecmp (o, "true") == 0)
    return (TRUE);

  if (g_strcasecmp (o, "1") == 0)
    return (FALSE);
  if (g_strcasecmp (o, "no") == 0)
    return (FALSE);
  if (g_strcasecmp (o, "false") == 0)
    return (FALSE);

  return (defaultval);
}

/* ------------------------------------------------------------------------- *\
 * Database option - Integer
\* ------------------------------------------------------------------------- */
int
get_database_option_int (configuration tree, const char *databasename,
			 const char *key, int defaultval)
{
  const char *o = get_database_option (tree, databasename, key);
  if (o)
    return (atoi (o));
  return (defaultval);
}

/* ------------------------------------------------------------------------- *\
 * HACK - temporary function
\* ------------------------------------------------------------------------- */
const char *
get_first_active_database (configuration tree)
{
  configuration_data *c = (configuration_data *) tree;
  configuration_data *i = NULL;
  GList *l = c->database;
  char *dbname;

  while (l)
    {
      i = (configuration_data *) l->data;
      dbname = i->name;
      if (get_database_option_bool (tree, dbname, "active", FALSE))
	return (dbname);
      l = g_list_next (l);
    }
  return (NULL);
}
