/*
   objectcache.c -  Object cache.
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
   
   $Id: objectcache.c,v 1.58 2001/07/25 15:40:18 reinhard Exp $
*/

#include "config.h"

#include <glib.h>
#include <string.h>

#include "geas-server.h"
#include "objectcache.h"
#include "objectstore/objectstore.h"
#include "config/configuration.h"
#include "oql/oql.h"

/* ie, turn off temp. debug output */
#define OC_TEMP_DEBUG (-83)     /* DEBUGLEVEL_OFF */

/** \file objectcache.c
 * \brief Store object data in RAM
 * The main cache is a hash table, indexed on object classnames.
 * Each entry is a further hash table indexed on object identifiers
 * 
 * additional index hash data structures will store pointers to entries in
 * the main table, indexed on field values.
 * 
 * Notes:
 * struct _ObjectData:
 *   the classname member is not duplicated, so should not be freed. It
 *   is intended to be the same copy as stored in the _CacheClassData
 *   so stored once per class in the cache
 */

static GHashTable *cachedata = NULL;
static GHashTable *indexes = NULL;
static unsigned int objectcache_count = 0;
static unsigned int hack_count = 0;
static guint32 cache_max_size = 0;
static GList *flushable = NULL;

void
oc_print_stats( void )
{
    printf( "Object Cache: %d entries   %d flushable\n" , hack_count ,
	    hack_count );
}

static void oc_squeeze_cache (unsigned int spaces_to_make);
static void oc_free_object_data (_ObjectData * o);
static _ObjectData *oc_allocate_object_data (const char *classname,
                                             const char *key);
static _FieldData *oc_alloc_field_data (const char *name);
static void oc_free_field_data (gpointer key, _FieldData * f,
                                gpointer userdata);
static ObjectData *oc_objectstore_query_to_objectcache (QueryData * q);
static void oc_remove_from_index (_CacheIndex * idx, _ObjectData * o);
static void oc_add_object_to_index (_ObjectData * object,
                                    oc_index_identifier * indexid);
static void oc_free_index_identifier_list (GList * list);
static oc_index_identifier *oc_make_index_identifier (_ObjectData * object,
                                                      _odl_index * i);
static char *oc_make_object_index_hash (_ObjectData * o, _odl_index * idx);
static GList *oc_object_to_start_of_list (GList * l, ObjectData * obj);
static GList *oc_object_to_end_of_list (GList * l, ObjectData * obj);

void _fill_values (gchar * key, _FieldData * field, GHashTable * values);

/* ------------------------------------------------------------------------- *\
 * prepare cache for lots of changes
\* ------------------------------------------------------------------------- */
void oc_freeze()
{
}

/* ------------------------------------------------------------------------- *\
 * finish making large numbers of changes
\* ------------------------------------------------------------------------- */
void oc_thaw()
{
}


/* ------------------------------------------------------------------------- *\
 * set user data in object
\* ------------------------------------------------------------------------- */
void
oc_set_object_userdata( ObjectData *object , void *data )
{
  ((_ObjectData *)object)->userdata = data;
}

/* ------------------------------------------------------------------------- *\
 * get user data from object
\* ------------------------------------------------------------------------- */
void *
oc_get_object_userdata( ObjectData *object )
{
return ((_ObjectData *)object)->userdata;
}

/* ------------------------------------------------------------------------- *\
 * used to clear all data from the cache
\* ------------------------------------------------------------------------- */
void
oc_empty_cache ()
{
  int maxlen = cache_max_size;
  oc_shrink_cache (0);
  cache_max_size = maxlen;
}

/* ------------------------------------------------------------------------- *\
 * Make the cache smaller
\* ------------------------------------------------------------------------- */
void
oc_shrink_cache (unsigned int new_max_length)
{
  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  trace_functioncall ();

  /* allow infinite numbers of objects anyway */
  if( cache_max_size == 0 ) return;

  /* this can't extend it */
  if (new_max_length >= cache_max_size) {
    timer_fail_operation( TIMER_CACHE);
    return;
  }

  if( new_max_length == 0 )
   {
     /* make an infinte size cache */
     cache_max_size = 0;
     return;
   }

  if (new_max_length < 0)
    cache_max_size += new_max_length;   /* shrink by -N spaces */
  else
    cache_max_size = new_max_length;

  /* 0 == empty cache */
  if (cache_max_size < 0)
    cache_max_size = 0;

  oc_squeeze_cache (5);

  /* still need space for 1 item in the cache, for reading/writing */
  if (cache_max_size < 1)
    cache_max_size = 1;

  timer_done_operation( TIMER_CACHE );
}

/* ------------------------------------------------------------------------- *\
 * Make the cache larger
\* ------------------------------------------------------------------------- */
void
oc_extend_cache (unsigned int new_max_length)
{
  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  trace_functioncall ();

  /* allow infinite numbers of objects anyway */
  if( cache_max_size == 0 ) return;

  /* this can't shrink it */
  if (new_max_length <= cache_max_size)
    return;

  if (new_max_length < 0)
    cache_max_size -= new_max_length;   /* extend by - N spaces */
  else
    cache_max_size = new_max_length;

  cache_max_size = new_max_length;
  /* only allocates more space, no need to remove any objects */

  timer_done_operation( TIMER_CACHE );
}

/** \brief make space in caxche
 * This ensuress that another 'spaces_to_make' objects can be added to
 * the cache before running out of space, assuming no other objects
 * are deleted
 */
static void
oc_squeeze_cache (unsigned int spaces_to_make)
{
  _ObjectData *obj;
  GList *last;
  gboolean squeezed = FALSE;

  /* allow infinite numbers of objects anyway */
  if( cache_max_size == 0 ) return;

  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  trace_functioncall ();

  spaces_to_make = 1;
  if (spaces_to_make > cache_max_size)
    {
      spaces_to_make = cache_max_size;
    }
  while ((spaces_to_make + objectcache_count) > cache_max_size && flushable)
    {
      squeezed = TRUE;

      /* grab one off the tail of the flush list and flush it */
      last = g_list_last (flushable);
      if (last)
        {
          obj = last->data;
#if 0
          message ("flushing object %08lx > %08lx/%08lx", obj,
                   obj->classname, obj->key);
          fflush (NULL);
          message ("flushing object %s/%s", obj->classname, obj->key);
          fflush (NULL);
#endif
          /* write to database, if needed */
          if (obj->indatabase == 0)
            {
              oc_flush_object_to_store (obj);
            }
          /* remove from cache (has to be in database now, but check anyway) */
          if (obj->indatabase != 0)
            {
              oc_remove_object (obj);
            }
          else
            {
              break;
            }
        }
    }
  if (objectcache_count > cache_max_size)
    {
      errormsg
        ("The cache has grown to %d entries, with a maximum of %d, and none can be flushed.",
         objectcache_count, cache_max_size);
    }
#if 0
  if (squeezed)
    {
      message ("New cache size: %d objects", objectcache_count);
    }
#endif
  timer_done_operation( TIMER_CACHE );
}

/* ------------------------------------------------------------------------- *\
 * Allocate object cache memory
\* ------------------------------------------------------------------------- */
void
create_object_cache (guint32 max_length)
{
  trace_functioncall ();

  if (cachedata)
    {
      free_object_cache ();
    }
  cache_max_size = max_length;
  cachedata = g_hash_table_new (g_str_hash, g_str_equal);
  indexes = g_hash_table_new (g_str_hash, g_str_equal);
  objectcache_count = 0;
  hack_count = 0;
}

/* ------------------------------------------------------------------------- *\
 * Free object cache memory
\* ------------------------------------------------------------------------- */
void
free_object_cache ()
{
  /* TODO */
  fatal_error ("Not yet implemented");
}

/* ========================================================================= *\
 * Searching
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Find an object by the key, no matter if it's cached or not
\* ------------------------------------------------------------------------- */
ObjectData *
oc_find_object_by_key (const char *classname, const char *key)
{
  ObjectData *obj;
  QueryData *q;

  trace_functioncall ();
  obj = oc_find_cached_object_by_key (classname, key);
  if (obj)
    {
      return (obj);
    }
  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  timer_start_profile( TIMER_FUNC_FIND_BY_KEY );

  /* not in cache, so load from database */
  q = oql_load_object_by_key (classname, key);
  if (q)
    {
      obj = oc_objectstore_query_to_objectcache (q);
    }
  oql_free_query (q);
  timer_update_profile( TIMER_FUNC_FIND_BY_KEY );
  timer_done_operation( TIMER_CACHE );
  return (obj);
}

/* ------------------------------------------------------------------------- *\
 * Find a cached object by the key
\* ------------------------------------------------------------------------- */
ObjectData *
oc_find_cached_object_by_key (const char *classname, const char *key)
{
  ObjectData *obj;
  char lookup[1024];
  char *p;

  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  timer_start_profile( TIMER_FUNC_FIND_CACHED_BY_KEY );
  trace_functioncall ();
  if (strncmp (classname, "root::", 6) == 0)
    {
      p = (char *) &classname[6];
    }
  else
    {
      p = (char *) classname;
    }
  /* find the classname's hash table - if none object can't be in the */
  /* cache yet */
//  lookup = g_strdup_printf ("%s-%s", p, key);
timer_start_profile( TIMER_FUNC_LOOKUP );
  sprintf( lookup , "%s-%s" , p , key );
  obj = g_hash_table_lookup (cachedata, lookup);
timer_update_profile( TIMER_FUNC_LOOKUP );
//if( !obj ) {
// printf( "not found: '%s'\n" , lookup );
// abort();
//}
//  g_free (lookup);
  timer_update_profile( TIMER_FUNC_FIND_CACHED_BY_KEY );
  timer_done_operation( TIMER_CACHE );
  return (obj);
}

/* ========================================================================= *\
 * Object management
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Add a new, empty object to the cache
\* ------------------------------------------------------------------------- */
ObjectData *
oc_add_empty_object (const char *classname, const char *key)
{
  ObjectData *obj = NULL;
  char *oid = g_strdup (key);
  char *p;

  /* printf( "adding object: currently %d entries in cache\n" , objectcache_count ); */

  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  trace_functioncall ();

  /* if there's still any '-' characters, remove them */
//  if (strchr (key, '-') != NULL)
//    {
//      cleanup_oid (oid);
//    }

  if (strncmp (classname, "root::", 6) == 0)
    {
      p = (char *) &classname[6];
    }
  else
    {
      p = (char *) classname;
    }
  /* if the object already exists, return it */
  obj = oc_find_cached_object_by_key (p, oid);
  if (obj)
    {
      g_free (oid);
      return (obj);
    }
/*  printf( "adding object: %d entries  %s/%s\n" , objectcache_count ,classname,key); */

  /* make sure the cache doesn't have too many entries */
  oc_squeeze_cache (5);

  /* create a new object */
  obj = oc_allocate_object_data (p, oid);
  g_free (oid);
  oid = NULL;
  if (!obj)
    {
      timer_fail_operation( TIMER_CACHE);
      return (NULL);
    }
  /* printf( "created: %08lx > %08lx/%08lx  (%s/%s)\n", obj,
     obj->classname,obj->key , obj->classname,obj->key ); */
  /* fflush( NULL ); */

  /* printf( "adding %08lx to flushable\n" , obj ); */
  flushable = g_list_prepend (flushable, obj);

  /* store it */
  /* printf( "add %s/%s\n" , classname , key ); */
  g_hash_table_insert (cachedata, obj->hashkey, obj);
  objectcache_count += 1;

  /* only count 'user' objects */
  if( strncmp(classname,"geas::",6) != 0 )
     hack_count += 1;

#if 0
  printf ("objectcache_count = %d\n", objectcache_count);
  printf ("now has %d entries\n", objectcache_count);
#endif
  /* done */
  timer_done_operation( TIMER_CACHE );
  return (obj);
}

/* ------------------------------------------------------------------------- *\
 * Remove an object from the cache
\* ------------------------------------------------------------------------- */
gboolean
oc_remove_object (ObjectData * object)
{
  _ObjectData *obj;
  /* GList *idxs, *ids; */
  /*  odl_class *c; */

  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  trace_functioncall ();

  obj = g_hash_table_lookup (cachedata, object->hashkey);
  if (!obj)
    {
      /* object wasn't in the cache */
      timer_fail_operation( TIMER_CACHE);
      return (FALSE);
    }
  flushable = g_list_remove (flushable, object);
  if (g_list_find (flushable, object))
    {
      printf ("still in flushable list: %px (bad 0)\n", object);
      abort ();
    }
  /* remove it */
  g_hash_table_remove (cachedata, obj->hashkey);
  objectcache_count -= 1;
  if( strncmp(object->classname,"geas::",6) != 0 )
     hack_count -= 1;

  oc_free_object_data (object);

  /* printf( "objectcache_count = %d\n" , objectcache_count ); */

  timer_done_operation( TIMER_CACHE );
  return (TRUE);
}

/* hack for lists */
gboolean
oc_remove_object2(ObjectData * object)
{
  g_hash_table_remove (cachedata, object->hashkey);
//printf( "hey?\n" );
//  oc_free_object_data (object);
//printf( "hehehe?\n" );
  return TRUE;
}

/* ------------------------------------------------------------------------- *\
 * Delete an object from the cache and from the database
\* ------------------------------------------------------------------------- */
gboolean
oc_delete_object (const char *classname, const char *keystr)
{
  _ObjectData *obj;
  struct query_result *result;
  int err;
  char *errmsg;

  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  trace_functioncall ();

  /* remove from cache */
  obj = oc_find_cached_object_by_key (classname, keystr);
  if (obj)
    oc_remove_object (obj);

  /* remove from database */
  result = delete_from_objectstore (classname, keystr, &err, &errmsg);
  if (result)
    free_query_result (result);
  if (errmsg)
    {
      errormsg (errmsg);
      g_free (errmsg);
      timer_fail_operation( TIMER_CACHE);
      return (FALSE);
    }

  timer_done_operation( TIMER_CACHE );
  return (TRUE);
}

/* ------------------------------------------------------------------------- *\
 * Get the classname of an object
\* ------------------------------------------------------------------------- */
const char *
oc_get_object_class (ObjectData * object)
{
  return (object->classname);
}

/* ------------------------------------------------------------------------- *\
 * Get the key of an object
\* ------------------------------------------------------------------------- */
const char *
oc_get_object_key (ObjectData * object)
{
  return (object->key);
}

/* ------------------------------------------------------------------------- *\
 * See if a key exists in a given class
\* ------------------------------------------------------------------------- */
gboolean
oc_validate_object (const char *classname, const char *keystr)
{
  trace_functioncall ();
  if (oc_find_object_by_key (classname, keystr))
    return (TRUE);
  else
    return FALSE;
}

/* ------------------------------------------------------------------------- *\
 * Flush an object into the database
\* ------------------------------------------------------------------------- */
void
_fill_values (gchar * key, _FieldData * field, GHashTable * values)
{
  g_hash_table_insert (values, key, field->value);
}

void
oc_flush_object_to_store (ObjectData * obj)
{
  struct query_result *r = NULL;
  GHashTable *values;

  timer_start_operation( TIMER_CACHE , __PRETTY_FUNCTION__ );
  timer_start_profile( TIMER_FUNC_FLUSH );
  trace_functioncall ();
  /* TODO: error checking, updating object flags */

  /* create hash with pure name-values pairs */
  values = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_foreach (obj->fields, (GHFunc) _fill_values, values);

  r = write_to_objectstore (obj->classname, obj->key, values,
                            obj->indatabase, NULL, NULL);
  g_hash_table_destroy (values);
  if (r == NULL || r->success != TRUE)
    {
      free_query_result (r);
      timer_fail_operation( TIMER_CACHE);
      return;
    }
  free_query_result (r);
  flushable = oc_object_to_end_of_list (flushable, obj);
  oc_set_object_flag (obj, of_indatabase, FLAG_ON);
  if (oc_get_object_flag (obj, of_indatabase) != FLAG_ON)
    {
      printf ("TODO: error handler at %s/%d\n", __FILE__, __LINE__);
      exit (0);
    }
  timer_update_profile( TIMER_FUNC_FLUSH );
  timer_done_operation( TIMER_CACHE );
}

/* ========================================================================= *\
 * Object flags
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Set an object flag
\* ------------------------------------------------------------------------- */
void
oc_set_object_flag (ObjectData * object, enum object_flag flag,
                    gboolean state)
{
  trace_functioncall ();
  switch (flag)
    {
    case of_indatabase:
      object->indatabase = (state == FLAG_ON ? 1 : 0);
      break;
    case of_deleted:
      object->deleted = (state == FLAG_ON ? 1 : 0);
      break;
    }

}

/* ------------------------------------------------------------------------- *\
 * Read an object flag
\* ------------------------------------------------------------------------- */
gboolean
oc_get_object_flag (ObjectData * object, enum object_flag flag)
{
  trace_functioncall ();
  switch (flag)
    {
    case of_indatabase:
      return object->indatabase ? FLAG_ON : FLAG_OFF;
    case of_deleted:
      return object->deleted ? FLAG_ON : FLAG_OFF;
    }
  return (FLAG_OFF);
}

/* ========================================================================= *\
 * Data access
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Get the data of a field of an object
\* ------------------------------------------------------------------------- */
char *
oc_get_object_field (ObjectData * object, const char *fieldname)
{
  _FieldData *f = NULL;
  odl_class *cl = NULL;
  odl_field *ft = NULL;
  QueryData *q = NULL;
  /* ObjectData *o = NULL; */

  trace_functioncall ();
  timer_start_profile( TIMER_FUNC_GET_FIELD );

  flushable = oc_object_to_start_of_list (flushable, object);

  /* special case */
  if (g_strcasecmp ("objectid", fieldname) == 0)
    {
      timer_update_profile( TIMER_FUNC_GET_FIELD );
      return (g_strdup (object->key));
    }
  /* look for field name in object */
  f = g_hash_table_lookup (object->fields, fieldname);
  if (f)
    {
      /* found in cache, returnm it */
      timer_update_profile( TIMER_FUNC_GET_FIELD );
      return (g_strdup (f->value));
    }
  /* get class and field info */
  cl = odl_find_class (all_classes, object->classname, NULL);
  if (!cl) {
    timer_update_profile( TIMER_FUNC_GET_FIELD );
    return (NULL);
  }
  ft = odl_class_get_field (cl, fieldname);
  if (!ft) {
    timer_update_profile( TIMER_FUNC_GET_FIELD );
    return (NULL);
  }
  switch (odl_field_get_type (ft))
    {
    case FT_basic:
      /* basic field in this class - attempt to reload it */
      q = oql_load_object_field_by_key (object->classname, fieldname,
                                        object->key);
      if (q)
        {
          object = oc_objectstore_query_to_objectcache (q);
          oql_free_query (q);
          if (object)
            {
              f = g_hash_table_lookup (object->fields, fieldname);
              if (f)
                {
                  /* found in cache, returnm it */
		  timer_update_profile( TIMER_FUNC_GET_FIELD );
                  return (g_strdup (f->value));
                }
              /* still not found? damn.. */
	      timer_update_profile( TIMER_FUNC_GET_FIELD );
              return (NULL);
            }
        }
      break;
    case FT_lookup:
      /* lookup field not found - load it */
      {
        GList *s, *t;
        /* char *keystr; */
        char *value;
        /* char *targetfield; */
        /* char *errmsg; */
        odl_class *c;
        char *loadclass = (char *) odl_field_get_sourceclass (ft);
        ObjectData *o;
        /* int idfield; */
        /* struct query_result *result; */

        q = create_base_query (loadclass);
        if (!q)
          {
	    timer_update_profile( TIMER_FUNC_GET_FIELD );
            return (NULL);
          }
        /* add constraints to find just members of this lookup */
        /* ... WHERE loadclass.t1 = this.s1 AND loadclass.t2 = this.s2
           ... */
        s = odl_field_get_source_fields (ft);
        t = odl_field_get_this_fields (ft);
        while (s && t)
          {
            c = odl_field_defined_in (cl, (const char *) t->data);
            if (!c)
              {
                oql_free_query (q);
	        timer_update_profile( TIMER_FUNC_GET_FIELD );
                return (NULL);
              }
            o =
              oc_find_object_by_key (odl_class_get_full_name (c),
                                     object->key);
            if (!o)
              {
                oql_free_query (q);
	        timer_update_profile( TIMER_FUNC_GET_FIELD );
                return (NULL);
              }
            value = (char *) oc_get_object_field (o, t->data);
            if (!oql_add_query_constraint
                (q, NULL, (const char *) value, "=", loadclass,
                 (const char *) s->data))
              {
                oql_free_query (q);
	        timer_update_profile( TIMER_FUNC_GET_FIELD );
                return (NULL);
              }
            s = g_list_next (s);
            t = g_list_next (t);
          }
        if (s != NULL || t != NULL)
          {
            /* serious error in class def */
            criticalerror ("%s.%s has unbalanced field lists",
                           odl_class_get_full_name (cl), fieldname);
            oql_free_query (q);
	    timer_update_profile( TIMER_FUNC_GET_FIELD );
            return (NULL);
          }
        if (q)
          {
            o = oc_objectstore_query_to_objectcache (q);
            oql_free_query (q);
            if (o)
              {
                char *retval;
                retval =
                  oc_get_object_field (o, odl_field_get_sourcefield (ft));
                f =
                  g_hash_table_lookup (o->fields,
                                       odl_field_get_sourcefield (ft));
                if (f)
                  {
                    f->readonly = TRUE;
                  }
	        timer_update_profile( TIMER_FUNC_GET_FIELD );
                return (retval);
              }
          }
      }
      break;
    default:
      /* no other types handled yet */
      break;
    }

  /* didn't find it */
  timer_update_profile( TIMER_FUNC_GET_FIELD );
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * Set the data of a field of the object
\* ------------------------------------------------------------------------- */
gboolean
oc_set_object_field_quick (ObjectData * object, const char *fieldname,
                     const char *value, gboolean readonly)
{
  _FieldData *f;

  timer_start_profile( TIMER_FUNC_SET_FIELD );

  if (g_strcasecmp ("objectid", fieldname) == 0)
    {
      timer_update_profile( TIMER_FUNC_SET_FIELD );
      return (FALSE);
    }
  f = g_hash_table_lookup (object->fields, fieldname);
  if (!f)
    {
      f = oc_alloc_field_data (fieldname);
      if (f)
        {
          g_hash_table_insert (object->fields, f->name, f);
        }
    }
  if (f)
    {
      if (f->value)
        {
          g_free (f->value);
        }
      f->value = g_strdup (value);
    }

  timer_update_profile( TIMER_FUNC_SET_FIELD );
  return FALSE;
}

gboolean
oc_set_object_field (ObjectData * object, const char *fieldname,
                     const char *value, gboolean readonly)
{
  _FieldData *f;
  GList *idxs, *l, *ids;
  odl_class *c;

  trace_functioncall ();
  timer_start_profile( TIMER_FUNC_SET_FIELD );
  if (g_strcasecmp ("objectid", fieldname) == 0)
    {
      timer_update_profile( TIMER_FUNC_SET_FIELD );
      return (FALSE);
    }
  f = g_hash_table_lookup (object->fields, fieldname);
  if (!f)
    {
      f = oc_alloc_field_data (fieldname);
      if (f)
        {
          g_hash_table_insert (object->fields, f->name, f);
        }
    }

  if (f)
    {
      /* the first time this is called with 'readonly' set to true */
      /* the object's readonly field is set to true */
      /* effectively, this allows write-once behaviour so that */
      /* a readonly field can be set the first time */
      if (f->readonly)
        {
	  timer_update_profile( TIMER_FUNC_SET_FIELD );
          return (FALSE);
        }
      if (readonly)
        {
          f->readonly = TRUE;
        }
      /* make a list of all indexes that this field is attached to */
      idxs = NULL;
      ids = NULL;
      c = odl_find_class (all_classes, object->classname, NULL);
      if (c)
        {
          GList *l = c->indexes;
          while (l)
            {
              _odl_index *i = (_odl_index *) l->data;
              if (!i->primary
                  && odl_namelist_contains (i->fields, fieldname, FALSE))
                {
                  idxs = g_list_append (idxs, i);
                }
              l = g_list_next (l);
            }
        }
      if (idxs)
        {
          l = idxs;
          while (l)
            {
              oc_index_identifier *identifier = NULL;
              _odl_index *i = (_odl_index *) l->data;

              identifier = oc_make_index_identifier (object, i);
              ids = g_list_append (ids, identifier);

              l = l->next;
            }
          g_list_free (idxs);
        }

      /* remove the object from any indexes in the 'ids' list */
      if (ids)
        {
          GList *l = object->indexes;
          while (l)
            {
              _CacheIndex *i = l->data;
              if (odl_namelist_contains (ids, i->hashkey, TRUE))
                {
                  oc_remove_from_index (i, object);
                }
              l = g_list_next (l);
            }
        }
      /* update the value */
      if (f->value)
        {
          g_free (f->value);
        }
      f->value = g_strdup (value);
      /* add the object to any indexes in the 'ids' list */
      if (ids)
        {
          GList *l = ids;
          while (l)
            {
              /* TODO: verify that all NOT NULL fields in the index have data */
              oc_add_object_to_index (object, l->data);
              l = g_list_next (l);
            }
        }

      /* free list of index ids */
      if (ids)
        {
          oc_free_index_identifier_list (ids);
        }
      flushable = oc_object_to_start_of_list (flushable, object);
      timer_update_profile( TIMER_FUNC_SET_FIELD );
      return (TRUE);
    }
  timer_update_profile( TIMER_FUNC_SET_FIELD );
  return (FALSE);
}

/* ------------------------------------------------------------------------- *\
 * Free the memory of an object in cache
\* ------------------------------------------------------------------------- */
static void
oc_free_object_data (_ObjectData * o)
{
  GList *l;

  trace_functioncall ();
  /* printf( "freeing: %08lx\n" , o ); */
  if (g_list_find (flushable, o))
    {
      printf ("Still in flushable list : %px (very bad)\n", o);
      abort ();
    }
  if (o)
    {
      if (o->indexes)
        {
          l = o->indexes;
          while (l)
            {
              oc_remove_from_index (l->data, o);
              l = g_list_next (l);
            }
          g_list_free (o->indexes);
          o->indexes = NULL;
        }
      if (o->hashkey)
        {
          g_free (o->hashkey);
        }
      o->hashkey = NULL;
      if (o->key)
        {
          g_free (o->key);
        }
      o->key = NULL;
      if (o->classname)
        {
          g_free (o->classname);
        }
      o->classname = NULL;
      if (o->fields)
        {
          g_hash_table_foreach (o->fields, (GHFunc) oc_free_field_data, NULL);
          g_hash_table_destroy (o->fields);
        }
      o->fields = NULL;
      g_free (o);
    }
}

/* ------------------------------------------------------------------------- *\
 * Allocate the memory for an object in cache
\* ------------------------------------------------------------------------- */
static _ObjectData *
oc_allocate_object_data (const char *classname, const char *key)
{
  _ObjectData *o;

  trace_functioncall ();

  o = g_malloc (sizeof (_ObjectData));
  if (o)
    {
      /* store object ID info */
      o->classname = g_strdup (classname);
      o->key = g_strdup (key);
      o->hashkey = g_strdup_printf ("%s-%s", classname, key);
      /* field storage */
      o->fields = g_hash_table_new (g_str_hash, g_str_equal);
      /* clear flags */
      o->deleted = 0;
      o->indatabase = 0;
      if (!o->classname || !o->key || !o->fields || !o->hashkey)
        {
          oc_free_object_data (o);
          o = NULL;
        }
      /* new objects aren't in any indexes */
      o->indexes = NULL;
      o->userdata = NULL;
    }
  return (o);
}

/* ------------------------------------------------------------------------- *\
 * Allocate memory for a filed of an object
\* ------------------------------------------------------------------------- */
static _FieldData *
oc_alloc_field_data (const char *name)
{
  _FieldData *f = g_new0 (_FieldData, 1);

  trace_functioncall ();
  if (f)
    {
      f->readonly = FALSE;
      f->name = g_strdup (name);
      f->value = NULL;

      if (!f->name)
        {
          oc_free_field_data (NULL, f, NULL);
          f = NULL;
        }
    }
  return (f);
}

/* ------------------------------------------------------------------------- *\
 * Free the memory of a filed of an object
\* ------------------------------------------------------------------------- */
static void
oc_free_field_data (gpointer key, _FieldData * f, gpointer userdata)
{
  trace_functioncall ();
  if (f)
    {
      if (f->name)
        {
          g_free (f->name);
        }
      if (f->value)
        {
          g_free (f->value);
        }
      g_free (f);
    }
}

/* ------------------------------------------------------------------------- *\
 * Read objects from the database and put the data in the cache
\* ------------------------------------------------------------------------- */
static ObjectData *
oc_objectstore_query_to_objectcache (QueryData * q)
{
  ObjectData *first = NULL;
  struct query_result *result;
  ObjectData *o;
  DatabaseResultRow r;
  int row, field, idfield;

  trace_functioncall ();
  /* perform query */
  result = query_objectstore (q, NULL, NULL);
  if (!result)
    {
      return (NULL);
    }
  if (!result->success || result->rows_affected == 0)
    {
      free_query_result (result);
      return (NULL);
    }
  /* any objects get added to the cache */
  idfield = oql_query_get_field_position (q, "objectid");
  if (idfield == (-1))
    {
      errormsg ("no objectid in result from query %s\n",
                oql_query_as_sql (q, OQL_DBTYPE_CACHEONLY));
      free_query_result (result);
      return (NULL);
    }
  for (row = 0; row < result->rows_affected; row++)
    {
      r = get_result_row (result, row);
      o = oc_add_empty_object (get_result_class_name (result),
                               get_field_in_row (r, idfield));
      if (!o)
        {
          errormsg ("%s/%d: A TODO, really  :)", __FILE__, __LINE__);
          exit (0);
        }
      if (first == NULL)
        {
          first = o;
        }
      for (field = 0; field < result->field_count; field++)
        {
          if (field != idfield)
            {
/*          printf( "setting field %s to %s\n" , get_result_field_name(result, field) , get_field_in_row(r, field) ); */
              oc_set_object_field (o,
                                   get_result_field_name (result,
                                                          field),
                                   get_field_in_row (r, field), FALSE);
            }
        }
      oc_set_object_flag (o, of_indatabase, FLAG_ON);
    }
  free_query_result (result);
  /* return the first object found */
  return (first);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static oc_index_identifier *
oc_make_index_identifier (_ObjectData * object, _odl_index * i)
{
  oc_index_identifier *id = g_new0 (oc_index_identifier, 1);
  trace_functioncall ();
  if (id)
    {
      GList *f;
      GString *buf;

      buf = g_string_new (object->classname);
      g_string_append (buf, "/");
      f = i->fields;
      while (f)
        {
          g_string_append (buf, f->data);
          if (f->next)
            g_string_append (buf, ",");
          f = f->next;
        }
      id->classname = g_strdup (object->classname);
      id->indexid = buf->str;
      id->index = i;
      buf->str = NULL;
      g_string_free (buf, TRUE);
    }
  return (id);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
oc_free_index_identifier_list (GList * list)
{
  GList *l;
  oc_index_identifier *i;
  trace_functioncall ();

  l = list;
  while (l)
    {
      i = list->data;
      if (i)
        {
          if (i->classname)
            {
              g_free (i->classname);
            }
          if (i->indexid)
            {
              g_free (i->indexid);
            }
          g_free (i);
        }

      l = g_list_next (l);
    }
  if (list)
    {
      g_list_free (list);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
oc_remove_from_index (_CacheIndex * idx, _ObjectData * o)
{
  _CacheIndexLink *link;
  char *objhash = NULL;
  /* GString *buf = NULL; */
  GList *l;
  trace_functioncall ();

  /* remove object 'o' from index 'idx' */
  objhash = oc_make_object_index_hash (o, idx->index);
  /* printf( "This object (%s/%s) is in index table '%s' (%s) (remove it here)\n" ,
     o->classname , o->key , idx->hashkey , objhash ); */

  /* remove it from the index in idx */
  link = g_hash_table_lookup (idx->hashtable, objhash);
  if (link)
    {
      /* printf( "Found the link\n" ); */
      l = link->objects;
      while (l)
        {
          ObjectData *o2;

          o2 = l->data;
          if (strcmp (o2->key, o->key) == 0)
            {
              link->objects = g_list_remove (link->objects, o2);
              /* printf( "Remove done\n" ); */
              l = NULL;
            }
          else
            {
              l = g_list_next (l);
            }
        }
      if (link->objects == NULL)
        {
          g_hash_table_remove (idx->hashtable, objhash);
          g_free (link->hashkey);
          g_free (link);
          /* printf( "Complete remove done\n" ); */
        }
    }
  g_free (objhash);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
oc_add_object_to_index (_ObjectData * object, oc_index_identifier * indexid)
{
  _CacheIndex *idx = NULL;
  _CacheIndexLink *link;
  char *objhash = NULL;
  /* GString *buf = NULL; */
  /* GList *l; */

  trace_functioncall ();

  /* insert object object into index table */
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: add object %s/%s to index %s\n",
              object->classname, object->key, indexid->indexid);
    }
  objhash = oc_make_object_index_hash (object, indexid->index);
  if (!objhash)
    {
      return;
    }
  /* find the index, if it exists */
  idx = g_hash_table_lookup (indexes, indexid->indexid);
  if (!idx)
    {
      if (debuglevel == OC_TEMP_DEBUG)
        {
          printf ("OCCACHE: adding new index '%s'\n", indexid->indexid);
        }
      /* nope, so create a new index */
      idx = g_new0 (_CacheIndex, 1);
      if (!idx)
        {
          return;
        }
      idx->hashkey = g_strdup (indexid->indexid);
      idx->hashtable = g_hash_table_new (g_str_hash, g_str_equal);
      idx->index = indexid->index;
      g_hash_table_insert (indexes, idx->hashkey, idx);
    }
  /* find the index link that should hold this object */
  link = g_hash_table_lookup (idx->hashtable, objhash);
  if (!link)
    {
      /* first time an objexct with this key was placed in the index */
      if (debuglevel == OC_TEMP_DEBUG)
        printf ("OCCACHE: adding new link\n");
      link = g_new0 (_CacheIndexLink, 1);
      if (!link)
        {
          g_free (objhash);
          return;
        }
      link->hashkey = g_strdup (objhash);
      link->objects = NULL;
      g_hash_table_insert (idx->hashtable, link->hashkey, link);
    }
  g_free (objhash);
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: link hash = %s\n", link->hashkey);
    }
  /* add the object to the link */
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: placing object in index\n");
    }
  link->objects = g_list_append (link->objects, object);
  object->indexes = g_list_append (object->indexes, idx);
}

/**
 * Use the query to determine an appropriate index to search for the
 * requested object.
 * \return A pointer to an _ObjectData structure containing the details
 * of an object matching the query
 */
ObjectData *
oc_search_for_single_object (_QueryData * q)
{
  _CacheIndex *idx = NULL;
  _CacheIndexLink *link;
  char *classname;
  GString *buf;
  /* GString *hash; */
  GList *l;
  /* gboolean needcomma = TRUE; */
  char *key = NULL;

  trace_functioncall ();
//   return( NULL ); /* XYZ */
  timer_start_profile( TIMER_FUNC_E );
  if (!q)
    {
  timer_update_profile( TIMER_FUNC_E );
      return (NULL);
    }
  /* name of class to search for */
  if (!q->classes)
    {
  timer_update_profile( TIMER_FUNC_E );
      return (NULL);
    }
  classname = q->classes->data;
  if (!classname)
    {
  timer_update_profile( TIMER_FUNC_E );
      return (NULL);
    }
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: Searching for single object in cache\n");
    }
  /* if just 'objectID' field, use main index */
  if (g_list_length (q->fields) == 1)
    {
      if (g_strcasecmp ("objectid", q->fields->data) == 0)
        {
          if (debuglevel == OC_TEMP_DEBUG)
            printf ("OCCACHE: use objectID to search main cache\n");
        }
    }
  timer_update_profile( TIMER_FUNC_E );

  timer_start_profile( TIMER_FUNC_F );
  /* fields to match */
  buf = g_string_new (classname);
  g_string_append (buf, "/");
  l = q->conditions;
  while (l)
    {
      _QueryCondition *c = l->data;

      if (g_strcasecmp (c->targetclass, classname) == 0 &&
          c->relation[0] == '=')
        {
          g_string_append (buf, c->field);
          if (l->next)
            {
              g_string_append (buf, ",");
            }
          if (strcmp (c->field, "objectid") == 0)
            {
              key = c->value;
            }
        }
      l = l->next;
    }
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: Look in cache '%s'\n", buf->str);
    }
  timer_update_profile( TIMER_FUNC_F );


  timer_start_profile( TIMER_FUNC_G );
  idx = g_hash_table_lookup (indexes, buf->str);
  if (!idx)
    {
      if (strcmp (&buf->str[strlen (classname) + 1], "objectid") == 0)
        {
          /* look in main table */
          g_string_free (buf, TRUE);
  timer_update_profile( TIMER_FUNC_G );
          return (oc_find_cached_object_by_key (classname, key));
        }
      if (debuglevel == OC_TEMP_DEBUG)
        {
          printf ("could not find index %s\n", buf->str);
        }
      g_string_free (buf, TRUE);
  timer_update_profile( TIMER_FUNC_G );
      return (NULL);
    }
  g_string_free (buf, TRUE); 
  timer_update_profile( TIMER_FUNC_G );


  /* make appropriate object hash key for finding 'classname' in 'idx' */
  timer_start_profile( TIMER_FUNC_H );
  buf = g_string_new ("");
  l = idx->index->fields;
  while (l)
    {
      GList *cl;
      char *field;
      _QueryCondition *c;

      /* find condition 'classname.(l->data) = value' */
      if (debuglevel == OC_TEMP_DEBUG)
        {
          printf ("OCCACHE:  index field: %s\n", (char *) (l->data));
        }
      cl = q->conditions;
      field = NULL;
      while (cl)
        {
          c = cl->data;
          if (g_strcasecmp (c->field, l->data) == 0)
            {
              if (debuglevel == OC_TEMP_DEBUG)
                {
                  printf ("OCCACHE:    value: %s\n", c->value);
                }
              field = c->value;
              cl = NULL;
            }
          else
            cl = g_list_next (cl);
        }

      if (field)
        {
          g_string_append (buf, field);
          if (l->next)
            {
              g_string_append (buf, ":");
            }
        }
      else
        {
          if (debuglevel == OC_TEMP_DEBUG)
            {
              printf ("OCCACHE: failed\n");
            }
          g_string_free (buf, TRUE);
  timer_update_profile( TIMER_FUNC_H );
          return (NULL);
        }

      l = g_list_next (l);
    }
  timer_update_profile( TIMER_FUNC_H );

  /* found an index, now look for the object in that index */
  timer_start_profile( TIMER_FUNC_I );
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: looking for '%s' in index\n", buf->str);
    }
  link = g_hash_table_lookup (idx->hashtable, buf->str);
  g_string_free (buf, TRUE);

  if (!link)
    {
  timer_update_profile( TIMER_FUNC_I );
      return (NULL);
    }
  if (!link->objects)
    {
  timer_update_profile( TIMER_FUNC_I );
      return (NULL);
    }
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: found\n");
    }
  fflush (NULL);
  timer_update_profile( TIMER_FUNC_I );
  return (link->objects->data);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
GList *
oc_search_for_objects (_QueryData * q)
{
  /* This is not yet implemented */
  fatal_error ("Not implemented: Searching for multiple objects in cache\n");
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static char *
oc_make_object_index_hash (_ObjectData * object, _odl_index * idx)
{
  GList *l;
  GString *buf;
  char *retval = NULL;
  odl_class *c;
  odl_field *f;

  trace_functioncall ();
  /* get the contents of the object's fields and add them to the object hash */
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: creating object's index hash key\n");
    }
  l = idx->fields;
  buf = g_string_new ("");
  c = odl_find_class (all_classes, object->classname, NULL);
  while (l)
    {
      /* TODO add a quote character to fields to quote any : characters */
      char *field;

      field = oc_get_object_field (object, l->data);
      f = odl_class_get_field (c, l->data);
      if (field)
        {
          g_string_append (buf, field);
        }
      else if ((f->properties & ODL_PROP_NOTNULL) != ODL_PROP_NOTNULL)
        {
          /* NULL field, allowed */
          g_string_append (buf, "(null)");
        }
      else
        {
          /* NULL field, with NOT NULL property set */
          g_string_free (buf, TRUE);
          if (debuglevel == OC_TEMP_DEBUG)
            {
              printf
                ("OCCACHE: NOT NULL field (%s) is NULL - can not index\n",
                 (char *) (l->data));
            }
          return (NULL);
        }
      if (l->next)
        {
          g_string_append (buf, ":");
        }
      l = g_list_next (l);
    }
  retval = buf->str;
  buf->str = NULL;
  g_string_free (buf, FALSE);
  if (debuglevel == OC_TEMP_DEBUG)
    {
      printf ("OCCACHE: object hash key: %s\n", retval);
    }
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static GList *
oc_object_to_start_of_list (GList * l, ObjectData * obj)
{
  if (g_list_find (l, obj))
    {
      l = g_list_remove (l, obj);
      if (g_list_find (l, obj))
        {
          printf ("%px is still in list (bad)\n", obj);
          abort ();
        }
      l = g_list_prepend (l, obj);
#if 0
      if (g_list_find (l, obj))
        {
          printf ("%px is back in list (good)\n", obj);
        }
#endif
    }
  return (l);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static GList *
oc_object_to_end_of_list (GList * l, ObjectData * obj)
{
  if (g_list_find (l, obj))
    {
      l = g_list_remove (l, obj);
      if (g_list_find (l, obj))
        {
          printf ("%px is still in list (bad)\n", obj);
          abort ();
        }
      l = g_list_append (l, obj);
#if 0
      if (g_list_find (l, obj))
        {
          printf ("%08lx is back in list (good)\n", obj);
        }
#endif
    }
  return (l);
}
