/*
   Compare existing tables to class definition
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 
   $Id: compare.c,v 1.3 2001/07/25 14:53:11 reinhard Exp $
*/

#include "config.h"
#include "compare.h"
#include <stdio.h>
#include <string.h>
#include <glib.h>
#include "classdata.h"
#include "oql/oql.h"

static DBchange *find_named_change (GList *l, char *name);

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
show_database_definition (DatabaseDefinition * def)
{
  FILE *fp = stdout;
  GList *tl, *cl;
  DatabaseTable *t;
  DatabaseColumn *c;

  if (!def)
    {
      fprintf (fp, "Empty database definition\n\n");
      return;
    }

  fprintf (fp, "Database Definition: %s\n", def->name);
  tl = def->tables;
  if (!tl)
    {
      fprintf (fp, "Empty database definition\n");
    }
  while (tl)
    {
      t = (DatabaseTable *) tl->data;
      fprintf (fp, "  Table %s\n", t->name);
      cl = t->columns;
      while (cl)
        {
          c = (DatabaseColumn *) cl->data;
          fprintf (fp, "    Column %s\t%s [%d]\n", c->name,
                   odl_datatype_name (c->datatype), c->datatype);
          cl = g_list_next (cl);
        }
      fprintf (fp, "\n");       /* space between tables */
      tl = g_list_next (tl);
    }
  fprintf (fp, "---\n");        /* space at end of database */
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DatabaseDefinition *
create_database_definition (const char *name)
{
  DatabaseDefinition *db = g_new0 (DatabaseDefinition, 1);
  g_assert (name != NULL);
  g_assert (db != NULL);

  if (db)
    {
      db->tables = NULL;
      db->name = g_strdup (name);
      if (!db->name)
        {
          free_database_definition (db);
          db = NULL;
        }
    }
  return (db);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
free_database_column_list (GList * list)
{
  DatabaseColumn *c;
  GList *l;
  g_assert (list != NULL);

  l = list;
  if (l)
    {
      while (l)
        {
          c = (DatabaseColumn *) l->data;
          if (c)
            {
              if (c->name)
                g_free (c->name);
              g_free (c);
            }
          l = g_list_next (l);
        }
      g_list_free (list);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
free_database_table_list (GList * list)
{
  DatabaseTable *t;
  GList *l;
  g_assert (list != NULL);

  l = list;
  if (l)
    {
      while (l)
        {
          t = (DatabaseTable *) l->data;
          if (t)
            {
              if (t->name)
                g_free (t->name);
              if (t->columns)
                free_database_column_list (t->columns);
              g_free (t);
            }
          l = g_list_next (l);
        }
      g_list_free (list);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_database_definition (DatabaseDefinition * def)
{
  g_assert (def != NULL);

  if (def)
    {
      if (def->name)
        g_free (def->name);
      if (def->tables)
        free_database_table_list (def->tables);
      g_free (def);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
gboolean
add_database_table (DatabaseDefinition * def, const char *name)
{
  DatabaseTable *t = g_new0 (DatabaseTable, 1);
  g_assert (def != NULL);
  g_assert (name != NULL);
  g_assert (t != NULL);

  if (!t)
    {
      return (FALSE);
    }
  t->name = NULL;
  t->columns = NULL;
  if (name)
    {
      t->name = g_strdup (name);
    }

  def->tables = g_list_append (def->tables, t);
  return (TRUE);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
count_database_tables (DatabaseDefinition * def)
{
  g_assert (def != NULL);
  /* g_assert (def->tables != NULL); */

  return (g_list_length (def->tables));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DatabaseTable *
get_database_table (DatabaseDefinition * def, int which)
{
  g_assert (def != NULL);
  g_assert (def->tables != NULL);

  return ((DatabaseTable *) g_list_nth_data (def->tables, which));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static DatabaseTable *
find_database_table (DatabaseDefinition *def, const char *name, DBType db)
{
  GList *l;
  char *quoted;

  g_assert (def != NULL);
  g_assert (name != NULL);

  l = def->tables;
  while (l)
    {
      /* We have to quote this because the database can return the unquoted
         names */
      quoted = oql_quote_column (((DatabaseTable *) l->data)->name, NULL, db);
      if (!g_strcasecmp (quoted, name))
        {
          return ((DatabaseTable *) l->data);
        }
      l = g_list_next (l);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static DatabaseColumn *
find_databasetable_field (DatabaseTable * t, const char *name, DBType db)
{
  GList *l;
  char *quoted;

  g_assert (t != NULL);
  g_assert (name != NULL);

  l = t->columns;
  while (l)
    {
      /* We have to quote this because the database can return the unquoted
         names */
      quoted = oql_quote_column (NULL, ((DatabaseColumn *) l->data)->name, db);
      if (!g_strcasecmp (quoted, name))
        {
          g_free (quoted);
          return (l->data);
        }
      g_free (quoted);
      l = g_list_next (l);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DatabaseColumn *
add_database_column (DatabaseTable * table, const char *name,
                     enum odl_datatype type)
{
  DatabaseColumn *c = g_new0 (DatabaseColumn, 1);
  g_assert (table != NULL);
  g_assert (name != NULL);
  g_assert (c != NULL);

  if (!c)
    {
      return (NULL);
    }
  if (name)
    {
      c->name = g_strdup (name);
    }
  else
    {
      c->name = NULL;
    }
  c->datatype = type;
  table->columns = g_list_append (table->columns, c);
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DatabaseChange *
create_database_change (void)
{
  DatabaseChange *c = g_new0 (DatabaseChange, 1);
  g_assert (c != NULL);

  c->adds = NULL;
  c->removes = NULL;
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
free_database_change (DatabaseChange * change)
{
}

/* ------------------------------------------------------------------------- *\
 * add or remove a table. get ptr to DBchange for adding column info 
\* ------------------------------------------------------------------------- */
DBchange *
dbchange_add_table (DatabaseChange * dbchange, const char *name)
{
  DBchange *c = g_new0 (DBchange, 1);
  char *p;
  g_assert (c != NULL);
  g_assert (dbchange != NULL);
  g_assert (name != NULL);

  if (!c)
    {
      return (NULL);
    }
  c->type = DBCH_ADD_TABLE;
  p = (char *) name;
  if (strncmp (name, "root_", 5) == 0)
    {
      p = (char *) &name[5];
    }
  c->name = g_strdup (p);
  c->datatype = DT_unknown;
  c->format = NULL;
  c->columns = NULL;
  c->notnull = FALSE;
  dbchange->adds = g_list_append (dbchange->adds, c);
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DBchange *
dbchange_remove_table (DatabaseChange * dbchange, const char *name)
{
  DBchange *c = g_new0 (DBchange, 1);
  g_assert (c != NULL);
  g_assert (dbchange != NULL);
  g_assert (name != NULL);

  if (!c)
    {
      return (NULL);
    }
  c->type = DBCH_REMOVE_TABLE;
  c->name = g_strdup (name);
  c->datatype = DT_unknown;
  c->format = NULL;
  c->columns = NULL;
  c->notnull = FALSE;
  dbchange->removes = g_list_append (dbchange->removes, c);
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * add or remove a column to/from a table
\* ------------------------------------------------------------------------- */
DBchange *
dbchange_add_column (DBchange * change, const char *name,
                     enum odl_datatype datatype, const char *format,
                     gboolean notnull)
{
  DBchange *c = g_new0 (DBchange, 1);
  g_assert (c != NULL);
  g_assert (change != NULL);
  g_assert (name != NULL);

  if (!c)
    {
      return (NULL);
    }
  c->type = DBCH_ADD_COLUMN;
  c->name = g_strdup (name);
  c->datatype = datatype;
  if (format)
    {
      c->format = g_strdup (format);
    }
  else
    {
      c->format = NULL;
    }
  c->columns = NULL;
  c->notnull = notnull;
  change->columns = g_list_append (change->columns, c);
  return (c);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DBchange *
dbchange_remove_column (DBchange * change, const char *name)
{
  return NULL;
}


/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
DBchange *
find_named_change (GList * l, char *name)
{
  g_assert (name != NULL);
  while (l) /* if l == NULL, then return 'not in list' value */
    {
      if (g_strcasecmp (((DBchange *) l->data)->name, name) == 0)
        {
          return (l->data);
        }
      l = g_list_next (l);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * scan a database definition and compare to the current class definitions
 * if there's a difference, produce a DatabaseChange object, which can be
 * used to make
\* ------------------------------------------------------------------------- */
static gboolean
is_class_required (GList * list, const char *name)
{
  char *tmp;
  g_assert (list != NULL);
  g_assert (name != NULL);
  while (list)
    {
      if (!list || !list->data)
        {
          tmp = "??";
        }
      else
        {
          tmp = odl_mangle_qualified_name (list->data);
        }
      if (g_strcasecmp (name, tmp) == 0)
        {
          g_free (tmp);
          return (TRUE);
        }
      g_free (tmp);
      list = g_list_next (list);
    }
  return (FALSE);
}

/* ------------------------------------------------------------------------- *\
 * Make a complete list of needed changes in the database schema
\* ------------------------------------------------------------------------- */
DatabaseChange *
compare_classes_to_database (odl_tree * tree, GList * requiredclasses,
                             DatabaseDefinition * database, DBType db)
{
  GList *l, *fl;
  DatabaseChange *c = create_database_change ();
  DatabaseTable *t;
  DBchange *ch;
  odl_class *cl;
  g_assert (tree != NULL);
  g_assert (requiredclasses != NULL);
  g_assert (database != NULL);
  g_assert (c != NULL);

  /* for each table found, if there isn't a matching required class, remove
     the table */
  if (!database)
    {
      return (NULL);
    }
  l = database->tables;
  while (l)
    {
      if (is_class_required
          (requiredclasses, ((DatabaseTable *) l->data)->name) == FALSE)
        {
          /* class matching this table isn't required */
          dbchange_remove_table (c, ((DatabaseTable *) l->data)->name);
        }
      l = g_list_next (l);
    }

  /* for each class, if there isn't a matching table, add the table */
  l = requiredclasses;
  while (l)
    {
      char *tmp;

      tmp = oql_quote_column ((const char *) l->data, NULL, db);
      t = find_database_table (database, tmp, db);

      if (!t)
        {
          GList *x;
          ch = dbchange_add_table (c, tmp);

          /* adding full table, so note the basic fields required */
          cl = odl_find_class (tree, (const char *) l->data, NULL);
          fl = odl_class_get_fields (cl, FT_basic);     /* no inherited fields */

          x = fl;
          while (x)
            {
              odl_field *f = x->data;
              dbchange_add_column (ch,
                                   odl_field_get_name (f),
                                   odl_field_get_datatype (f),
                                   odl_field_get_format (f),
                                   odl_field_has_property (f,
                                                           ODL_PROP_NOTNULL));
              x = g_list_next (x);
            }
          odl_fieldlist_free (fl);
        }

      g_free (tmp);
      l = g_list_next (l);
    }

  /* for each column in an existing table: if the matching class doesn't
     have that field, remove it */
  l = database->tables;
  while (l)
    {
      /* find any fields in the table that don't match a column in the
         class (remove them) */
      /* TODO */
      l = g_list_next (l);
    }

  l = requiredclasses;
  while (l)
    {
      char *tmp;

      tmp = oql_quote_column ((const char *) l->data, NULL, db);
      t = find_database_table (database, tmp, db);
      g_free (tmp);
      tmp = NULL;
      cl = odl_find_class (tree, (const char *) l->data, NULL);
      if (t && cl)
        {
          GList *l2;
          /* find any fields in the class not matching columns in the
             table (add them) */

          fl = odl_class_get_fields (cl, FT_basic);     /* no inherited fields */
          l2 = fl;
          while (l2)
            {
              odl_field *f = (odl_field *) l2->data;
              char *quoted = oql_quote_column (NULL, odl_field_get_name (f),
                                               db);
              if (!find_databasetable_field (t, quoted, db))
                {
                  ch = find_named_change (c->adds, t->name);
                  if (!ch)
                    {
                      char *tmp;
                      /* first change related to this table,
                         so make the "add table" item */
                      tmp = odl_mangle_qualified_name (l->data);
                      ch = dbchange_add_table (c, tmp);
                      g_free (tmp);
                      ch->type = DBCH_MODIFY_TABLE;
                    }
                  if (ch)
                    {
                      dbchange_add_column (ch,
                                           odl_field_get_name (f),
                                           odl_field_get_datatype (f),
                                           odl_field_get_format (f),
                                           odl_field_has_property (f,
                                                                   ODL_PROP_NOTNULL));

                    }
                }
              g_free (quoted);
              l2 = g_list_next (l2);
            }

          odl_fieldlist_free (fl);
        }
      l = g_list_next (l);
    }
  return (c);
}
