/*
   postgresql_access.c - Interface to PostgreSQL Database
                         ALL PostgreSQL code should go into this file.

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: postgresql.c,v 1.11 2001/07/25 12:51:08 reinhard Exp $
*/

#include "config.h"
#include <string.h>
#include "objectstore.h"
#include "objectstore_private.h"
#include "geas-server.h"

#ifdef USE_POSTGRESQL
#include "libpq-fe.h"

/* Private Functions */
struct active_connection * postgresql_get_connection (struct database_handle *hnd);
void postgres_notice_processor (void *arg, const char *msg);
void postgresql_add_column_data (GString *buf, GList *l, const char *separator);
struct query_result *
postgresql_update_tables (struct database_handle *hnd,
                          gboolean remove_items, int *errorcode,
                          char **errormsg);

/* ========================================================================= *\
 * Database specific data structures
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Connection
\* ------------------------------------------------------------------------- */
struct postgresql_connection
{
  /* the first line must be: struct active_connection base; */
  struct active_connection base;

  /* any postgresql specific data goes here */
  /* one structure is allocated per connection to a database */
  PGconn *handle;
};

/* ------------------------------------------------------------------------- *\
 * Database handle
\* ------------------------------------------------------------------------- */
struct postgresql_handle
{
  /* the first line must be: struct database_handle base; */
  struct database_handle base;

  /* any postgresql specific data goes here */
  /* one structure is allocated per distinct database in use */
};

/* ========================================================================= *\
 * Database specific functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * A hack (begin transaction) - supposed to improve database speed, no other
 * effect
\* ------------------------------------------------------------------------- */
void postgresql_hack_begin(struct database_handle *ph);
void postgresql_hack_begin(struct database_handle *ph)
{
  struct postgresql_connection *conn;
  conn = (struct postgresql_connection *) ph->get_connection (ph);
  PQexec (conn->handle, "BEGIN TRANSACTION" );
  conn->base.available = TRUE;
}

/* ------------------------------------------------------------------------- *\
 * A hack (commit transaction) - supposed to improve database speed, no other
 * effect
\* ------------------------------------------------------------------------- */
void postgresql_hack_commit(struct database_handle *ph);
void postgresql_hack_commit(struct database_handle *ph)
{
  struct postgresql_connection *conn;
  conn = (struct postgresql_connection *) ph->get_connection (ph);
  PQexec (conn->handle, "COMMIT TRANSACTION" );
  conn->base.available = TRUE;
}

/* ------------------------------------------------------------------------- *\
 * handle postgresql produced NOTICE messages
\* ------------------------------------------------------------------------- */
void
postgres_notice_processor (void *arg, const char *msg)
{
  /* only print notices when debugging, for now */
  debug_output (DEBUGLEVEL_10, msg);
}

/* ------------------------------------------------------------------------- *\
 * find a connection to the database that is available for use
\* ------------------------------------------------------------------------- */
struct active_connection *
postgresql_get_connection (struct database_handle *hnd)
{
  struct postgresql_handle *h = (struct postgresql_handle *) hnd;
  struct active_connection *retval = NULL;
  unsigned int i;

  /*
     * find available connection:
     * first available and connected connection 
     * else first available and non connected and connect it immediately
     * else first connected with lowest queue
     * else fail
     */

  /* available and connected */
  /* TODO: make connection check depend on actual database state, to detect *
     unexpected disconnections */
  for (i = 0; i < h->base.num_connections; i++)
    {
      retval =
        (struct active_connection *) &((struct postgresql_connection *)
                                       h->base.connections)[i];
      if (retval->available && retval->connected)
        {
          retval->index = i;
          retval->available = FALSE;
          return (retval);
        }
    }

  /* available and not connected, if connection succeeds */
  for (i = 0; i < h->base.num_connections; i++)
    {
      retval =
        (struct active_connection *) &((struct postgresql_connection *)
                                       h->base.connections)[i];
      if (retval->available && !retval->connected)
        {
          /* try to connect to database */
          if (hnd->connect (hnd, i))
            {
              return (retval);
            }
        }
    }

  fprintf (stderr, "\nTODO: handle no connection\n");
  fprintf (stderr,
           "needs to wait until a connection becomes available, as another\n");
  fprintf (stderr, "thread has the connection\n\n");
  abort ();
}

/* ------------------------------------------------------------------------- *\
 * activate a connection, so it can be used
\* ------------------------------------------------------------------------- */
static gboolean
postgresql_connect (struct database_handle *hnd, unsigned long int index)
{
  gboolean retval = FALSE;
  struct postgresql_connection *c;
  char *pgoptions, *pgtty;
  char tmpport[20];
  PGresult *res;

  pgtty = NULL;
  pgoptions = NULL;

  /* range check */
  if (index >= hnd->num_connections)
    {
      errormsg
        ("Invalid connection index, your trying to access a handle not registered");
      return (FALSE);
    }

  /* get connection, point it to the handle connection */
  c =
    (struct postgresql_connection *) &((struct postgresql_connection *)
                                       hnd->connections)[index];

  /* TODO: imaginary thread safety for andrewm. */
  if (!c->base.connected)
    {
      sprintf (tmpport, "%lu", hnd->port);
      c->handle =
        PQsetdbLogin (hnd->hostname, tmpport, pgoptions, pgtty,
                      hnd->dbname, hnd->username, hnd->password);
      PQsetNoticeProcessor (c->handle, postgres_notice_processor, NULL);

      /* check to see that the backend connection was successfully made */
      if (c->handle == NULL)
        {
          errormsg ("failed to initilise postgresql");
          return (FALSE);
        }

      /* it kinda got a connection but got refused for some reason */
      if (PQstatus (c->handle) == CONNECTION_BAD)
        {
          errormsg ("Connection to database '%s' failed: %s\n", hnd->dbname,
                    PQerrorMessage (c->handle));
          PQfinish (c->handle);
          c->handle = NULL;
          return (FALSE);
        }
      /* do a test listen */
      res = PQexec (c->handle, "LISTEN TBL2");
      if (PQresultStatus (res) != PGRES_COMMAND_OK)
        {
          errormsg ("LISTEN command failed on postgresql connection\n");
          PQclear (res);
          PQfinish (c->handle);
          c->handle = NULL;
          return (FALSE);
        }
      PQclear (res);

      c->base.connected = TRUE;
      retval = TRUE;
    }
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * deactivate a connection
\* ------------------------------------------------------------------------- */
static gboolean
postgresql_disconnect (struct database_handle *hnd, unsigned long int index)
{
  struct postgresql_connection *c;

  /* range check */
  if (index >= hnd->num_connections)
    return (FALSE);

  /* get connection */
  c =
    (struct postgresql_connection *) &((struct postgresql_connection *)
                                       hnd->connections)[index];

  /* if not connected, 'succeed' at disconnecting */
  if (!c->base.connected)
    return (TRUE);

  /* TODO: andru's imagainary thread safety */
  PQfinish (c->handle);
  c->handle = NULL;
  c->base.connected = FALSE;
  return (TRUE);
}

/* ------------------------------------------------------------------------- *\
 * perform an arbitrary query
\* ------------------------------------------------------------------------- */
static struct query_result *
postgresql_execute_query (struct database_handle *ph,
                          QueryData * query, int *errorcode, char **errormsg)
{
  int dbtype;
  struct query_result *result = new_query_result ();       /* result pointer */
  /* convert to actual type */
  struct postgresql_connection *conn;
  odl_class *c;
  char *fieldname;

  PGresult *res = NULL;
  int row;
  unsigned int tries = 0;
  int PQerrorcode;

  /* claim a connection */
  conn = (struct postgresql_connection *) ph->get_connection (ph);
  if (!conn)
    {
      errormsg ("No available connection : should have aborted by now.\n");
      abort ();
    }

  if (!result)
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("out of memory");
      return (NULL);
    }

  dbtype = OQL_DBTYPE_POSTGRESQL;
  debug_output (DEBUGLEVEL_HIGH, "SQL query: '%s'",
                oql_query_as_sql (query, dbtype));

  c = odl_find_class (all_classes, oql_query_get_classname (query), NULL);
retry:
  tries++;
/* printf( "[%s]\n" , oql_query_as_sql(query,dbtype) ); */
  timer_start_profile( TIMER_FUNC_EXECQUERY );
  res = PQexec (conn->handle, oql_query_as_sql (query, dbtype));
  timer_update_profile( TIMER_FUNC_EXECQUERY );
  PQerrorcode = PQresultStatus (res);
#if 0
  if (PQerrorcode != PGRES_FATAL_ERROR)
    {
      char buf[16];
      printf ("msg: '%s'\n", PQresultErrorMessage (res));
      printf ("Fake database failure? (Y/N) ");
      fflush (NULL);
      gets (buf);
      if (buf[0] == 'y' || buf[0] == 'Y')
        {
          PQerrorcode = PGRES_FATAL_ERROR;
          printf ("PQerrorcode = %d\n", PQerrorcode);
        }
    }
#endif
  if (PQerrorcode == PGRES_COMMAND_OK)        /* query does not return rows,
                                           and didn't - ok */
    {
      result->success = TRUE;
      result->rows_affected = 0;
      /* message ("[ok, no rows expected] rows affected: %d",
        result->rows_affected ); */
    }
  else if (PQerrorcode == PGRES_TUPLES_OK)        /* returned some rows of
                                                   data */
    {
      /* message ("[ok, data rows expected] rows affected: %d",
        result->rows_affected ); */
  timer_start_profile( TIMER_FUNC_HANDLERESULTS );
      result->rows_affected = PQntuples (res);
      result->field_count = PQnfields (res);
      result->success = TRUE;

      /* for each entry, extract object ID */
      for (row = 0; row < result->rows_affected; row++)
        {
          int i;
          DatabaseResultRow_t r = NULL;

          /* the result set is a linked list of rows */
          /* each row is in turn a linked list of strings */
          /* where entry N is field N, listed left to right */
          /* in the SELECT query: */
          /* SELECT objectID,name,age : */
          /* objectID = field 0, name = field 1, age = field 2 */
          /* first row returned = row 0 (start of list), 2nd = row 1 (next
             in list) */
          for (i = 0; i < result->field_count; i++)
            {
              char *val = PQgetvalue (res, row, i);
              char *p = val;
              /* printf( " >  '%s'  " , val ); */
              while (p[strlen (p) - 1] == ' ')
                p[strlen (p) - 1] = '\0';
              /* printf( " '%s'\n" , val ); */
              fieldname = (char *) oql_query_get_field_name (query, i);
              val = oql_translate_from_read (val, c, fieldname, dbtype);
              /* add_field_to_result_row(r,val); */
              r = g_list_append(r,val); /* add_field_to_result_row */
            }
          result->data = g_list_prepend (result->data, r);
        }
      if (res != NULL)
        PQclear (res);
      res = NULL;
  timer_update_profile( TIMER_FUNC_HANDLERESULTS );
    }
  else
    {
      switch (PQerrorcode)
        {
        case PGRES_FATAL_ERROR:
          debug_output (DEBUGLEVEL_7, "PostgreSQL reported a fatal error");
          break;
        case PGRES_NONFATAL_ERROR:
          debug_output (DEBUGLEVEL_7,
                        "PostgreSQL reported a non fatal error");
          break;
        case PGRES_BAD_RESPONSE:
          debug_output (DEBUGLEVEL_7, "PostgreSQL reported a bad response");
          break;
        case PGRES_COPY_IN:
          debug_output (DEBUGLEVEL_7, "PostgreSQL reported a copy in error");
          break;
        case PGRES_COPY_OUT:
          debug_output (DEBUGLEVEL_7, "PostgreSQL reported a copy out error");
          break;
        case PGRES_EMPTY_QUERY:
          debug_output (DEBUGLEVEL_7, "PostgreSQL reported an empty query");
          break;
        }
      debug_output (DEBUGLEVEL_7, "PostgreSQL Error message: %s (%d)",
                    PQresultErrorMessage (res), PQerrorcode);

      if (strncmp
          (PQresultErrorMessage (res),
           "ERROR:  ExecAppend: Fail to add null value in not null attribute",
           64) == 0)
        {
          PQerrorcode = PGRES_NONFATAL_ERROR;
        }

      if (PQerrorcode == PGRES_FATAL_ERROR)
        {
          /* something really bad happened: */
          /*   disconnect & reconnect then try again */
          if (res != NULL)
            PQclear (res);
          res = NULL;
          debug_output (DEBUGLEVEL_8, "disconnecting...");
          ph->disconnect (ph, conn->base.index);
          debug_output (DEBUGLEVEL_8, "reconnecting...");
          ph->connect (ph, conn->base.index);
          if (tries <= ph->retries)
            {
              /* we can try again, now  :) */
              if (res != NULL)
                PQclear (res);
              res = NULL;
              goto retry;
            }
          else
            {
              errormsg
                ("Retries exceeded limit with fatal database errors. (limit of %d attempts)",
                 1 + ph->retries);
              errormsg ("Last error message: %s", PQresultErrorMessage (res));
	      errormsg ("Last query: %s" , oql_query_as_sql (query, dbtype) );
            }
        }

      if (errorcode)
        *errorcode = (-1);
      if (errormsg)
        *errormsg = g_strdup (PQresultErrorMessage (res));
      result->success = FALSE;
      if (res != NULL)
        PQclear (res);
      res = NULL;
    }
  /* release this connection */
  conn->base.available = TRUE;
  result->data = g_list_reverse (result->data);
  return (result);
}

/* ------------------------------------------------------------------------- *\
 * delete an object from the database, given a classname and object key
\* ------------------------------------------------------------------------- */
static struct query_result *
postgresql_delete_object (struct database_handle *ph, const char *classname,
                          const char *key, int *errorcode, char **errormsg)
{
  struct query_result *retval = NULL;
  QueryData *q = oql_delete_object (classname, key);

  if (q)
    {
      retval = postgresql_execute_query (ph, q, errorcode, errormsg);
      oql_free_query (q);
    }
  else
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("Could not create query");
    }
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * delete all records that have a specific value in a specific key
\* ------------------------------------------------------------------------- */
static struct query_result *
postgresql_delete_all_objects (struct database_handle *ph,
                               const char *classname, const char *field,
                               const char *key, int *errorcode, char **errormsg)
{
  struct query_result *retval = NULL;
  QueryData *q = oql_delete_all_objects (classname, field, key);

  if (q)
    {
      retval = postgresql_execute_query (ph, q, errorcode, errormsg);
      oql_free_query (q);
    }
  else
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("Could not create query");
    }
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * save an object to the database from the object cache
\* ------------------------------------------------------------------------- */
static struct query_result *
postgresql_write_object (struct database_handle *ph, const gchar *classname,
                         const gchar *key, GHashTable *values, gboolean update,
                         int *errorcode, char **errormsg)
{
  struct query_result *retval = NULL;
  QueryData *q;
  q = oql_write_object (classname, key, values, update);
  if (q)
    {
      retval = postgresql_execute_query (ph, q, errorcode, errormsg);
      oql_free_query (q);
    }
  else
    {
      if (errorcode)
        *errorcode = (-1);        /* TODO: meaningful error codes */
      if (errormsg)
        *errormsg = g_strdup ("Could not create write query");
    }
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * free memory
\* ------------------------------------------------------------------------- */
static void
postgresql_delete_database_handle (struct database_handle *hnd)
{
  unsigned int i;
  struct postgresql_handle *h = (struct postgresql_handle *) hnd;

  if (h)
    {
      free_generic_database (hnd);

      if (h->base.connections)
        {
          for (i = 0; i < h->base.num_connections; i++)
            {
              struct postgresql_connection *c
                =
                (struct postgresql_connection *)
                &((struct postgresql_connection *) hnd->connections)[i];

              if (c->handle)
                PQfinish (c->handle);
            }
          g_free (h->base.connections);
        }

      g_free (h);
    }
}

/* ------------------------------------------------------------------------- *\
 * read the tables and columns from the current database
\* ------------------------------------------------------------------------- */
static DatabaseDefinition *
postgresql_read_database_definition (struct database_handle *hnd)
{
  struct postgresql_handle *h = (struct postgresql_handle *) hnd;
  PGconn *handle;
  PGresult *res;
  char *buf;
  char tmpport[20];
  int i, row;
  DatabaseDefinition *db = NULL;

  db = create_database_definition (h->base.name);
  if (!db)
    return (NULL);

  sprintf (tmpport, "%lu", hnd->port);
  /* connect to database */
  handle =
    PQsetdbLogin (hnd->hostname, tmpport, NULL, NULL, hnd->dbname,
                  hnd->username, hnd->password);

  if (handle == NULL)
    {
      errormsg ("handle came back as null.");
      free_database_definition (db);
      return (NULL);
    }
  PQsetNoticeProcessor (handle, postgres_notice_processor, NULL);
  if (PQstatus (handle) == CONNECTION_BAD)
    {
      errormsg ("Connection to database '%s' failed: %s\n", hnd->dbname,
                PQerrorMessage (handle));
      PQfinish (handle);
      handle = NULL;
      free_database_definition (db);
      return (NULL);
    }

  /* get list of tables */
  res =
    PQexec (handle,
            "SELECT tablename FROM pg_tables WHERE tableowner != 'postgres'");
  if (!res)
    {
      errormsg
        ("Failed to get postgresql table definitions from database %s.",
         hnd->name);
      PQfinish (handle);
      free_database_definition (db);
      return (NULL);
    }
  for (i = 0; i < PQntuples (res); i++)
    {
      add_database_table (db, PQgetvalue (res, i, 0));
      /* message( " tab: %s" , row[0] ); */
    }
  PQclear (res);

  /* for each table: */
  for (i = 0; i < count_database_tables (db); i++)
    {
      DatabaseTable *tab = get_database_table (db, i);
      enum odl_fieldtype type;

      /* get list of fields and types */
      /* printf( "handling table '%s'\n" , tab->name ); */
      buf = g_strdup_printf ("select a.attname, t.typname "
                             "from pg_class c, pg_attribute a, pg_type t "
                             "where c.relname = '%s' and "
                             "a.attnum > 0 and "
                             "a.attrelid = c.oid and "
                             "a.atttypid = t.oid", tab->name);
      debug_output (DEBUGLEVEL_HIGH, "query will be\n%s", buf);

      res = PQexec (handle, buf);
      g_free (buf);
      if (PQresultStatus (res) != PGRES_TUPLES_OK)
        {
          errormsg
            ("Failed to get postgresql table %s definition from database %s.",
             tab->name, hnd->name);
          PQclear (res);
          free_database_definition (db);
          return (NULL);
        }
      if (PQntuples (res) <= 0)
        message ("Warning: no fields in table %s in database %s",
                 tab->name, hnd->name);

      for (row = 0; row < PQntuples (res); row++)
        {
          char *name;

          if (g_strcasecmp (PQgetvalue (res, row, 1), "VARCHAR") == 0)
            type = DT_char;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "INT2") == 0)
            type = DT_int16;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "INT4") == 0)
            type = DT_int32;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "INT8") == 0)
            type = DT_int64;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "BOOL") == 0)
            type = DT_boolean;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "TEXT") == 0)
            type = DT_text;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "DATE") == 0)
            type = DT_date;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "TIME") == 0)
            type = DT_time;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "DATETIME") == 0)
            type = DT_datetime;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "FLOAT8") == 0)
            type = DT_float;
          else if (g_strcasecmp (PQgetvalue (res, row, 1), "CHAR") == 0)
            type = DT_object;
          else
            type = DT_unknown;

          name = g_strdup (PQgetvalue (res, row, 0));
          add_database_column (tab, name, type);
          g_free (name);
        }
      PQclear (res);
    }
  /* close the database */
  PQfinish (handle);
  return (db);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
postgresql_add_column_data (GString *buf, GList *l, const char *separator)
{
  DBchange *c;
  char *name;

  while (l)
    {
      c = (DBchange *) l->data;
      name = oql_quote_column (NULL, c->name, OQL_DBTYPE_POSTGRESQL);
      g_string_append (buf, name);
      g_free (name);
      g_string_append (buf, " ");
      switch (c->datatype)
        {
        case DT_char:
          g_string_sprintfa (buf, "VARCHAR");
          if (c->format)
            g_string_sprintfa (buf, "(%s)", c->format);
          else
            g_string_sprintfa (buf, "(1)");
          break;
        case DT_int16:
          g_string_sprintfa (buf, "INT2");
          break;
        case DT_int32:
        case DT_int:
        case DT_unsignedint:
          g_string_sprintfa (buf, "INT4");
          break;
        case DT_int64:
          g_string_sprintfa (buf, "INT8");
          break;
        case DT_boolean:
        case DT_bool:
          g_string_sprintfa (buf, "BOOL");
          break;
        case DT_text:
          g_string_sprintfa (buf, "TEXT");
          break;
        case DT_date:
          g_string_sprintfa (buf, "DATE");
          break;
        case DT_time:
          g_string_sprintfa (buf, "TIME");
          break;
        case DT_datetime:
          g_string_sprintfa (buf, "DATETIME");
          break;
        case DT_float:
          g_string_sprintfa (buf, "FLOAT8");
          break;
        case DT_object:
          g_string_sprintfa (buf, "CHAR(32)");
          break;
        default:
          /* g_critical ("unknown data type: %d %s", c->datatype,
                      odl_datatype_name (c->datatype)); */
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
                 "unknown data type: %d %s", c->datatype,
                 odl_datatype_name (c->datatype));
          g_string_sprintfa (buf, "UNKNOWN_TYPE");
          break;
        }
      if (c->notnull)
        g_string_append (buf, " NOT NULL");
      else
        g_string_append (buf, " ");

      g_string_append (buf, separator);
      l = g_list_next (l);
    }
}

/* ------------------------------------------------------------------------- *\
 * Update the table scheme in the db to fit the current class definition
\* ------------------------------------------------------------------------- */
struct query_result *
postgresql_update_tables (struct database_handle *hnd,
                          gboolean remove_items, int *errorcode,
                          char **errormsg)
{
  char *name;
  GString *buf;
  DatabaseChange *changes = NULL;
  GList *required = NULL;
  GList *tmp;
  char tmpport[20];
  DatabaseDefinition *database = NULL;
  PGconn *handle;
  PGresult *res;
  FILE *changefile;

  debug_output (DEBUGLEVEL_HIGH, "pg - updating the tables");

  /* hard code this to OFF - is intended for remote admin */
  remove_items = FALSE;

  /* clear error indicators */
  if (errorcode)
    *errorcode = 0;
  if (errormsg)
    *errormsg = NULL;

  /* make a list of required classnames in this database */
  required = odl_tree_list_classes (all_classes);
  /* currently only a single SQL database is allowed at one time */
  /* so all classes are required */
  /* TODO: remove names from list if not required in */

  /* read current table definitions from database */
  database = postgresql_read_database_definition (hnd);
  /* show_database_definition( database ); */

  /* compare to current classes for this database */
  /* if table doesn't match a class, report the table as unnecessary */
  /* if column doesn't match a class field, report it as unnecessary */
  /* if class doesn't match a table, record the table as required */
  /* if data field doesn't match a column, record the column as required */
  changes = compare_classes_to_database (all_classes, required, database,
                                         OQL_DBTYPE_POSTGRESQL);
  odl_namelist_free (required);
  required = NULL;

  if (!changes)
    {
      /* no changes required! yay! */
      /* don't care about a result - error msgs alreayd printed */
      message ("No database changes.");
      return (NULL);
    }

  changefile =
    fopen (get_global_option_str
           (configdata, "databasechangefile", "database.changes.txt"), "a");
  if (!changefile)
    {
      errormsg ("Could not open log file '%s' for database changes.",
                get_global_option_str (configdata, "databasechangefile",
                                       "database.changes.txt"));
      return (NULL);
    }
  /* display remove suggestions */
  tmp = changes->removes;
  if (tmp)
    fprintf (changefile,
             "These changes should be made to database '%s', unless there has been an\n"
             "error in the configuration of GEAS:\n\n", hnd->name);
  while (tmp)
    {
      remove_message (changefile, (DBchange *) tmp->data);
      tmp = g_list_next (tmp);
    }
  fprintf (changefile, "\n\n");

  if (!changes->adds)
    {
      /* no additions, so don't both with the rest */
      fclose (changefile);
      return (NULL);
    }
  /* connect to database */
  sprintf (tmpport, "%lu", hnd->port);
  handle =
    PQsetdbLogin (hnd->hostname, tmpport, NULL, NULL, hnd->dbname,
                  hnd->username, hnd->password);
  if (!handle)
    {
      errormsg ("Failed to initialise Postgres connection.");
      fclose (changefile);
      return (NULL);
    }
  PQsetNoticeProcessor (handle, postgres_notice_processor, NULL);

  /* update tables on server */
  tmp = changes->adds;
  fprintf (changefile, "These changes are being made automatically:\n\n");
  while (tmp)
    {
      DBchange *add = (DBchange *) tmp->data;

      /* display add requirement */
      add_message (changefile, (DBchange *) tmp->data);

      if (add->type == DBCH_ADD_TABLE)
        {
          /* for each new table, construct a CREATE command */
          buf = g_string_new ("");
          name = oql_quote_column (add->name, NULL, OQL_DBTYPE_POSTGRESQL);
          g_string_sprintf (buf, "CREATE TABLE %s (", name);
          g_free (name);
          postgresql_add_column_data (buf, add->columns, ", ");
          g_string_append (buf, "PRIMARY KEY (ObjectID) )");

          /* execute CREATE query */
          g_strdown (buf->str);
          /* printf( "CREATE QUERY: [%s]\n" , buf->str ); */
          res = PQexec (handle, buf->str);
          if (PQresultStatus (res) == PGRES_COMMAND_OK)
            {
              /* query appeared to succeed */
              message ("[%s] succeeded", buf->str);
            }
          else
            {
              /* TODO: error handling */
              errormsg ("[%s] failed", buf->str);
              errormsg ("%s", PQresultErrorMessage (res));
            }


        }
      else if (add->type == DBCH_MODIFY_TABLE)
        {

          GList *tmplist;
          GList *l = add->columns;
          while (l)
            {
              buf = g_string_new ("");
              name =
                oql_quote_column (add->name, NULL, OQL_DBTYPE_POSTGRESQL);
              g_string_sprintf (buf, "ALTER TABLE %s ADD ", name);
              g_free (name);

              /* tmplst hack because postgresql can only add one column at a time */
              /* and I want to reuse postgresql_add_column_data */
              tmplist = g_list_append (NULL, l->data);
              postgresql_add_column_data (buf, tmplist, "");
              g_list_free (tmplist);


              /* execute ALTER query */
              g_strdown (buf->str);
              /* printf( "ALTER QUERY: [%s]\n" , buf->str ); */
              res = PQexec (handle, buf->str);
              if (PQresultStatus (res) == PGRES_COMMAND_OK)
                {
                  /* query appeared to succeed */
                  message ("[%s] succeeded", buf->str);
                }
              else
                {
                  /* TODO: error handling */
                  message ("[%s] failed", buf->str);
                  errormsg ("%s", PQresultErrorMessage (res));
                }
              g_string_free (buf, TRUE);

              l = g_list_next (l);
            }
        }
      tmp = g_list_next (tmp);
    }
  fprintf (changefile, "\n");

  PQfinish (handle);
  fclose (changefile);
  /* message( "Done updating database" ); */

  /* don't care about a result - error msgs alreayd printed */
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * initialise a connection to a specific postgresql database
\* ------------------------------------------------------------------------- */
struct database_handle *
postgresql_create_database_handle (configuration config, const char *dbname)
{
  unsigned int i;
  struct database_handle *h = g_malloc (sizeof (struct database_handle));

  if (h)
    {
      /* clear all <datatbasetype>_handle variables */
      /* none here */

      /* store general data */
      if (!generic_database_config
          (config, dbname, (struct database_handle *) h))
        {
          postgresql_delete_database_handle ((struct database_handle *) h);
          return (NULL);
        }

      /* read postgresql options from configuration file */
      /* should be the only change required in this function */
      /* h->whatever = postgresql specific options */
      /* if fail, delete handle and return NULL */

      /* store database operation functions */
      h->connect = postgresql_connect;
      h->disconnect = postgresql_disconnect;
      h->get_connection = postgresql_get_connection;
      h->execute = postgresql_execute_query;
      h->delete_database = postgresql_delete_database_handle;
      h->delete_object = postgresql_delete_object;
      h->delete_all_objects = postgresql_delete_all_objects;
      h->write_object = postgresql_write_object;
      h->update_tables = postgresql_update_tables;
      h->begintransaction = postgresql_hack_begin;
      h->committransaction = postgresql_hack_commit;

      /* create connection data storage */
      h->connections =
        (struct active_connection *) g_new0 (struct postgresql_connection,
                                             h->num_connections);

      /* abort if resource acquisition failed */
      if (!h->connections /* || !other data */ )
        {
          postgresql_delete_database_handle (h);
          return (NULL);
        }

      /* initialise connections */
      for (i = 0; i < h->num_connections; i++)
        {
          struct postgresql_connection *c
            =
            (struct postgresql_connection *)
            &((struct postgresql_connection *) h->connections)[i];

          /* set general state */
          c->base.available = TRUE;
          c->base.connected = FALSE;

          /* any postgresql specific data */
          c->handle = NULL;        /* no connection yet */
        }
    }
  return (h);
}

#else
/* not defined to use postgresql */

/* ------------------------------------------------------------------------- *\
 * Dummy Definition of postgresql database
\* ------------------------------------------------------------------------- */
struct database_handle *
postgresql_create_database_handle (configuration config, const char *dbname)
{
  return NULL;
}

#endif
