-----------------------------------------------------------
-- Daan Leijen (c) 1999-2000, daan@cs.uu.nl
--
-- $version: 17 May 2000, release version 0.3$
-----------------------------------------------------------
module ParsecExpr( Assoc(..), Operator(..), OperatorTable
                 , buildExpressionParser
                 ) where

import Char
import Parsec


-----------------------------------------------------------
-- Assoc and OperatorTable
-----------------------------------------------------------
data Assoc              = AssocNone 
                        | AssocLeft
                        | AssocRight
                        
data Operator a         = Infix (Parser (a -> a -> a)) Assoc
                        | Prefix (Parser (a -> a))
                        | Postfix (Parser (a -> a))

type OperatorTable a    = [[Operator a]]



-----------------------------------------------------------
-- Convert an OperatorTable and basic term parser into
-- a full fledged expression parser
-----------------------------------------------------------
buildExpressionParser :: OperatorTable a -> Parser a -> Parser a
buildExpressionParser operators simpleExpr
    = foldl (makeParser) simpleExpr operators
    where
      makeParser term ops
        = let (rassoc,lassoc,nassoc
               ,prefix,postfix)      = foldr splitOp ([],[],[],[],[]) ops
              
              rassocOp   = choice rassoc
              lassocOp   = choice lassoc
              nassocOp   = choice nassoc
              prefixOp   = choice prefix  <?> ""
              postfixOp  = choice postfix <?> ""
              
              ambigious assoc op= try $
                                  do{ op; fail ("ambiguous use of a " ++ assoc 
                                                 ++ " associative operator")
                                    }
              
              ambigiousRight    = ambigious "right" rassocOp
              ambigiousLeft     = ambigious "left" lassocOp
              ambigiousNon      = ambigious "non" nassocOp 
              
              termP      = do{ pre  <- prefixP
                             ; x    <- term     
                             ; post <- postfixP
                             ; return (post (pre x))
                             }
              
              postfixP   = postfixOp <|> return id
              
              prefixP    = prefixOp <|> return id
                                         
              rassocP x  = do{ f <- rassocOp
                             ; y  <- do{ z <- termP; rassocP1 z }
                             ; return (f x y)
                             }
                           <|> ambigiousLeft
                           <|> ambigiousNon
                           
              rassocP1 x = rassocP x <|> return x                           
                           -- <|> return x
                           
              lassocP x  = do{ f <- lassocOp
                             ; y <- termP
                             ; lassocP1 (f x y)
                             }
                           <|> ambigiousRight
                           <|> ambigiousNon
                           
              lassocP1 x = lassocP x <|> return x                           
                           -- <|> return x
                           
              nassocP x  = do{ f <- nassocOp
                             ; y <- termP
                             ;    ambigiousRight
                              <|> ambigiousLeft
                              <|> ambigiousNon
                              <|> return (f x y)
                             }                                                          
                           -- <|> return x                                                      
                           
           in  do{ x <- termP
                 ; rassocP x <|> lassocP  x <|> nassocP x <|> return x
                   <?> "operator"
                 }
                

      splitOp (Infix op assoc) (rassoc,lassoc,nassoc,prefix,postfix)
        = case assoc of
            AssocNone  -> (rassoc,lassoc,op:nassoc,prefix,postfix)
            AssocLeft  -> (rassoc,op:lassoc,nassoc,prefix,postfix)
            AssocRight -> (op:rassoc,lassoc,nassoc,prefix,postfix)
            
      splitOp (Prefix op) (rassoc,lassoc,nassoc,prefix,postfix)
        = (rassoc,lassoc,nassoc,op:prefix,postfix)
        
      splitOp (Postfix op) (rassoc,lassoc,nassoc,prefix,postfix)
        = (rassoc,lassoc,nassoc,prefix,op:postfix)
      
