{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to chapter 7 (Callback Registration) of the
GLUT3 API docs.
-}

module GLUT_CBGlobal (
   MenuStatusAction,      menuStatusFunc, MenuUsage(..),   -- @glut_api_geq3@
   IdleAction,            idleFunc,
   TimerAction,           timerFunc
) where

import Prelude            hiding (Either(Left))
import IOExts             (fixIO)
import Foreign            (FunPtr, freeHaskellFunPtr)
import CForeign           (CInt, CUInt)

import GL_BasicTypes      (WindowPosition, windowPosition)
import GLUT_Constants     (glut_MENU_NOT_IN_USE, glut_MENU_IN_USE)
import GLUT_CallbackUtils (GlobalCallback(..), gFunc)

---------------------------------------------------------------------------
-- global callbacks

-- Note: glutMenuStateFunc is deprecated

-- @glut_api_geq3@
data MenuUsage =
     MenuNotInUse
   | MenuInUse
   deriving (Eq,Ord)

unmarshalMenuUsage :: CInt -> MenuUsage
unmarshalMenuUsage menuUsage
   | menuUsage == glut_MENU_NOT_IN_USE = MenuNotInUse
   | menuUsage == glut_MENU_IN_USE     = MenuInUse
   | otherwise                         = error "unmarshalMenuUsage"

type MenuStatusAction = MenuUsage -> WindowPosition -> IO ()

menuStatusFunc :: Maybe MenuStatusAction -> IO ()
menuStatusFunc = gFunc MenuStatusCB (\act -> mkMenuStatusFunc (\s x y -> act (unmarshalMenuUsage s)
                                                                             (windowPosition x y)))
                       menuStatusFunc_

foreign export dynamic mkMenuStatusFunc ::
   (CInt -> CInt -> CInt -> IO ()) -> IO (FunPtr (CInt -> CInt -> CInt -> IO ()))
foreign import "glutMenuStatusFunc" menuStatusFunc_ ::
   (FunPtr (CInt -> CInt -> CInt -> IO ())) -> IO ()

--------------------

type IdleAction = IO ()

idleFunc :: Maybe IdleAction -> IO ()
idleFunc = gFunc IdleCB mkIdleFunc idleFunc_

foreign export dynamic mkIdleFunc :: IdleAction -> IO (FunPtr IdleAction)
foreign import "glutIdleFunc" idleFunc_ :: FunPtr IdleAction -> IO ()

--------------------

type TimerAction = IO ()

-- Note 1: No Maybe here, either, because timers can't be cancelled.
-- Note 2: Because Haskell has closures, we don't need the ugly value passing like C.
-- Note 3: Timers are one-shot, and there can be arbitrary many timers active at once.
--         Consequently, the above machinery for callback memory management is
--         unsuitable (and too complicated, too). We do the freeing for ourselves in
--         a wrapper around the user's action.
timerFunc :: Int -> TimerAction -> IO ()
timerFunc msecs act = do
   newCBAddr <- fixIO (\a -> mkTimerFunc (const (do freeHaskellFunPtr a; act)))
   timerFunc_ (fromIntegral msecs) newCBAddr 0

foreign export dynamic mkTimerFunc :: (CInt -> IO ()) -> IO (FunPtr (CInt -> IO()))
foreign import "glutTimerFunc" timerFunc_ :: CUInt -> (FunPtr (CInt -> IO())) -> CInt -> IO ()
