module StdControlClass (Controls(..), Windowhandle.ControlState) where


--	********************************************************************************
--	Clean to Haskell Standard Object I/O library, version 1.2
--	
--	StdControlClass define the standard set of controls instances.
--	********************************************************************************


import CleanExtras
import CleanStdList
import CleanStdMisc
import Commondef
import IOstate
import Osfont
import Ostypes
import Oswindow
import StdControlAttribute
import Windowhandle
#include "preprocess.h"
         -- Compile with -cpp; set MVAR=1 for MVAR version


class Controls cdef where
#if MVAR
	controlToHandles :: cdef -> GUI [ControlState]
#else
	controlToHandles :: cdef ls ps -> GUI ps [ControlState ls ps]
#endif

#if MVAR
#else
instance (Controls c) => Controls (AddLS c) where
	controlToHandles (AddLS addLS addDef)
		= do {
			cs <- controlToHandles addDef;
			return [wElementHandleToControlState
				(WExtendLSHandle addLS (map controlStateToWElementHandle cs))
			       ]
		     }

instance (Controls c) => Controls (NewLS c) where
	controlToHandles (NewLS newLS newDef)
		= do {
			cs <- controlToHandles newDef;
			return [wElementHandleToControlState
				(WChangeLSHandle newLS (map controlStateToWElementHandle cs))
			       ]
		      }
#endif

instance (Controls c) => Controls (ListLS c) where
	controlToHandles (ListLS cDefs)
		= do {
			css <- sequence (map controlToHandles cDefs);
#if MVAR
			return (map controlStateToWElementHandle (flatten css))
#else
			return [wElementHandleToControlState (WListLSHandle (map controlStateToWElementHandle (flatten css)))]
#endif
		     }

instance Controls NilLS where
	controlToHandles NilLS
		= return []

#if MVAR
instance (Controls c) => Controls [c] where
	controlToHandles cDefs
		= do {
			css <- sequence (map controlToHandles cDefs);
			return (map controlStateToWElementHandle (flatten css))
		     }

instance Controls () where
	controlToHandles ()
		= return []
#endif

instance (Controls c1,Controls c2) => Controls (TupLS c1 c2) where
	controlToHandles (c1 :+: c2)
		= do {
			cs1 <- controlToHandles c1;
			cs2 <- controlToHandles c2;
			return (cs1 ++ cs2)
		     }

instance Controls ButtonControl where
	controlToHandles (ButtonControl textLine atts)
		= do {
			wMetrics <- accIOEnv ioStGetOSWindowMetrics;
			size     <- liftIO (getButtonSize wMetrics textLine cWidth);
			return
				[wElementHandleToControlState
					(WItemHandle 
						{ wItemId         = getIdAttribute atts
						, wItemNr         = 0
						, wItemKind       = IsButtonControl
						, wItemInfo       = WButtonInfo (ButtonInfo {buttonInfoText=textLine})
						, wItemAtts       = filter (not . redundantAttribute) atts
						, wItemPos        = zero
						, wItemSize       = size
						, wItemPtr        = osNoWindowPtr
						, wItemLayoutInfo = undef
						})
				]
		  }
		where
			cWidth = getControlWidthAttribute atts
			
			getButtonSize :: OSWindowMetrics -> String -> Maybe ControlWidth -> IO Size
			getButtonSize wMetrics _ (Just (PixelWidth reqW))
				= return (Size {w=wOK,h=hOK})
				where
					wOK = max (osGetButtonControlMinWidth wMetrics) reqW
					hOK = osGetButtonControlHeight wMetrics
			getButtonSize wMetrics _ (Just (TextWidth wtext))
				= do {
					width <- getDialogFontTextWidth wMetrics wtext;
					let wOK = max (osGetButtonControlMinWidth wMetrics) width
					in  return (Size {w=wOK,h=hOK})
				  }
				where
					hOK = osGetButtonControlHeight wMetrics
			getButtonSize wMetrics _ (Just (ContentWidth wtext))
				= do {
					(width,hOK) <- osGetButtonControlSize wMetrics wtext;
					let wOK = max (osGetButtonControlMinWidth wMetrics) width
					in  return (Size {w=wOK,h=hOK})
				  }
			getButtonSize wMetrics text Nothing
				= do {
					(width,hOK) <- osGetButtonControlSize wMetrics text;
					let wOK = max (osGetButtonControlMinWidth wMetrics) width
					in  return (Size {w=wOK,h=hOK})
				  }

instance Controls EditControl where
	controlToHandles (EditControl textLine cWidth nrLines atts)
		= do {
			wMetrics <- accIOEnv ioStGetOSWindowMetrics;
			size     <- liftIO (getEditSize wMetrics nrLines cWidth);
			return
				[wElementHandleToControlState
					(WItemHandle 
						{ wItemId         = getIdAttribute atts
						, wItemNr         = 0
						, wItemKind       = IsEditControl
						, wItemInfo       = WEditInfo (EditInfo
							                  { editInfoText    = textLine
							                  , editInfoWidth   = w size
							                  , editInfoNrLines = nrLines
							                  })
						, wItemAtts       = filter (not . redundantAttribute) atts
						, wItemPos        = zero
						, wItemSize       = size
						, wItemPtr        = osNoWindowPtr
						, wItemLayoutInfo = undef
						})
				]
		  }
		where
			getEditSize :: OSWindowMetrics -> Int -> ControlWidth -> IO Size
			getEditSize wMetrics nrLines (PixelWidth reqW)
				= do {
					(width,hOK) <- osGetEditControlSize wMetrics reqW nrLines;
					let wOK = max (osGetEditControlMinWidth wMetrics) width
					in  return (Size {w=wOK,h=hOK})
				  }
			getEditSize wMetrics nrLines (TextWidth wtext)
				= do {
					width        <- getDialogFontTextWidth wMetrics wtext;
					(width1,hOK) <- osGetEditControlSize wMetrics width nrLines;
					let wOK = max (osGetEditControlMinWidth wMetrics) width1
					in  return (Size {w=wOK,h=hOK})
				  }
			getEditSize wMetrics nrLines (ContentWidth wtext)
				= do {
					width        <- getDialogFontTextWidth wMetrics (wtext++"mm");
					(width1,hOK) <- osGetEditControlSize wMetrics width nrLines;
					let wOK = max (osGetEditControlMinWidth wMetrics) width1
					in  return (Size {w=wOK,h=hOK})
				  }

instance Controls TextControl where
	controlToHandles (TextControl textLine atts)
		= do {
			wMetrics <- accIOEnv ioStGetOSWindowMetrics;
			size     <- liftIO (getTextSize wMetrics textLine cWidth);
			return
				[wElementHandleToControlState
					(WItemHandle 
						{ wItemId         = getIdAttribute atts
						, wItemNr         = 0
						, wItemKind       = IsTextControl
						, wItemInfo       = WTextInfo (TextInfo {textInfoText=textLine})
						, wItemAtts       = filter (not . redundantAttribute) atts
						, wItemPos        = zero
						, wItemSize       = size
						, wItemPtr        = osNoWindowPtr
						, wItemLayoutInfo = undef
						})
				]
		  }
		where
			cWidth = getControlWidthAttribute atts
			
			getTextSize :: OSWindowMetrics -> String -> Maybe ControlWidth -> IO Size
			getTextSize wMetrics _ (Just (PixelWidth reqW))
				= return (Size {w=wOK,h=hOK})
				where
					wOK = max (osGetTextControlMinWidth wMetrics) reqW
					hOK = osGetTextControlHeight wMetrics
			getTextSize wMetrics _ (Just (TextWidth wtext))
				= do {
					width <- getDialogFontTextWidth wMetrics wtext;
					let wOK = max (osGetTextControlMinWidth wMetrics) width
					    hOK = osGetTextControlHeight wMetrics
					in  return (Size {w=wOK,h=hOK})
				  }
			getTextSize wMetrics _ (Just (ContentWidth wtext))
				= do {
					(width,hOK) <- osGetTextControlSize wMetrics wtext;
					let wOK = max (osGetTextControlMinWidth wMetrics) width
					in  return (Size {w=wOK,h=hOK})
				  }
			getTextSize wMetrics text Nothing
				= do {
					(width,hOK) <- osGetTextControlSize wMetrics text;
					let wOK = max (osGetTextControlMinWidth wMetrics) width
					in  return (Size {w=wOK,h=hOK})
				  }

-- getDialogFontTextWidth is not the proper implementation because accScreenPicture class not yet implemented.
-- This version gives a crude estimation in terms of max width times text length.
getDialogFontTextWidth :: OSWindowMetrics -> String -> IO Int
getDialogFontTextWidth wMetrics text
	= do {
		(_,_,_,maxwidth) <- osGetfontmetrics False 0 (osmFont wMetrics);
		return ((length text) * maxwidth)
	  }

getIdAttribute :: [IF_MVAR(ControlAttribute,ControlAttribute ls ps)] -> Maybe Id
getIdAttribute atts
	| hasId     = Just (getControlIdAtt idAtt)
	| otherwise = Nothing
	where
		(hasId,idAtt) = cselect isControlId undef atts

getControlWidthAttribute :: [IF_MVAR(ControlAttribute,ControlAttribute ls ps)] -> Maybe ControlWidth
getControlWidthAttribute atts
	| hasControlWidth = Just (getControlWidthAtt widthAtt)
	| otherwise       = Nothing
	where
		(hasControlWidth,widthAtt) = cselect isControlWidth undef atts

redundantAttribute :: IF_MVAR(ControlAttribute,ControlAttribute ls ps) -> Bool
redundantAttribute (ControlId _) = True
redundantAttribute _             = False
