// MM1ALG.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "mm1alg.h"

/*################################################################################################*/

mm1_geomopt::mm1_geomopt(mm1_eng * p1, i32s p2, f64 p3) : conjugate_gradient(p2, p3)
{
	eng = p1;
	
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			AddVar(& eng->crd[n1][n2], & eng->d1[n1][n2]);
		}
	}
}

mm1_geomopt::~mm1_geomopt(void)
{
}

f64 mm1_geomopt::GetValue(void)
{
	eng->Compute(0);	// request energy
	return eng->energy;
}

f64 mm1_geomopt::GetGradient(void)
{
	eng->Compute(1);	// request energy and gradient
	return eng->energy;
}

/*################################################################################################*/

mm1_moldyn::mm1_moldyn(mm1_eng * p1, f64 p2, f64 p3)
{
	eng = p1;
	
	vel = new f64[eng->GetModel()->atom_list.size() * 3];		// [1.0E+3 m/s]
	acc = new f64[eng->GetModel()->atom_list.size() * 3];		// [1.0E+12 m/s^2]
	
	mass = new f64[eng->GetModel()->atom_list.size()];
	
	iter_mm1al it1 = eng->GetModel()->atom_list.begin(); i32s n1 = 0;
	while (it1 != eng->GetModel()->atom_list.end())
	{
		mass[n1] = (* it1).el.GetAtomicMass();
		mass[n1] *= 1.6605402e-27 * 6.0221367e+23;
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			vel[n1 * 3 + n2] = 0.0;
			acc[n1 * 3 + n2] = 0.0;
		}
		
		it1++; n1++;
	}
	
	temp = p2; step1 = p3;
	step2 = step1 * step1;		// [1.0E-15 s]
}

mm1_moldyn::~mm1_moldyn(void)
{
	delete[] vel;
	delete[] acc;
	
	delete[] mass;
}

void mm1_moldyn::TakeMDStep(void)
{
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			f64 tmp1 = step1 * vel[n1 * 3 + n2] * 1.0e-3;
			f64 tmp2 = step2 * acc[n1 * 3 + n2] * 0.5e-9;
			eng->crd[n1][n2] += tmp1 + tmp2;
			
			vel[n1 * 3 + n2] += step1 * acc[n1 * 3 + n2] * 0.5e-6;
		}
	}
	
	eng->Compute(1);
	pot = eng->energy;
	
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			acc[n1 * 3 + n2] = -eng->d1[n1][n2] / mass[n1];
			vel[n1 * 3 + n2] += step1 * acc[n1 * 3 + n2] * 0.5e-6;
		}
	}
	
	kin = KineticEnergy();
}

f64 mm1_moldyn::KineticEnergy(void)
{
	f64 energy = 0.0;
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		f64 sum = 0.0;
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 tmp = vel[n1 * 3 + n2];
			sum += tmp * tmp;
		}
		
		energy += 500.0 * mass[n1] * sum;
	}
	
	return energy;
}

f64 mm1_moldyn::ConvTempEKin(f64 p1)
{
	return (3.0 / 2.0) * eng->GetModel()->atom_list.size() * 8.314510 * p1 / 1000.0;
}

f64 mm1_moldyn::ConvEKinTemp(f64 p1)
{
	return (2.0 / 3.0) * p1 * 1000.0 / (eng->GetModel()->atom_list.size() * 8.314510);
}

void mm1_moldyn::SetEKin(f64 p1)
{
	f64 tmp1 = p1 / KineticEnergy();
	f64 tmp2 = (tmp1 < 0.0 ? 0.0 : sqrt(tmp1));
	
	i32u tmp3 = eng->GetModel()->atom_list.size() * 3;
	for (i32u n1 = 0;n1 < tmp3;n1++) vel[n1] *= tmp2;
}

/*################################################################################################*/

mm1_random_search::mm1_random_search(mm1_mdl * p1, i32s p2, i32s p3, i32s p4, i32s p5, i32s p6)
{
	mdl = p1;
	molnum = p2;
	in_crdset = p3;
	out_crdset = p4;
	
	cycles = p5;
	optsteps = p6;
	
	mdl->GatherGroups();	// mm1_intcrd requires this!!!
	ic = new mm1_intcrd((* mdl), molnum, in_crdset);
	
	eng = mdl->CreateDefaultEngine(); go = NULL;
	
	counter1 = 0;
	counter2 = NOT_DEFINED;
	
	if (!ic->GetVariableCount())
	{
		cout << "ERROR: no rotatable bonds!!!" << endl;
		counter1 = cycles;	// skip the search...
	}
	
	CopyCRD(mdl, eng, in_crdset); CopyCRD(eng, mdl, out_crdset);
	eng->Compute(0); min_energy = eng->energy;
}

mm1_random_search::~mm1_random_search(void)
{
	if (go != NULL) delete go;
	delete eng;
	
	delete ic;
}

i32s mm1_random_search::TakeStep(void)
{
	if (counter1 < cycles)
	{
		if (counter2 == NOT_DEFINED)	// start a new cycle...
		{
			counter1++; counter2 = 0;
			
			fGL rotprob = 1.0 / sqrt((fGL) ic->GetVariableCount());
			for (i32s n1 = 0;n1 < ic->GetVariableCount();n1++)
			{
				fGL random;
				
				random = (fGL) rand() / (fGL) RAND_MAX;
				if (random > rotprob) continue;
				
				random = (fGL) rand() / (fGL) RAND_MAX;
				ic->SetVariable(n1, 2.0 * M_PI * random);
			}
			
			ic->UpdateCartesian();
			mdl->CenterCRDSet(in_crdset);
			CopyCRD(mdl, eng, in_crdset);
			
			if (go != NULL) delete go;
			go = new mm1_geomopt(eng, 50, 0.001);
		}
		
		for (i32s n1 = 0;n1 < UPDATEFRQ;n1++)	// optimize...
		{
			counter2++;
			go->TakeCGStep(conjugate_gradient::Newton2An);
		}
		
		CopyCRD(eng, mdl, in_crdset);
		
		i32s retval = counter2;
		if (counter2 >= optsteps)
		{
			eng->Compute(0);
			if (eng->energy < min_energy)
			{
				CopyCRD(eng, mdl, out_crdset);
				min_energy = eng->energy;
			}
			
			counter2 = NOT_DEFINED;
		}
		
		return retval;
	}
	else return -1;
}

/*################################################################################################*/

mm1_systematic_search::mm1_systematic_search(mm1_mdl * p1, i32s p2, i32s p3, i32s p4, i32s p5, i32s p6)
{
	mdl = p1;
	molnum = p2;
	in_crdset = p3;
	out_crdset = p4;
	
	divisions = p5;
	optsteps = p6;
}

mm1_systematic_search::~mm1_systematic_search(void)
{
}

i32s mm1_systematic_search::TakeStep(void)
{
	return -1;
}

/*################################################################################################*/

// eof
