// DOCVIEW.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "docview.h"	// config.h is here -> we get ENABLE-macros here...

#ifdef ENABLE_GRAPHICS
#include "views.h"

#include "v3d.h"
#include "Vector.h"	// OpenBabel utils for torsion, angle measure
using namespace OpenBabel;

#include <iomanip>
#include <strstream>
#include <algorithm>
using namespace std;

/*################################################################################################*/

iGLu docview::list_counter = 1;		// zero is not a valid display list number...

docview::docview(ostream * p1, graphics_class_factory & p2) : model_simple(p1, p2)
{
	graphics_factory = & p2;
	
	project_path = NULL;
	project_filename = NULL;
	
	SetDefaultProjectFileName();
	
	select_buffer = new iGLu[SB_SIZE];
	selected_object = NULL;
	
#ifdef ENABLE_TREELIST_VIEW

	tlv = NULL;
	
#endif	// ENABLE_TREELIST_VIEW

}

docview::~docview(void)
{
	selected_object = NULL;
	
#ifdef ENABLE_TREELIST_VIEW

	if (tlv != NULL) DestroyTreeListView();
	
#endif	// ENABLE_TREELIST_VIEW

	while (graphics_view_vector.size() > 0)
	{
		graphics_view * ref;
		ref = graphics_view_vector.back();
		RemoveGraphicsView(ref, true);
	}
	
	while (object_vector.size() > 0)
	{
		smart_object * ref;
		ref = object_vector.back();
		RemoveObject(ref);
	}
	
	if (project_path != NULL) delete[] project_path;
	if (project_filename != NULL) delete[] project_filename;
	
	delete[] select_buffer;
}

bool docview::AddTP(void * owner, transparent_primitive & tp)
{
	if (!tp.TestOwner(owner)) return false;		// this is just a double check, to make
	tp_vector.push_back(tp); return true;		// sure that we have correct "owner"...
}

void docview::UpdateMPsForAllTPs(void * owner)
{
	for (i32u n1 = 0;n1 < tp_vector.size();n1++)
	{
		if (tp_vector[n1].TestOwner(owner)) tp_vector[n1].GetData()->UpdateMP();
	}
}

void docview::RemoveAllTPs(void * owner)
{
	i32u n1 = 0;
	while (n1 < tp_vector.size())
	{
		vector<transparent_primitive>::iterator iter;
		
		if (!tp_vector[n1].TestOwner(owner))
		{
			n1++;
		}
		else
		{
			delete tp_vector[n1].GetData();		// destroy the data object!!!
			
			iter = (tp_vector.begin() + n1);
			tp_vector.erase(iter);
		}
	}
}

void docview::SetProjectPath(const char * path)
{
	if (project_path != NULL) delete[] project_path;
	
	project_path = new char[strlen(path) + 1];
	strcpy(project_path, path);
}

void docview::SetProjectFileName(const char * filename)
{
	if (project_filename != NULL) delete[] project_filename;
	
	project_filename = new char[strlen(filename) + 1];
	strcpy(project_filename, filename);
}

void docview::SetDefaultProjectFileName(void)
{
	static i32s id_counter = 1;
	
	ostrstream str(buffer, sizeof(buffer));
	str << "untitled" << setw(2) << setfill('0') << id_counter++ << ends;
	
	SetProjectFileName(buffer);
}

void docview::ParseProjectFileNameAndPath(const char * string)
{
	char * localstring1 = new char[strlen(string) + 1];
	strcpy(localstring1, string);
	
	i32s lastdir = NOT_DEFINED;
	for (i32s n1 = 0;n1 < (i32s) strlen(localstring1);n1++)
	{
		if (localstring1[n1] == DIR_SEPARATOR) lastdir = n1;
	}
	
	char * localstring2 = localstring1;
	
	// set project_path if needed...
	// set project_path if needed...
	// set project_path if needed...
	
	if (lastdir != NOT_DEFINED)
	{
		localstring2[lastdir] = 0;
		SetProjectPath(localstring2);
		
		localstring2 = & localstring2[lastdir + 1];
	}
	
	// and now set project_filename, without extension...
	// and now set project_filename, without extension...
	// and now set project_filename, without extension...
	
	i32s lastext = NOT_DEFINED;
	for (i32s n1 = 0;n1 < (i32s) strlen(localstring2);n1++)
	{
		if (localstring2[n1] == EXT_SEPARATOR) lastext = n1;
	}
	
	if (lastext != NOT_DEFINED)
	{
	  // This only removes an extension if it matches *our* extension,
	  // which makes problems for imported files
	  // e.g. nh3.mol.mmg1p (!) instead of nh3.mol or nh3.mm1gp
	  //		char * localstring3 = & localstring2[lastext + 1];
	  //		bool has_extension = !strcmp(localstring3, GetProjectFileNameExtension());
	  //		if (has_extension) localstring2[lastext] = 0;

	  // instead
	  localstring2[lastext] = 0;
	}
	
	SetProjectFileName(localstring2);
	
	delete[] localstring1;
}

const char * docview::GetProjectFileName(bool with_extension)
{
	ostrstream ostr(buffer, sizeof(buffer));
	
	ostr << project_filename;
	if (with_extension) ostr << EXT_SEPARATOR << GetProjectFileNameExtension();
	ostr << ends;
	
	return buffer;
}

const char * docview::GetFullProjectFileName(void)
{
	ostrstream ostr(buffer, sizeof(buffer));
	
	if (project_path != NULL) ostr << project_path << DIR_SEPARATOR;
	ostr << project_filename << EXT_SEPARATOR << GetProjectFileNameExtension() << ends;
	
	return buffer;
}

iGLu docview::GetDisplayListIDs(iGLu p1)
{
	iGLu first = list_counter;
	list_counter += p1;
	
	return first;
}

void docview::DeleteDisplayLists(iGLu p1, iGLu p2)
{
	for (i32u n1 = 0;n1 < graphics_view_vector.size();n1++)
	{
		graphics_view_vector[n1]->SetCurrent();
		glDeleteLists(p1, p2);
	}
}

/*##############################################*/
/*##############################################*/

i32s docview::IsLight(const dummy_object * p1)
{
	i32s index = NOT_DEFINED;
	for (i32u n1 = 0;n1 < light_vector.size();n1++)
	{
		if (light_vector[n1] == p1) index = n1;
	}
	
	return index;
}

bool docview::SelectLight(const dummy_object * p1)
{
	i32s n1 = IsLight(p1);
	if (n1 < 0) return false;
	
	selected_object = light_vector[n1];
	event_SelectedObjectChanged();
	
#ifdef TARGET2

	cout << "selected " << light_vector[n1]->GetObjectName() << " which is ";
	cout << (light_vector[n1]->owner != NULL ? "local " : "global ") << endl;
	
#endif	// TARGET2

	return true;
}

bool docview::AddGlobalLight(light * p1)
{
	iGLs max_local_size = 0;
	for (i32u n1 = 0;n1 < camera_vector.size();n1++)
	{
		iGLs local_size = CountLocalLights(camera_vector[n1]);
		if (local_size > max_local_size) max_local_size = local_size;
	}
	
	iGLs total_lights = CountGlobalLights() + max_local_size;
	iGLs max_lights; glGetIntegerv(GL_MAX_LIGHTS, & max_lights);
	if (total_lights == max_lights) return false;
	
	light_vector.push_back(p1);
	SetGlobalLightNumbers();
	
	for (i32u n1 = 0;n1 < camera_vector.size();n1++)
	{
		SetLocalLightNumbers(camera_vector[n1]);
		SetupLights(camera_vector[n1]);
	}
	
#ifdef ENABLE_TREELIST_VIEW

	if (tlv != NULL) tlv->LightAdded(p1);
	
#endif	// ENABLE_TREELIST_VIEW

	selected_object = light_vector.back();
	event_SelectedObjectChanged();
	
	return true;
}

bool docview::AddLocalLight(light * p1, camera * p2)
{
	iGLs total_lights = CountGlobalLights() + CountLocalLights(p2);
	iGLs max_lights; glGetIntegerv(GL_MAX_LIGHTS, & max_lights);
	if (total_lights == max_lights) return false;
	
	p1->owner = p2; light_vector.push_back(p1);
	SetLocalLightNumbers(p2); SetupLights(p2);
	
#ifdef ENABLE_TREELIST_VIEW

	if (tlv != NULL) tlv->LightAdded(p1);
	
#endif	// ENABLE_TREELIST_VIEW

	selected_object = light_vector.back();
	event_SelectedObjectChanged();
	
	return true;
}

bool docview::RemoveLight(dummy_object * p1)
{
	i32s n1 = IsLight(p1);
	if (n1 < 0) return false;
	
#ifdef ENABLE_TREELIST_VIEW

	if (tlv != NULL) tlv->LightRemoved(light_vector[n1]);
	
#endif	// ENABLE_TREELIST_VIEW

	camera * owner = light_vector[n1]->owner;
	light_vector.erase(light_vector.begin() + n1);
	delete p1;
	
	if (owner != NULL)
	{
		SetLocalLightNumbers(owner);
		SetupLights(owner);
	}
	else
	{
		SetGlobalLightNumbers();
		for (i32u n1 = 0;n1 < camera_vector.size();n1++)
		{
			SetLocalLightNumbers(camera_vector[n1]);
			SetupLights(camera_vector[n1]);
		}
	}
	
	return true;
}

iGLs docview::CountGlobalLights(void)
{
	iGLs sum = 0; i32u n1 = 0;
	while (n1 < light_vector.size())
	{
		if (light_vector[n1++]->owner == NULL) sum++;
	}
	
	return sum;
}

iGLs docview::CountLocalLights(camera * p1)
{
	iGLs sum = 0; i32u n1 = 0;
	while (n1 < light_vector.size())
	{
		if (light_vector[n1++]->owner == p1) sum++;
	}
	
	return sum;
}

bool docview::SetupLights(camera * p1)
{
	if (p1->docv != this) return false;
	iGLs max_lights; glGetIntegerv(GL_MAX_LIGHTS, & max_lights);
	
	for (i32u n1 = 0;n1 < graphics_view_vector.size();n1++)
	{
		if (graphics_view_vector[n1]->cam != p1) continue;
		
		graphics_view_vector[n1]->SetCurrent();
		for (iGLs n2 = 0;n2 < max_lights;n2++)
		{
			glDisable((GLenum) (GL_LIGHT0 + n2));
		}
		
		for (i32u n2 = 0;n2 < light_vector.size();n2++)
		{
			bool test1 = (light_vector[n2]->owner != NULL);
			bool test2 = (light_vector[n2]->owner != p1);
			if (test1 && test2) continue;
			
			light_vector[n2]->SetupProperties(); bool test = false;
			if (light_vector[n2]->owner == NULL && p1->use_global_lights) test = true;
			if (light_vector[n2]->owner == p1 && p1->use_local_lights) test = true;
			if (test) glEnable((GLenum) light_vector[n2]->number);
		}
	}
	
	return true;
}

void docview::SetGlobalLightNumbers(void)
{
	iGLs counter = 0;
	for (i32u n1 = 0;n1 < light_vector.size();n1++)
	{
		if (light_vector[n1]->owner != NULL) continue;
		light_vector[n1]->number = GL_LIGHT0 + counter++;
	}
}

void docview::SetLocalLightNumbers(camera * p1)
{
	iGLs counter = CountGlobalLights();
	for (i32u n1 = 0;n1 < light_vector.size();n1++)
	{
		if (light_vector[n1]->owner != p1) continue;
		light_vector[n1]->number = GL_LIGHT0 + counter++;
	}
}

/*##############################################*/
/*##############################################*/

i32s docview::IsObject(const dummy_object * p1)
{
	i32s index = NOT_DEFINED;
	for (i32u n1 = 0;n1 < object_vector.size();n1++)
	{
		if (object_vector[n1] == p1) index = n1;
	}
	
	return index;
}

bool docview::SelectObject(const dummy_object * p1)
{
	i32s n1 = IsObject(p1);
	if (n1 < 0) return false;
	
	selected_object = object_vector[n1];
	event_SelectedObjectChanged();
	
#ifdef TARGET2

	cout << "selected object #" << n1 << " which is ";
	cout << object_vector[n1]->GetObjectName() << endl;
	
#endif	// TARGET2

	return true;
}

void docview::AddObject(smart_object * p1)
{
	object_vector.push_back(p1);
	selected_object = object_vector.back();
	event_SelectedObjectChanged();
	
#ifdef ENABLE_TREELIST_VIEW

	if (tlv != NULL) tlv->ObjectAdded(p1);
	
#endif	// ENABLE_TREELIST_VIEW

}

bool docview::RemoveObject(dummy_object * p1)
{
	i32s n1 = IsObject(p1);
	if (n1 < 0) return false;
	
#ifdef ENABLE_TREELIST_VIEW

	if (tlv != NULL) tlv->ObjectRemoved(object_vector[n1]);
	
#endif	// ENABLE_TREELIST_VIEW

	object_vector.erase(object_vector.begin() + n1);
	delete p1; return true;
}

// these are the measuring functions, that only take coordinates as input (so they are model-independent)...
// these are the measuring functions, that only take coordinates as input (so they are model-independent)...
// these are the measuring functions, that only take coordinates as input (so they are model-independent)...

float measure_len(float * c1, float * c2)
{
	v3d<float> v1(c1, c2);
	return v1.len();
}

float measure_ang(float * c1, float * c2, float * c3)
{
	Vector v1, v2;
	v1 = Vector(c1[0] - c2[0], c1[1] - c2[1], c1[2] - c2[2]);
	v2 = Vector(c3[0] - c2[0], c3[1] - c2[1], c3[2] - c2[2]);
	return VectorAngle(v1, v2);
}

float measure_tor(float * c1, float * c2, float * c3, float * c4)
{
	Vector v1, v2, v3, v4;
	v1 = Vector(c1[0], c1[1], c1[2]) * 10.0f;
	v2 = Vector(c2[0], c2[1], c2[2]) * 10.0f;
	v3 = Vector(c3[0], c3[1], c3[2]) * 10.0f;
	v4 = Vector(c4[0], c4[1], c4[2]) * 10.0f;
	return CalcTorsionAngle(v1, v2, v3, v4);
}

/*##############################################*/
/*##############################################*/

graphics_view * docview::AddGraphicsView(bool detached)
{
	fGL focus = GetDefaultFocus();
	camera_vector.push_back(new camera(ol_static(), focus, this));
	
	graphics_view_vector.push_back(graphics_factory->ProduceGraphicsView(this, camera_vector.back(), detached));
	
	// also add a default light, so that we can see something...
	// also add a default light, so that we can see something...
	// also add a default light, so that we can see something...
	
	AddLocalLight(new directional_light(ol_static()), camera_vector.back());
	
	UpdateAllWindowTitles();
	return graphics_view_vector.back();
}

// here the camera pointer "p1" should already be in the camera_vector; see the above case!!!

graphics_view * docview::AddGraphicsView(camera * p1, bool detached)
{
	graphics_view_vector.push_back(graphics_factory->ProduceGraphicsView(this, p1, detached));
	
	// setup the old lights that we already have...
	// setup the old lights that we already have...
	// setup the old lights that we already have...
	
	SetupLights(p1);
	
	UpdateAllWindowTitles();
	return graphics_view_vector.back();
}

bool docview::IsThisLastGraphicsView(graphics_view * p1)
{
	i32s other_views_for_this_cam = 0;
	for (i32u n1 = 0;n1 < graphics_view_vector.size();n1++)
	{
		bool test1 = (graphics_view_vector[n1] == p1);
		bool test2 = (graphics_view_vector[n1]->cam == p1->cam);
		
		if (!test1 && test2) other_views_for_this_cam++;
	}
	
	if (!other_views_for_this_cam && camera_vector.size() < 2) return true;
	else return false;
}

bool docview::RemoveGraphicsView(graphics_view * p1, bool force)
{
	i32s n1 = NOT_DEFINED;
	for (i32u n2 = 0;n2 < camera_vector.size();n2++)
	{
		if (camera_vector[n2] == p1->cam) n1 = n2;
	}
	
	if (n1 == NOT_DEFINED)		// this should never happen...
	{
		cout << "WARNING: could not find the camera at docview::RemoveGraphicsView()." << endl;
		return false;
	}
	
	i32s other_views_for_this_cam = 0;	// this is also calculated at IsThisLastGraphicsView().
	
	i32s n2 = NOT_DEFINED;
	for (i32u n3 = 0;n3 < graphics_view_vector.size();n3++)
	{
		if (graphics_view_vector[n3] == p1) n2 = n3;
		else if (graphics_view_vector[n3]->cam == p1->cam) other_views_for_this_cam++;
	}
	
	if (n2 == NOT_DEFINED)		// this should never happen...
	{
		cout << "WARNING: could not find the view at docview::RemoveGraphicsView()." << endl;
		return false;
	}
	
	if (!force && (!other_views_for_this_cam && camera_vector.size() < 2))	// refuse to close the last window!!!
	{
		err->ErrorMessage("This is the last graphics view for\nthis project - can't close it.");
		return false;
	}
	
	// if this was the last view for a camera, we have to remove also the camera.
	// but before that we must remove all local lights that relate to the camera...
	
	// a light object can also be selected_object, so we must compare the pointers
	// and invalidate selected_object if necessary...
	
	if (!other_views_for_this_cam)
	{
		i32u n3 = 0;
		while (n3 < light_vector.size())
		{
			if (light_vector[n3]->owner == camera_vector[n1])
			{
				if (selected_object != NULL)
				{
					if (selected_object == light_vector[n3])
					{
						selected_object = NULL;
						event_SelectedObjectChanged();
					}
				}
				
				#ifdef ENABLE_TREELIST_VIEW
				
				if (tlv != NULL) tlv->LightRemoved(light_vector[n3]);
				
				#endif	// ENABLE_TREELIST_VIEW
				
				delete light_vector[n3];
				light_vector.erase(light_vector.begin() + n3);
			}
			else n3++;
		}
		
		SetGlobalLightNumbers();
		for (n3 = 0;n3 < camera_vector.size();n3++)
		{
			if (n1 == (i32s) n3) continue;
			
			SetLocalLightNumbers(camera_vector[n3]);
			SetupLights(camera_vector[n3]);
		}
	}
	
	// now it's time to remove the window...
	
	delete graphics_view_vector[n2];
	graphics_view_vector.erase(graphics_view_vector.begin() + n2);
	
	// and now we finally can remove the camera, if needed...
	
	if (!other_views_for_this_cam)
	{
		delete camera_vector[n1];
		camera_vector.erase(camera_vector.begin() + n1);
	}
	
	UpdateAllWindowTitles();
	return true;
}

#ifdef ENABLE_TREELIST_VIEW

treelist_view * docview::CreateTreeListView(void)
{
	if (tlv != NULL) return tlv;
	else
	{
		tlv = graphics_factory->ProduceTreeListView(this);
		return tlv;
	}
}

bool docview::DestroyTreeListView(void)
{
	if (!tlv) return false;
	else
	{
		delete tlv;
		tlv = NULL;
		
		return true;
	}
}

#endif	// ENABLE_TREELIST_VIEW

void docview::UpdateAllViews(void)
{
	UpdateAllGraphicsViews();
}

void docview::UpdateAllWindowTitles(void)
{
        char buffer[1024];

	for (i32u n1 = 0;n1 < camera_vector.size();n1++)
	{
		i32s count = 1;
		for (i32u n2 = 0;n2 < graphics_view_vector.size();n2++)
		{
			if (graphics_view_vector[n2]->cam != camera_vector[n1]) continue;
			
			ostrstream str(buffer, sizeof(buffer));
			str << "camera #" << (n1 + 1) << " view #" << count++ << ends;
			graphics_view_vector[n2]->SetTitle(buffer);
			
			// should we also update treelist view title, if it exists? or does it have a fixed name???
			// should we also update treelist view title, if it exists? or does it have a fixed name???
			// should we also update treelist view title, if it exists? or does it have a fixed name???
		}
	}
}

void docview::UpdateAllGraphicsViews(bool flag)
{
	for (i32u n1 = 0;n1 < graphics_view_vector.size();n1++)
	{
		graphics_view_vector[n1]->Update(flag);
	}
}

void docview::UpdateGraphicsViews(camera * cam, bool flag)
{
	for (i32u n1 = 0;n1 < graphics_view_vector.size();n1++)
	{
		if (graphics_view_vector[n1]->cam != cam) continue;
		graphics_view_vector[n1]->Update(flag);
	}
}

void docview::UpdateGraphicsView(graphics_view * gv, bool flag)
{
	gv->Update(flag);
}

void docview::UpdatePlottingViews(void)
{
	for (i32u n1 = 0;n1 < plotting_view_vector.size();n1++)
	{
		// energy-level diagrams might require scaling here?!?!?!
		// detect them and call the scaling function... ANY BETTER WAY TO DO THIS???
		enlevdiag_view * eldv = dynamic_cast<enlevdiag_view *>(plotting_view_vector[n1]);
		if (eldv != NULL) eldv->SetCenterAndScale();
		
		plotting_view_vector[n1]->Update();
	}
}

void docview::RenderAllTPs(graphics_view * gv, rmode)
{
	// here we will render those transparent primitives...
	// all models should call this before returning from their versions!!!
	
	// first we must update the distances for all TP's, and sort them...
	
	const fGL * ref1 = gv->cam->GetLocData()->crd;
	const fGL * ref2 = gv->cam->GetLocData()->zdir.data;
	
	for (i32u n1 = 0;n1 < tp_vector.size();n1++)
	{
		tp_vector[n1].UpdateDistance(ref1, ref2);
	}
	
	sort(tp_vector.begin(), tp_vector.end());
	
	// and then we will just render them...
	// it looks better if we disable depth buffer changes...
	
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDepthMask(false); glEnable(GL_BLEND);
	
	for (i32u n1 = 0;n1 < tp_vector.size();n1++)
	{
		tp_vector[n1].GetData()->Render();
	}
	
	glDisable(GL_BLEND); glDepthMask(true);
}

/*##############################################*/
/*##############################################*/

void docview::DoDeleteCurrentObject(void)
{
	if (selected_object != NULL)
	{
		bool test1 = RemoveLight(selected_object);
		bool test2 = test1; if (!test1) test2 = RemoveObject(selected_object);
		
		if (test2)
		{
			selected_object = NULL;
			event_SelectedObjectChanged();
			UpdateAllGraphicsViews();
		}
	}
}

void docview::DoSwitchLocalLights(camera * cam, bool report)
{
	cam->use_local_lights = !cam->use_local_lights;
	if (report) cout << "local lights = " << (cam->use_local_lights ? "on" : "off") << endl;
	SetupLights(cam); UpdateGraphicsViews(cam);
}

void docview::DoSwitchGlobalLights(camera * cam, bool report)
{
	cam->use_global_lights = !cam->use_global_lights;
	if (report) cout << "global lights = " << (cam->use_global_lights ? "on" : "off") << endl;
	SetupLights(cam); UpdateGraphicsViews(cam);
}


void docview::InitGL(void)
{
        const fGL default_background[4] = { 0.0, 0.0, 0.0, 1.0};
	fGL *background = model_prefs->ColorRGBA("Graphics/BkgndColor", default_background);
	glClearColor(background[0], background[1], background[2], background[3]);
	delete [] background;
	
	glDepthFunc(GL_LESS); glEnable(GL_DEPTH_TEST);
	glMateriali(GL_FRONT_AND_BACK, GL_SHININESS, 64);
	
	const fGL default_reflectance[4] = { 0.9, 0.9, 0.9, 1.0 };
	fGL *specular_reflectance = model_prefs->ColorRGBA("Graphics/SpecularReflectance", default_reflectance);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular_reflectance);
	delete [] specular_reflectance;
	
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glEnable(GL_COLOR_MATERIAL);
	
	const fGL default_ambient[4] = { 0.2, 0.2, 0.2, 1.0 };
	fGL *ambient_intensity = model_prefs->ColorRGBA("Graphics/AmbientIntensity", default_ambient);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, ambient_intensity);
	delete [] ambient_intensity;
	
	glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, true);
	
	glFogi(GL_FOG_MODE, GL_EXP);
	glFogf(GL_FOG_DENSITY, model_prefs->Double("Graphics/FogDensity",0.15));
	
	const fGL default_fog[4] = { 0.0, 0.0, 0.0, 0.0 };
	fGL *fog_color = model_prefs->ColorRGBA("Graphics/FogColor", default_fog);
	glFogfv(GL_FOG_COLOR, fog_color);
	delete [] fog_color;
	
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	
	glSelectBuffer(SB_SIZE, select_buffer);
}

/*################################################################################################*/

transparent_primitive::transparent_primitive(void)
{
	owner = NULL; data = NULL;
	z_distance = 0.0;
}

transparent_primitive::transparent_primitive(void * p1, transparent_primitive_data & p2)
{
	owner = p1; data = & p2;
	z_distance = 0.0;
}

transparent_primitive::transparent_primitive(const transparent_primitive & p1)
{
	owner = p1.owner; data = p1.data;
	z_distance = p1.z_distance;
}

transparent_primitive::~transparent_primitive(void)
{
}

bool transparent_primitive::TestOwner(void * p1) const
{
	return (owner == p1);
}

transparent_primitive_data * transparent_primitive::GetData(void) const
{
	return data;
}

void transparent_primitive::UpdateDistance(const fGL * crd_c, const fGL * zdir)
{
	fGL crd_p[3];
	crd_p[0] = data->midpoint[0] - crd_c[0];
	crd_p[1] = data->midpoint[1] - crd_c[1];
	crd_p[2] = data->midpoint[2] - crd_c[2];
	
	// if we mark the vector from crd_c to crd_p as v1, and angle between v1 and zdir as angle
	// alpha, we have
	//
	//	zdist = cos(alpha) * |v1|			, where |v1| = length of v1. since
	//
	//	cos(alpha) = ip(v1,zdir) / (|zdir| * |v1|)	, we have
	//
	//	zdist = ip(v1,zdir) / |zdir|			, and if zdir is a unit vector,
	//
	//	zdist = ip(v1,zdir)
	
	z_distance = crd_p[0] * zdir[0] + crd_p[1] * zdir[1] + crd_p[2] * zdir[2];
}

bool transparent_primitive::operator<(const transparent_primitive & p1) const
{
	return (z_distance > p1.z_distance);	// inverted order...
}

/*################################################################################################*/

transparent_primitive_data::transparent_primitive_data(void)
{
}

transparent_primitive_data::~transparent_primitive_data(void)
{
}

/*################################################################################################*/

tpd_tri_3c::tpd_tri_3c(fGL * c1, fGL * p1, fGL * c2, fGL * p2, fGL * c3, fGL * p3)
{
	color[0] = c1;
	color[1] = c2;
	color[2] = c3;
	
	point[0] = p1;
	point[1] = p2;
	point[2] = p3;
	
	UpdateMP();
}

tpd_tri_3c::~tpd_tri_3c(void)
{
}

void tpd_tri_3c::Render(void)
{
	glBegin(GL_TRIANGLES);
	
	glColor4fv(color[0]);
	glVertex3fv(point[0]);
	
	glColor4fv(color[1]);
	glVertex3fv(point[1]);
	
	glColor4fv(color[2]);
	glVertex3fv(point[2]);
	
	glEnd();	// GL_TRIANGLES
}

void tpd_tri_3c::UpdateMP(void)
{
	midpoint[0] = (point[0][0] + point[1][0] + point[2][0]) / 3.0;
	midpoint[1] = (point[0][1] + point[1][1] + point[2][1]) / 3.0;
	midpoint[2] = (point[0][2] + point[1][2] + point[2][2]) / 3.0;
}

/*################################################################################################*/

tpd_quad_4c::tpd_quad_4c(fGL * c1, fGL * p1, fGL * c2, fGL * p2, fGL * c3, fGL * p3, fGL * c4, fGL * p4)
{
	color[0] = c1;
	color[1] = c2;
	color[2] = c3;
	color[3] = c4;
	
	point[0] = p1;
	point[1] = p2;
	point[2] = p3;
	point[3] = p4;
	
	UpdateMP();
}

tpd_quad_4c::~tpd_quad_4c(void)
{
}

void tpd_quad_4c::Render(void)
{
	glBegin(GL_QUADS);
	
	glColor4fv(color[0]);
	glVertex3fv(point[0]);
	
	glColor4fv(color[1]);
	glVertex3fv(point[1]);
	
	glColor4fv(color[2]);
	glVertex3fv(point[2]);
	
	glColor4fv(color[3]);
	glVertex3fv(point[3]);
	
	glEnd();	// GL_QUADS
}

void tpd_quad_4c::UpdateMP(void)
{
	midpoint[0] = (point[0][0] + point[1][0] + point[2][0] + point[3][0]) / 4.0;
	midpoint[1] = (point[0][1] + point[1][1] + point[2][1] + point[3][1]) / 4.0;
	midpoint[2] = (point[0][2] + point[1][2] + point[2][2] + point[3][2]) / 4.0;
}

/*################################################################################################*/

#endif	// ENABLE_GRAPHICS

// eof
