/* GtkEditor - a source editor widget for GTK
 * Copyright (C) 1998 Thomas Mailund.
 *
 * The editor widget was written by Thomas Mailund, so bugs should be
 * reported to <mailund@daimi.au.dk>, not the gtk ppl.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>

#include "internal.h"
#include "gtkeditor-regex.h"

#define INIT_BUFFER_SIZE 512	/* this should do for most cases...I think */

/* --<local prototypes>------------------------------------------------ */
/* I don't use the following 2 functions right now, but I don't dare
 * throw them away yet...who knows when they will be usefull again? */
static gint         regex_search_string (Regex            *regex,
					 char             *string,
					 Match            *m);
static gint         regex_match_string  (Regex            *regex,
					 char             *string);


/* --<creation and destruction>---------------------------------------- */
/* compile_regex -- compiles pattern into a regex structure that can
 * be used in regex_search and regex_match.  Returns TRUE if
 * succesfull, FALSE if not.*/
gboolean
_gtk_editor_compile_regex (const gchar *pattern, Regex *regex)
{
  memset (&regex->buf, 0, sizeof(regex->buf));
  regex->buf.translate = NULL;
  regex->buf.fastmap = g_malloc (256);
  regex->buf.allocated = 0;
  regex->buf.buffer = NULL;
  regex->buf.can_be_null = 0;	/* so we wont allow that in patterns! */
  regex->buf.no_sub = 0;
  if (re_compile_pattern (pattern, strlen (pattern), &regex->buf) == 0) {
    /* success...now try to compile a fastmap */
    if ( re_compile_fastmap (&regex->buf) != 0 ) {
      /* error...no fastmap */
      g_free (regex->buf.fastmap);
      regex->buf.fastmap = NULL;
    }
    return TRUE;
  } else {
    return FALSE;
  }
}

/* regex_free -- wrapper around regfree, to abstract from the differen
 * regex libs.  This function does *not* free the Regex structure!  If
 * this has to be done call g_free on the structure after calling
 * regex_free. */
void
_gtk_editor_regex_free (Regex *regex)
{
  if (regex->buf.regs_allocated) {
    g_free (regex->reg.start);
    g_free (regex->reg.end);
  }
  /* this is pretty much regfree */
  g_free (regex->buf.buffer);
  regex->buf.buffer = NULL;
  regex->buf.allocated = 0;
  regex->buf.used = 0;
  g_free (regex->buf.fastmap);
  regex->buf.fastmap = NULL;
  regex->buf.fastmap_accurate = 0;
  g_free (regex->buf.translate);
  regex->buf.translate = NULL;
}

/* gtk_editor_check_pattern -- checks wether the pattern is sound...I
 * know no better way, than to compile it...and since I have no
 * knowledge of what happens with it afterwards, I need to free it
 * again :( Perhaps caching of patterns should be considered... */
gboolean
gtk_editor_check_pattern (const gchar *pattern)
{
  Regex tmp;
  if (_gtk_editor_compile_regex (pattern, &tmp)) {
    _gtk_editor_regex_free (&tmp);
    return TRUE;
  } else {
    return FALSE;
  }
}

/* --<search stuff>---------------------------------------------------- */
/* regex_search_string -- searches for regex in string. If found,
 * returns index of beginning of the match, otherwise returns < 0.  If
 * found the match interval is [m.from,m.to[.  m.from is allways equal
 * to the return value, but m.to is undefined if no match. */
static gint
regex_search_string (Regex *regex, char *string, Match *m)
{
  gint len = strlen (string);
  m->from = re_search (&regex->buf, string, len, 0, len, &regex->reg);
  if (m->from > -1) m->to = regex->reg.end[0];
  return m->from;
}

/* regex_match_string -- tries to match regex in beginning of
 * string. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string. */
static gint
regex_match_string (Regex *regex, char *string)
{
  gint len = strlen (string);
  return re_match (&regex->buf, string, len, 0, NULL);
}

/* regex_search -- searches for regex in text from position
 * 'start'. If 'forward' is TRUE, it searches forward, otherwise
 * backwards.  If found, returns index of beginning of the match,
 * otherwise returns < 0.  If found the match interval is
 * [m.from,m.to[.  m.from is always equal to the return value, but
 * m.to is undefined if no match.  I search in GtkSCText, since I don't
 * really need an editor for this function. This will also make it
 * easier to move this function to the text widget, should we want
 * that. */
gint
_gtk_editor_regex_search (GtkSCText *text, guint start, Regex *regex,
	      gboolean forward, Match *m)
{
  g_return_val_if_fail (start <= text->text_end - text->gap_size, -1);
  g_return_val_if_fail (regex != NULL, -1);
  g_return_val_if_fail (m != NULL, -1);

  m->from = re_search_2 (&regex->buf,
			 /* text before gap */
			 text->text.ch, text->gap_position,
			 /* text after gap */
			 text->text.ch + text->gap_position + text->gap_size,
			 text->text_end - (text->gap_position + text->gap_size),
			 /* from 'start' and forward to the end */
			 start,
			 (forward ? text->text_end - text->gap_size - start
			  : -start),
			 &regex->reg,
			 text->text_end - text->gap_size - start);

  if (m->from > -1) m->to = regex->reg.end[0];
  return m->from;    
}

/* select_regex -- a wrapper around _regex_search_ and
 * _regex_search_back_.  They behave the same besides the search
 * direction.  If forward is TRUE, searches forward, otherwise
 * searches backwards. */
gint
_gtk_editor_select_regex (GtkEditor *editor, gchar *regex, gboolean forward)
{
  Regex re;
  Match m;

  g_return_val_if_fail (editor != NULL, -1);
  g_return_val_if_fail (regex != NULL, -1);

  if (!_gtk_editor_compile_regex (regex, &re)) {
    _gtk_editor_regex_free (&re);
    return -1;
  }
  if (_gtk_editor_regex_search (GTK_SCTEXT (editor), GTK_SCTEXT (editor)->point.index,
		    &re, forward, &m) > -1) {
    gtk_editable_set_position (GTK_EDITABLE (editor), m.from);
    gtk_editable_select_region (GTK_EDITABLE (editor), m.from, m.to);
    _gtk_editor_regex_free (&re);
    return 1;
  } else {
    _gtk_editor_regex_free (&re);
    return 0;
  }
}


/* regex_match -- tries to match regex at the 'pos' position in the
 * text. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string.  The reason for workin on GtkSCText is the same as in
 * regex_search. */
gint
_gtk_editor_regex_match (GtkSCText *text, guint pos, Regex *regex)
{
  g_return_val_if_fail (pos <= text->text_end - text->gap_size, -1);
  g_return_val_if_fail (regex != NULL, -1);

  return re_match_2 (&regex->buf,
		     /* text before gap */
		     text->text.ch, text->gap_position,
		     /* text after gap */
		     text->text.ch + text->gap_position + text->gap_size,
		     text->text_end - (text->gap_position + text->gap_size),
		     /* from pos and not after the end */
		     pos, &regex->reg, text->text_end - text->gap_size);
}


/* --<highlight>------------------------------------------------------- */
/* gtk_editor_pentry_new -- creates new pattern entry for
 * highlighting.  Designed for changing, and for easy use with
 * install_patterns. */
GList*
gtk_editor_pentry_new (const gchar *name, const gchar *pattern,
			GdkFont *font, const GdkColor *fore,
			const GdkColor  *back, GList *next)
{
  GList *tmp = g_list_alloc ();
  PEntry *pe = g_new (PEntry, 1);

  if ( !_gtk_editor_compile_regex (pattern, &(pe->regex)) ) {
    /* compile error! */
    /* FIXME: another error handling...a callback? */
    g_warning ("regex compile error!");
    g_free (pe);
    return next;
  }

  /* the regex compiled fine...now copy the rest of the arguments */
  pe->name = g_strdup (name);
  pe->pattern = g_strdup (pattern);

  pe->no_prop = 1;
  pe->prop = g_new (GtkSCTextProperties, 1);
  pe->prop->font = _gtk_editor_fontdup (font);
  pe->prop->fore = _gtk_editor_coldup (fore);
  pe->prop->back = _gtk_editor_coldup (back);

  pe->refcount = 1;		/* the list is refering to this one */

  tmp->data = (gpointer)pe;
  tmp->next = next;

  return tmp;
}

/* gtk_editor_pentry_new_with_regs -- creates new pattern entry for
 * highlighting.  Designed for changing, and for easy use with
 * install_patterns. */
GList*
gtk_editor_pentry_new_with_regs (const gchar *name, const gchar *pattern,
				 guint no_prop, GtkSCTextProperties *prop,
				 GList *next)
{
  GList *tmp = g_list_alloc ();
  PEntry *pe = g_new (PEntry, 1);

  if ( !_gtk_editor_compile_regex (pattern, &(pe->regex)) ) {
    /* compile error! */
    /* FIXME: another error handling...a callback? */
    g_warning ("regex compile error!");
    g_free (pe);
    return next;
  }

  /* the regex compiled fine...now copy the rest of the arguments */
  pe->name = g_strdup (name);
  pe->pattern = g_strdup (pattern);

  pe->no_prop = no_prop;
  pe->prop = prop;

  pe->refcount = 1;		/* the list is refering to this one */

  tmp->data = (gpointer)pe;
  tmp->next = next;

  return tmp;
}


static void
destroy_pentry (PEntry *pe)
{
  int i;

  if ( --(pe->refcount) <= 0) {
    g_free (pe->name);
    g_free (pe->pattern);
    _gtk_editor_regex_free (&(pe->regex));
    for (i = 0; i < pe->no_prop; i++) {
      if (pe->prop[i].font) gdk_font_unref (pe->prop[i].font);
      g_free (pe->prop[i].fore);
      g_free (pe->prop[i].back);
    }
    g_free (pe);
  }
}

/* gtk_editor_free_pentries -- frees the entries in the 'entries'
 * list */
void
gtk_editor_free_pentries (GList *entries)
{
  for (; entries; entries = entries->next) {
    destroy_pentry ((PEntry*)entries->data);
  }

  g_list_free (entries);
}

/* new_patterns -- creates a new pattern structure.  All entries must
 * contain valid patterns/regexps...the will have if constructed with
 * the right functions.  If this is not the case, the regex won't be
 * compiled, which is a critical error. */
GtkEditorHilitePatterns*
_gtk_editor_new_patterns (GList *entries)
{
  /* the new patterns struct */
  GtkEditorHilitePatterns *new = g_new (GtkEditorHilitePatterns, 1);
  PEntry *pe;

  /* buffer for collecting all patterns */
  int size, used;
  char *all_patterns;
  int len;

  /* init all_patterns buffer */
  all_patterns = g_new (char, INIT_BUFFER_SIZE);
  size = INIT_BUFFER_SIZE;
  used = 0;

  /* init new */
  new->patterns = NULL;

  /* loop through patterns */
  while (entries) {

    pe = (PEntry*)entries->data;

    /* fix all_patterns */
    len = strlen (pe->pattern);

    /* do we have room for this? */
    if ( ((len+strlen("\\|")) > (size - used)) ) {
      while ( ((len+strlen("\\|")) > (size - used)) )
	size *= 2;
      all_patterns = g_realloc (all_patterns, size);
    }

    strcpy (all_patterns+used, pe->pattern);
    used += len;

    strcpy (all_patterns+used, "\\|");
    used += strlen ("\\|");

    /* insert entry in list */
    pe->refcount++;
    new->patterns = g_list_prepend (new->patterns, (gpointer)pe);

    entries = entries->next;
  }
  /* remove last '\\|' */
  all_patterns[used-2] = '\0';

  if ( !_gtk_editor_compile_regex (all_patterns, &(new->all)) ) {
    /* critical error!!! */
    g_error ("all patterns fucked up!!!");
  }
  
  g_free (all_patterns);	/* no more need for this */

  return new;
}

/* destroy_patterns -- yep! */
void
_gtk_editor_destroy_patterns (GtkEditorHilitePatterns *patterns)
{
  GList *tmp;

  if ( !patterns )
    return;			/* *very* destroyed already */

  tmp = patterns->patterns;

  while (tmp) {			/* free entries */
    destroy_pentry ((PEntry*)tmp->data);
    tmp = tmp->next;
  }
  g_list_free (patterns->patterns); /* free list */
  _gtk_editor_regex_free (&(patterns->all));

  /* and finaly the struct */
  g_free (patterns);
}

void
_gtk_editor_philite_interval (GtkEditor *editor, guint from, guint to)
{
  Match m;
  GList *tmp;
  int next = 0, k = 0;
  guint i;
  PEntry *pe;

  /* if no patterns, simply return */
  if (!editor->patterns) {
    return;
  }

  while ( from <= to && next >= 0 ) {

    /* ********************************* */
    /* periodically hacks into msg queue */

#if 0
    /* oops!  This makes keeping track of the point durring highlighting
     * nearly imposible... the point will jump around...I need a fix for this
     * someday...
     */
    while( gtk_events_pending() )
      gtk_main_iteration();
#endif

    /* ********************************** */


    /* FIXME: don't search futher than 'to'! */
    next = _gtk_editor_regex_search (GTK_SCTEXT (editor), from,
				     &editor->patterns->all, TRUE, &m);

    if (next >= 0) {

      if (m.to <= to) {
	/* we got something...now, find out which pattern! */
	for (tmp = editor->patterns->patterns; tmp; tmp = tmp->next) {
	  if ( (k = _gtk_editor_regex_match (GTK_SCTEXT (editor), next,
					     &((PEntry*)tmp->data)->regex))
	       > 0) {
	    pe = (PEntry*)tmp->data;
	      
	    for (i = 0; i < pe->no_prop; i++) {
	      if (pe->regex.reg.start[i] > -1) {
		gtk_sctext_set_property (GTK_SCTEXT (editor), 
				       pe->regex.reg.start[i],
				       pe->regex.reg.end[i],
				       pe->prop[i].font, pe->prop[i].fore, 
				       pe->prop[i].back, NULL);
	      }
	    }
	    
	    break;		/* leave for loop */
	  }
	}
      }

      /* increment search_from */
      if (k <= 0) {
	from = m.to;
      } else {
	from += k;
      }
    }
  }
}
