/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "main.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>

#include "util.h"

void
giftui_print_info (const gchar *fmt, ...)
{
	va_list args;
	
	fprintf (stdout, "%s : ", PACKAGE);
	va_start (args, fmt);
	vfprintf (stdout, fmt, args);
	va_end (args);
	
	return;
}

void
giftui_print_err (const gchar *fmt, ...)
{
	va_list args;
	
	fprintf (stderr, "%s : ", PACKAGE);
	va_start (args, fmt);
	vfprintf (stderr, fmt, args);
	va_end (args);
	
	return;
}

/**/

gchar *
str_convert_to_utf8 (const gchar *str)
{
	gsize r, w;
	
	g_return_val_if_fail (str != NULL, NULL);
	
	return g_convert (str, strlen (str), "UTF-8", "ISO-8859-1", &r, &w, NULL);
}

gchar *
str_convert_to_ascii (const gchar *str)
{
	gsize r, w;
	
	g_return_val_if_fail (str != NULL, NULL);
	
	return g_convert (str, g_utf8_strlen (str, strlen (str)), "ISO-8859-1", "UTF-8",
			  &r, &w, NULL);
}

/**/

/*
  Zettaoctets, yottaoctets, ...
 */
gchar *
size_str_human_extra (const gchar *g_size)
{
	gulong size;
	gdouble test;
	
	g_return_val_if_fail (g_size != NULL, NULL);
	
	size = atol (g_size);
	
	if (size < KBLOCK_KO)
		return g_strdup_printf (_("%lu G"), size);
	
	if (size > (KBLOCK_KO * KBLOCK_KO * KBLOCK_KO))
	{
		test = ((gdouble) size) / (KBLOCK_KO * KBLOCK_KO * KBLOCK_KO);
		if ((test - floor (test)) >= 0.01)
			return g_strdup_printf (_("%.2f E"), test);
		return g_strdup_printf (_("%.0f E"), test);
	}
	
	if (size > (KBLOCK_KO * KBLOCK_KO))
	{
		test = ((gdouble) size) / (KBLOCK_KO * KBLOCK_KO);
		if ((test - floor (test)) >= 0.01)
			return g_strdup_printf (_("%.2f P"), test);
		return g_strdup_printf (_("%.0f P"), test);
	}
	
	test = ((gdouble) size) / KBLOCK_KO;
	if ((test - floor (test)) >= 0.01)
		return g_strdup_printf (_("%.2f T"), test);
	return g_strdup_printf (_("%.0f T"), test);
}

gchar *
size_str_human (gulong size)
{
	gulong kb;
	gdouble test;
	
	if (size < KBLOCK_KO)
		return g_strdup_printf (_("%lu B"), size);
		
	kb = size / KBLOCK_KO;
	
	if (kb > (KBLOCK_KO * KBLOCK_KO * KBLOCK_KO))
	{
		test = ((gdouble) kb) / (KBLOCK_KO * KBLOCK_KO * KBLOCK_KO);
		if ((test - floor (test)) >= 0.01)
			return g_strdup_printf (_("%.2f T"), test);
		return g_strdup_printf (_("%.0f T"), test);
	}
	
	if (kb > (KBLOCK_KO * KBLOCK_KO))
	{
		test = ((gdouble) kb) / (KBLOCK_KO * KBLOCK_KO);
		if ((test - floor (test)) >= 0.01)
			return g_strdup_printf (_("%.2f G"), test);
		return g_strdup_printf (_("%.0f G"), test);
	}
	
	if (kb > KBLOCK_KO)
	{
		test = ((gdouble) kb) / KBLOCK_KO;
		if ((test - floor(test)) >= 0.01)
			return g_strdup_printf (_("%.2f M"), test);
		return g_strdup_printf (_("%.0f M"), test);
	}
	
	test = ((gdouble) size) / KBLOCK_KO;
	if ((test - floor (test)) >= 0.01)
		return g_strdup_printf (_("%.2f K"), test);
	return g_strdup_printf (_("%.0f K"), test);
}

gfloat
size_percent (gulong total, gulong current)
{
	gfloat percent;
	
	if (!total)
		return 0;
	
	percent = (gfloat) current / total;
	percent = percent * 100;
	
	return percent;
}

gchar *
time_str_human (gulong secs)
{
	guint minutes;
	guint hours;
	guint days;
	
	minutes = secs / 60;
	hours = minutes / 60;
	days = hours / 24;
	
	if (!secs)
		return g_strdup ("0s");
	if (secs < 60)
		return g_strdup_printf ("%lus", secs);
	if (minutes < 60)
		return g_strdup_printf ("%um %lus", minutes, secs % 60);
	if (hours < 24)
		return g_strdup_printf ("%uh %um", hours, minutes % 60);
	
	return g_strdup_printf ("%ud %uh", days, hours % 24);
}

gchar *
speed_str_human (gulong speed)
{
	gchar *sp, *s;
	
	if (!speed)
		return g_strdup_printf (_("0 K/s"));
	
	s = size_str_human (speed);
	sp = g_strdup_printf (_("%s/s"), s);
	g_free (s);
	
	return sp;
}

/**/

gchar *
network_name_from_url (const gchar *url)
{
	gchar *name;
	
	g_return_val_if_fail (url != NULL, NULL);
	
	if ((name = strchr (url, ':')))
		return g_ascii_strdown (url, (gint) (name - url));
	
	return NULL;
}

/**/

gboolean
network_list_update (GArray **array, const gchar *network)
{
	gchar *net;
	
	g_return_val_if_fail (array != NULL, FALSE);
	g_return_val_if_fail (network != NULL, FALSE);
	
	if (*array)
	{
		gchar **networks;
		
		networks = (gchar **) (*array)->data;
		while (*networks && strcasecmp (*networks, network))
			networks++;
		
		if (!(*networks))
		{
			net = g_strdup (network);
			g_array_append_val (*array, net);
			
			return TRUE;
		}
		
		return FALSE;
	}
	
	*array = g_array_new (TRUE, TRUE, sizeof (gchar *));
	net = g_strdup (network);
	g_array_append_val (*array, net);
	
	return TRUE;
}

gchar *
network_list_get_index (GArray *array, guint index)
{
	gchar **network;
	
	g_return_val_if_fail (array != NULL, NULL);
	g_return_val_if_fail (index <= array->len, NULL);
	
	network = (gchar **) array->data;
	
	return network[index];
}

void
network_list_free (GArray *array)
{
	g_return_if_fail (array != NULL);
	
	g_array_free (array, TRUE);
	
	return;
}
