/*
 * gkrellongrun.c
 *
 * GKrellM LongRun Plugin
 * Initial work by: (c) 2001 Nozomi Sato (nozomi@palette.plala.or.jp)
 * Copyright (c) 2001-2002 Masaharu FUJITA
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * You may contact the author by:
 *	e-mail: m@fjts.org
 *	url: http://fjts.org/~m/Soft/GKrelLongRun/ (in Japanese)
 *	     http://fjts.org/~m/Soft/GKrelLongRun/index_e.html (in English)
 */

#include <gkrellm/gkrellm.h>
#include <unistd.h>
#include <utime.h>
#include <dirent.h>

#include "interface.h"

extern gint x_label_position();

#define GKRELLONGRUN "GKrelLongRun"
#define KRELL_MAX 100
#define SCALE_MAX 500
#define DEFAULT_TEXT_FORMAT	"$M\\n$P%"

gint meter_style_id;
gint chart_style_id;
GtkWidget *config_radio[LongRunDataTypeNum+1];
gint slider_label_type;
longrun_t longrun;

/* ======================================================================== */

void
set_longrun_label()
{
	if ( longrun.parts[Mode].label != NULL )
		g_free(longrun.parts[Mode].label);
	longrun.parts[Mode].label =
			g_strdup(longrun_mode_label[longrun.data[LongRunData]]);

	if ( longrun.parts[Frequency].label != NULL )
		g_free(longrun.parts[Frequency].label);
	longrun.parts[Frequency].label =
			g_strdup_printf("%dMHz", longrun.data[FrequencyData]);

	if ( longrun.parts[Voltage].label != NULL )
		g_free(longrun.parts[Voltage].label);
	longrun.parts[Voltage].label =
			g_strdup_printf("%.2fV", longrun.data[VoltageData] / 1000.0);

	if ( longrun.parts[Meter].label != NULL )
		g_free(longrun.parts[Meter].label);
	longrun.parts[Meter].label =
			g_strdup_printf("%d%%", longrun.data[PercentageData]);
}

static void
set_slider_label()
{
	switch (slider_label_type) {
	case LongRunData:
		longrun.parts[Slider].label = g_strdup_printf("%s",
				longrun_mode_label[longrun.data[LongRunData]]);
		break;
	case FrequencyData:
		longrun.parts[Slider].label = g_strdup_printf("%dMHz",
				longrun.data[FrequencyData]);
		break;
	case VoltageData:
		longrun.parts[Slider].label = g_strdup_printf("%.2fV",
				longrun.data[VoltageData]/1000.0);
		break;
	case PercentageData:
		longrun.parts[Slider].label = g_strdup_printf("%d%%",
				longrun.data[PercentageData]);
		break;
	default:
		longrun.parts[Slider].label = strdup("");
	}
}

GtkItemFactory *gkrellongrun_factory;

GtkItemFactory *
gkrellongrun_menu_factory()
{
	int size;
	GtkItemFactory *factory;
	GtkAccelGroup *accel = gtk_accel_group_new();

#if ((VERSION_MAJOR>1)||(VERSION_MAJOR==1 && VERSION_MINOR==1 && VERSION_REV>3)||(VERSION_MAJOR==1 && VERSION_MINOR>1))
	gtk_window_add_accel_group(GTK_WINDOW(gkrellm_get_top_window()), accel);
#else
	gtk_window_add_accel_group(GTK_WINDOW(top_window), accel);
#endif

	factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", accel);
	size = get_gkrellongrun_itemfactory_size();
	gtk_item_factory_create_items(factory, size,
			gkrellongrun_factory_entry, NULL);

	return factory;
}

static void
format_chart_text(gchar *buf, gint size)
{
	Chart *cp = longrun.gkchart.chart;
	gchar c, *s;
	gint len;

	--size;
	*buf = '\0';
	for (s = longrun.gkchart.text_format; *s != '\0' && size > 0; ++s) {
		len = 1;
		if (*s == '$' && *(s + 1) != '\0') {
			if ((c = *(s + 1)) == 'M')
				len = snprintf(buf, size, "%s", longrun_mode_label[longrun.data[LongRunData]]);
			else if (c == 'F')
				len = snprintf(buf, size, "%d", longrun.data[FrequencyData]);
			else if (c == 'V')
				len = snprintf(buf, size, "%.2f", longrun.data[VoltageData] / 1000.0);
			else if (c == 'P')
				len = snprintf(buf, size, "%d", longrun.data[PercentageData]);
			++s;
		} else 
			*buf = *s;
		size -= len;
		buf += len;
	}
	*buf = '\0';
}

static void
draw_chart_extra()
{
	char buf[128];

	if (!longrun.gkchart.chart)
		return;

	format_chart_text(buf, sizeof(buf));
	gkrellm_draw_chart_text(longrun.gkchart.chart, chart_style_id, buf);
}

static void draw_chart(Chart *chart)
{
	gkrellm_draw_chartdata(chart);
	if (longrun.gkchart.draw_extra_info)
		draw_chart_extra();
	gkrellm_draw_chart_to_screen(chart);
}

static gint
gkrellongrun_chart_expose_event( GtkWidget *widget, GdkEventExpose *event )
{
	GdkPixmap *pixmap = NULL;
  
	if ( widget == longrun.gkchart.chart->drawing_area )
		pixmap = longrun.gkchart.chart->pixmap;
	else if ( widget == longrun.gkchart.chart->panel->drawing_area )
		pixmap = longrun.gkchart.chart->panel->pixmap;

	if ( pixmap ) {
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
		event->area.x, event->area.y, event->area.x, event->area.y,
		event->area.width, event->area.height);
	}
  
	return FALSE;
}

static gint
gkrellongrun_expose_event( GtkWidget *widget, GdkEventExpose *event )
{
	GdkPixmap *pixmap = NULL;
	gint i;

	for ( i = 0 ; i < GKrelLongRunPartsTypeNum ; i++ )
		if ( widget == longrun.parts[i].panel->drawing_area ) {
			pixmap = longrun.parts[i].panel->pixmap;
			break;
		}

	if ( pixmap ) {	
		gdk_draw_pixmap( widget->window,
			GK.draw1_GC, pixmap,
			event->area.x, event->area.y, event->area.x,
			event->area.y, event->area.width, event->area.height );
		return FALSE;
	}
	return TRUE;
}

static void
gkrellongrun_chart_button_press(GtkWidget *widget, GdkEventButton *event,
			gpointer data)
{
	switch (event->button) {
	case 1:
		longrun.gkchart.draw_extra_info =
			!longrun.gkchart.draw_extra_info;
		gkrellm_config_modified ();
		break;
	case 2:
		gtk_menu_popup(GTK_MENU(gkrellongrun_factory->widget),
				NULL, NULL, NULL, NULL,
				event->button, event->time);
		break;
	case 3:
		gkrellm_chartconfig_window_create(longrun.gkchart.chart);
		break;
	}
}

static void
gkrellongrun_button_press(GtkWidget *widget, GdkEventButton *event,
			gpointer data)
{
	switch (event->button) {
		case 3:
			gtk_menu_popup(GTK_MENU(gkrellongrun_factory->widget),
				NULL, NULL, NULL, NULL,
				event->button, event->time);
			break;
	}
}

static void
create_gkrellongrun_parts(GtkWidget *vbox, gint first, Style *style, gint type)
{
	Panel *panel;
	Decal *decal;
	TextStyle *textstyle;

	if ( first )
		longrun.parts[type].panel = gkrellm_panel_new0();

	if ( type == Meter ) {
		longrun.parts[type].krell = gkrellm_create_krell(
					longrun.parts[type].panel,
					gkrellm_krell_meter_image(
						meter_style_id), style );

		gkrellm_update_krell(longrun.parts[type].panel,
					longrun.parts[type].krell,
					longrun.data[PercentageData]);
	}

	if ( type == Slider ) {
		longrun.parts[type].krell = gkrellm_create_krell(
					longrun.parts[type].panel,
					gkrellm_krell_slider_image(),
					gkrellm_krell_slider_style());
		longrun.parts[type].krell->y0 =
			(longrun.parts[type].panel->h - 
			longrun.parts[type].krell->h_frame)/2;
		longrun.parts[type].krell->full_scale = 100;

		gkrellm_update_krell(longrun.parts[type].panel,
					longrun.parts[type].krell,
					longrun.data[PercentageData]);
	}


	longrun.parts[type].panel->textstyle =
			gkrellm_meter_textstyle(meter_style_id);

	longrun.parts[type].decal =
			gkrellm_create_decal_text( longrun.parts[type].panel,
			"M8", longrun.parts[type].panel->textstyle, style,
			-1, -1, -1 );

	gkrellm_configure_panel( longrun.parts[type].panel, NULL, style );
	gkrellm_create_panel( vbox, longrun.parts[type].panel,
			gkrellm_bg_meter_image(meter_style_id) );

	panel = longrun.parts[type].panel;
	decal = longrun.parts[type].decal;

	textstyle = &decal->text_style;
	if ( type != Slider )
		longrun.parts[type].decal->x_off = x_label_position(
				panel->label->position, decal->w,
				gdk_string_width( textstyle->font,
					longrun.parts[type].label ), 0 );

	if ( longrun.parts[type].enable )
		gkrellm_monitor_height_adjust( panel->h );
	else
		gtk_widget_hide( panel->hbox );

	gkrellm_draw_decal_text( panel, decal, longrun.parts[type].label, -1 );
	gkrellm_draw_layers( panel );

	if ( first ) {
		gtk_signal_connect( GTK_OBJECT(panel->drawing_area),
				"expose_event",
				(GtkSignalFunc)gkrellongrun_expose_event,
				NULL );
		gtk_signal_connect( GTK_OBJECT(panel->drawing_area),
				"button_press_event",
				(GtkSignalFunc)gkrellongrun_button_press,
				NULL );
	}
}


static void
create_gkrellongrun_chart(GtkWidget *vbox, gint first)
{
	Style *style;
	Krell *k;
	ChartData *cd;
  
	if ( first ) {
		longrun.gkchart.chart = gkrellm_chart_new0();
		longrun.gkchart.chart->panel = gkrellm_panel_new0();
	}
  
	longrun.gkchart.display = GDK_WINDOW_XDISPLAY(vbox->window);
	longrun.gkchart.window = GDK_WINDOW_XWINDOW(vbox->window);
	style = gkrellm_panel_style(chart_style_id);
  
	gkrellm_chart_create(vbox, longrun.gkchart.monitor,
		longrun.gkchart.chart, &longrun.gkchart.chart_config);
	cd = gkrellm_add_default_chartdata(longrun.gkchart.chart, "Percentage");
	gkrellm_monotonic_chartdata(cd, FALSE);
	gkrellm_set_chartconfig_grid_resolution(longrun.gkchart.chart_config,
			SCALE_MAX / FULL_SCALE_GRIDS);
	gkrellm_set_chartconfig_auto_grid_resolution(
			longrun.gkchart.chart_config, FALSE);
	gkrellm_alloc_chartdata(longrun.gkchart.chart);
	gkrellm_set_draw_chart_function(longrun.gkchart.chart, draw_chart,
			longrun.gkchart.chart);
  
	k = gkrellm_create_krell(longrun.gkchart.chart->panel,
			gkrellm_krell_panel_image(chart_style_id), style);

	gkrellm_monotonic_krell_values(k, FALSE);
	gkrellm_set_krell_full_scale(k, KRELL_MAX, 1);
	gkrellm_panel_configure(longrun.gkchart.chart->panel,
			g_strdup("LongRun"), style);
	gkrellm_panel_create(vbox, longrun.gkchart.monitor,
			longrun.gkchart.chart->panel);

	if (!longrun.gkchart.enable) {
		gkrellm_monitor_height_adjust(-longrun.gkchart.chart->panel->h);
		gkrellm_monitor_height_adjust(-longrun.gkchart.chart->h);
		gtk_widget_hide(longrun.gkchart.chart->panel->hbox);
		gtk_widget_hide(longrun.gkchart.chart->hbox);
	}
  
	if ( first ) {
		gtk_signal_connect(
			GTK_OBJECT(longrun.gkchart.chart->drawing_area),
			"expose_event",
			GTK_SIGNAL_FUNC(gkrellongrun_chart_expose_event), NULL);
		gtk_signal_connect(
			GTK_OBJECT(longrun.gkchart.chart->panel->drawing_area),
			"expose_event",
			GTK_SIGNAL_FUNC(gkrellongrun_chart_expose_event), NULL);
		gtk_signal_connect(
			GTK_OBJECT(longrun.gkchart.chart->drawing_area),
			"button_press_event",
			GTK_SIGNAL_FUNC(gkrellongrun_chart_button_press), NULL);
	}
}

static void
create_gkrellongrun(GtkWidget *vbox, gint first)
{
	Style *style;
	gint i;

	style = gkrellm_meter_style( meter_style_id );

	for ( i = 0 ; i < GKrelLongRunPartsTypeNum ; i++ )
		create_gkrellongrun_parts(vbox, first, style, i);

	create_gkrellongrun_chart(vbox, first);
}

static void
update_gkrellongrun_text(gint type)
{
	gkrellm_draw_decal_text( longrun.parts[type].panel,
			longrun.parts[type].decal,
			longrun.parts[type].label, -1 );
	gkrellm_draw_layers( longrun.parts[type].panel );
}

static void
update_gkrellongrun()
{
	Krell *krell;

	if ( GK.second_tick ) {
		gkrellm_store_chartdata(longrun.gkchart.chart, 0,
			longrun.data[PercentageData], 0);
		draw_chart(longrun.gkchart.chart);
	} else {
		read_longrun_data();

		update_gkrellongrun_text(Mode);
		update_gkrellongrun_text(Frequency);
		update_gkrellongrun_text(Voltage);
		update_gkrellongrun_text(Meter);

		longrun.parts[Meter].krell->full_scale = 100;
		longrun.parts[Meter].krell->previous = 0;
		gkrellm_update_krell( longrun.parts[Meter].panel,
				longrun.parts[Meter].krell,
				longrun.data[PercentageData]);
		gkrellm_draw_layers( longrun.parts[Meter].panel );

		set_slider_label();
		longrun.parts[Slider].krell->previous = 0;
		gkrellm_update_krell(longrun.parts[Slider].panel,
			longrun.parts[Slider].krell,
			longrun.data[PercentageData]);
		gkrellm_draw_decal_text(longrun.parts[Slider].panel,
			longrun.parts[Slider].decal,
			longrun.parts[Slider].label, -1);
		gkrellm_draw_layers(longrun.parts[Slider].panel);
	}

	krell = KRELL(longrun.gkchart.chart->panel);
	gkrellm_update_krell(longrun.gkchart.chart->panel, krell,
				longrun.data[PercentageData]);
	gkrellm_draw_panel_layers(longrun.gkchart.chart->panel);
}

static void
create_gkrellongrun_tab(GtkWidget *tab)
{
	GtkWidget *tabs;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *text;
	GtkWidget *vbox1, *hbox;
	GList *list;
	gchar *gkrellongrun_about_text = NULL;

	GSList *label_group = NULL;
	gint i;
	gchar *label_name[] = {
		"Show mode", "Show frequency",
		"Show voltage", "Show percentage", "None"
	};

	gchar *gkrellongrun_info_text[] = {
		N_("<b>Chart Labels\n"),
		N_("Substitution variables for the format string for chart labels:\n"),
		N_("\t$M    Mode of LongRun\n"),
		N_("\t$F    Frequency\n"),
		N_("\t$V    Voltage\n"),
		N_("\t$P    Percentage\n"),
	};

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab), tabs, TRUE, TRUE, 0);

	/* Setup */
	vbox = gkrellm_create_tab( tabs, _("Setup") );

	gkrellm_check_button( vbox, &longrun.parts[Mode].enable_button,
			longrun.parts[Mode].enable, TRUE, 0, _("Enable mode") );
	gkrellm_check_button( vbox, &longrun.parts[Frequency].enable_button,
			longrun.parts[Frequency].enable, TRUE, 0,
			_("Enable frequency") );
	gkrellm_check_button( vbox, &longrun.parts[Voltage].enable_button,
			longrun.parts[Voltage].enable, TRUE, 0,
			_("Enable voltage") );
	gkrellm_check_button( vbox, &longrun.parts[Meter].enable_button,
			longrun.parts[Meter].enable, TRUE, 0,
			_("Enable meter") );
	gkrellm_check_button( vbox, &longrun.parts[Slider].enable_button,
			longrun.parts[Slider].enable, TRUE, 0,
			_("Enable slider") );
	gkrellm_check_button( vbox, &longrun.gkchart.enable_button,
			longrun.gkchart.enable, TRUE, 0,
			_("Enable chart") );

	/* Slider Setup */
	vbox = gkrellm_create_tab(tabs, "Slider Setup");

	for ( i = 0 ; i <= LongRunDataTypeNum ; i++ ) {
		config_radio[i] = gtk_radio_button_new_with_label(label_group,
                                                      label_name[i]);
		label_group = gtk_radio_button_group(
				GTK_RADIO_BUTTON(config_radio[i]));

		if ( i == slider_label_type )
			gtk_toggle_button_set_active(
				GTK_TOGGLE_BUTTON(config_radio[i]), TRUE);

		gtk_widget_show(config_radio[i]);
		gtk_box_pack_start(GTK_BOX(vbox), config_radio[i],
					TRUE, TRUE, 0);
	}

	/* Chart Setup */
	vbox = gkrellm_create_tab(tabs, "Chart Setup");
	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Chart Labels"),
					4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	longrun.gkchart.text_format_combo = gtk_combo_new();
	gtk_widget_set_usize (GTK_WIDGET(longrun.gkchart.text_format_combo), 350, 0);
	gtk_box_pack_start(GTK_BOX(hbox), longrun.gkchart.text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, _(DEFAULT_TEXT_FORMAT));
	list = g_list_append(list, _("$M\\n$FMHz\\n$VV\\n$P%"));
	list = g_list_append(list, _("$M\\n$FMHz"));
	list = g_list_append(list, _("$M\\n$VV"));
	list = g_list_append(list, _("$FMHz"));
	gtk_combo_set_popdown_strings(GTK_COMBO(longrun.gkchart.text_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(longrun.gkchart.text_format_combo)->entry), longrun.gkchart.text_format);

	/* Infomation */
	vbox = gkrellm_create_tab( tabs, _("Infomation") );
	text = gkrellm_scrolled_text(vbox, NULL, GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(gkrellongrun_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text,
					_(gkrellongrun_info_text[i]));

	/* About */
	vbox = gkrellm_create_tab( tabs, _("About") );
	gkrellongrun_about_text = g_strdup_printf(
		"%s %s\n" \
		"GKrellM LongRun Plugin\n" \
		"\n" \
		"Initial work by: (c) 2001 Nozomi Sato\n" \
		"nozomi@palette.plala.or.jp\n" \
		"\n" \
		"Copyright (C) 2001-2002 Masaharu FUJITA\n" \
		"m@fjts.org\n" \
		"http://fjts.org/~m/Soft/GKrelLongRun/\n" \
		"\n" \
		"Released under the GNU General Public License\n",
		GKRELLONGRUN, VERSION);

	label = gtk_label_new(gkrellongrun_about_text);
	g_free(gkrellongrun_about_text);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 2);
}

static void
apply_gkrellongrun_config()
{
	gint i;
	gint old_enable;
	gchar *s;
	GKrelLongRunParts *p;

	s = gkrellm_entry_get_text(&(GTK_COMBO(longrun.gkchart.text_format_combo)->entry));
	gkrellm_dup_string(&longrun.gkchart.text_format, s);

	for ( i = 0 ; i < GKrelLongRunPartsTypeNum ; i++ ) {
		p = &longrun.parts[i];

		old_enable = p->enable;
		p->enable = GTK_TOGGLE_BUTTON(p->enable_button)->active;

		/* Applied new user setting. */
		if ( p-> enable != old_enable ) {
			if ( p -> enable ) {
				gkrellm_monitor_height_adjust( p->panel->h );
				gtk_widget_show( p->panel->hbox );
			} else {
				gkrellm_monitor_height_adjust( - p->panel->h );
				gtk_widget_hide( p->panel->hbox );
			}
		}
	}

	for ( i = 0 ; i <= LongRunDataTypeNum ; i++ ) {
		if ( GTK_TOGGLE_BUTTON(config_radio[i])->active ) {
			slider_label_type = i;
			break;
		}
	}

	set_slider_label();

	old_enable = longrun.gkchart.enable;
	longrun.gkchart.enable =
		GTK_TOGGLE_BUTTON(longrun.gkchart.enable_button)->active;

	if (old_enable != longrun.gkchart.enable) {
		if (longrun.gkchart.enable) {
			gkrellm_monitor_height_adjust(
				longrun.gkchart.chart->panel->h);
			gkrellm_monitor_height_adjust(
				longrun.gkchart.chart->h);
			gtk_widget_show(longrun.gkchart.chart->panel->hbox);
			gtk_widget_show(longrun.gkchart.chart->hbox);
		} else {
			gkrellm_monitor_height_adjust(
				-longrun.gkchart.chart->panel->h);
			gkrellm_monitor_height_adjust(
				-longrun.gkchart.chart->h);
			gtk_widget_hide(longrun.gkchart.chart->panel->hbox);
			gtk_widget_hide(longrun.gkchart.chart->hbox);
		}
	}
}

static void
save_gkrellongrun_config(FILE *f)
{
	fprintf(f, "%s enable_mode %d\n", GKRELLONGRUN,
				longrun.parts[Mode].enable);
	fprintf(f, "%s enable_frequency %d\n", GKRELLONGRUN,
				longrun.parts[Frequency].enable);
	fprintf(f, "%s enable_voltage %d\n", GKRELLONGRUN,
				longrun.parts[Voltage].enable);
	fprintf(f, "%s enable_meter %d\n", GKRELLONGRUN,
				longrun.parts[Meter].enable);
	fprintf(f, "%s enable_slider %d\n", GKRELLONGRUN,
				longrun.parts[Slider].enable);
	fprintf(f, "%s enable_chart %d\n", GKRELLONGRUN,
				longrun.gkchart.enable);
	fprintf(f, "%s slider_label_type %d\n", GKRELLONGRUN,
				slider_label_type);
	fprintf(f, "%s draw_extra_info %d\n", GKRELLONGRUN,
				longrun.gkchart.draw_extra_info);
	fprintf(f, "%s text_format %s\n", GKRELLONGRUN,
				longrun.gkchart.text_format);

	gkrellm_save_chartconfig(f, longrun.gkchart.chart_config,
					GKRELLONGRUN, NULL);
}

static void
load_gkrellongrun_config(gchar *arg)
{
	gchar config[32], item[CFG_BUFSIZE];

	if ( sscanf(arg, "%s %[^\n]", config, item) != 2 )
		return;

	if ( strcmp(config, "enable_mode") == 0 )
		sscanf(item, "%d", &longrun.parts[Mode].enable);
	else if ( strcmp(config, "enable_frequency") == 0 )
		sscanf(item, "%d", &longrun.parts[Frequency].enable);
	else if ( strcmp(config, "enable_voltage") == 0 )
		sscanf(item, "%d", &longrun.parts[Voltage].enable);
	else if ( strcmp(config, "enable_meter") == 0 )
		sscanf(item, "%d", &longrun.parts[Meter].enable);
	else if ( strcmp(config, "enable_slider") == 0 )
		sscanf(item, "%d", &longrun.parts[Slider].enable);
	else if ( strcmp(config, "enable_chart") == 0 )
		sscanf(item, "%d", &longrun.gkchart.enable);
	else if ( strcmp(config, "slider_label_type") == 0 )
		sscanf(item, "%d", &slider_label_type);
	else if ( strcmp(config, "draw_extra_info") == 0 )
		sscanf(item, "%d", &longrun.gkchart.draw_extra_info);
	else if ( strcmp(config, "text_format") == 0 )
		gkrellm_dup_string(&longrun.gkchart.text_format, item);
	else if ( strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD) == 0 )
		gkrellm_load_chartconfig(&longrun.gkchart.chart_config,
						item, 1);
}

static Monitor gkrellongrun_mon  =
{
	"GKrelLongRun",			/* Name, for config tab. */
	0,				/* Id, 0 if a plugin */
	create_gkrellongrun,		/* The create_plugin() function	*/
	update_gkrellongrun,		/* The update_plugin() function	*/

	create_gkrellongrun_tab,	/* Create_plugin_tab() config func*/
	apply_gkrellongrun_config,	/* The apply_plugin_config() function*/ 

	save_gkrellongrun_config,	/* The save_plugin_config() function */
	load_gkrellongrun_config,	/* The load_plugin_config() function */
	GKRELLONGRUN,			/* Config Keyword */

	NULL,			/* Undefined 2 */
	NULL,			/* Undefined 1 */
	NULL,			/* Undefined 0 */

	MON_APM,		/* Insert plugin before this monitor */
	NULL,			/* Handle if a plugin, filled by GKrellM */
	NULL			/* Path if a plugin, filled by GKrellM */
};

Monitor *
init_plugin()
{
	int i;


	if (check_cpu())
		return NULL;
	read_longrun_data();

	gkrellongrun_factory = gkrellongrun_menu_factory();
	longrun.gkchart.monitor = &gkrellongrun_mon;
	meter_style_id =
		gkrellm_add_meter_style( &gkrellongrun_mon, GKRELLONGRUN );
	chart_style_id =
		gkrellm_add_meter_style( &gkrellongrun_mon, GKRELLONGRUN );

	for ( i = 0 ; i < GKrelLongRunPartsTypeNum ; i++ )
		longrun.parts[i].enable = TRUE;
	slider_label_type = FrequencyData;

	longrun.gkchart.enable = TRUE;
	longrun.gkchart.draw_extra_info = TRUE;
	longrun.gkchart.chart = NULL;
	longrun.gkchart.text_format =  g_strdup(_(DEFAULT_TEXT_FORMAT));

	return &gkrellongrun_mon;
}
