/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*******************
 * The file array. *
 *******************/

#include "gliv.h"

#ifdef HAVE_FTW
#include <ftw.h>
#else
#include "../lib/ftw.h"
#endif

#include <string.h>             /* memcpy(), memmove(), strcmp() */
#include <time.h>               /* time() */
#include <unistd.h>             /* getpid() */
#include <dirent.h>             /* DIR, opendir(), readdir(), closedir() */
#include <stdlib.h>             /* qsort() */

extern gliv_image *current_image;

static GPtrArray *filenames_array;

#define FILE_ID(x) (g_ptr_array_index(filenames_array, x))

gchar *get_file_name_by_id(guint id)
{
    return FILE_ID(id);
}

/* Used in loading.c */
void file_array_remove(guint id)
{
    g_ptr_array_remove_index(filenames_array, id);
}

G_GNUC_PURE guint array_length(void)
{
    return filenames_array->len;
}

/*** Additions. ***/

static void add_file_to_list(gchar * filename)
{
    if (extension_is_ok(filename) == TRUE)
        g_ptr_array_add(filenames_array, filename);
}

/* Called only by ftw(). */
static gint ftw_add_to_list(const gchar * file, const struct stat *s, gint flag)
{
    if (flag == FTW_F && S_ISDIR(s->st_mode) == FALSE)
        add_file_to_list(g_strdup(file));
    return 0;
}

static void add_to_list(gchar * name, gboolean recursive)
{
    struct stat st;
    struct dirent *dir_ent;
    DIR *dir;
    gchar *full_path;

    if (stat(name, &st) == -1)
        return;

    if (S_ISDIR(st.st_mode) == TRUE) {
        if (recursive == TRUE)
            /* Traverse recursively the directory. */
            ftw(name, ftw_add_to_list, 32);
        else {
            /* Add all files in the directory. */
            dir = opendir(name);
            if (dir == NULL)
                return;

            dir_ent = readdir(dir);
            while (dir_ent != NULL) {
                full_path = g_strconcat(name, "/", dir_ent->d_name, NULL);

                stat(full_path, &st);

                if (S_ISDIR(st.st_mode) == TRUE)
                    g_free(full_path);
                else
                    add_file_to_list(full_path);

                dir_ent = readdir(dir);
            }
            closedir(dir);
        }
        g_free(name);
    } else
        /* Add the file. */
        add_file_to_list(name);
}

/*** Sorting ***/

/* Used by qsort. */
G_GNUC_PURE static gint compar(gconstpointer a, gconstpointer b)
{
    return strcmp((*((gchar **) a)), (*((gchar **) b)));
}

/* The arrays used here are not NULL-terminated since their length are known. */

/*
 * *dirs_array    : Directory array.
 * *dirs_nb_array : For each directory, how many of their files are in the list.
 * *file_dir_array: For each file, the id of their directory in *dir_list.
 *
 * Return value   : number of directories.
 */
static guint get_dir_list(guint begin, guint num, gchar *** dirs_array,
                          guint ** dirs_nb_array, guint ** file_dir_array)
{
    guint file_id, dir_id, nb_dirs;
    gchar *dir;
    gchar **dirs;
    guint *dirs_nb, *file_dir;

    dirs = NULL;
    dirs_nb = NULL;
    file_dir = g_new(guint, num);
    nb_dirs = 0;

    /* For each file. */
    for (file_id = 0; file_id < num; file_id++) {
        dir = g_path_get_dirname(get_file_name_by_id(begin + file_id));
        /* Search the dir it belongs to. */
        for (dir_id = 0; dir_id < nb_dirs; dir_id++) {
            if (g_str_equal(dir, dirs[dir_id]) == TRUE)
                /* The dir is already in the list. */
                break;
        }

        if (dir_id == nb_dirs) {
            /* The dir was not in the list, so add it. */
            nb_dirs++;

            dirs = g_renew(gchar *, dirs, nb_dirs);
            dirs_nb = g_renew(guint, dirs_nb, nb_dirs);

            dirs[dir_id] = dir;
            dirs_nb[dir_id] = 1;
        } else {
            /* The dir was already there. */
            g_free(dir);
            dirs_nb[dir_id]++;
        }

        /* The file 'file_id' is in dir 'dir_id'. */
        file_dir[file_id] = dir_id;
    }

    *dirs_array = dirs;
    *dirs_nb_array = dirs_nb;
    *file_dir_array = file_dir;

    return nb_dirs;
}

/* Sort the array, and keep the other arrays synchronized. */
static guint get_dir_list_sorted(guint begin, guint num,
                                 guint ** dirs_nb_array,
                                 guint ** file_dir_array)
{
    guint dir_id, dir_id2, nb_dirs, file_id;
    gchar **dirs, **dirs_array_copy;
    guint *dirs_nb, *dirs_nb_old, *file_dir;
    guint *sorted_dir;

    nb_dirs = get_dir_list(begin, num, &dirs, dirs_nb_array, file_dir_array);

    dirs_nb_old = *dirs_nb_array;
    file_dir = *file_dir_array;

    sorted_dir = g_new(guint, nb_dirs);
    dirs_array_copy = g_memdup(dirs, nb_dirs * sizeof(gchar *));

    /* We'll have sub-directories after their parent. */
    qsort(dirs, nb_dirs, sizeof(gchar *), compar);

    /* Now we track the moves made by qsort(). */
    for (dir_id = 0; dir_id < nb_dirs; dir_id++)
        for (dir_id2 = 0; dir_id2 < nb_dirs; dir_id2++)
            if (dirs_array_copy[dir_id] == dirs[dir_id2]) {
                /* After sorting 'dir_id' became 'dir_id2'. */
                sorted_dir[dir_id] = dir_id2;
                break;
            }

    g_free(dirs_array_copy);

    /* We reflect the *dirs_array changes in dirs_nb. */

    dirs_nb = g_new(guint, nb_dirs);

    for (dir_id = 0; dir_id < nb_dirs; dir_id++)
        dirs_nb[sorted_dir[dir_id]] = dirs_nb_old[dir_id];

    g_free(dirs_nb_old);
    *dirs_nb_array = dirs_nb;

    /* We reflect the *dirs_array changes in sorted_dir. */

    for (file_id = 0; file_id < num; file_id++)
        file_dir[file_id] = sorted_dir[file_dir[file_id]];

    g_free(sorted_dir);

    return nb_dirs;
}

gchar **get_file_list_sorted(guint begin, guint num)
{
    guint *dirs_nb_array;
    guint *file_dir_array;
    guint nb_dirs;
    guint file_id, dir_id, i, pos;
    guint *dirs_nb_filled;
    gchar **array;

    nb_dirs = get_dir_list_sorted(begin, num, &dirs_nb_array, &file_dir_array);

    array = g_new(gchar *, num);
    dirs_nb_filled = g_new0(guint, nb_dirs);

    for (file_id = 0; file_id < num; file_id++) {
        /* The file 'file_id' is in dir 'dir_id'. */
        dir_id = file_dir_array[file_id];
        pos = 0;
        for (i = 0; i < dir_id; i++)
            pos += dirs_nb_array[i];
        pos += dirs_nb_filled[dir_id];
        dirs_nb_filled[dir_id]++;
        array[pos] = get_file_name_by_id(begin + file_id);
    }

    g_free(file_dir_array);
    g_free(dirs_nb_filled);

    /*
     * Now in the array, each file is under its directory,
     * so we sort the file list for each directory.
     */

    pos = 0;
    for (dir_id = 0; dir_id < nb_dirs; dir_id++) {
        qsort(array + pos, dirs_nb_array[dir_id], sizeof(gchar *), compar);
        pos += dirs_nb_array[dir_id];
    }

    g_free(dirs_nb_array);

    return array;
}

static void reorder_array(guint begin, guint num, gboolean shuffle)
{
    guint i, r;
    gpointer tmp;
    GRand *rand;
    gchar **array;

    if (num <= 1)
        return;

    if (shuffle == TRUE) {
        rand = g_rand_new_with_seed((guint) time(NULL) + getpid());

        for (i = begin; i < begin + num; i++) {
            r = g_rand_int_range(rand, begin, begin + num);
            tmp = FILE_ID(i);
            FILE_ID(i) = FILE_ID(r);
            FILE_ID(r) = tmp;
        }

        g_rand_free(rand);

    } else {
        /* shuffle == FALSE */
        array = get_file_list_sorted(begin, num);
        memcpy(&FILE_ID(begin), array, num * sizeof(gpointer));
        g_free(array);
    }
}

/*** Initialization ***/

void init_list(gchar ** files, guint num, gboolean recursive, gboolean shuffle)
{
    guint i;

    filenames_array = g_ptr_array_new();

    for (i = 0; i < num; i++)
        add_to_list(files[i], recursive);

    reorder_array(0, array_length(), shuffle);
}

/*** Misc. operations. ***/

/*
 * This function is used when we want an image in a freshly inserted bunch
 * to be at a position which is obviously before the beginning of the bunch,
 * that's why the searching loop starts at 'i = position'.
 *
 * A negative position means current_image + |position|.
 */
void place_at_position(gchar * filename, gint position)
{
    guint i;

    if (position < 0)
        position = current_image->number - position;

    for (i = position; i < array_length(); i++) {
        if (g_str_equal(FILE_ID(i), filename) == TRUE) {
            g_free(FILE_ID(i));
            FILE_ID(i) = FILE_ID(position);
            FILE_ID(position) = filename;
            return;
        }
    }
}

/* Place the last 'nb' files at the position 'where'. */
static void array_insert_the_end(guint where, guint nb)
{
    gpointer *backup;
    guint pos;

    /* Backup what will be moved. */
    backup = g_new(gpointer, nb);
    pos = array_length() - nb;
    memcpy(backup, &FILE_ID(pos), nb * sizeof(gpointer));

    /* Make some room for the insertion. */
    g_memmove(&FILE_ID(where) + nb, &FILE_ID(where),
              (pos - where) * sizeof(gpointer));

    /* The insertion. */
    memcpy(&FILE_ID(where), backup, nb * sizeof(gpointer));

    g_free(backup);
}

void insert_after_current_position(gchar ** filename, guint nb,
                                   gboolean shuffle)
{
    guint offset;
    guint i;

    offset = array_length();

    for (i = 0; i < nb; i++)
        add_to_list(filename[i], FALSE);

    reorder_array(offset, array_length() - offset, shuffle);

    /* Put the extension at its place in the array. */
    array_insert_the_end(current_image->number + 1, array_length() - offset);
}
