/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/********************************
 * Loading through ImageMagick. *
 ********************************/

#include "gliv.h"

#include <sys/stat.h>           /* S_IRWXU */
#include <sys/types.h>          /* pid_t */
#include <sys/wait.h>           /* waitpid() */
#include <stdio.h>              /* tmpnam(), remove(), perror() */
#include <time.h>               /* time(), nanosleep() */
#include <unistd.h>             /* *dir(), getpid(), fork(), _exit() */
#include <stdlib.h>             /* system() */
#include <math.h>               /* sqrt() */
#include <signal.h>             /* signal(), SIGCHLD */

/*
 * Used to store the connection between an unloadable file
 * by gdk-pixbuf and its PNG equivalent made by ImageMagick.
 */
static GHashTable *hash = NULL;

/*
 * We chdir() before doing the ImageMagick stuff, because it can leave
 * some garbage if interrupted. We will delete the dir at the end.
 */
static const gchar *gliv_wd, *magick_wd;

/* When the parent receives a SIGCHLD it is set to TRUE. */
static gboolean child_exited;

#define RATIO 1.618             /* The golden number : why not ? */
static void png_concatenate(gchar * basename)
{
    gint nb = 0;
    gchar *filename, *old_src;
    gchar *cmd, *src;

    src = g_strdup("");

    /* Build the list of files to concatenate. */
    for (;;) {
        filename = g_strdup_printf("%s.%d", basename, nb);
        if (g_file_test(filename, G_FILE_TEST_EXISTS) == FALSE) {
            g_free(filename);
            break;
        }

        old_src = src;
        src = g_strconcat(src, " ", filename, NULL);

        g_free(filename);
        g_free(old_src);
        nb++;
    }
    nb--;

    /* Concatenate them. */
    cmd =
        g_strdup_printf("%s -quality 100 -geometry \'1x1<\'  -tile %dx%d %s %s",
                        MAGICK_MONTAGE, (gint) (sqrt(RATIO * nb) + 1),
                        (gint) (sqrt(nb / RATIO) + 1), src, basename);
    system(cmd);
    g_free(cmd);

    /* Delete the previous files. */
    filename = g_strdup_printf("%s.%d", basename, nb);
    while (nb >= 0) {
        /* nb is decreasing so sprintf will always have enough room. */
        sprintf(filename, "%s.%d", basename, nb);
        remove(filename);
        nb--;
    }

    g_free(filename);
}

G_GNUC_NORETURN static void child_process(gchar * filename, gboolean tile,
                                          gchar * png_name)
{
    gchar *cmd;
    const gchar *dir;
    gint ret;

    dir = g_path_is_absolute(filename) ? "" : gliv_wd;

    /* The conversion. */
    cmd = g_strdup_printf("%s -quality 100 \"%s/%s\" \"png:%s\"",
                          MAGICK_CONVERT, dir, filename, png_name);

    ret = system(cmd);
    g_free(cmd);

    if (ret == 0) {
        if (tile == TRUE && g_file_test(png_name, G_FILE_TEST_EXISTS) == FALSE)
            png_concatenate(png_name);
    }

    /* We don't want the child process to run the atexit() function. */
    _exit(ret);
}

static void sig_chld(gint unused)
{
    child_exited = TRUE;
}

static gchar *convert_to_png(gchar * filename, gboolean tile)
{
    pid_t pid;
    gchar *png_name;
    gint status;
    const struct timespec req = {.tv_sec = 0,.tv_nsec = 10000000 };

    child_exited = FALSE;
    signal(SIGCHLD, sig_chld);
    png_name = g_strdup_printf("%ld_%s.png", random(),
                               g_path_get_basename(filename));

    pid = fork();

    if (pid == -1) {
        g_free(png_name);
        png_name = NULL;
        perror("fork");
    } else if (pid == 0)
        child_process(filename, tile, png_name);
    else {
        /* Parent. */
        while (child_exited == FALSE) {
            while (gtk_events_pending() != 0)
                gtk_main_iteration_do(FALSE);
            nanosleep(&req, NULL);
        }

        /* The child process exited. */

        waitpid(pid, &status, WNOHANG);

        if (WIFEXITED(status) == 0 || WEXITSTATUS(status) != 0) {
            /* The conversion failed. */
            g_free(png_name);
            png_name = NULL;
        }
    }

    return png_name;
}

static void image_magick_init(void)
{
    gchar *tmp_dir;

    hash = g_hash_table_new(g_str_hash, g_str_equal);

    gliv_wd = g_get_current_dir();

    magick_wd = g_strdup_printf("%s/gliv/%s_%ld_%d", g_get_tmp_dir(),
                                g_path_get_basename(tmpnam(NULL)),
                                time(NULL), getpid());

    tmp_dir = g_path_get_dirname(magick_wd);
    mkdir(tmp_dir, S_IRWXU);
    g_free(tmp_dir);

    mkdir(magick_wd, S_IRWXU);
}

GdkPixbuf *load_with_imagemagick(gchar * filename, gboolean tile)
{
    gchar *png_name;
    GdkPixbuf *im;

    if (hash == NULL)
        image_magick_init();

    chdir(magick_wd);

    png_name = g_hash_table_lookup(hash, filename);

    if (png_name == NULL) {
        /* First time this image is loaded. */
        png_name = convert_to_png(filename, tile);

        if (png_name != NULL) {
            g_hash_table_insert(hash, filename, png_name);
            im = GDK_PIXBUF_LOAD(png_name);
        } else
            im = NULL;
    } else
        /* This image has already been converted. */
        im = GDK_PIXBUF_LOAD(png_name);

    chdir(gliv_wd);

    return im;
}

static void suppress_file(gpointer unused, gpointer filename)
{
    remove(filename);
}

/*
 * Delete the temporary directory we created.
 * This function is called on exit so there is
 * no need freeing memory.
 */
void image_magick_finish(void)
{
    gchar *cmd, *tmp_dir, *msg;
    gint res;

    if (hash != NULL) {
        g_hash_table_foreach(hash, (GHFunc) suppress_file, NULL);
        cmd = g_strdup_printf("rm -fr %s", magick_wd);
        system(cmd);
        hash = NULL;

        tmp_dir = g_path_get_dirname(magick_wd);
        res = rmdir(tmp_dir);
        if (res < 0) {
            msg = _("This directory can be deleted if GLiv is not running");
            system(g_strdup_printf("echo \"%s\" > %s/README", msg, tmp_dir));
        }
    }
}
