/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*******************
 * The Open dialog *
 *******************/

#include "gliv.h"

#include <stdlib.h>             /* exit() */
#include <glob.h>               /* glob_t, glob(), GLOB_NOSORT */
#include <gdk/gdkkeysyms.h>     /* GDK_Escape */

static gboolean shuffle_flag = FALSE;
static gboolean dir_flag = TRUE;

static void toggle_flag(GtkToggleButton * button, gboolean * flag)
{
    *flag = gtk_toggle_button_get_active(button);
}

static void add_button(GtkFileSelection * dialog, gchar * label,
                       gboolean * value)
{
    GtkCheckButton *button;

    label = locale_to_utf8(add_mnemonic(label));
    button = GTK_CHECK_BUTTON(gtk_check_button_new_with_mnemonic(label));

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), *value);

    g_signal_connect(button, "toggled", G_CALLBACK(toggle_flag), value);

    gtk_container_add(GTK_CONTAINER(dialog->action_area), GTK_WIDGET(button));
}

/* A GtkFileSelection with a shuffle button. */
static GtkFileSelection *decorated_dialog(void)
{
    GtkFileSelection *dialog;
    gchar *label;

    label = locale_to_utf8(_("GLiv : Select files or use wild cards"));
    dialog = GTK_FILE_SELECTION(gtk_file_selection_new(label));

    gtk_file_selection_set_select_multiple(dialog, TRUE);

    add_button(dialog, _("Shuffle"), &shuffle_flag);
    add_button(dialog, _("All files in the directory"), &dir_flag);

    return dialog;
}

static void single_selection(GtkFileSelection * dialog, gboolean append)
{
    glob_t pglob;
    gchar *name;
    gchar **glob_res;
    guint i;

    /* Use what is in the entry. */
    glob(gtk_file_selection_get_filename(dialog), GLOB_NOSORT, NULL, &pglob);

    if (pglob.gl_pathc == 1) {
        /* No wildcard expansion. */

        if (append == TRUE)
            open_file(g_strdup(pglob.gl_pathv[0]), dir_flag, shuffle_flag);
        else {
            name = (dir_flag == TRUE) ? g_path_get_dirname(pglob.gl_pathv[0]) :
                g_strdup(pglob.gl_pathv[0]);

            init_list(&name, 1, FALSE, shuffle_flag);

            if (dir_flag == TRUE)
                place_at_position(g_strdup(pglob.gl_pathv[0]), 0);
        }
    } else {
        /* Add all expansions from glob(). */

        glob_res = g_new(gchar *, pglob.gl_pathc);
        for (i = 0; i < pglob.gl_pathc; i++)
            glob_res[i] = g_strdup(pglob.gl_pathv[i]);

        if (append == TRUE) {
            insert_after_current_position(glob_res, pglob.gl_pathc,
                                          shuffle_flag);
            open_file(pglob.gl_pathv[0], FALSE, FALSE);
        } else
            init_list(glob_res, pglob.gl_pathc, FALSE, shuffle_flag);

        g_free(glob_res);
    }
    globfree(&pglob);
}

static void process_selection(GtkFileSelection * dialog, gboolean append)
{
    gchar **array;
    guint nb;

    array = gtk_file_selection_get_selections(dialog);

    for (nb = 0; array[nb] != NULL; nb++);

    if (nb == 1)
        single_selection(dialog, append);

    else if (append == TRUE) {
        insert_after_current_position(array, nb, shuffle_flag);
        open_next_image();

    } else
        init_list(array, nb, FALSE, shuffle_flag);
}

/* Processes selection made by the GtkFileSelection. */
static void add_selected(GtkFileSelection * dialog, gboolean append)
{
    process_selection(dialog, append);

    gtk_widget_destroy(GTK_WIDGET(dialog));
}

/* Callback for the File->Open dialog. */
static gboolean file_selected(GtkFileSelection * widget)
{
    add_selected(widget, TRUE);
    return TRUE;
}

/* Leaves if ESC is pressed. */
static gboolean key_press_event(GtkWidget * widget, GdkEventKey * event,
                                gpointer data)
{
    if (event->keyval == GDK_Escape) {
        if (GPOINTER_TO_INT(data) == TRUE)
            /* First open dialog. */
            exit(0);
        else
            /* Dialog from menu. */
            gtk_widget_destroy(widget);
        return TRUE;
    }
    return FALSE;
}

/* Called if there is no filename on the command line. */
void show_open_dialog_first(void)
{
    GtkFileSelection *dialog;

    dialog = decorated_dialog();

    g_signal_connect(dialog->ok_button, "clicked", G_CALLBACK(gtk_main_quit),
                     NULL);

    g_signal_connect(dialog->cancel_button, "clicked", G_CALLBACK(exit), NULL);

    g_signal_connect(dialog, "delete_event", G_CALLBACK(exit), NULL);

    g_signal_connect(dialog, "key_press_event", G_CALLBACK(key_press_event),
                     GINT_TO_POINTER(TRUE));

    gtk_widget_show_all(GTK_WIDGET(dialog));
    gtk_main();

    /* The dialog widget has gone. */

    add_selected(dialog, FALSE);
}

/* Called by File->Open (or the 'o' key). */
gboolean menu_open(void)
{
    static GtkFileSelection *dialog = NULL;

    if (GTK_IS_WIDGET(dialog) == TRUE)
        return TRUE;

    dialog = decorated_dialog();

    g_signal_connect_swapped(dialog->ok_button, "clicked",
                             G_CALLBACK(file_selected), dialog);

    g_signal_connect_swapped(dialog->cancel_button, "clicked",
                             G_CALLBACK(gtk_widget_destroy), dialog);

    g_signal_connect(dialog, "key_press_event", G_CALLBACK(key_press_event),
                     GINT_TO_POINTER(FALSE));

    show_dialog(GTK_WIDGET(dialog), NULL, TRUE);
    return TRUE;
}
