/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/************************
 * Rendering functions. *
 ************************/

#include "gliv.h"

#include <gtkgl/gtkglarea.h>
#include <GL/gl.h>

extern rt_struct *rt;
extern options_struct *options;
extern gliv_image *current_image;
extern GtkWidget *gl_widget;

static guint idle_id = 0;

static void call_lists(gboolean smooth)
{
    guint id;
    gint filter;

    if (rt->filtering_enabled != smooth) {
        rt->filtering_enabled = smooth;
        filter = (smooth == TRUE) ? GL_LINEAR : GL_NEAREST;

        /* All textures have the same parameters. */
        for (id = 0; id < current_image->nb_tiles; id++) {
            glBindTexture(GL_TEXTURE_2D, current_image->tex_ids[id]);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter);
        }
    }

    for (id = 0; id < current_image->nb_tiles; id++)
        if (matrix_tile_visible(current_image->tiles + id) == TRUE)
            glCallList(current_image->list + id);
}

static void draw_checker(void)
{
    static guint tex_id = 0;
    guint i;
    guchar *texture;
    gdouble half_w, half_h;
    gboolean matrix_changed;

    matrix_changed = get_matrix_has_changed();

    if (rt->alpha_checks_changed == TRUE) {
        if (tex_id == 0)
            glGenTextures(1, &tex_id);

        texture = g_new(guchar, 12);
        for (i = 0; i < 3; i++) {
            texture[i] = texture[9 + i] = options->alpha1[i] >> 8;
            texture[3 + i] = texture[6 + i] = options->alpha2[i] >> 8;
        }

        glBindTexture(GL_TEXTURE_2D, tex_id);

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 2, 2, 0, GL_RGB,
                     GL_UNSIGNED_BYTE, texture);

        g_free(texture);
        rt->alpha_checks_changed = FALSE;
    } else
        glBindTexture(GL_TEXTURE_2D, tex_id);

    if (matrix_changed == FALSE)
        /* glMatrixMode(GL_MODELVIEW); */
        /* Save the matrix only if we will not replace it when redrawing. */
        glPushMatrix();

    glLoadIdentity();

    half_w = rt->wid_size->width / 2.0;
    half_h = rt->wid_size->height / 2.0;

    glBegin(GL_QUADS);

    glTexCoord2d(0.0, 0.0);
    glVertex2d(-half_w, -half_h);

    glTexCoord2d(half_w / 16.0, 0.0);
    glVertex2d(half_w, -half_h);

    glTexCoord2d(half_w / 16.0, half_h / 16.0);
    glVertex2d(half_w, half_h);

    glTexCoord2d(0.0, half_h / 16.0);
    glVertex2d(-half_w, half_h);

    glEnd();

    if (matrix_changed == FALSE)
        /* glMatrixMode(GL_MODELVIEW); */
        glPopMatrix();
}

static gboolean redraw(void)
{
    clear_zoom_frame();

    gdk_gl_wait_gdk();

    glDisable(GL_DITHER);
    if (current_image->has_alpha == TRUE && options->alpha_checks == TRUE)
        draw_checker();
    else
        glClear(GL_COLOR_BUFFER_BIT);
    glEnable(GL_DITHER);

    write_gl_matrix();

    call_lists(is_filtering_needed());

    gtk_gl_area_swap_buffers(GTK_GL_AREA(gl_widget));

    gdk_gl_wait_gl();

    /* To remove the idle func. */
    idle_id = 0;
    return FALSE;
}

void render(void)
{
    prioritize_textures(current_image, TRUE);

    matrix_reset();
    fill_ident(current_image);

    rt->filtering_enabled = FALSE;

    if (options->fullscreen == FALSE)
        goto_window();

    if (options->maximize == TRUE || options->scaledown == TRUE)
        matrix_set_max_zoom(-1, -1, TRUE);

    refresh(REFRESH_NOW | REFRESH_STATUS | APPEND_HISTORY);

    /* A bit dumb, but needed on some cards. */
    refresh(REFRESH_NOW);

    g_free(current_image->ident);
    current_image->ident = NULL;

}

void zoom_in(gdouble ratio)
{
    gint pointer_x, pointer_y;
    gdouble x, y;

    if (options->zoom_pointer == TRUE) {
        /* The pointer is the zoom center. */
        gdk_window_get_pointer(gl_widget->window, &pointer_x, &pointer_y, NULL);
        x = (gdouble) pointer_x;
        y = (gdouble) pointer_y;
    } else {
        /* The zoom center is the midle of the window. */
        x = rt->wid_size->width / 2.0;
        y = rt->wid_size->height / 2.0;
    }

    matrix_zoom(ratio, x, y);
    refresh(REFRESH_IMAGE | REFRESH_STATUS | APPEND_HISTORY);
}

void refresh(guint what)
{
    /* GTK does that automatically but with more overhead. */
    if ((what & REFRESH_IMAGE) != 0) {
        if (idle_id == 0)
            idle_id = gtk_idle_add_priority(GDK_PRIORITY_REDRAW,
                                            (GtkFunction) redraw, NULL);
    } else if ((what & REFRESH_NOW) != 0) {
        if (idle_id != 0)
            gtk_idle_remove(idle_id);
        redraw();
    }

    if ((what & REFRESH_STATUS) != 0)
        update_status_bar();

    if ((what & APPEND_HISTORY) != 0)
        append_history();
}
