/*
 * Gmail. A Gnome email client.
 * Copyright (C) 1999 Wayne Schuller
 *
 * print.c - code for printing messages using gnome-print
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"

#include <libgnomeprint/gnome-printer.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-meta.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <libgnomeprint/gnome-print-pixbuf.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gnome-printer-dialog.h>
#include <libgnomeprint/gnome-print-master.h>
#include <libgnomeprint/gnome-print-master-preview.h>
#include <libgnomeprint/gnome-print-dialog.h>

extern 	_GmailApp_	*GmailApp;
extern 	Message * load_vfolder_msg(gint id);
extern void free_message(Message *message);
extern void gmail_wrap_string(gchar *str, int width);

void print_cb (GtkWidget *widget, void *data);
void print_preview_cb (GtkWidget *widget, void *data);
static void create_printer_page (GnomePrintContext *pc, Message *msg);
static void print_msg (Message *msg);
static gint gmail_print_text(gchar *text, gint x, gint y, gint inc, GnomePrintContext *pc, GnomeFont *font);

/* Stolen from gnumeric 0.65 print-cell.c */
static int g_unichar_to_utf8 (gint c, gchar *outbuf);
int print_show_iso8859_1 (GnomePrintContext *pc, char const *text);
int print_show (GnomePrintContext *pc, char const *text);
/* are these needed ? */
double get_width_string_n (GnomeFont *font, char const* text, guint n);
double get_width_string (GnomeFont *font, char const* text);

/*
 * This is cut & pasted from glib 1.3
 *
 * We need it only for iso-8859-1 converter and it will be
 * abandoned, if glib 2.0 or any other unicode library will
 * be introduced.
 */

static int
g_unichar_to_utf8 (gint c, gchar *outbuf)
{
  size_t len = 0;
  int first;
  int i;

  if (c < 0x80)
    {
      first = 0;
      len = 1;
    }
  else if (c < 0x800)
    {
      first = 0xc0;
      len = 2;
    }
  else if (c < 0x10000)
    {
      first = 0xe0;
      len = 3;
    }
   else if (c < 0x200000)
    {
      first = 0xf0;
      len = 4;
    }
  else if (c < 0x4000000)
    {
      first = 0xf8;
      len = 5;
    }
  else
    {
      first = 0xfc;
      len = 6;
    }

  if (outbuf)
    {
      for (i = len - 1; i > 0; --i)
	{
	  outbuf[i] = (c & 0x3f) | 0x80;
	  c >>= 6;
	}
      outbuf[0] = c | first;
    }

  return len;
}

#ifndef _PROPER_I18N
/*
 * print_show_iso8859_1
 *
 * Like gnome_print_show, but expects an ISO 8859.1 string.
 *
 * NOTE: This function got introduced when gnome-print switched to UTF-8,
 * and will disappear again once Gnumeric makes the switch. Deprecated at
 * birth!
 */
int
print_show_iso8859_1 (GnomePrintContext *pc, char const *text)
{
	gchar *p, *utf, *udyn, ubuf[4096];
	gint len, ret, i;

	g_return_val_if_fail (pc && text, -1);

	if (!*text)
		return 0;

	/* We need only length * 2, because iso-8859-1 is encoded in 1-2 bytes */
	len = strlen (text);
	if (len * 2 > sizeof (ubuf)) {
		udyn = g_new (gchar, len * 2);
		utf = udyn;
	} else {
		udyn = NULL;
		utf = ubuf;
	}
	p = utf;

	for (i = 0; i < len; i++) {
		p += g_unichar_to_utf8 (((guchar *) text)[i], p);
	}

	ret = gnome_print_show_sized (pc, utf, p - utf);

	if (udyn)
		g_free (udyn);

	return ret;
}
#endif

int
print_show (GnomePrintContext *pc, char const *text)
{
#ifdef _PROPER_I18N
	wchar_t* wcs,wcbuf[4096];
	char* utf8,utf8buf[4096];
	
	size_t conv_status;
	int n = strlen (text);
	int retval;
	const int wcbuf_len = sizeof (wcbuf) / sizeof (wcbuf[0]);

	g_return_val_if_fail (pc && text, -1);	
	
	if ( n > wcbuf_len)
		wcs = g_new (wchar_t,n);
	else
		wcs = wcbuf;

	conv_status = mbstowcs (wcs, text, n);

	if (conv_status == (size_t)(-1)){
		if (wcs != wcbuf)
			g_free (wcs);
		return 0;
	};
	if (conv_status * 6 > sizeof (utf8buf))
		utf8 = g_new (gchar, conv_status * 6);
	else
		utf8 = utf8buf;

	{
		size_t i;
		char* p = utf8;
		for(i = 0; i < conv_status; ++i)
			p += g_unichar_to_utf8 ( (gint) wcs[i], p);
		if (wcs != wcbuf)
			g_free (wcs);			
		retval = gnome_print_show_sized (pc, utf8, p - utf8);			
	}	

	if (utf8 != utf8buf)
		g_free (utf8);
	return retval;		
#else
	return print_show_iso8859_1 (pc, text);
#endif
};

double
get_width_string_n (GnomeFont *font, char const* text, guint n)
{
#ifdef _PROPER_I18N
	wchar_t* wcs, wcbuf[4000];
	size_t conv_status, i;
	double total = 0;	
	
	if ( n > (sizeof(wcbuf)/sizeof(wcbuf[0])))
		wcs = g_new (wchar_t,n);
	else
		wcs = wcbuf;

	conv_status = mbstowcs (wcs, text, n);

	if (conv_status == (size_t)(-1)){
		if (wcs != wcbuf)
			g_free (wcs);
		return 0;
	};
	for (i = 0; i < conv_status; ++i)
		total += gnome_font_get_glyph_width (font, 
				gnome_font_lookup_default (font, wcs[i]));

	if (wcs != wcbuf)
		g_free (wcs);
	return total;
#else
	return gnome_font_get_width_string_n (font, text, n);
#endif
};


double
get_width_string (GnomeFont *font, char const* text)
{
	return get_width_string_n (font, text, strlen(text));
};
/* End of material stolen from gnumeric */

/* Gmail Callback for the print button. */
void 
print_cb (GtkWidget *widget, void *data)
{
	Message *msg;
	GtkWidget *text;
	GladeXML *xml;
	GString *body;
	gint id = -1;

	g_return_if_fail(GmailApp->msgpane != NULL);

	id = (gint) gtk_object_get_data(GTK_OBJECT(GmailApp->msgpane), "id");

	/* Get id of currently display message. */
	/* Load the message. */
	msg = load_vfolder_msg(id);

	/* Get the text of the message from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));
	text = glade_xml_get_widget(xml, "textmsg");
	body = g_string_new(gtk_editable_get_chars(GTK_EDITABLE (text), 0, -1));

	/* Replace the current msg body with the one from the widget. */
	g_string_free(msg->message, TRUE);
	msg->message = body;

	/* Print the message. */
	print_msg(msg);

	free_message(msg);
}

/* Gmail Callback for the print preview button. 
 * FIXME: This repeats code from print_cb
 */
void 
print_preview_cb (GtkWidget *widget, void *data)
{
	Message *msg;
	GnomePrintContext *pc;
	GnomePrintMasterPreview *pmp;
	GnomePrintMaster *gpm;
	gint copies=1, collate=TRUE; 
	gint id = -1;
	GtkWidget *text;
	GladeXML *xml;
	GString *body;

	g_return_if_fail(GmailApp->msgpane != NULL);
	id = (gint) gtk_object_get_data(GTK_OBJECT(GmailApp->msgpane), "id");

	/* Get id of currently display message. */
	/* Load the message. */
	msg = load_vfolder_msg(id);

	/* Get the text of the message from the widget. */
	xml = glade_get_widget_tree(GTK_WIDGET (GmailApp->msgpane));
	text = glade_xml_get_widget(xml, "textmsg");
	body = g_string_new(gtk_editable_get_chars(GTK_EDITABLE (text), 0, -1));

	/* Replace the current msg body with the one from the widget. */
	g_string_free(msg->message, TRUE);
	msg->message = body;

	/* transfer dialog data to output context */
	gpm = gnome_print_master_new();
	gnome_print_master_set_copies(gpm, copies, collate);

	pc = gnome_print_master_get_context(gpm);
	create_printer_page (pc, msg);
	gnome_print_master_close(gpm);
	pmp = gnome_print_master_preview_new(gpm, "Test preview");
	gtk_widget_show(GTK_WIDGET(pmp));

	free_message(msg);
}

/* This is called from print_cb.
 * It opens the print dialog widget, and allows them the choice of printing
 * or viewing the print preview.
 * This is based on one of the gnome-print test programs.
 */
static void
print_msg (Message *msg)
{
	GnomePrintContext *pc;
	GnomePrintMaster *gpm;
	gint copies=1, collate=TRUE; 
	gboolean do_preview = FALSE;
	GnomePrintDialog *gpd;



	gpd = (GnomePrintDialog *) gnome_print_dialog_new("Print test", GNOME_PRINT_DIALOG_COPIES);
	gnome_print_dialog_set_copies(gpd, copies, collate);

	switch (gnome_dialog_run(GNOME_DIALOG(gpd))) {
	case GNOME_PRINT_PRINT:
		do_preview = FALSE;
		break;
	case GNOME_PRINT_PREVIEW:
		do_preview = TRUE;
		break;
	case GNOME_PRINT_CANCEL:
		gnome_dialog_close(GNOME_DIALOG(gpd));
		return;
		break;
	}

	/* printf("printer = %p\n", gnome_print_dialog_get_printer(gpd)); */

	/* transfer dialog data to output context */
	gpm = gnome_print_master_new();
	gnome_print_dialog_get_copies(gpd, &copies, &collate);
	gnome_print_master_set_copies(gpm, copies, collate);
	gnome_print_master_set_printer(gpm, gnome_print_dialog_get_printer(gpd));
	gnome_dialog_close(GNOME_DIALOG(gpd));

	pc = gnome_print_master_get_context(gpm);
	create_printer_page (pc, msg);
	gnome_print_master_close(gpm);

	if (do_preview) {
		GnomePrintMasterPreview *pmp;
		pmp = gnome_print_master_preview_new(gpm, "Test preview");
		gtk_widget_show(GTK_WIDGET(pmp));
	} else {
		gnome_print_master_print(gpm);
	}
}

/* This function actually does the job of rendering the message into
 * a gnome print page.
 * This is the function to modify to improve the look of the gmail
 * printing.
 */
static void
create_printer_page (GnomePrintContext *pc, Message *msg)
{
	gint x;	/* The left margin. */
	gint y; /* The top margin (vertical start point). */
	gint inc; /* The amount of spacing between lines. */
	gchar *text, *headers;
	GnomeFont *font;

	gnome_print_beginpage(pc, "Gmail");
	gnome_print_gsave (pc);

	font = gnome_font_new_closest ("Helvetica", GNOME_FONT_BOOK, 0, 14);
	if (font == NULL) {
		g_warning ("Font not found\n");
		return;
	}
	gnome_print_setfont (pc, font);

	headers = g_strdup_printf("Subject: %s\nFrom: %s\nTo: %s\nDate: %s\nMessage:\n", msg->subject, msg->from, msg->to, msg->date);

	gmail_wrap_string(headers, gnome_config_get_int("/gmail/UserInfo/wrap_len"));

	x = 30; inc = 30;
	y = 800; /* Top of the page. */
	y = gmail_print_text(headers, x, y, inc, pc, font);
	g_free(headers);

	/* Print the body of the email. */
	inc = 20;
	gmail_wrap_string(msg->message->str, gnome_config_get_int("/gmail/UserInfo/wrap_len"));
	y = gmail_print_text(msg->message->str, x, y, inc, pc, font);

	/* Print the footer. */
	gtk_object_unref (GTK_OBJECT (font));
	font = gnome_font_new_closest ("Helvetica", GNOME_FONT_LIGHT, 1, 9);
	gnome_print_setfont (pc, font);
	gnome_print_moveto (pc, 330, 20);

	text = g_strdup_printf("Printed by Gmail %s. (http://gmail.linuxpower.org)", VERSION);
	gnome_print_show (pc, text);
	g_free(text);

	gnome_print_showpage(pc);

	/*  gnome_print_context_close (pc);*/

	gtk_object_unref (GTK_OBJECT (font));
}

/* gmail_print_text
 * Utiltity function for printing a set amount of text. Takes into account
 * newlines, and starts a new page when we hit one.
 */
static gint
gmail_print_text(gchar *text, gint x, gint y, gint inc, GnomePrintContext *pc, GnomeFont *font) {
	gchar *copy = g_strdup(text);
	gchar *tmp = copy;

	/* FIXME: This code unfinished. */
	while (TRUE) { 
		gchar *text;

		gnome_print_moveto (pc, x, y);	/* Increment a newline. */
		text = strtok(copy, "\n");	/* Grab the first line. */
		if (text == NULL) break; /* EOF. */
  		print_show (pc, text); /* Print the line. Modified for encoding. */
		copy = NULL; /* To make subsequent strtok calls work. */
		y -= inc;

		/* Check if we need to start a new page. */
		if (y < 40) {
  			gnome_print_showpage (pc);
			gnome_print_beginpage(pc, "new page");
			gnome_print_setfont (pc, font);
			y = 800;
		}
	}
	g_free(tmp);

	return(y);
}
