/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <signal.h>

#include "config.h"
#ifdef ZVT
#   include <stdlib.h>
#   include <zvt/zvtterm.h>
#endif

#include "settings.h"
#include "editor.h"
#include "notebook.h"
#include "dialog.h"
#include "misc.h"
#include "statusbar.h"
#include "packages.h"

#define NONE  -1
#define ERROR -2

extern char **environ;
char        **env_copy = NULL;
char *parse_filename(const char *name, world *w);
char *parse_command(const char *filename);
int   check_e_editor_change(e_editor_t *ee);
int tab_editor_key_pressed(GtkWidget *input, GdkEventKey *event, editor_t *e);

void uploade_inline_create(editor_t *e, const char *upload) {
	upload_inline_t *u = g_malloc(sizeof(upload_inline_t));
	u->upload_command = g_strdup(upload);
	e->upload = u;
	e->upload_type = UPLOAD_INLINE;
}

void uploade_mcp_create(editor_t *e, const char *ref, const char *type) {
	upload_mcp_t *u = g_malloc(sizeof(upload_mcp_t));
	u->ref = g_strdup(ref);
	u->type = g_strdup(type);
	e->upload = u;
	e->upload_type = UPLOAD_MCP;
}


void upload_inline_destroy(editor_t *e) {
	upload_inline_t *u = e->upload;
	g_free(u->upload_command);
	g_free(u);
	e->upload = NULL;
}

void upload_inline(world *w, upload_inline_t *up, GList *lines,
		   int has_newlines) {
	GList *l;
	int len;
	if(debug) printf("Uploading an inline editor\n");
	if(gm_world_is_connected(w)) {
		gm_world_writeln(w, up->upload_command);
		if(has_newlines) {
			for(l = lines; l; l = g_list_next(l)) {
				gm_world_write(w, l->data);
			}
			if((l = g_list_last(lines))) {
				len = strlen(l->data);
				if(len && ((char *) l->data)[len - 1] != '\n') {
					gm_world_write(w, "\n");
				}
			}
			gm_world_write(w, ".\n");
		} else { /* ! has_newlines */
			for(l = lines; l; l = g_list_next(l)) {
				gm_world_writeln(w, l->data);
			}
			gm_world_write(w, "\n.\n");
		}
	} else {
		gm_dialog_popup_only(_("Error"),
				     _("World isn't connected,\n"
				       "can't upload the data from the editor"),
				     FALSE);
	}
}

void upload_mcp(editor_t *e, GList *lines, int has_newlines) {
	upload_mcp_t *u = e->upload;
	if(debug) printf("MCP upload requested\n");
	gm_mcp_upload_editor(e, u->ref, u->type, lines, has_newlines);
}

void editor_upload(editor_t *e, GList *lines, int has_newlines) {
	if(e->upload_type == UPLOAD_INLINE) {
		upload_inline(e->belongs_to, (upload_inline_t *) e->upload, lines,
			      has_newlines);
	} else { /* e->upload_type == UPLOAD_MCP */
		upload_mcp(e, lines, has_newlines);
	}
	gm_notebook_set(gm_notebook_get_pos((void *) e->belongs_to));
}

void upload_mcp_destroy(editor_t *e) {
	upload_mcp_t *u = e->upload;
	g_free(u->ref);
	g_free(u->type);
	g_free(u);
	e->upload = NULL;
}

void upload_destroy(editor_t *e) {
	if(debug) printf("\t\t\tDetroying upload capabilities\n");
	if(!e->upload) {
		if(debug) printf("\t\t\tEditor didn't had any upload capabilities !\n");
		return;
	}
	if(e->upload_type == UPLOAD_INLINE) {
		upload_inline_destroy(e);
	} else if(e->upload_type == UPLOAD_MCP) {
		upload_mcp_destroy(e);
	} else if(debug) {
		printf("MAJOR ERROR: DON'T KNOW HOW TO DESTROY THIS UPLOAD TYPE\n");
	}
}

FILE *create_file(const char *filename) {
	FILE *file = fopen(filename, "w");
	char *s;
	if(!file) {
		s = g_strdup_printf(_("Error opening file \"%s\":\n"
				      "%s\n"), filename, g_strerror(errno));
		gm_dialog_popup(_("Error"), s, B_OK, TRUE);
		g_free(s);
	}
	return file;
}

e_editor_t *external_editor_new(world *w, const char *name) {
#ifdef ZVT
	e_tab_editor_t   *ete = NULL;
#endif
	e_notab_editor_t *ene = NULL;
	e_editor_t *ee;
	editor_t *e;
#ifdef ZVT
	if(settings->editor_needs_term) {
		ete = g_malloc(sizeof(e_tab_editor_t));
		ee = E_EDITOR(ete);
		e = EDITOR(ete);
	} else {
#endif
		ene = g_malloc(sizeof(e_notab_editor_t));
		ee = E_EDITOR(ene);
		e = EDITOR(ene);
#ifdef ZVT
	}
#endif
	e->type = EDITOR_EXTERNAL_TYPE;
	e->belongs_to = w;
	e->name = g_strdup(name);
	e->done_receiving = FALSE;
	e->ok_to_close = FALSE;
	e->upload = NULL;
	ee->filename = parse_filename(name, w);
	ee->command = parse_command(ee->filename);
	ee->file    = create_file(ee->filename);
	ee->check_change_id = NONE;
	ee->has_notebook_page = settings->editor_needs_term;
	if(debug) printf("External editor created:\n"
			 "\tbelongs_to \"%s\"\n"
			 "\tname       \"%s\"\n"
			 "\tfilename   \"%s\"\n"
			 "\tcommand    \"%s\"\n"
			 "\thas_notebook_page %d\n",
			 w->p->name,
			 e->name,
			 ee->filename,
			 ee->command,
			 ee->has_notebook_page);
#ifdef ZVT
	if(ee->has_notebook_page) {
		ete->main = ete->term = NULL;
	} else {
#endif
		ene->pid = -1;
		ene->check_closed_id = -1;
#ifdef ZVT
	}
#endif
	return ee;
}

int tab_editor_key_pressed(GtkWidget *input, GdkEventKey *event,
			   editor_t *e) {
	int handled = TRUE;

	/* Tab switching ... */
	if((event->keyval == GDK_Tab ||
	    event->keyval == GDK_KP_Tab ||
	    event->keyval == GDK_ISO_Left_Tab) &&
	   event->state & settings->hotkey_mask) {
		if(event->state & GDK_SHIFT_MASK)
			gm_notebook_prev();
		else
			gm_notebook_next();
	} else if(event->keyval >= GDK_0 && event->keyval <= GDK_9 && 
		  event->state & settings->hotkey_mask) {
		gm_notebook_set(event->keyval - GDK_0 > 0 ? event->keyval-GDK_0-1 : 9);
	} else if(settings->hotkey_arrow && event->state & settings->hotkey_mask &&
		  event->keyval == GDK_Left) {
		gm_notebook_prev();
	} else if(settings->hotkey_arrow && event->state & settings->hotkey_mask &&
		  event->keyval == GDK_Right) {
		gm_notebook_next();
		/* END switching ... */

	} else {
		/* Unhandled key combo */
		handled = FALSE;
	}

	if(handled) {
		gtk_signal_emit_stop_by_name(GTK_OBJECT(input), "key_press_event");
	}
	return handled;

}

int i_tab_editor_key_pressed(GtkWidget *input, GdkEventKey *event,
			     editor_t *e) {
	int handled = tab_editor_key_pressed(input, event, e);
	if(handled) {
		return handled;
	}
	/* do anything other than switching tabs with the keyboard */
	return handled;
}

void i_editor_upload_cb(GtkWidget *upload_button, i_editor_t *ie) {
	GList *lines = NULL;
	char *text;

	if(debug) printf("I_Editor Upload button clicked\n");
	text = gtk_editable_get_chars(GTK_EDITABLE(ie->text), 0,
				      gtk_text_get_length(GTK_TEXT(ie->text)));
	lines = str_to_GList_sep(text, '\n');
	editor_upload(EDITOR(ie), lines, FALSE);
	g_list_foreach(lines, (GFunc) g_free, NULL);
	g_list_free(lines);
	g_free(text);
}

void i_editor_close_cb(GtkWidget *close_button, i_editor_t *ie) {
	if(debug) printf("I_Editor Close button clicked\n");
	gm_editor_close(EDITOR(ie));
}

void create_i_editor_widgets(i_editor_t *ie) {
	GtkWidget *hbb, *scrollw;

	ie->main = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(ie->main);

	hbb = gtk_hbutton_box_new();
	gtk_widget_show(hbb);
	gtk_box_pack_start(GTK_BOX(ie->main), hbb, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbb), 3);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbb), GTK_BUTTONBOX_START);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbb), 3);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbb), 0, 0);

	ie->upload_button = gtk_button_new_with_label(_("Upload"));
	gtk_widget_show(ie->upload_button);
	gtk_container_add(GTK_CONTAINER(hbb), ie->upload_button);
	GTK_WIDGET_UNSET_FLAGS(ie->upload_button, GTK_CAN_FOCUS);
	gtk_signal_connect(GTK_OBJECT(ie->upload_button), "clicked",
			   GTK_SIGNAL_FUNC(i_editor_upload_cb), ie);

	ie->close_button = gtk_button_new_with_label(_("Close"));
	gtk_widget_show(ie->close_button);
	gtk_container_add(GTK_CONTAINER(hbb), ie->close_button);
	GTK_WIDGET_UNSET_FLAGS(ie->close_button, GTK_CAN_FOCUS);
	gtk_signal_connect(GTK_OBJECT(ie->close_button), "clicked",
			   GTK_SIGNAL_FUNC(i_editor_close_cb), ie);

	scrollw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(scrollw);
	gtk_box_pack_start(GTK_BOX(ie->main), scrollw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollw),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	ie->text = gtk_text_new(NULL, NULL);
	gtk_widget_show(ie->text);
	gtk_container_add(GTK_CONTAINER(scrollw), ie->text);
	gtk_text_set_editable(GTK_TEXT(ie->text), TRUE);
	gtk_signal_connect(GTK_OBJECT(ie->text), "key_press_event",
			   GTK_SIGNAL_FUNC(i_tab_editor_key_pressed), ie);
	gtk_widget_set_style(GTK_WIDGET(ie->text), input_style);


	ie->tab_label = gtk_label_new("");
	gtk_widget_show(ie->tab_label);
}

i_editor_t *internal_editor_new(world *w, const char *name) {
	i_editor_t *ie = g_malloc(sizeof(i_editor_t));
	editor_t *e = EDITOR(ie);

	e->type = EDITOR_INTERNAL_TYPE;
	e->belongs_to = w;
	e->name = g_strdup(name);
	e->done_receiving = FALSE;
	e->ok_to_close = FALSE;
	e->upload = NULL;

	create_i_editor_widgets(ie);

	return ie;
}

editor_t *editor_new(world *w, const char *name) {
	editor_t *e;
	if(settings->editor_external) {
		e = EDITOR(external_editor_new(w, name));
	} else {
		e = EDITOR(internal_editor_new(w, name));
	}
	w->editors = g_list_append(w->editors, (void *) e);
	return e;
}

void editor_new_from_oob(world *w, const char *name, const char *upload) {
	editor_t *e = editor_new(w, name);
	uploade_inline_create(e, upload);
	w->current_editor = e;
}

void gm_editor_new_from_oob(world *w, const char *line) {
	char *name, *upload;
	int namelen;

	if(strncmp(line, "edit ", 5) != 0) {
		return;
	}
	name = strstr(line, "name: ");
	if(name == NULL) return;
	upload = strstr(name + 6, "upload: ");
	if(upload == NULL) return;
	namelen = upload - name - 7;
	name = g_strdup(name + 6);
	name[namelen] = 0;
	upload = g_strdup(upload + 8);
	if(debug) printf("Editor command reconised: name=\"%s\", upload=\"%s\"\n",
			 name, upload);
	editor_new_from_oob(w, name, upload);
	g_free(name);
	g_free(upload);
}

editor_t *gm_editor_new_from_mcp(world *w, const char *name, const char *ref,
				 const char *type) {
	editor_t *e = editor_new(w, name);
	uploade_mcp_create(e, ref, type);
	return e;
}

int gm_editor_ok_to_close(editor_t *e) {
	if(debug) printf("Checking if ok to close editor \"%s\"\n", e->name);
	return e->ok_to_close;
}

void editor_close_internal(i_editor_t *ie) {
	editor_t *e = EDITOR(ie);
	if(debug) printf("\t\tClosing INTERNAL editor \"%s\"\n", EDITOR(ie)->name);
	gm_notebook_remove_editor(e);

	e->belongs_to->editors = g_list_remove(e->belongs_to->editors, (void *) e);
	if(e->belongs_to->current_editor == e) {
		e->belongs_to->current_editor = NULL;
	}
	g_free(e->name);
	upload_destroy(e);
	g_free(e);
}

void editor_close_external(e_editor_t *ee) {
	editor_t *e = EDITOR(ee);
	if(debug) printf("\t\tClosing EXTERNAL editor \"%s\"\n", e->name);
	if(ee->check_change_id != NONE) {
		if(ee->check_change_id != ERROR) {
			gtk_timeout_remove(ee->check_change_id);
		}
		if(!check_e_editor_change(ee)) {
			gm_dialog_popup_only(_("Error"),
					     _("An error occured checking if the data\n"
					       "of this editor needs to be uploaded.\n"
					       "Changes in the editor since the last\n"
					       "upload have been lost :("), FALSE);
		}
	}
#ifdef ZVT
	if(IS_E_TAB_EDITOR(ee) && E_TAB_EDITOR(ee)->main) {
		gm_notebook_remove_editor(e);
		gtk_widget_destroy(E_TAB_EDITOR(ee)->main);
	} else { /* IS_E_NOTAB_EDITOR */
#endif
		if(E_NOTAB_EDITOR(ee)->pid != -1) {
			kill(E_NOTAB_EDITOR(ee)->pid, SIGHUP);
		}
		if(E_NOTAB_EDITOR(ee)->check_closed_id != -1) {
			gtk_timeout_remove(E_NOTAB_EDITOR(ee)->check_closed_id);
		}
#ifdef ZVT
	}
#endif
	e->belongs_to->editors = g_list_remove(e->belongs_to->editors, (void *) e);
	if(e->belongs_to->current_editor == e) {
		e->belongs_to->current_editor = NULL;
	}
	if(settings->editor_delete_file) {
		remove(ee->filename);
	}

	g_free(e->name);
	upload_destroy(e);
	g_free(ee->command);
	g_free(ee->filename);
	g_free(e);
}


void gm_editor_close(editor_t *e) {
	if(IS_I_EDITOR(e)) {
		editor_close_internal(I_EDITOR(e));
	} else if(IS_E_EDITOR(e)) {
		editor_close_external(E_EDITOR(e));
	}
}

void i_editor_add_line(i_editor_t *ie, const char *line, int linelen) {
	gtk_text_insert(GTK_TEXT(ie->text), NULL, NULL, NULL, line, linelen);
	gtk_text_insert(GTK_TEXT(ie->text), NULL, NULL, NULL, "\n", 1);
}

void e_editor_add_line(e_editor_t *ee, const char *line, int linelen) {
	char *s;
	if(debug) printf("E_Editor: %s(%d)\n", line, linelen);
	if(fputs(line, ee->file) == EOF || fputc('\n', ee->file) == EOF) {
		s = g_strdup_printf(_("Error writing to \"%s\":\n"
				      "%s\n"), ee->filename, g_strerror(errno));
		gm_dialog_popup(_("Error"), s, B_OK, TRUE);
		g_free(s);
	}
}

void gm_editor_add_line(editor_t *e, const char *line, int linelen) {
	if(IS_I_EDITOR(e)) {
		i_editor_add_line(I_EDITOR(e), line, linelen);
	} else { /* IS_E_EDITOR */
		e_editor_add_line(E_EDITOR(e), line, linelen);
	}
}

void add_editor_to_notebook(editor_t *e, GtkWidget *main) {
	int pos;
#ifdef ZVT
	GtkWidget *tab  = IS_I_EDITOR(e) ?
		I_EDITOR(e)->tab_label : E_TAB_EDITOR(e)->tab_label;
#else
	GtkWidget *tab  = I_EDITOR(e)->tab_label;
#endif
	if(debug) printf("\tAdding to notebook\n");

	pos = gm_notebook_get_pos((void *) (e->belongs_to));
	if(pos != -1) {
		pos += g_list_length(e->belongs_to->editors);
	}
	gm_notebook_add_type(pos, TK_EDITOR, (void *) e, main, tab);
}

int i_editor_done(i_editor_t *ie) {
	add_editor_to_notebook(EDITOR(ie), ie->main);
	return TRUE;
}

void copy_enviroment() { /* copy()ed form zterm - zvt terminal example */
#if 1 /* new version */
	char **p;
	int i = 0;

	for (p = environ; *p; p++);
	env_copy = g_malloc(sizeof(char *) * (p - environ + 3));
	for (p = environ; *p; p++) {
		if (strncmp(*p, "TERM=", 5) == 0 ||
		    strncmp (*p, "COLUMNS=", 8) == 0 ||
		    strncmp (*p, "LINES=", 6) == 0)
			continue;
		env_copy[i++] = *p;
	}
	env_copy[i++] = "TERM=xterm";
	env_copy[i] = NULL;
#else /* old version */
	char **env, **p;
	int winid_pos, i;

	env = environ;
	for (p = env; *p; p++);
	i = p - env;
	env_copy = (char **) g_malloc (sizeof (char *) * (i + 3));
	for (i = 0, p = env; *p; p++) {
		if (strncmp (*p, "TERM=", 5) == 0) {
			env_copy [i++] = "TERM=xterm";
		} else if ((strncmp (*p, "COLUMNS=", 8) == 0) ||
			   (strncmp (*p, "LINES=", 6) == 0)) {
			continue;
		} else {
			env_copy [i++] = *p;
		}
	}
	env_copy [i++] = "COLORTERM=gmoo-editor";
	winid_pos = i++;
	env_copy [winid_pos] = "TEST";
	env_copy [i] = NULL;
#endif
}

int check_e_editor_change(e_editor_t *ee) {
	editor_t *e = EDITOR(ee);
	struct stat stats;
	GList *lines;

	if(stat(ee->filename, &stats) == 0) {
		if(stats.st_mtime > ee->last_modified) {
			if(debug) printf("Editor \"%s\" changed to:\n", e->name);
			lines = get_lines_from_file(ee->filename);
			editor_upload(e, lines, TRUE);
			g_list_foreach(lines, (GFunc) g_free, NULL);
			g_list_free(lines);
			ee->last_modified = stats.st_mtime;
		} else if (debug)
			printf("Editor \"%s\" _not_ changed\n", e->name);
	}
	/* else some error occured statting the file. Most likely the editor is just
	 * saveing the file, and doesn't allow us to stat it now. like, it is recreating
	 * it ;)
	 * Besides, if there really was an error, it will pop up when the user closes
	 * the editor.
	 */
	return TRUE; /* repeat this timeout */
}

#ifdef ZVT
int start_e_tab_editor(e_tab_editor_t *ete) {
	ZvtTerm *term = ZVT_TERM(ete->term);
	char *cmd[4];
	char *s;

	if(!env_copy) {
		copy_enviroment();
	}
	cmd[0] = "/bin/sh";
	cmd[1] = "-c";
	cmd[2] = E_EDITOR(ete)->command;
	cmd[3] = NULL;

	if(debug) printf("\tCreating new process\n");
	switch(zvt_term_forkpty(term,
				ZVT_TERM_DO_UTMP_LOG|ZVT_TERM_DO_WTMP_LOG)) {
	case -1:
		s = g_strconcat("Could not fork child process:\n", g_strerror(errno),
				NULL);
		gm_dialog_popup_only(_("Error"), s, TRUE);
		g_free(s);
		return FALSE;
	case 0:
		execve(cmd[0], cmd, env_copy);
		s = g_strconcat(_("Could not fork child process:\n"),
				g_strerror(errno), NULL);
		gm_dialog_popup_only(_("Error"), s, TRUE);
		g_free(s);
		_exit(127);
	}
	if(debug) printf("Done opening\n");
	return TRUE; /* sucessfull opening */
}

void tab_editor_died(ZvtTerm *term, e_tab_editor_t *ete) {
	world *w = EDITOR(ete)->belongs_to;
	if(debug)
		printf("Editor \"%s\" of world \"%s\" died\n", EDITOR(ete)->name,
		       w->p->name);
	editor_close_external(E_EDITOR(ete));
}

int e_tab_editor_done(e_tab_editor_t *ete) {
	ete->main = gtk_frame_new(NULL);
	gtk_widget_ref(ete->main);
	gtk_widget_show(ete->main);
	gtk_frame_set_shadow_type(GTK_FRAME(ete->main), GTK_SHADOW_IN);

	ete->term = zvt_term_new();
	gtk_widget_show(ete->term);
	zvt_term_init(ete->term, EDITOR(ete)->belongs_to->p);
	gtk_signal_connect(GTK_OBJECT(ete->term), "child_died",
			   GTK_SIGNAL_FUNC(tab_editor_died), ete);
	gtk_signal_connect(GTK_OBJECT(ete->term), "key_press_event",
			   GTK_SIGNAL_FUNC(tab_editor_key_pressed), ete);
	gtk_container_add(GTK_CONTAINER(ete->main), ete->term);


	ete->tab_label = gtk_label_new("");
	gtk_widget_show(ete->tab_label);

	add_editor_to_notebook(EDITOR(ete), ete->main);

	return start_e_tab_editor((void *) ete);
}
#endif

void notab_editor_died(e_notab_editor_t *ene) {
	world *w = EDITOR(ene)->belongs_to;
	if(debug)
		printf("Editor \"%s\" of world \"%s\" died\n", EDITOR(ene)->name,
		       w->p->name);

	ene->pid = -1;
	ene->check_closed_id = -1;
	editor_close_external(E_EDITOR(ene));
}

int check_e_notab_editor_closed(e_notab_editor_t *ene) {
	switch(waitpid(ene->pid, NULL, WNOHANG)) {
	case -1: /* error */
		if(debug) printf("ERROR WITH WAITING FOR EDITOR PID !!\n");
		return FALSE;
	case 0:  /* child still running */
		return TRUE;
	default: /* child exited */
		notab_editor_died(ene);
		return FALSE; /* don't repeat theis timeout */
	}
}

int e_notab_editor_done(e_notab_editor_t *ene) {
	char *cmd[4];
	char *s;
	pid_t pid;

	if(!env_copy) {
		copy_enviroment();
	}
	cmd[0] = "/bin/sh";
	cmd[1] = "-c";
	cmd[2] = E_EDITOR(ene)->command;
	cmd[3] = NULL;

	pid = fork();
	if(pid == -1) {
		s = g_strconcat("Could not fork clild process:\n", g_strerror(errno),
				NULL);
		gm_dialog_popup_only(_("Error"), s, TRUE);
		g_free(s);
		return FALSE;
	} 
	if(pid == 0) { /* child process */
		execve(cmd[0], cmd, env_copy);
		s = g_strdup_printf("Could not execute \"%s\":\n%s", cmd[2],
				    g_strerror(errno));
		gm_dialog_popup_only(_("Error"), s, TRUE);
		g_free(s);
		_exit(127);
	} /* end child process */
	ene->pid = pid;
	ene->check_closed_id = gtk_timeout_add(1000,
					       (GtkFunction)
					       check_e_notab_editor_closed,
					       (void *) ene);
	return TRUE; /* sucsessfull start */
}

int e_editor_done(e_editor_t *ee) {
	struct stat stats;
	int ok;

	fclose(ee->file);
	ee->file = NULL;
#ifdef ZVT
	if(IS_E_TAB_EDITOR(ee)) {
		ok = e_tab_editor_done(E_TAB_EDITOR(ee));
	} else {
#endif
		ok = e_notab_editor_done(E_NOTAB_EDITOR(ee));
#ifdef ZVT
	}
#endif
	stat(ee->filename, &stats);
	ee->last_modified = stats.st_mtime;
	ee->check_change_id = gtk_timeout_add(1000,
					      (GtkFunction) check_e_editor_change,
					      (void *) ee);
	return ok;
}

void gm_editor_all_data_received(editor_t *e) {
	int ok;
	if(debug) printf("Editor done receiving data\n");
	if(IS_I_EDITOR(e)) {
		ok = i_editor_done(I_EDITOR(e));
	} else { /* IS_E_EDITOR */
		ok = e_editor_done(E_EDITOR(e));
	}
	if(!ok) {
		gm_dialog_popup_only(_("Error"), _("Error opening editor !"), FALSE);
		gm_editor_close(e);
	}
}

void gm_editor_got_focus(editor_t *e) {
	char *text;
	if(IS_I_EDITOR(e)) {
		gtk_widget_grab_focus(I_EDITOR(e)->text);
	}
	text = g_strconcat("Editor editing ", e->name, NULL);
	gm_statusbar_set(text);
	g_free(text);
}

void i_editor_update_tab_label(i_editor_t *ie, int pos) {
	char *s;
	char *name = settings->tab_limit_length ?
		limit_string_to_length(EDITOR(ie)->name, settings->tab_max_length) :
		EDITOR(ie)->name;
	if(settings->tab_pos_on_tab && pos > -1) {
		s = g_strdup_printf("(%d) %s", pos, name);
		gtk_label_set_text(GTK_LABEL(ie->tab_label), s);
		g_free(s);
	} else {
		gtk_label_set_text(GTK_LABEL(ie->tab_label), name);
	}
	if(settings->tab_limit_length) {
		g_free(name);
	}
}

#ifdef ZVT
void e_tab_editor_update_tab_label(e_tab_editor_t *ete, int pos) {
	char *s;
	char *name = settings->tab_limit_length ?
		limit_string_to_length(EDITOR(ete)->name, settings->tab_max_length) :
		EDITOR(ete)->name;
	if(settings->tab_pos_on_tab && pos > -1) {
		s = g_strdup_printf("(%d) %s", pos, name);
		gtk_label_set_text(GTK_LABEL(ete->tab_label), s);
		g_free(s);
	} else {
		gtk_label_set_text(GTK_LABEL(ete->tab_label), name);
	}
	if(settings->tab_limit_length) {
		g_free(name);
	}
}
#endif

void gm_editor_set_position(editor_t *e, int pos) {
	pos = pos < 10 ? (pos + 1) % 10 : -1;
	if(IS_I_EDITOR(e)) {
		i_editor_update_tab_label(I_EDITOR(e), pos);
#ifdef ZVT
	} else if(IS_E_TAB_EDITOR(e)) {
		e_tab_editor_update_tab_label(E_TAB_EDITOR(e), pos);
#endif
	}
}

/****************************************************************************/
/****************************************************************************/

#define DELIMITER  '_'
#define DELIMITERS "/\\><\"';() $"
char *parse_filename(const char *name, world *w) {
	char *s;
	char *filename;
	s = g_strconcat(w->p->name, ":", name, NULL);
	s = g_strdelimit(s, DELIMITERS, DELIMITER);
	filename = g_strconcat(gm_settings_get_editor_dir(), "/", s, ".moo", NULL);
	g_free(s);
	return filename;
}

char *parse_command(const char *filename) {
	char *exec, *command;
	if(settings->editor_use_env) {
		exec = g_getenv("EDITOR");
	} else {
		exec = settings->editor_command;
	}
	command = g_strconcat(exec, " ", filename, NULL);
	return command;
}

