/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdarg.h>

#include "config.h"
#include "properties_creator.h"
#include "window.h"

#define PC_WIN "__pc_win"
#define TARGET "__target"

void pc_close_cb(GtkWidget*, pc_win*);
void pc_apply_cb(GtkWidget*, pc_win*);

void pc_delete_cb(GtkWidget *w, void *v, pc_win *win) {
    if(debug) printf("Deleting window\n");
    pc_close_cb(win->close, win);
}

void pc_ok_cb(GtkWidget *but, pc_win *win) {
    if((win->apply_cb)((void *) win->data))
	pc_close_cb(win->close, win);
}

void pc_apply_cb(GtkWidget *but, pc_win *win) {
    (win->apply_cb)((void *) win->data);
    gtk_widget_set_sensitive(win->ok, FALSE);
    gtk_widget_set_sensitive(win->apply, FALSE);
}

void pc_close_cb(GtkWidget *but, pc_win *win) {
    (win->close_cb)((void *) win->data);
    pc_destroy(win);
}

void pc_help_cb(GtkWidget *but, pc_win *win) {
    int pos = gtk_notebook_get_current_page(GTK_NOTEBOOK(win->notebook));
    (win->help_cb)(pos, (char *) (g_slist_nth(win->section_names, pos)->data),
		   (void *) win->data);
}

void pc_changed_cb(GtkWidget *button, pc_win *win) {
    gtk_widget_set_sensitive(win->ok, TRUE);
    gtk_widget_set_sensitive(win->apply, TRUE);
}

void pc_option_cb(GtkWidget *item, GtkWidget *option) {
    int *target = gtk_object_get_data(GTK_OBJECT(option), TARGET);
    int value  = g_list_index(GTK_MENU_SHELL(GTK_OPTION_MENU(option)
					     ->menu)->children,
			      GTK_OPTION_MENU(option)->menu_item);
    if(debug) printf("Option menu value changed from %d to %d\n", *target, value);
    *target = value;
}

void pc_check_cb(GtkWidget *check, int *target) {
    int value  = GTK_TOGGLE_BUTTON(check)->active;
    if(debug) printf("Check button changed from %d to %d\n", *target, value);
    *target = value;
}

void pc_entry_cb(GtkWidget *entry, char **target) {
    char *value  = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
    if(debug) printf("Entry changed from \"%s\" to \"%s\"\n", *target, value);
    g_free(*target);
    *target = value;
}

void pc_spin_cb(GtkWidget *spin, int *target) {
    int value = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
    if(debug) printf("Spinbutton changed from %d to %d\n", *target, value);
    *target = value;
}

void pc_scale_draw_value(GtkWidget *scale) {
    int show_int =
	GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(scale), "show_int"));
    int show_hex =
	GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(scale), "show_hex"));
    int *val = gtk_object_get_data(GTK_OBJECT(scale), "variable");
    GtkWidget *label = gtk_object_get_data(GTK_OBJECT(scale), "label");
    char s[20] = "";
    if(show_int) {
	if(show_hex) {
	    sprintf(s, "%d (%02x)", *val, *val);
	} else {
	    sprintf(s, "%d", *val);
	}
    } else if(show_hex) {
	sprintf(s, "%02X", *val);
    }
    gtk_label_set_text(GTK_LABEL(label), s);
}

void pc_scale_cb(GtkAdjustment *adj, GtkWidget *scale) {
    int *var = gtk_object_get_data(GTK_OBJECT(scale), "variable");
    *var = (int)adj->value;
    pc_scale_draw_value(scale);
}
    
void pc_font_ok_cb(GtkWidget *ok_button, GtkWidget *entry) {
    GtkFontSelectionDialog *fsd = 
	GTK_FONT_SELECTION_DIALOG(gtk_widget_get_toplevel(ok_button));
    gtk_entry_set_text(GTK_ENTRY(entry),
		       gtk_font_selection_dialog_get_font_name(fsd));
    gtk_entry_set_position(GTK_ENTRY(entry), 0);
}

void pc_font_cb(GtkWidget *button, GtkWidget *entry) {
    GtkWidget *fsd;
    GtkFontSelectionDialog *ffsd;
    gchar *spacings[] = { "c", "m", NULL };

    fsd = gtk_font_selection_dialog_new(_("Select font"));
    ffsd = GTK_FONT_SELECTION_DIALOG (fsd);
    gtk_font_selection_dialog_set_filter (ffsd,
					  GTK_FONT_FILTER_BASE, GTK_FONT_ALL,
					  NULL, NULL, NULL, NULL, spacings, 
					  NULL);
    gtk_font_selection_dialog_set_font_name
	(ffsd, gtk_entry_get_text(GTK_ENTRY(entry)));

    gtk_window_set_modal(GTK_WINDOW(fsd), TRUE);
    gtk_signal_connect(GTK_OBJECT(ffsd->ok_button), "clicked",
		       GTK_SIGNAL_FUNC(pc_font_ok_cb), entry);
    gtk_signal_connect_object(GTK_OBJECT(ffsd->ok_button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(fsd));
    gtk_signal_connect_object(GTK_OBJECT(ffsd->cancel_button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(fsd));
    gtk_widget_show(fsd);
}

void pc_file_ok_cb(GtkWidget *ok_button, GtkWidget *entry) {
    GtkFileSelection *fsd =
	GTK_FILE_SELECTION(gtk_widget_get_toplevel(ok_button));
    gtk_entry_set_text(GTK_ENTRY(entry),
		       gtk_file_selection_get_filename(fsd));
    gtk_entry_set_position(GTK_ENTRY(entry), 0);
}

void pc_file_cb(GtkWidget *button, GtkWidget *entry) {
    GtkFileSelection *fsd;

    fsd = GTK_FILE_SELECTION(gtk_file_selection_new(_("Select file")));
    gtk_file_selection_set_filename(fsd,
				    gtk_entry_get_text(GTK_ENTRY(entry)));
    
    gtk_window_set_modal(GTK_WINDOW(fsd), TRUE);
    gtk_signal_connect(GTK_OBJECT(fsd->ok_button), "clicked",
		       GTK_SIGNAL_FUNC(pc_file_ok_cb), entry);
    gtk_signal_connect_object(GTK_OBJECT(fsd->ok_button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(fsd));
    gtk_signal_connect_object(GTK_OBJECT(fsd->cancel_button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(fsd));
    gtk_widget_show(GTK_WIDGET(fsd));
}

void pc_ctree_select_row(GtkWidget *ctree, GtkCTreeNode *node,
			 int col, pc_win *win) {
    int row = g_slist_index(win->section_nodes, (void *) node);

    if(debug) printf("Row %d selected\n", row);
    gtk_notebook_set_page(GTK_NOTEBOOK(win->notebook), row);
}

/************** Standard operations *******************/
pc_win *pc_new(const char *name, pc_apply_func apply_cb, pc_func close_cb,
	       pc_help_func help_cb, int show_apply, const void  *data) {
    pc_win *win = g_malloc(sizeof(pc_win));
    GtkWidget *vbox;
    GtkWidget *hseperator;
    GtkWidget *hbb;
    GtkWidget *tmp;
    GtkWidget *hbox;

    if(debug) printf("Creating properties dialog\n");
    
    /*    win->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);*/
    win->window = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(win->window),
				 GTK_WINDOW(gm_window_get_window()));
    gtk_window_set_title(GTK_WINDOW(win->window), name);
    gtk_signal_connect(GTK_OBJECT(win->window), "delete_event",
		       GTK_SIGNAL_FUNC(pc_delete_cb), win);

    gtk_window_set_position(GTK_WINDOW(win->window), GTK_WIN_POS_MOUSE);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(win->window), vbox);

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);

    tmp = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(tmp);
    gtk_box_pack_start(GTK_BOX(hbox), tmp, FALSE, FALSE, 0);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(tmp),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    win->ctree = gtk_ctree_new(1, 0);
    gtk_widget_show(win->ctree);
    gtk_container_add(GTK_CONTAINER(tmp), win->ctree);
    gtk_clist_column_titles_hide(GTK_CLIST(win->ctree));
    gtk_widget_set_usize(win->ctree, 130, -1);
    gtk_clist_set_column_width(GTK_CLIST(win->ctree), 0, 1);
    gtk_clist_set_selection_mode(GTK_CLIST(win->ctree), GTK_SELECTION_BROWSE);
    gtk_signal_connect(GTK_OBJECT(win->ctree), "tree_select_row",
    		       GTK_SIGNAL_FUNC(pc_ctree_select_row), win);

    tmp = gtk_frame_new(NULL);
    gtk_widget_show(tmp);
    gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 0);
    gtk_frame_set_shadow_type(GTK_FRAME(tmp), GTK_SHADOW_IN);

    win->notebook = gtk_notebook_new();
    gtk_widget_show(win->notebook);
    gtk_container_add(GTK_CONTAINER(tmp), win->notebook);
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(win->notebook), FALSE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK(win->notebook), FALSE);

    hseperator = gtk_hseparator_new();
    gtk_widget_show(hseperator);
    gtk_box_pack_start(GTK_BOX(vbox), hseperator, FALSE, TRUE, 0);

    hbb = gtk_hbutton_box_new();
    gtk_widget_show(hbb);
    gtk_box_pack_start(GTK_BOX(vbox), hbb, FALSE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbb), 5);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbb), 0);
    gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbb), 3, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbb), GTK_BUTTONBOX_END);

    tmp = gtk_button_new_with_label (_("OK"));
    gtk_widget_show(tmp);
    gtk_container_add(GTK_CONTAINER(hbb), tmp);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		       GTK_SIGNAL_FUNC(pc_ok_cb), win);
    win->ok = tmp;

    tmp = gtk_button_new_with_label (_("Apply"));
    if(show_apply) gtk_widget_show(tmp);
    gtk_container_add(GTK_CONTAINER(hbb), tmp);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		       GTK_SIGNAL_FUNC(pc_apply_cb), win);
    win->apply = tmp;

    tmp = gtk_button_new_with_label (_("Close"));
    gtk_widget_show(tmp);
    gtk_container_add(GTK_CONTAINER(hbb), tmp);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		       GTK_SIGNAL_FUNC(pc_close_cb), win);
    win->close = tmp;

    tmp = gtk_button_new_with_label (_("Help"));
    gtk_widget_show(tmp);
    gtk_container_add(GTK_CONTAINER(hbb), tmp);
    GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
    gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
		       GTK_SIGNAL_FUNC(pc_help_cb), win);
    win->help = tmp;

    gtk_widget_set_sensitive(win->ok,    FALSE);
    gtk_widget_set_sensitive(win->apply, FALSE);
    gtk_widget_grab_default (win->close);
    gtk_widget_grab_focus   (win->close);

    win->section_names = NULL;
    win->section_nodes = NULL;
    win->data = data;
    win->apply_cb = apply_cb;
    win->close_cb = close_cb;
    win->help_cb  = help_cb;

    gtk_object_set_data(GTK_OBJECT(win->window), PC_WIN, win);

    return win;
}

void pc_show(pc_win *win) {
    GtkCTreeNode *node = win->section_nodes ? win->section_nodes->data : NULL;
    if(node) {
	gtk_ctree_select(GTK_CTREE(win->ctree), node);
    }
    gtk_widget_show(win->window);
}

void pc_hide(pc_win *win) {
    gtk_widget_hide(win->window);
}

void pc_destroy(pc_win *win) {
    gtk_widget_hide(win->window);
    gtk_widget_destroy(win->window);
    g_slist_foreach(win->section_names, (GFunc) g_free, NULL);
    g_slist_free(win->section_names);
    g_slist_free(win->section_nodes);
    g_free(win);
}

void pc_modal(pc_win *win, int on) {
    gtk_window_set_modal(GTK_WINDOW(win->window), on);
}

/*************** Adding sections ******************/
#define get_pc_win(w) gtk_object_get_data(GTK_OBJECT(\
gtk_widget_get_toplevel(GTK_WIDGET(w))), PC_WIN)
#define tree_add(tree, parent, text)\
gtk_ctree_insert_node(GTK_CTREE(tree), parent, NULL, text, 0,\
		      NULL, NULL, NULL, NULL, FALSE, TRUE)\

pc_sec *pc_section_new(pc_win *win,
		       const char *desc) {
    GtkWidget *table;
    GtkCTreeNode *node;
    char *text[1];

    if(debug) printf("Creating section \"%s\"\n", desc);

    table = gtk_table_new(0, 3, FALSE);
    gtk_widget_show(table);
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);   
    gtk_table_set_row_spacings(GTK_TABLE(table), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table), 5); 

    gtk_notebook_append_page(GTK_NOTEBOOK(win->notebook), table, NULL);
    win->section_names = g_slist_append(win->section_names, g_strdup(desc));

    text[0] = (char *) desc;
    node = tree_add(win->ctree, NULL, text);
    win->section_nodes = g_slist_append(win->section_nodes, node);

    return GTK_TABLE(table);
}

pc_sec *pc_subsection_new(pc_sec *sec,
			  const char *desc) {
    GtkWidget *table;
    int pos;
    char *text[1];
    pc_win *win = get_pc_win(sec);
    GtkCTreeNode *parent, *node;

    if(debug) printf("Creating section \"%s\"\n", desc);

    table = gtk_table_new(0, 3, FALSE);
    gtk_widget_show(table);
    gtk_container_set_border_width(GTK_CONTAINER(table), 5);   
    gtk_table_set_row_spacings(GTK_TABLE(table), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table), 5); 

    gtk_notebook_append_page(GTK_NOTEBOOK(win->notebook), table, NULL);
    win->section_names = g_slist_append(win->section_names, g_strdup(desc));

    pos = gtk_notebook_page_num(GTK_NOTEBOOK(win->notebook), GTK_WIDGET(sec));
    parent = gtk_ctree_node_nth(GTK_CTREE(win->ctree), pos);
    
    text[0] = (char *) desc;
    node = tree_add(win->ctree, parent, text);
    win->section_nodes = g_slist_append(win->section_nodes, node);

    return GTK_TABLE(table);
}

/*************** Adding variables ******************/
pc_var *pc_radio_new(pc_sec *sec, pc_var *radio, const char *desc,
		       int *variable) {
    GtkWidget *but;
    GSList *group = radio ? gtk_radio_button_group(GTK_RADIO_BUTTON(radio)):NULL;
    int pos;

    if(debug) printf("\tCreating radio button \"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    but = gtk_radio_button_new_with_label(group, desc);
    gtk_widget_show(but);
    gtk_table_attach(sec, but, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
    gtk_label_set_justify(GTK_LABEL(GTK_BIN(but)->child), GTK_JUSTIFY_LEFT);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(but), *variable);

    gtk_signal_connect(GTK_OBJECT(but), "clicked",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(but));
    gtk_signal_connect(GTK_OBJECT(but), "clicked",
		       GTK_SIGNAL_FUNC(pc_check_cb), variable);

    return but;
}

pc_var *pc_option_new(pc_sec    *sec,
		      const char  *desc,
		      int         *variable,
                      ...) {
    GtkWidget *label;
    GtkWidget *option;
    GtkWidget *menu;
    GtkWidget *item;
    int pos;
    va_list l;
    char *s;

    if(debug) printf("\tCreating optionmenu \"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    label = gtk_label_new(desc);
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(sec, label, 0, 1, pos, pos + 1, GTK_FILL, 0, 0, 0);

    option = gtk_option_menu_new();
    gtk_widget_show(option);
    gtk_table_attach(sec, option, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    menu = gtk_menu_new();
    gtk_widget_show(menu);
    gtk_option_menu_set_menu(GTK_OPTION_MENU(option), menu);

    va_start(l, variable);
    s = va_arg(l, char *);
    while(s) {
        pos++;
        if(debug) printf("\tAdding \"%s\"\n", s);
        item = gtk_menu_item_new_with_label(s);
        gtk_widget_show(item);
        gtk_container_add(GTK_CONTAINER(menu), item);
        gtk_signal_connect(GTK_OBJECT(item), "activate",
                           GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(option));
        gtk_signal_connect(GTK_OBJECT(item), "activate",
                           GTK_SIGNAL_FUNC(pc_option_cb), option);
        s = va_arg(l, char *);
    }
    va_end(l);

    gtk_option_menu_set_history(GTK_OPTION_MENU(option), *variable);
    gtk_object_set_data(GTK_OBJECT(option), TARGET,     variable);

    return option;
}

pc_var *pc_check_new (pc_sec    *sec,
		      const char  *desc,
		      int         *variable) {
    GtkWidget *check;
    int pos;

    if(debug) printf("\tCreating check button \"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    check = gtk_check_button_new_with_label(desc);
    gtk_widget_show(check);
    gtk_table_attach(sec, check, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);
    gtk_label_set_justify(GTK_LABEL(GTK_BIN(check)->child), GTK_JUSTIFY_LEFT);

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), *variable);
    gtk_signal_connect(GTK_OBJECT(check), "clicked",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(check));
    gtk_signal_connect(GTK_OBJECT(check), "clicked",
		       GTK_SIGNAL_FUNC(pc_check_cb), variable);

    return check;
}

pc_var *pc_spin_new  (pc_sec    *sec,
		      const char  *desc,
		      int          lower,
		      int          upper,
		      int         *variable) {
    GtkWidget *label;
    GtkWidget *spin;
    GtkObject* adj;
    int pos;

    if(debug) printf("\tCreating spinbutton \"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    label = gtk_label_new(desc);
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(sec, label, 0, 1, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    adj = gtk_adjustment_new(*variable, lower, upper, 1, 10, 10);
    spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 0);
    gtk_widget_show(spin);
    gtk_table_attach(sec, spin, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    gtk_signal_connect(GTK_OBJECT(spin), "changed",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(spin));
    gtk_signal_connect(GTK_OBJECT(spin), "changed",
		       GTK_SIGNAL_FUNC(pc_spin_cb), variable);

    return spin;
}

pc_var *pc_entry_new (pc_sec    *sec,
		      const char  *desc,
		      char       **variable) {
    GtkWidget *label;
    GtkWidget *entry;
    int pos;

    if(debug) printf("\tCreating entry \"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    label = gtk_label_new(desc);
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(sec, label, 0, 1, pos, pos + 1, GTK_FILL, 0, 0, 0);

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), *variable);
    gtk_entry_set_position(GTK_ENTRY(entry), 0);
    gtk_widget_show(entry);
    gtk_table_attach(sec, entry, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(entry));
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(pc_entry_cb), variable);

    return entry;
}

pc_var *pc_font_new  (pc_sec    *sec,
		      const char  *desc,
		      char       **variable) {
    GtkWidget *label;
    GtkWidget *entry;
    GtkWidget *button;
    int pos;

    if(debug) printf("\tCreating font selector \"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    label = gtk_label_new(desc);
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(sec, label, 0, 1, pos, pos + 1, GTK_FILL, 0, 0, 0);

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), *variable);
    gtk_entry_set_position(GTK_ENTRY(entry), 0);
    gtk_widget_show(entry);
    gtk_table_attach(sec, entry, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    button = gtk_button_new_with_label(_("Browse..."));
    gtk_widget_show(button);
    gtk_table_attach(sec, button, 2, 3, pos, pos + 1, GTK_FILL, 0, 0, 0);

    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(entry));
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(pc_entry_cb), variable);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(pc_font_cb), entry);


    return entry;
}

pc_var *pc_file_new  (pc_sec    *sec,
		      const char  *desc,
		      char       **variable) {
    GtkWidget *label;
    GtkWidget *entry;
    GtkWidget *button;
    int pos;

    if(debug) printf("\tCreating file selector\"%s\"\n", desc);

    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    label = gtk_label_new(desc);
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(sec, label, 0, 1, pos, pos + 1, GTK_FILL, 0, 0, 0);

    entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(entry), *variable);
    gtk_entry_set_position(GTK_ENTRY(entry), 0);
    gtk_widget_show(entry);
    gtk_table_attach(sec, entry, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    button = gtk_button_new_with_label(_("Browse..."));
    gtk_widget_show(button);
    gtk_table_attach(sec, button, 2, 3, pos, pos + 1, GTK_FILL, 0, 0, 0);

    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(entry));
    gtk_signal_connect(GTK_OBJECT(entry), "changed",
		       GTK_SIGNAL_FUNC(pc_entry_cb), variable);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(pc_file_cb), entry);


    return entry;
}

pc_var *pc_scale_new  (pc_sec    *sec,
		      const char  *desc,
		      int show_int_val,
		      int show_hex_val,
		      int lower,
		      int upper,
		      int *variable) {
    GtkWidget *label;
    GtkWidget *scale;
    GtkWidget *value;
    int pos;

    if(debug) printf("\tCreating scale \"%s\"\n", desc);
    pos = sec->nrows;
    gtk_table_resize(sec, pos + 1, 3);

    label = gtk_label_new(desc);
    gtk_widget_show(label);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(sec, label, 0, 1, pos, pos + 1, GTK_FILL, 0, 0, 0);

    scale = gtk_hscale_new(GTK_ADJUSTMENT(gtk_adjustment_new(*variable,
							     lower, upper,
							     1, 1, 1)));
    gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
    gtk_widget_show(scale);
    gtk_table_attach(sec, scale, 1, 2, pos, pos + 1, GTK_FILL|GTK_EXPAND, 0, 0, 0);

    value = gtk_label_new("");
    gtk_widget_show(value);
    gtk_table_attach(sec, value, 2, 3, pos, pos + 1, GTK_FILL, 0, 0, 0);

    gtk_object_set_data(GTK_OBJECT(scale), "show_int",
			GINT_TO_POINTER(show_int_val));
    gtk_object_set_data(GTK_OBJECT(scale), "show_hex",
			GINT_TO_POINTER(show_hex_val));
    gtk_object_set_data(GTK_OBJECT(scale), "variable", variable);
    gtk_object_set_data(GTK_OBJECT(scale), "label", value);
    
    pc_scale_draw_value(scale);
    
    gtk_signal_connect(GTK_OBJECT(GTK_RANGE(scale)->adjustment),
		       "value_changed",
		       GTK_SIGNAL_FUNC(pc_changed_cb), get_pc_win(scale));
    gtk_signal_connect(GTK_OBJECT(GTK_RANGE(scale)->adjustment),
		       "value_changed",
		       GTK_SIGNAL_FUNC(pc_scale_cb), scale);

    return scale;
}


/********** Locking, can be only used with check variables *************/
void pc_set_sensitivity(GtkWidget *check, GtkWidget *w) {
    gtk_widget_set_sensitive(w, GTK_TOGGLE_BUTTON(check)->active);
}

void pc_set_through_sensitivity(GtkWidget *control, GtkStateType old,
				GtkWidget *w) {
    if(!GTK_TOGGLE_BUTTON(control)->active ||
       GTK_WIDGET_STATE(control) == GTK_STATE_INSENSITIVE) {
	gtk_widget_set_sensitive(w, FALSE);
    } else {
	gtk_widget_set_sensitive(w, TRUE);
    }
}

void pc_check_lock(pc_var *check, ...) {
    GtkWidget *w;
    va_list l;
    va_start(l, check);
    w = va_arg(l, GtkWidget *);
    while(w) {
	gtk_signal_connect(GTK_OBJECT(check), "clicked",
			   GTK_SIGNAL_FUNC(pc_set_sensitivity), w);
	pc_set_sensitivity(check, w);
	w = va_arg(l, GtkWidget *);
    }
}

void pc_set_sensitivity_r(GtkWidget *check, GtkWidget *w) {
    gtk_widget_set_sensitive(w, ! GTK_TOGGLE_BUTTON(check)->active);
}

void pc_check_lock_reversed(pc_var *check, ...) {
    GtkWidget *w;
    va_list l;
    va_start(l, check);
    w = va_arg(l, GtkWidget *);
    while(w) {
	gtk_signal_connect(GTK_OBJECT(check), "clicked",
			   GTK_SIGNAL_FUNC(pc_set_sensitivity_r), w);
	pc_set_sensitivity_r(check, w);
	w = va_arg(l, GtkWidget *);
    }
}

void pc_set_sensitivity2(void *pointer, GtkWidget *w) {
    GtkToggleButton *t0 = gtk_object_get_data(GTK_OBJECT(w), "check0");
    GtkToggleButton *t1 = gtk_object_get_data(GTK_OBJECT(w), "check1");

    gtk_widget_set_sensitive(w, t0->active && t1->active);
}

void pc_check_lock2(pc_var *check0, pc_var *check1, ...) {
    GtkWidget *w;
    va_list l;
    va_start(l, check1);
    w = va_arg(l, GtkWidget *);
    while(w) {
	gtk_signal_connect(GTK_OBJECT(check0), "clicked",
			   GTK_SIGNAL_FUNC(pc_set_sensitivity2), w);
	gtk_signal_connect(GTK_OBJECT(check1), "clicked",
			   GTK_SIGNAL_FUNC(pc_set_sensitivity2), w);
	gtk_object_set_data(GTK_OBJECT(w), "check0", check0);
	gtk_object_set_data(GTK_OBJECT(w), "check1", check1);
	pc_set_sensitivity2(NULL, w);
	w = va_arg(l, GtkWidget *);
    }
}

void pc_check_lock_through(pc_var *check, ...) {
    GtkWidget *w;
    va_list l;
    va_start(l, check);
    w = va_arg(l, GtkWidget *);
    while(w) {
	gtk_signal_connect(GTK_OBJECT(check), "state_changed",
			   GTK_SIGNAL_FUNC(pc_set_through_sensitivity), w);
	pc_set_through_sensitivity(check, 0, w);
	w = va_arg(l, GtkWidget *);
    }
}
